"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToTranscribe = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary The LambdaToTranscribe class.
 */
class LambdaToTranscribe extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToTranscribe class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToTranscribeProps} props - user provided props for the construct.
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // All our tests are based upon this behavior being on, so we're setting
        // context here rather than assuming the client will set it
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        // Check props
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        // Check source bucket props
        const sourceS3Props = {
            existingBucketObj: props.existingSourceBucketObj,
            bucketProps: props.sourceBucketProps,
            loggingBucketProps: props.sourceLoggingBucketProps,
            logS3AccessLogs: props.logSourceS3AccessLogs
        };
        defaults.CheckS3Props(sourceS3Props);
        // Check destination bucket props (only if not using same bucket)
        if (!props.useSameBucket) {
            const destinationS3Props = {
                existingBucketObj: props.existingDestinationBucketObj,
                bucketProps: props.destinationBucketProps,
                loggingBucketProps: props.destinationLoggingBucketProps,
                logS3AccessLogs: props.logDestinationS3AccessLogs
            };
            defaults.CheckS3Props(destinationS3Props);
        }
        // Setup VPC if required
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.TRANSCRIBE);
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.S3);
        }
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc,
        });
        // Setup source S3 Bucket
        let sourceBucket;
        if (!props.existingSourceBucketObj) {
            const buildSourceBucketResponse = defaults.buildS3Bucket(this, {
                bucketProps: props.sourceBucketProps,
                loggingBucketProps: props.sourceLoggingBucketProps,
                logS3AccessLogs: props.logSourceS3AccessLogs
            }, `${id}-source-bucket`);
            this.sourceBucket = buildSourceBucketResponse.bucket;
            this.sourceLoggingBucket = buildSourceBucketResponse.loggingBucket;
            sourceBucket = this.sourceBucket;
        }
        else {
            sourceBucket = props.existingSourceBucketObj;
        }
        this.sourceBucketInterface = sourceBucket;
        // Setup destination S3 Bucket
        let destinationBucket;
        if (props.useSameBucket) {
            destinationBucket = sourceBucket;
            this.destinationBucketInterface = sourceBucket;
        }
        else {
            if (!props.existingDestinationBucketObj) {
                const buildDestinationBucketResponse = defaults.buildS3Bucket(this, {
                    bucketProps: props.destinationBucketProps,
                    loggingBucketProps: props.destinationLoggingBucketProps,
                    logS3AccessLogs: props.logDestinationS3AccessLogs
                }, `${id}-destination-bucket`);
                this.destinationBucket = buildDestinationBucketResponse.bucket;
                this.destinationLoggingBucket = buildDestinationBucketResponse.loggingBucket;
                destinationBucket = this.destinationBucket;
            }
            else {
                destinationBucket = props.existingDestinationBucketObj;
            }
            this.destinationBucketInterface = destinationBucket;
        }
        // Configure environment variables
        const sourceBucketEnvName = props.sourceBucketEnvironmentVariableName || 'SOURCE_BUCKET_NAME';
        const destinationBucketEnvName = props.destinationBucketEnvironmentVariableName || 'DESTINATION_BUCKET_NAME';
        this.lambdaFunction.addEnvironment(sourceBucketEnvName, sourceBucket.bucketName);
        this.lambdaFunction.addEnvironment(destinationBucketEnvName, destinationBucket.bucketName);
        // Grant Lambda permissions to S3 buckets
        sourceBucket.grantRead(this.lambdaFunction.grantPrincipal);
        destinationBucket.grantWrite(this.lambdaFunction.grantPrincipal);
        // Grant Lambda permissions to Transcribe service
        this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'transcribe:StartTranscriptionJob',
                'transcribe:GetTranscriptionJob',
                'transcribe:ListTranscriptionJobs'
            ],
            resources: ['*']
        }));
    }
}
exports.LambdaToTranscribe = LambdaToTranscribe;
_a = JSII_RTTI_SYMBOL_1;
LambdaToTranscribe[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-transcribe.LambdaToTranscribe", version: "2.97.0" };
//# sourceMappingURL=data:application/json;base64,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