"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const resources = require("@aws-solutions-constructs/resources");
class OpenApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.apiDefinitionAsset && (props.apiDefinitionBucket || props.apiDefinitionKey)) {
            throw new Error('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified, but not both');
        }
        const apiDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
        const apiDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
        if (apiDefinitionBucket === undefined || apiDefinitionKey === undefined) {
            throw new Error('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
        }
        if (props.apiIntegrations === undefined || props.apiIntegrations.length < 1) {
            throw new Error('At least one ApiIntegration must be specified in the apiIntegrations property');
        }
        // store a counter to be able to uniquely name lambda functions to avoid naming collisions
        let lambdaCounter = 0;
        this.apiLambdaFunctions = props.apiIntegrations.map(apiIntegration => {
            if (apiIntegration.existingLambdaObj || apiIntegration.lambdaFunctionProps) {
                return {
                    id: apiIntegration.id,
                    lambdaFunction: defaults.buildLambdaFunction(this, {
                        existingLambdaObj: apiIntegration.existingLambdaObj,
                        lambdaFunctionProps: apiIntegration.lambdaFunctionProps
                    }, `${apiIntegration.id}ApiFunction${lambdaCounter++}`)
                };
            }
            else {
                throw new Error(`One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: ${apiIntegration.id}`);
            }
        });
        // Map each id and lambda function pair to the required format for the template writer custom resource
        const apiIntegrationUris = this.apiLambdaFunctions.map(apiLambdaFunction => {
            // the placeholder string that will be replaced in the OpenAPI Definition
            const uriPlaceholderString = apiLambdaFunction.id;
            // the endpoint URI of the backing lambda function, as defined in the API Gateway extensions for OpenAPI here:
            // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
            const uriResolvedValue = `arn:${aws_cdk_lib_1.Aws.PARTITION}:apigateway:${aws_cdk_lib_1.Aws.REGION}:lambda:path/2015-03-31/functions/${apiLambdaFunction.lambdaFunction.functionArn}/invocations`;
            return {
                id: uriPlaceholderString,
                value: uriResolvedValue
            };
        });
        // This custom resource will overwrite the string placeholders in the openapi definition with the resolved values of the lambda URIs
        const apiDefinitionWriter = resources.createTemplateWriterCustomResource(this, 'Api', {
            templateBucket: apiDefinitionBucket,
            templateKey: apiDefinitionKey,
            templateValues: apiIntegrationUris,
            timeout: props.internalTransformTimeout ?? cdk.Duration.minutes(1),
            memorySize: props.internalTransformMemorySize ?? 1024
        });
        const specRestApiResponse = defaults.CreateSpecRestApi(this, {
            ...props.apiGatewayProps,
            apiDefinition: apigateway.ApiDefinition.fromBucket(apiDefinitionWriter.s3Bucket, apiDefinitionWriter.s3Key)
        }, props.logGroupProps);
        this.apiGateway = specRestApiResponse.api;
        this.apiGatewayCloudWatchRole = specRestApiResponse.role;
        this.apiGatewayLogGroup = specRestApiResponse.logGroup;
        // Redeploy the API any time the incoming API definition changes (from asset or s3 object)
        this.apiGateway.latestDeployment?.addToLogicalId(apiDefinitionKey);
        this.apiLambdaFunctions.forEach(apiLambdaFunction => {
            // Redeploy the API any time one of the lambda functions changes
            this.apiGateway.latestDeployment?.addToLogicalId(apiLambdaFunction.lambdaFunction.functionArn);
            // Grant APIGW invocation rights for each lambda function
            apiLambdaFunction.lambdaFunction.addPermission('PermitAPIGInvocation', {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.apiGateway.arnForExecuteApi('*')
            });
        });
    }
}
exports.OpenApiGatewayToLambda = OpenApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayToLambda[_a] = { fqn: "@aws-solutions-constructs/aws-openapigateway-lambda.OpenApiGatewayToLambda", version: "2.44.0" };
//# sourceMappingURL=data:application/json;base64,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