"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const resources = require("@aws-solutions-constructs/resources");
class OpenApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        CheckOpenapiProps(props);
        const apiDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
        const apiDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
        // store a counter to be able to uniquely name lambda functions to avoid naming collisions
        let lambdaCounter = 0;
        this.apiLambdaFunctions = props.apiIntegrations.map(apiIntegration => {
            if (apiIntegration.existingLambdaObj && apiIntegration.lambdaFunctionProps) {
                throw new Error(`Error - Cannot provide both lambdaFunctionProps and existingLambdaObj in an ApiIntegrationfor the api integration with id: ${apiIntegration.id}`);
            }
            if (apiIntegration.existingLambdaObj || apiIntegration.lambdaFunctionProps) {
                return {
                    id: apiIntegration.id,
                    lambdaFunction: defaults.buildLambdaFunction(this, {
                        existingLambdaObj: apiIntegration.existingLambdaObj,
                        lambdaFunctionProps: apiIntegration.lambdaFunctionProps
                    }, `${apiIntegration.id}ApiFunction${lambdaCounter++}`)
                };
            }
            else {
                throw new Error(`One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: ${apiIntegration.id}`);
            }
        });
        // Map each id and lambda function pair to the required format for the template writer custom resource
        const apiIntegrationUris = this.apiLambdaFunctions.map(apiLambdaFunction => {
            // the placeholder string that will be replaced in the OpenAPI Definition
            const uriPlaceholderString = apiLambdaFunction.id;
            // the endpoint URI of the backing lambda function, as defined in the API Gateway extensions for OpenAPI here:
            // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
            const uriResolvedValue = `arn:${aws_cdk_lib_1.Aws.PARTITION}:apigateway:${aws_cdk_lib_1.Aws.REGION}:lambda:path/2015-03-31/functions/${apiLambdaFunction.lambdaFunction.functionArn}/invocations`;
            return {
                id: uriPlaceholderString,
                value: uriResolvedValue
            };
        });
        // This custom resource will overwrite the string placeholders in the openapi definition with the resolved values of the lambda URIs
        const apiDefinitionWriter = resources.createTemplateWriterCustomResource(this, 'Api', {
            // CheckAlbProps() has confirmed the existence of these values
            templateBucket: apiDefinitionBucket,
            templateKey: apiDefinitionKey,
            templateValues: apiIntegrationUris,
            timeout: props.internalTransformTimeout ?? cdk.Duration.minutes(1),
            memorySize: props.internalTransformMemorySize ?? 1024
        });
        const specRestApiResponse = defaults.CreateSpecRestApi(this, {
            ...props.apiGatewayProps,
            apiDefinition: apigateway.ApiDefinition.fromBucket(apiDefinitionWriter.s3Bucket, apiDefinitionWriter.s3Key)
        }, props.logGroupProps);
        this.apiGateway = specRestApiResponse.api;
        this.apiGatewayCloudWatchRole = specRestApiResponse.role;
        this.apiGatewayLogGroup = specRestApiResponse.logGroup;
        // Redeploy the API any time the incoming API definition changes (from asset or s3 object)
        this.apiGateway.latestDeployment?.addToLogicalId(apiDefinitionKey);
        this.apiLambdaFunctions.forEach(apiLambdaFunction => {
            // Redeploy the API any time one of the lambda functions changes
            this.apiGateway.latestDeployment?.addToLogicalId(apiLambdaFunction.lambdaFunction.functionArn);
            // Grant APIGW invocation rights for each lambda function
            apiLambdaFunction.lambdaFunction.addPermission('PermitAPIGInvocation', {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.apiGateway.arnForExecuteApi('*')
            });
        });
    }
}
exports.OpenApiGatewayToLambda = OpenApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayToLambda[_a] = { fqn: "@aws-solutions-constructs/aws-openapigateway-lambda.OpenApiGatewayToLambda", version: "2.68.0" };
function CheckOpenapiProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if (props.apiDefinitionAsset && (props.apiDefinitionBucket || props.apiDefinitionKey)) {
        errorMessages += 'Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified, but not both\n';
        errorFound = true;
    }
    const apiDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
    const apiDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
    if (apiDefinitionBucket === undefined || apiDefinitionKey === undefined) {
        errorMessages += 'Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified\n';
        errorFound = true;
    }
    if (props.apiIntegrations === undefined || props.apiIntegrations.length < 1) {
        errorMessages += 'At least one ApiIntegration must be specified in the apiIntegrations property\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
//# sourceMappingURL=data:application/json;base64,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