"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const logs = require("aws-cdk-lib/aws-logs");
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, defaults.generateIntegStackName(__filename));
(0, core_1.SetConsistentFeatureFlags)(stack);
stack.templateOptions.description = 'Integration Test for aws-openapigateway-lambda';
const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'ApiDefinitionAsset', {
    path: path.join(__dirname, 'openapi/apiDefinition.yaml')
});
const messagesLambda = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
        currentVersionOptions: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
        },
    }
});
// We need a different log group, because using the function name
// causes CDK to create a log group with the same name (that is not destroyed)
// so there's a collision with the default log group
const replacementLogGroup = new logs.LogGroup(stack, 'our-log-group', {
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
});
defaults.addCfnSuppressRules(replacementLogGroup, [
    {
        id: 'W84',
        reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
    }
]);
const photosLambda = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        functionName: 'PhotosLambdaAliasTestFromAsset',
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
        logGroup: replacementLogGroup,
    }
});
const messagesAlias = new lambda.Alias(stack, 'messages-alias', {
    version: messagesLambda.currentVersion,
    aliasName: 'messagesAlias',
});
new lib_1.OpenApiGatewayToLambda(stack, 'OpenApiGatewayToLambda', {
    apiDefinitionAsset,
    apiIntegrations: [
        {
            id: 'MessagesHandler',
            existingLambdaObj: messagesAlias
        },
        {
            id: 'PhotosHandler',
            existingLambdaObj: photosLambda
        }
    ]
});
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', {
    testCases: [
        stack
    ]
});
//# sourceMappingURL=data:application/json;base64,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