# Copyright (c) 2014 Baidu.com, Inc. All Rights Reserved
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
# except in compliance with the License. You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software distributed under the
# License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied. See the License for the specific language governing permissions
# and limitations under the License.

"""
This module provides a client class for esg.
"""
from __future__ import unicode_literals

import copy
import json
import uuid

from baidubce import bce_base_client
from baidubce import compat
from baidubce.auth import bce_v1_signer
from baidubce.http import bce_http_client
from baidubce.http import handler
from baidubce.http import http_methods
from baidubce.services.esg import esg_model
from baidubce.utils import required


class EsgClient(bce_base_client.BceBaseClient):
    """
    Bcc base sdk client
    """

    prefix = b'/v1'

    def __init__(self, config=None):
        bce_base_client.BceBaseClient.__init__(self, config)

    def _merge_config(self, config=None):
        if config is None:
            return self.config
        else:
            new_config = copy.copy(self.config)
            new_config.merge_non_none_values(config)
            return new_config

    def _send_request(self, http_method, path,
                      body=None, headers=None, params=None,
                      config=None, body_parser=None, prefix=None):
        config = self._merge_config(config)
        if body_parser is None:
            body_parser = handler.parse_json
        if prefix is None:
            prefix = EsgClient.prefix

        return bce_http_client.send_request(
            config, bce_v1_signer.sign, [handler.parse_error, body_parser],
            http_method, prefix + path, body, headers, params)

    @required(name=(bytes, str),  # ***Unicode***
              rules=list)
    def create_enterprise_security_group(self, name, rules=None,
                                         desc=None,
                                         tags=None,
                                         client_token=None,
                                         config=None):
        """
            Creating a newly esg with specified rules.

            :param name:
                The name of esg that will be created.
            :type name: string

            :param rules:
                The list of rules which define how the esg works.
            :type rules: list<esg_model.EnterpriseSecurityGroupRuleModel>

            :param desc:
                The optional parameter to describe the esg that will be created.
            :type desc: string

            :param tags:
                The optional list of tag to be bonded.
            :type tags: list<esg_model.TagModel>

            :param client_token:
                An ASCII string whose length is less than 64.
                The request will be idempotent if client token is provided.
                If the clientToken is not specified by the user,
                a random String generated by default algorithm will be used.
            :type client_token: string

            :param config:
                :type config: baidubce.BceClientConfiguration

            :return:
            :rtype baidubce.bce_response.BceResponse
        """
        path = b'/enterprise/security'
        params = {
            'clientToken': generate_client_token() if client_token is None else client_token
        }
        rule_list = [rule.__dict__ for rule in rules]
        body = {
            'name': name,
            'desc': desc,
            'rules': rule_list
        }
        if desc is not None:
            body['desc'] = desc
        if tags is not None:
            tag_list = [tag.__dict__ for tag in tags]
            body['tags'] = tag_list
        return self._send_request(http_methods.POST, path, json.dumps(body), params=params, config=config)

    def list_enterprise_security_groups(self, instance_id=None, marker=None, max_keys=None, config=None):
        """
        Listing EnterpriseSecurityGroup owned by the authenticated user.

            :param instance_id:
                The id of instance. The optional parameter to list the SecurityGroup.
                If it's specified,only the SecurityGroup related to the specified instance will be listed
            :type instance_id: string

            :param marker:
                The optional parameter marker specified in the original request to specify
                where in the results to begin listing.
                Together with the marker, specifies the list result which listing should begin.
                If the marker is not specified, the list result will listing from the first one.
            :type marker: string

            :param max_keys:
                The optional parameter to specifies the max number of list result to return.
                The default value is 1000.
            :type max_keys: int

            :param config:
                :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = b'/enterprise/security'
        params = {}
        if instance_id is not None:
            params['instanceId'] = instance_id
        if marker is not None:
            params['marker'] = marker
        if max_keys is not None:
            params['maxKeys'] = max_keys

        return self._send_request(http_methods.GET, path, params=params, config=config)

    @required(enterprise_security_group_id=(bytes, str))  # ***Unicode***
    def delete_enterprise_security_group(self, enterprise_security_group_id, config=None):
        """
            Deleting the specified EnterpriseSecurityGroup.

                :param enterprise_security_group_id:
                    The id of SecurityGroup that will be deleted.
                :type enterprise_security_group_id: string

                :param config:
                    :type config: baidubce.BceClientConfiguration
            :return:
            :rtype baidubce.bce_response.BceResponse
        """
        enterprise_security_group_id = compat.convert_to_bytes(enterprise_security_group_id)
        path = b'/enterprise/security/%s' % enterprise_security_group_id
        return self._send_request(http_methods.DELETE, path, config=config)

    @required(enterprise_security_group_id=(bytes, str),  # ***Unicode***
              rule=esg_model.EnterpriseSecurityGroupRuleModel)
    def authorize_enterprise_security_group_rule(self, enterprise_security_group_id, rules,
                                                 client_token=None,
                                                 config=None):
        """
        authorize a security group rule to the specified security group

            :param enterprise_security_group_id:
                The id of EnterpriseSecurityGroup that will be authorized.
            :type enterprise_security_group_id: string

            :param rules:
                The list of rules which define how the esg works.
            :type rules: list<esg_model.EnterpriseSecurityGroupRuleModel>

            :param client_token:
                An ASCII string whose length is less than 64.
                The request will be idempotent if client token is provided.
                If the clientToken is not specified by the user,
                a random String generated by default algorithm will be used.
            :type client_token: string

            :param config:
                :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        enterprise_security_group_id = compat.convert_to_bytes(enterprise_security_group_id)
        path = b'/enterprise/security/%s' % enterprise_security_group_id
        params = {'authorizeRule': ''}
        if client_token is None:
            params['clientToken'] = generate_client_token()
        else:
            params['clientToken'] = client_token
        rule_list = [rule.__dict__ for rule in rules]
        body = {
            'rules': rule_list
        }
        return self._send_request(http_methods.PUT, path, json.dumps(body), params=params, config=config)

    @required(enterprise_security_group_rule_id=(bytes, str))  # ***Unicode***
    def update_enterprise_security_group_rule(self, enterprise_security_group_rule_id,
                                              remark=None,
                                              protocol=None,
                                              portrange=None,
                                              source_ip=None,
                                              dest_ip=None,
                                              action=None,
                                              local_ip=None,
                                              priority=None,
                                              source_portrange=None,
                                              config=None):
        """
            uodate a enterprise security group rule from the specified security group
            :param enterprise_security_group_rule_id:
                The id of EnterpriseSecurityGroupRule that will be updated.
            :param: remark:
                The remark for the rule.
            :param: portrange:
                The port range to specify the port which the rule will work on.
                Available range is rang [0, 65535], the fault value is "" for all port.
            :param: source_portrange:
                The source port range to specify the port which the rule will work on.
                Available range is rang [0, 65535], the fault value is "" for all port.
            :param: protocol:
                The parameter specify which protocol will the rule work on, the fault value is "" for all protocol.
                Available protocol are tcp, udp and icmp.
            :param: source_ip:
                The source ip range with CIDR formats. The default value 0.0.0.0/0 (allow all ip address),
                other supported formats such as {ip_addr}/12 or {ip_addr}. Only supports IPV4.
                Only works for  direction = "ingress".
            :param: dest_ip:
                The destination ip range with CIDR formats. The default value 0.0.0.0/0 (allow all ip address),
                other supported formats such as {ip_addr}/12 or {ip_addr}. Only supports IPV4.
                Only works for  direction = "egress".
            :param: local_ip:
                The parameter specify the localIP (allow all ip address: all).
            :param: priority:
                The parameter specify the priority of the rule(range 1-1000).
            :param: action:
                The parameter specify the action of the rule, available value are "allow/deny".
            :param config:
                :type config: baidubce.BceClientConfiguration
            :return:
            :rtype baidubce.bce_response.BceResponse
        """
        enterprise_security_group_rule_id = compat.convert_to_bytes(enterprise_security_group_rule_id)
        path = b'/enterprise/security/rule/%s' % enterprise_security_group_rule_id
        body = {
            'remark': remark,
            'protocol': protocol,
            'portRange': portrange,
            'sourceIp': source_ip,
            'destIp': dest_ip,
            'action': action,
            'localIp': local_ip,
            'priority': priority,
            'sourcePortRange': source_portrange
        }
        return self._send_request(http_methods.PUT, path, json.dumps(body), params=None, config=config)

    @required(enterprise_security_group_rule_id=(bytes, str))  # ***Unicode***
    def delete_enterprise_security_group_rule(self, enterprise_security_group_rule_id, config=None):
        """
            delete a enterprise security group rule from the specified security group
            :param enterprise_security_group_rule_id:
                The id of EnterpriseSecurityGroupRule that will be deleted.
            :type enterprise_security_group_id: string

            :param config:
                :type config: baidubce.BceClientConfiguration
            :return:
            :rtype baidubce.bce_response.BceResponse
        """
        enterprise_security_group_rule_id = compat.convert_to_bytes(enterprise_security_group_rule_id)
        path = b'/enterprise/security/rule/%s' % enterprise_security_group_rule_id
        return self._send_request(http_methods.DELETE, path, params=None, config=config)


def generate_client_token_by_uuid():
    """
    The default method to generate the random string for client_token
    if the optional parameter client_token is not specified by the user.
    :return:
    :rtype string
    """
    return str(uuid.uuid4())


generate_client_token = generate_client_token_by_uuid
