import os


def which( name, path=None ):
    """
    Look for an executable file of the given name in the given list of directories,
    or the directories listed in the PATH variable of the current environment. Roughly the
    equivalent of the `which` program. Does not work on Windows.

    :param executable_name: the name of the program

    :type executable_name: str

    :param path: an iterable of directory paths to consider or None if the PATH environment
    variable should be used

    :returns: an iterator yielding the full path to every occurrance of an executable file of the
    given name in a directory on the given path or the PATH environment variable if no path was
    passed

    >>> next( which('ls') )
    '/bin/ls'
    >>> list( which('asdalskhvxjvkjhsdasdnbmfiewwewe') )
    []
    >>> list( which('ls', path=()) )
    []
    """
    if path is None:
        path = os.environ.get( 'PATH' )
        if path is None: return
        path = path.split( os.pathsep )
    for bin_dir in path:
        executable_path = os.path.join( bin_dir, name )
        if os.access( executable_path, os.X_OK ):
            yield executable_path
