from typing_extensions import Literal, overload
from dataclasses import dataclass
from pydantic import BaseModel
from binance.util import UserMixin, timestamp
from binance.types import validate_response

@dataclass
class Balance:
  asset: str
  borrowed: str
  free: str
  interest: str
  locked: str
  netAsset: str

class AccountResponse(BaseModel):
  """{
        "created" : true, // True means margin account created , false means margin account not created.
        "borrowEnabled": true,
        "marginLevel": "11.64405625",
        "collateralMarginLevel" : "3.2",
        "totalAssetOfBtc": "6.82728457",
        "totalLiabilityOfBtc": "0.58633215",
        "totalNetAssetOfBtc": "6.24095242",
        "TotalCollateralValueInUSDT": "5.82728457",
        "totalOpenOrderLossInUSDT": "582.728457",
        "tradeEnabled": true,
        "transferInEnabled": true,
        "transferOutEnabled": true,
        "accountType": "MARGIN_1",  // // MARGIN_1 for Cross Margin Classic, MARGIN_2 for Cross Margin Pro
        "userAssets": [
            {
                "asset": "BTC",
                "borrowed": "0.00000000",
                "free": "0.00499500",
                "interest": "0.00000000",
                "locked": "0.00000000",
                "netAsset": "0.00499500"
            },
            {
                "asset": "BNB",
                "borrowed": "201.66666672",
                "free": "2346.50000000",
                "interest": "0.00000000",
                "locked": "0.00000000",
                "netAsset": "2144.83333328"
            },
            {
                "asset": "ETH",
                "borrowed": "0.00000000",
                "free": "0.00000000",
                "interest": "0.00000000",
                "locked": "0.00000000",
                "netAsset": "0.00000000"
            },
            {
                "asset": "USDT",
                "borrowed": "0.00000000",
                "free": "0.00000000",
                "interest": "0.00000000",
                "locked": "0.00000000",
                "netAsset": "0.00000000"
            }
        ]
  }"""
  marginLevel: str
  collateralMarginLevel: str
  totalAssetOfBtc: str
  totalLiabilityOfBtc: str
  totalNetAssetOfBtc: str
  totalCollateralValueInUSDT: str
  tradeEnabled: bool
  transferInEnabled: bool
  transferOutEnabled: bool
  accountType: str
  userAssets: list[Balance]

  def free(self, asset: str):
    from decimal import Decimal
    for b in self.userAssets:
      if b.asset == asset:
        return Decimal(b.free)
    return Decimal(0)

@dataclass
class Account(UserMixin):
  recvWindow: int = 5000

  @UserMixin.with_client
  async def account(self) -> AccountResponse:
    """https://developers.binance.com/docs/margin_trading/account/Query-Cross-Margin-Account-Details"""
    params = {
      'timestamp': timestamp.now(),
      'recvWindow': self.recvWindow,
    }
    query = self.signed_query(params)
    r = await self.client.get(
      f'/sapi/v1/margin/account?{query}',
      headers={'X-MBX-APIKEY': self.api_key},
    )
    return validate_response(r.text, AccountResponse)