"""
Main script to provide the argument handling class extensions
"""

# pylint: disable=E1101
import sys

import str2bool


class argument_handling:
    """
    Extension for the Population class containing the code for source-file sampling functions

    ###################################################
    # Argument functions
    ###################################################

    # General flow of generating the arguments for the binary_c call:
    # - user provides parameter and value via set (or manually but that is risky)
    # - The parameter names of these input get compared to the parameter names in the self.defaults;
    #    with this, we know that its a valid parameter to give to binary_c.
    # - For a single system, the bse_options will be written as a arg line
    # - For a population the bse_options will get copied to a temp_bse_options dict and updated with
    #   all the parameters generated by the grid

    # I will not create the argument line by fully writing all the defaults and overriding user
    # input, that seems not necessary because by using the get_defaults() function we already
    # know for sure which parameter names are valid for the binary_c version
    # And because binary_c uses internal defaults, its not necessary to explicitly pass them.
    # I do however suggest everyone to export the binary_c defaults to a file, so that you know
    # exactly which values were the defaults.
    """

    def __init__(self, **kwargs):
        """
        Init function for the spacing_functions class
        """

        return

    def set(self, **kwargs) -> None:
        """
        Function to set the values of the population. This is the preferred method to set values
        of functions, as it provides checks on the input.

        the bse_options will get populated with all the those that have a key that is present
        in the self.defaults

        the grid_options will get updated with all the those that have a key that is present
        in the self.grid_options

        If neither of above is met; the key and the value get stored in a custom_options dict.

        Args:
            via kwargs all the arguments are either set to binary_c parameters, grid_options or custom_options (see above)
        """

        # Go over all the input
        for key, value in kwargs.items():
            # match to hostname if appropriate
            value = self._match_arg_to_host(arg={key: value})

            # Filter out keys for the bse_options
            if key in self.defaults:
                self.verbose_print(
                    "adding: {}={} to BSE_options".format(key, value),
                    self.grid_options["verbosity"],
                    2,
                )
                self.bse_options[key] = value

            # Extra check to check if the key fits one of parameter names that end with %d
            elif self._check_key_is_special_param(key):
                self.verbose_print(
                    "adding: {}={} to BSE_options by catching the %d".format(
                        key, value
                    ),
                    self.grid_options["verbosity"],
                    1,
                )
                self.bse_options[key] = value

            # Filter out keys for the grid_options
            elif key in self.grid_options.keys():
                self.verbose_print(
                    "adding: {}={} to grid_options".format(key, value),
                    self.grid_options["verbosity"],
                    1,
                )
                self.grid_options[key] = value

            # The of the keys go into a custom_options dict
            else:
                self.verbose_print(
                    "<<<< Warning: Key does not match previously known parameter: \
                    adding: {}={} to custom_options >>>>".format(
                        key, value
                    ),
                    self.grid_options["verbosity"],
                    0,  # NOTE: setting this to be 0 prevents mistakes being overlooked.
                )
                self.custom_options[key] = value

    def parse_cmdline(self) -> None:
        """
        Function to handle settings values via the command line in the form x=y, w=z, etc.

        Best to be called after all the .set(..) lines, and just before the .evolve() is called

        If you input any known parameter (i.e. contained in grid_options, defaults/bse_options
        or custom_options), this function will attempt to convert the input from string
        (because everything is string) to the type of the value that option had before.

        The values of the bse_options are initially all strings, but after user input they
        can change to ints.

        The value of any new parameter (which will go to custom_options) will be a string.
        """

        # get the cmd-line args in the form x=y
        cmdline_args = sys.argv[1:]

        if cmdline_args:
            self.verbose_print(
                "Found cmdline args. Parsing them now",
                self.grid_options["verbosity"],
                1,
            )

            # Grab the input and split them up, while accepting only non-empty entries
            # cmdline_args = args
            self.grid_options["_commandline_input"] = cmdline_args

            # expand args by hostname
            cmdline_args = self.expand_args_by_hostname(cmdline_args)

            # Make dict and fill it
            cmdline_dict = {}
            for cmdline_arg in cmdline_args:
                split = cmdline_arg.split("=")

                if len(split) == 2:
                    parameter = split[0]
                    value = split[1]

                    old_value_found = False

                    # Find an old value
                    if parameter in self.grid_options:
                        old_value = self.grid_options[parameter]
                        old_value_found = True

                    elif parameter in self.custom_options:
                        old_value = self.custom_options[parameter]
                        old_value_found = True

                    elif parameter in self.bse_options:
                        old_value = self.bse_options[parameter]
                        old_value_found = True

                    elif parameter in self.defaults:
                        # this will revert to a string type, always
                        old_value = self.defaults[parameter]
                        old_value_found = True

                    # (attempt to) convert type
                    if old_value_found:
                        if old_value is not None:
                            try:
                                self.verbose_print(
                                    "Converting type of {} from {} to {}".format(
                                        parameter, type(value), type(old_value)
                                    ),
                                    self.grid_options["verbosity"],
                                    3,
                                )
                                try:
                                    if isinstance(old_value, bool):
                                        value = str2bool.str2bool(value)
                                    else:
                                        value = type(old_value)(value)
                                    self.verbose_print(
                                        "Success!", self.grid_options["verbosity"], 2
                                    )
                                except Exception as e:
                                    print(
                                        "Failed to convert {param} value with type {type}: old_value is '{old}', new value is '{new}', {e}".format(
                                            param=parameter,
                                            old=old_value,
                                            type=type(old_value),
                                            new=split[1],
                                            e=e,
                                        )
                                    )
                                    self.exit(code=1)

                            except ValueError:

                                # might be able to eval the parameter, e.g.
                                # an expression like "2-1" can eval to "1"
                                # which would be valid
                                try:
                                    evaled = eval(value)
                                    value = type(old_value)(evaled)
                                    self.verbose_print(
                                        "Success! (evaled)",
                                        self.grid_options["verbosity"],
                                        2,
                                    )

                                except ValueError:
                                    self.verbose_print(
                                        "Tried to convert the given parameter {}/value {} to its correct type {} (from old value {}). But that wasn't possible.".format(
                                            parameter, value, type(old_value), old_value
                                        ),
                                        self.grid_options["verbosity"],
                                        0,
                                    )
                    # Add to dict
                    self.verbose_print(
                        "setting {} = {} ".format(parameter, value),
                        self.grid_options["verbosity"],
                        3,
                    )
                    cmdline_dict[parameter] = value

                else:
                    print(
                        "Error: I do not know how to process",
                        cmdline_arg,
                        " : cmdline args should be in the format x=y, yours appears not to be.",
                    )
                    self.exit(1)

            # unpack the dictionary into the setting function that handles where the values are set
            self.set(**cmdline_dict)

    def _return_argline(self, parameter_dict=None):
        """
        Function to create the string for the arg line from a parameter dict
        """

        #
        if not parameter_dict:
            parameter_dict = self.bse_options

        argline = "binary_c "

        # Combine all the key value pairs into string
        for param_name in sorted(parameter_dict):
            argline += "{} {} ".format(param_name, parameter_dict[param_name])
        argline = argline.strip()

        return argline

    def _check_key_is_special_param(self, param_key):
        """
        Function to check if the given key is part of the special parameter list
        """

        is_special_key = any(
            [
                True
                if (
                    param_key.startswith(param[:-2])
                    and len(param[:-2]) < len(param_key)
                )
                else False
                for param in self.special_params
            ]
        )

        return is_special_key

    def _check_full_system_dict_keys(self, full_system_dict):
        """
        Function to check the full system dict that will be turned in to a binary_c call
        """

        for key in full_system_dict.keys():
            if key not in self.available_keys:
                # Deal with special keys
                if not self._check_key_is_special_param(key):
                    msg = "Error: Found a parameter unknown to binary_c: {}. Abort".format(
                        key
                    )
                    raise ValueError(msg)
