import json
import datetime

def get_hash(data):
    """
    Generate SHA-256 hash value for given data.

    Args:
        data: Data to be hashed.

    Returns:
        str: The SHA-256 hash value of the data.
    """
    # Initial hash values (first 32 bits of the fractional parts of the square roots of the first 8 primes 2..19)
    h0 = 0x6a09e667
    h1 = 0xbb67ae85
    h2 = 0x3c6ef372
    h3 = 0xa54ff53a
    h4 = 0x510e527f
    h5 = 0x9b05688c
    h6 = 0x1f83d9ab
    h7 = 0x5be0cd19

    # Pre-processing: Padding the message to multiple of 512 bits
    data = json.dumps(data, sort_keys=True).encode()
    data_len_bits = len(data) * 8
    data += b'\x80'
    while len(data) % 64 != 56:
        data += b'\x00'
    data += data_len_bits.to_bytes(8, byteorder='big')

    # Process the message in successive 512-bit chunks
    for i in range(0, len(data), 64):
        chunk = data[i:i+64]

        # Message schedule (prepare 64 words from the chunk)
        w = [0] * 64
        for j in range(16):
            w[j] = int.from_bytes(chunk[j*4:j*4+4], byteorder='big')
        for j in range(16, 64):
            s0 = (rightrotate(w[j-15], 7) ^ rightrotate(w[j-15], 18) ^ (w[j-15] >> 3))
            s1 = (rightrotate(w[j-2], 17) ^ rightrotate(w[j-2], 19) ^ (w[j-2] >> 10))
            w[j] = (w[j-16] + s0 + w[j-7] + s1) & 0xFFFFFFFF

        # Initialize hash value for this chunk
        a, b, c, d, e, f, g, h = h0, h1, h2, h3, h4, h5, h6, h7

        # Main loop
        for j in range(64):
            S1 = (rightrotate(e, 6) ^ rightrotate(e, 11) ^ rightrotate(e, 25))
            ch = (e & f) ^ ((~e) & g)
            temp1 = (h + S1 + ch + k[j] + w[j]) & 0xFFFFFFFF
            S0 = (rightrotate(a, 2) ^ rightrotate(a, 13) ^ rightrotate(a, 22))
            maj = (a & b) ^ (a & c) ^ (b & c)
            temp2 = (S0 + maj) & 0xFFFFFFFF

            h = g
            g = f
            f = e
            e = (d + temp1) & 0xFFFFFFFF
            d = c
            c = b
            b = a
            a = (temp1 + temp2) & 0xFFFFFFFF

        # Add this chunk's hash to result so far
        h0 = (h0 + a) & 0xFFFFFFFF
        h1 = (h1 + b) & 0xFFFFFFFF
        h2 = (h2 + c) & 0xFFFFFFFF
        h3 = (h3 + d) & 0xFFFFFFFF
        h4 = (h4 + e) & 0xFFFFFFFF
        h5 = (h5 + f) & 0xFFFFFFFF
        h6 = (h6 + g) & 0xFFFFFFFF
        h7 = (h7 + h) & 0xFFFFFFFF

    # Produce the final hash value (big-endian)
    hash_result = (h0 << 224) | (h1 << 192) | (h2 << 160) | (h3 << 128) | (h4 << 96) | (h5 << 64) | (h6 << 32) | h7
    return hex(hash_result)[2:]

def rightrotate(x, n):
    return (x >> n) | (x << (32 - n)) & 0xFFFFFFFF

# Constants for SHA-256
k = [
    0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
    0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
    0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
    0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
    0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
    0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
    0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
    0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2,
]



data = []
previous_hash =[0]
main_hash =[]
time_node =[]


def add_block(x):
    hash_set = get_hash(x)
    time_stamp = datetime.datetime.now()

    previous_hash.append(hash_set)
    main_hash.append(hash_set)

    data.append(x)
    main_hash.append(hash_set)
    previous_hash.append(hash_set)

    time_node.append(time_stamp)


def read_block(y):
      print("block index :",y+1)
      print("time :",time_node[y+1])
      print("previous hash :",previous_hash[y])
      print("hash :",main_hash[y+1])
      print("data :",data[y])
      print("________________")

def show_allblock():
    for i in range(0,len(data)):
      
      print("block index :",i+1)
      print("previous hash :",previous_hash[i])
      print("hash :",main_hash[i+1])
      print("data :",data[i])
      print("________________")
      



    

