#!/usr/bin/env python3
"""
Example: Converting ColabFold A3M Monomer MSAs to Boltz2 Multimer CSV Format

This example demonstrates how to convert individual A3M MSA files (generated by
ColabFold for each monomer) into the paired CSV format required by Boltz2 for
multimer structure predictions.

Key Concepts:
- ColabFold generates separate A3M MSA files for each protein chain
- For multimer predictions, Boltz2 uses paired MSAs for co-evolutionary signal
- The pairing is done by matching organism/species identifiers across chains
- Each chain gets its own CSV file with columns: 'key', 'sequence'
- Sequences with the same 'key' value across different chain CSVs are paired

How it works:
1. Parse A3M files to extract sequences and organism identifiers
2. Match sequences from the same organism across chains (pairing)
3. Generate per-chain CSV files with matching 'key' values for paired sequences
4. Use these CSVs in Boltz2 prediction for multimer structure

Usage:
    python 15_a3m_to_multimer_csv.py
    
Or via CLI:
    boltz2 convert-msa chain_A.a3m chain_B.a3m -c A,B -o paired.csv
"""

import asyncio
from pathlib import Path

# Import the converter
from boltz2_client import (
    Boltz2Client,
    Polymer,
    PredictionRequest,
    A3MToCSVConverter,
    A3MParser,
    convert_a3m_to_multimer_csv,
    create_paired_msa_per_chain,
    AlignmentFileRecord,
)


def demonstrate_a3m_parsing():
    """Show how A3M files are parsed."""
    print("\n" + "=" * 60)
    print("Part 1: Understanding A3M File Format")
    print("=" * 60)
    
    # Example A3M content (simplified)
    example_a3m = """>Query|-|Query
MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
>UniRef100_A0A2N5EEG3	340	0.994	6.284E-105	0	167	168	0	167	384
MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
>UniRef100_UPI000755DEB5	340	0.994	6.284E-105	0	167	168	0	167	771
MKTVRQERLKSIVRILERSKDPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
>tr|A0A0B4J2F2|A0A0B4J2F2_HUMAN
MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLG
>tr|P12345|P12345_MOUSE
MKTVRQERLKSIVRVLERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLG
"""
    
    print("\nExample A3M content:")
    print("-" * 40)
    for line in example_a3m.split('\n')[:8]:
        print(f"  {line[:70]}{'...' if len(line) > 70 else ''}")
    print("  ...")
    
    # Parse the A3M
    msa = A3MParser.parse(example_a3m)
    
    print(f"\nParsed {len(msa.sequences)} sequences:")
    for seq in msa.sequences[:5]:
        print(f"  - ID: {seq.identifier:30s} | Organism: {seq.organism_id:20s} | Query: {seq.is_query}")
    
    return msa


def demonstrate_pairing():
    """Show how sequences are paired across chains."""
    print("\n" + "=" * 60)
    print("Part 2: Sequence Pairing Strategy")
    print("=" * 60)
    
    # Create two example MSAs representing different chains
    chain_a_a3m = """>Query|-|Query
MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
>UniRef100_A0A2N5EEG3	340	0.994
MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
>UniRef100_UPI000755DEB5	340	0.994
MKTVRQERLKSIVRILERSKDPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
>UniRef100_Q54321	320	0.95
MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLG
>UniRef100_B67890	315	0.90
MKTVRQERLKSIVRILERSKEPVSGAQLAEDLSVSRQVIVQDIAYLRSLG
"""
    
    chain_b_a3m = """>Query|-|Query
MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAEAPAD
>UniRef100_A0A2N5EEG3	340	0.994
MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAEAPAD
>UniRef100_UPI000755DEB5	340	0.994
MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAERPAD
>UniRef100_X99999	300	0.85
MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAE
>UniRef100_B67890	315	0.90
MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAEAPAD
"""
    
    print("\nChain A has sequences from organisms: A0A2N5EEG3, UPI000755DEB5, Q54321, B67890")
    print("Chain B has sequences from organisms: A0A2N5EEG3, UPI000755DEB5, X99999, B67890")
    print("\nCommon organisms (will be paired): A0A2N5EEG3, UPI000755DEB5, B67890")
    
    # Demonstrate conversion
    converter = A3MToCSVConverter()
    result = converter.convert_content(
        a3m_contents={'A': chain_a_a3m, 'B': chain_b_a3m}
    )
    
    print(f"\nGenerated {result.num_pairs} paired sequences")
    print("\nCSV Output:")
    print("-" * 40)
    for line in result.csv_content.split('\n')[:5]:
        print(f"  {line[:80]}{'...' if len(line) > 80 else ''}")
    
    return result


def demonstrate_file_conversion():
    """Show file-based conversion workflow."""
    print("\n" + "=" * 60)
    print("Part 3: File-Based Conversion")
    print("=" * 60)
    
    # Check for existing A3M files
    example_dir = Path(__file__).parent
    msa_files = list(example_dir.glob("**/*.a3m"))
    
    if len(msa_files) >= 2:
        print(f"\nFound {len(msa_files)} A3M files in examples directory:")
        for f in msa_files[:5]:
            print(f"  - {f.name}")
        
        # If we have at least 2 files, demonstrate conversion
        files_to_use = msa_files[:2]
        chain_ids = ['A', 'B']
        
        a3m_dict = {cid: path for cid, path in zip(chain_ids, files_to_use)}
        
        output_path = example_dir / "demo_paired_msa.csv"
        
        print(f"\nConverting files to: {output_path}")
        result = convert_a3m_to_multimer_csv(
            a3m_files=a3m_dict,
            output_path=output_path
        )
        
        print(f"✅ Created paired MSA with {result.num_pairs} sequences")
        
        return result
    else:
        print("\nNo A3M files found. Creating demo files...")
        
        # Create demo A3M files
        demo_dir = example_dir / "demo_msa"
        demo_dir.mkdir(exist_ok=True)
        
        chain_a_content = """>Query|demo|ChainA
MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
>UniRef100_DemoOrg1	340	0.99
MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
>UniRef100_DemoOrg2	330	0.95
MKTVRQERLKSIVRILERSKDPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG
"""
        
        chain_b_content = """>Query|demo|ChainB
MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAEAPAD
>UniRef100_DemoOrg1	340	0.99
MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAEAPAD
>UniRef100_DemoOrg2	330	0.95
MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAERPAD
"""
        
        chain_a_file = demo_dir / "chain_A.a3m"
        chain_b_file = demo_dir / "chain_B.a3m"
        
        chain_a_file.write_text(chain_a_content)
        chain_b_file.write_text(chain_b_content)
        
        print(f"  Created: {chain_a_file}")
        print(f"  Created: {chain_b_file}")
        
        # Convert
        output_path = demo_dir / "paired_msa.csv"
        result = convert_a3m_to_multimer_csv(
            a3m_files={'A': chain_a_file, 'B': chain_b_file},
            output_path=output_path
        )
        
        print(f"\n✅ Created paired MSA CSV: {output_path}")
        print(f"   Paired sequences: {result.num_pairs}")
        
        return result


async def demonstrate_boltz2_prediction(csv_result=None):
    """Show how to use the converted CSV with Boltz2."""
    print("\n" + "=" * 60)
    print("Part 4: Using Paired MSA with Boltz2")
    print("=" * 60)
    
    # Example sequences (matching what would be in the A3M files)
    chain_a_seq = "MKTVRQERLKSIVRILERSKEPVSGAQLAEELSVSRQVIVQDIAYLRSLGYNIVATPRGYVLAGG"
    chain_b_seq = "MVTPEGNVSLVDESLLVGVTDEDRAVRSAHQFYERLIGLWAPAVMEAAHELGVFAALAEAPAD"
    
    print("""
The Boltz2 NIM API requires SEPARATE CSVs for each chain:
- Each chain gets its own CSV with columns: 'key', 'sequence'
- Rows with the same 'key' value across chains are paired
- This enables the model to identify co-evolved sequences
""")
    
    print("Example per-chain CSVs:")
    print("-" * 40)
    
    csv_chain_a = f"""key,sequence
1,{chain_a_seq}
2,{chain_a_seq}"""
    
    csv_chain_b = f"""key,sequence
1,{chain_b_seq}
2,{chain_b_seq}"""
    
    print("Chain A CSV:")
    print(csv_chain_a)
    print("\nChain B CSV:")
    print(csv_chain_b)
    
    # Create per-chain MSA records
    msa_a = AlignmentFileRecord(alignment=csv_chain_a, format="csv", rank=0)
    msa_b = AlignmentFileRecord(alignment=csv_chain_b, format="csv", rank=0)
    
    # Each chain gets its own MSA
    protein_A = Polymer(
        id="A",
        molecule_type="protein",
        sequence=chain_a_seq,
        msa={"paired": {"csv": msa_a}}
    )
    
    protein_B = Polymer(
        id="B",
        molecule_type="protein",
        sequence=chain_b_seq,
        msa={"paired": {"csv": msa_b}}
    )
    
    request = PredictionRequest(
        polymers=[protein_A, protein_B],
        recycling_steps=3,
        sampling_steps=50
    )
    
    print("\nPrediction request created with paired MSA!")
    print(f"  Chain A: {len(chain_a_seq)} residues, own CSV")
    print(f"  Chain B: {len(chain_b_seq)} residues, own CSV")
    print(f"  Pairing: via matching 'key' values across CSVs")
    
    print("\nUsing the converter (recommended):")
    print("-" * 40)
    print("""
# Convert A3M files to paired CSVs
result = convert_a3m_to_multimer_csv(
    a3m_files={'A': Path('chain_A.a3m'), 'B': Path('chain_B.a3m')},
    output_path=Path('paired.csv')
)

# Create per-chain MSA data structures
msa_per_chain = create_paired_msa_per_chain(result)

# Use in polymers
protein_A = Polymer(id='A', sequence='...', msa=msa_per_chain['A'])
protein_B = Polymer(id='B', sequence='...', msa=msa_per_chain['B'])
""")


def show_cli_usage():
    """Show CLI usage for the converter."""
    print("\n" + "=" * 60)
    print("CLI Usage")
    print("=" * 60)
    
    print("""
Convert two monomer A3M files to paired CSV:

    boltz2 convert-msa chain_A.a3m chain_B.a3m -c A,B -o paired.csv

This generates:
  - paired.csv (combined, for reference)
  - paired_chain_A.csv (for Chain A)
  - paired_chain_B.csv (for Chain B)

Convert three chains with max pairs limit:

    boltz2 convert-msa chainA.a3m chainB.a3m chainC.a3m -c A,B,C -o paired.csv --max-pairs 1000

Use taxonomy-based pairing strategy:

    boltz2 convert-msa protein1.a3m protein2.a3m -c A,B -o output.csv --pairing-strategy taxonomy

Example YAML config using per-chain CSVs:

    version: 1
    sequences:
      - protein:
          id: A
          sequence: "MKTVRQERLK..."
          msa: "paired_chain_A.csv"  # Chain A's CSV
      - protein:
          id: B  
          sequence: "MVTPEGNVSL..."
          msa: "paired_chain_B.csv"  # Chain B's CSV

The pairing works because rows with the same 'key' value
in both CSVs are treated as coming from the same organism.
""")


def main():
    """Main entry point for the example."""
    print("=" * 60)
    print("A3M to Multimer CSV Converter Example")
    print("=" * 60)
    print("""
This example demonstrates converting ColabFold A3M monomer MSA files
to the paired CSV format required by Boltz2 for multimer predictions.

The key insight is that Boltz2 uses the CSV format for multimer MSAs:
- Two columns: 'key' and 'sequence'  
- Sequences with the same 'key' are paired (from same organism)
- Multiple chain sequences are separated by ':' in sequence column
""")
    
    # Part 1: A3M parsing
    demonstrate_a3m_parsing()
    
    # Part 2: Pairing strategy
    demonstrate_pairing()
    
    # Part 3: File conversion
    demonstrate_file_conversion()
    
    # Part 4: Boltz2 integration
    asyncio.run(demonstrate_boltz2_prediction())
    
    # CLI usage
    show_cli_usage()
    
    print("\n" + "=" * 60)
    print("Example Complete!")
    print("=" * 60)


if __name__ == "__main__":
    main()

