"""Helper utilities for type metadata on Pydantic fields.

Provides functionality to annotate fields with expected types without forcing
the type at runtime, useful for disambiguation when multiple resources share
the same ID but have different types.
"""

from dataclasses import dataclass
from typing import Annotated, Optional, Type, Union, get_args, get_origin

from pydantic import BaseModel


@dataclass
class ExpectedTypeMetadata:
    """Metadata to indicate the expected type for a field without forcing it."""
    expected_type: Type[BaseModel]


def get_expected_type(model: Type[BaseModel] | BaseModel, field_name: str) -> Optional[Type[BaseModel]]:
    """Retrieve the expected type metadata for a field, if present.
    
    Handles complex annotations like Optional[Annotated[Type, Metadata]].
    
    Args:
        model: The Pydantic model class or instance to inspect
        field_name: The name of the field to check for metadata
        
    Returns:
        The expected type from the metadata, or None if not present
        
    Example:
        >>> class MyModel(BaseModel):
        ...     resource: Annotated[BaseResource, ExpectedTypeMetadata(SpecificResource)]
        >>> get_expected_type(MyModel, "resource")
        <class 'SpecificResource'>
    """
    # Access model_fields from the class, not the instance
    model_class = model if isinstance(model, type) else type(model)
    field_info = model_class.model_fields.get(field_name)
    if not field_info:
        return None
    
    # First try the field_info.metadata (works for simple Annotated types)
    for metadata in field_info.metadata:
        if isinstance(metadata, ExpectedTypeMetadata):
            return metadata.expected_type
    
    # If not found, extract from the annotation (handles Optional[Annotated[...]] etc.)
    annotation = field_info.annotation
    return _extract_expected_type_from_annotation(annotation)


def _extract_expected_type_from_annotation(annotation) -> Optional[Type[BaseModel]]:
    """Recursively extract ExpectedTypeMetadata from a type annotation.
    
    Args:
        annotation: A type annotation to search for metadata
        
    Returns:
        The expected type from the metadata, or None if not present
    """
    origin = get_origin(annotation)
    
    # If it's Annotated, check its metadata
    if origin is Annotated:
        args = get_args(annotation)
        # args[0] is the actual type, args[1:] are metadata
        for metadata in args[1:]:
            if isinstance(metadata, ExpectedTypeMetadata):
                return metadata.expected_type
    
    # If it's Optional/Union, check each union member
    elif origin is Union:
        args = get_args(annotation)
        for arg in args:
            if arg is type(None):  # Skip None in Optional
                continue
            result = _extract_expected_type_from_annotation(arg)
            if result:
                return result
    
    return None
