////////////////////////////////////////////////////////////
//
// Copyright (C) 2016 - Actian Corp.
//
////////////////////////////////////////////////////////////

// This file exists in both project "zen.core" and "zen.docs".
// Make sure that both files are kept in sync.
//
// When updating this file make the corresponding changes to "btrieveCpp.h".

#ifndef _BTRIEVEC_H
#define _BTRIEVEC_H

#ifdef __cplusplus
extern "C" {
#endif

#ifndef LINKAGE
	#ifdef __GNUC__
		#define LINKAGE
	#else
		#define LINKAGE __declspec(dllimport)
	#endif
#endif

/// \brief The %Btrieve maximum record length.
#define BTRIEVE_MAXIMUM_RECORD_LENGTH (64 * 1024 - 1)

/// \brief The %Btrieve maximum key segment length.
#define BTRIEVE_MAXIMUM_KEY_LENGTH 1024

/// \brief The %Btrieve maximum bulk record count.
#define BTRIEVE_MAXIMUM_BULK_RECORD_COUNT 25804

/// \brief The %Btrieve status codes.
/// \par \SNIPPET
/// \snippet btest.c StatusCode
/// \see BtrieveStatusCodeToString
typedef enum {
    /// \brief 0 - No error.
    BTRIEVE_STATUS_CODE_NO_ERROR = 0,
    /// \brief 1 - Invalid function.
    BTRIEVE_STATUS_CODE_INVALID_FUNCTION = 1,
    /// \brief 2 - Input/Output error.
    BTRIEVE_STATUS_CODE_IO_ERROR = 2,
    /// \brief 3 - File not open.
    BTRIEVE_STATUS_CODE_FILE_NOT_OPEN = 3,
    /// \brief 4 - Key value not found.
    BTRIEVE_STATUS_CODE_KEY_VALUE_NOT_FOUND = 4,
    /// \brief 5 - Duplicate key value.
    BTRIEVE_STATUS_CODE_DUPLICATE_KEY_VALUE = 5,
    /// \brief 6 - Invalid index number.
    BTRIEVE_STATUS_CODE_INVALID_INDEX_NUMBER = 6,
    /// \brief 7 - Different index number.
    BTRIEVE_STATUS_CODE_DIFFERENT_INDEX_NUMBER = 7,
    /// \brief 8 - Position not set.
    BTRIEVE_STATUS_CODE_POSITION_NOT_SET = 8,
    /// \brief 9 - End of file.
    BTRIEVE_STATUS_CODE_END_OF_FILE = 9,
    /// \brief 10 - Modifiable key value error.
    BTRIEVE_STATUS_CODE_MODIFIABLE_KEYVALUE_ERROR = 10,
    /// \brief 11 - File name bad.
    BTRIEVE_STATUS_CODE_FILENAME_BAD = 11,
    /// \brief 12 - File not found.
    BTRIEVE_STATUS_CODE_FILE_NOT_FOUND = 12,
    /// \brief 13 - Extended file error.
    BTRIEVE_STATUS_CODE_EXTENDED_FILE_ERROR = 13,
    /// \brief 14 - Preimage open error.
    BTRIEVE_STATUS_CODE_PREIMAGE_OPEN_ERROR = 14,
    /// \brief 15 - Preimage input/output error.
    BTRIEVE_STATUS_CODE_PREIMAGE_IO_ERROR = 15,
    /// \brief 16 - Expansion error.
    BTRIEVE_STATUS_CODE_EXPANSION_ERROR = 16,
    /// \brief 17 - Close error.
    BTRIEVE_STATUS_CODE_CLOSE_ERROR = 17,
    /// \brief 18 - Disk full.
    BTRIEVE_STATUS_CODE_DISKFULL = 18,
    /// \brief 19 - Unrecoverable error.
    BTRIEVE_STATUS_CODE_UNRECOVERABLE_ERROR = 19,
    /// \brief 20 - Record manager inactive.
    BTRIEVE_STATUS_CODE_RECORD_MANAGER_INACTIVE = 20,
    /// \brief 21 - Key buffer too short.
    BTRIEVE_STATUS_CODE_KEYBUFFER_TOO_SHORT = 21,
    /// \brief 22 - Data length error.
    BTRIEVE_STATUS_CODE_DATALENGTH_ERROR = 22,
    /// \brief 23 - Position block length.
    BTRIEVE_STATUS_CODE_POSITIONBLOCK_LENGTH = 23,
    /// \brief 24 - Page size error.
    BTRIEVE_STATUS_CODE_PAGE_SIZE_ERROR = 24,
    /// \brief 25 - Create input/output error.
    BTRIEVE_STATUS_CODE_CREATE_IO_ERROR = 25,
    /// \brief 26 - Number of indexes.
    BTRIEVE_STATUS_CODE_NUMBER_OF_INDEXES = 26,
    /// \brief 27 - Invalid key position.
    BTRIEVE_STATUS_CODE_INVALID_KEY_POSITION = 27,
    /// \brief 28 - Invalid record length.
    BTRIEVE_STATUS_CODE_INVALID_RECORD_LENGTH = 28,
    /// \brief 29 - Invalid keylength.
    BTRIEVE_STATUS_CODE_INVALID_KEYLENGTH = 29,
    /// \brief 30 - Not a %Btrieve file.
    BTRIEVE_STATUS_CODE_NOT_A_BTRIEVE_FILE = 30,
    /// \brief 31 - File already extended.
    BTRIEVE_STATUS_CODE_FILE_ALREADY_EXTENDED = 31,
    /// \brief 32 - Extend input/output error.
    BTRIEVE_STATUS_CODE_EXTEND_IO_ERROR = 32,
    /// \brief 33 - %Btrieve cannot unload.
    BTRIEVE_STATUS_CODE_BTR_CANNOT_UNLOAD = 33,
    /// \brief 34 - Invalid extension name.
    BTRIEVE_STATUS_CODE_INVALID_EXTENSION_NAME = 34,
    /// \brief 35 - Directory error.
    BTRIEVE_STATUS_CODE_DIRECTORY_ERROR = 35,
    /// \brief 36 - Transaction error.
    BTRIEVE_STATUS_CODE_TRANSACTION_ERROR = 36,
    /// \brief 37 - Transaction is active.
    BTRIEVE_STATUS_CODE_TRANSACTION_IS_ACTIVE = 37,
    /// \brief 38 - Transaction file input/output error.
    BTRIEVE_STATUS_CODE_TRANSACTION_FILE_IO_ERROR = 38,
    /// \brief 39 - End transaction error.
    BTRIEVE_STATUS_CODE_END_TRANSACTION_ERROR = 39,
    /// \brief 40 - Transaction maximum files.
    BTRIEVE_STATUS_CODE_TRANSACTION_MAX_FILES = 40,
    /// \brief 41 - Operation not allowed.
    BTRIEVE_STATUS_CODE_OPERATION_NOT_ALLOWED = 41,
    /// \brief 42 - Incomplete accelerated access.
    BTRIEVE_STATUS_CODE_INCOMPLETE_ACCEL_ACCESS = 42,
    /// \brief 43 - Invalid record address.
    BTRIEVE_STATUS_CODE_INVALID_RECORD_ADDRESS = 43,
    /// \brief 44 - Null key path.
    BTRIEVE_STATUS_CODE_NULL_KEYPATH = 44,
    /// \brief 45 - Inconsistent key flags.
    BTRIEVE_STATUS_CODE_INCONSISTENT_KEY_FLAGS = 45,
    /// \brief 46 - Access to file denied.
    BTRIEVE_STATUS_CODE_ACCESS_TO_FILE_DENIED = 46,
    /// \brief 47 - Maximum open files.
    BTRIEVE_STATUS_CODE_MAXIMUM_OPEN_FILES = 47,
    /// \brief 48 - Invalid alternate collating sequence definition.
    BTRIEVE_STATUS_CODE_INVALID_ALT_SEQUENCE_DEF = 48,
    /// \brief 49 - Key type error.
    BTRIEVE_STATUS_CODE_KEY_TYPE_ERROR = 49,
    /// \brief 50 - Owner already set.
    BTRIEVE_STATUS_CODE_OWNER_ALREADY_SET = 50,
    /// \brief 51 - Invalid owner.
    BTRIEVE_STATUS_CODE_INVALID_OWNER = 51,
    /// \brief 52 - Error writing cache.
    BTRIEVE_STATUS_CODE_ERROR_WRITING_CACHE = 52,
    /// \brief 53 - Invalid interface.
    BTRIEVE_STATUS_CODE_INVALID_INTERFACE = 53,
    /// \brief 54 - Variable page error.
    BTRIEVE_STATUS_CODE_VARIABLE_PAGE_ERROR = 54,
    /// \brief 55 - Autoincrement error.
    BTRIEVE_STATUS_CODE_AUTOINCREMENT_ERROR = 55,
    /// \brief 56 - Incomplete index.
    BTRIEVE_STATUS_CODE_INCOMPLETE_INDEX = 56,
    /// \brief 57 - Expaned memory error.
    BTRIEVE_STATUS_CODE_EXPANED_MEM_ERROR = 57,
    /// \brief 58 - Compress buffer too short.
    BTRIEVE_STATUS_CODE_COMPRESS_BUFFER_TOO_SHORT = 58,
    /// \brief 59 - File already exists.
    BTRIEVE_STATUS_CODE_FILE_ALREADY_EXISTS = 59,
    /// \brief 60 - Reject count reached.
    BTRIEVE_STATUS_CODE_REJECT_COUNT_REACHED = 60,
    /// \brief 61 - Small extended get buffer error.
    BTRIEVE_STATUS_CODE_SMALL_EX_GET_BUFFER_ERROR = 61,
    /// \brief 62 - Invalid get expression.
    BTRIEVE_STATUS_CODE_INVALID_GET_EXPRESSION = 62,
    /// \brief 63 - Invalid extended insert buffer.
    BTRIEVE_STATUS_CODE_INVALID_EXT_INSERT_BUFF = 63,
    /// \brief 64 - Optimize limit reached.
    BTRIEVE_STATUS_CODE_OPTIMIZE_LIMIT_REACHED = 64,
    /// \brief 65 - Invalid extractor.
    BTRIEVE_STATUS_CODE_INVALID_EXTRACTOR = 65,
    /// \brief 66 - Referential integrity too many databases.
    BTRIEVE_STATUS_CODE_RI_TOO_MANY_DATABASES = 66,
    /// \brief 67 - Referential integrity SQL data dictionary files cannot open.
    BTRIEVE_STATUS_CODE_RIDDF_CANNOT_OPEN = 67,
    /// \brief 68 - Referential integrity cascade too deep.
    BTRIEVE_STATUS_CODE_RI_CASCADE_TOO_DEEP = 68,
    /// \brief 69 - Referential integrity cascade error.
    BTRIEVE_STATUS_CODE_RI_CASCADE_ERROR = 69,
    /// \brief 71 - Referential integrity violation.
    BTRIEVE_STATUS_CODE_RI_VIOLATION = 71,
    /// \brief 72 - Referential integrity referenced file cannot open.
    BTRIEVE_STATUS_CODE_RI_REFERENCED_FILE_CANNOT_OPEN = 72,
    /// \brief 73 - Referential integrity out of sync.
    BTRIEVE_STATUS_CODE_RI_OUT_OF_SYNC = 73,
    /// \brief 74 - End changed to abort.
    BTRIEVE_STATUS_CODE_END_CHANGED_TO_ABORT = 74,
    /// \brief 76 - Referential integrity conflict.
    BTRIEVE_STATUS_CODE_RI_CONFLICT = 76,
    /// \brief 77 - Can't loop in server.
    BTRIEVE_STATUS_CODE_CANT_LOOP_IN_SERVER = 77,
    /// \brief 78 - Dead lock.
    BTRIEVE_STATUS_CODE_DEAD_LOCK = 78,
    /// \brief 79 - Programming error.
    BTRIEVE_STATUS_CODE_PROGRAMMING_ERROR = 79,
    /// \brief 80 - Conflict.
    BTRIEVE_STATUS_CODE_CONFLICT = 80,
    /// \brief 81 - Lock error.
    BTRIEVE_STATUS_CODE_LOCKERROR = 81,
    /// \brief 82 - Lost position.
    BTRIEVE_STATUS_CODE_LOST_POSITION = 82,
    /// \brief 83 - Read outside transaction.
    BTRIEVE_STATUS_CODE_READ_OUTSIDE_TRANSACTION = 83,
    /// \brief 84 - Record in use.
    BTRIEVE_STATUS_CODE_RECORD_INUSE = 84,
    /// \brief 85 - File in use.
    BTRIEVE_STATUS_CODE_FILE_INUSE = 85,
    /// \brief 86 - File table full.
    BTRIEVE_STATUS_CODE_FILE_TABLE_FULL = 86,
    /// \brief 87 - No handles available.
    BTRIEVE_STATUS_CODE_NOHANDLES_AVAILABLE = 87,
    /// \brief 88 - Incompatible mode error.
    BTRIEVE_STATUS_CODE_INCOMPATIBLE_MODE_ERROR = 88,
    /// \brief 90 - Device table full.
    BTRIEVE_STATUS_CODE_DEVICE_TABLE_FULL = 90,
    /// \brief 91 - Server error.
    BTRIEVE_STATUS_CODE_SERVER_ERROR = 91,
    /// \brief 92 - Transaction table full.
    BTRIEVE_STATUS_CODE_TRANSACTION_TABLE_FULL = 92,
    /// \brief 93 - Incompatible lock type.
    BTRIEVE_STATUS_CODE_INCOMPATIBLE_LOCK_TYPE = 93,
    /// \brief 94 - Permission error.
    BTRIEVE_STATUS_CODE_PERMISSION_ERROR = 94,
    /// \brief 95 - Session no longer valid.
    BTRIEVE_STATUS_CODE_SESSION_NO_LONGER_VALID = 95,
    /// \brief 96 - Communications error.
    BTRIEVE_STATUS_CODE_COMMUNICATIONS_ERROR = 96,
    /// \brief 97 - Data message too small.
    BTRIEVE_STATUS_CODE_DATA_MESSAGE_TOO_SMALL = 97,
    /// \brief 98 - Internal transaction error.
    BTRIEVE_STATUS_CODE_INTERNAL_TRANSACTION_ERROR = 98,
    /// \brief 99 - Requester can't access runtime.
    BTRIEVE_STATUS_CODE_REQUESTER_CANT_ACCESS_RUNTIME = 99,
    /// \brief 100 - No cache buffers available.
    BTRIEVE_STATUS_CODE_NO_CACHE_BUFFERS_AVAIL = 100,
    /// \brief 101 - No operating system memory available.
    BTRIEVE_STATUS_CODE_NO_OS_MEMORY_AVAIL = 101,
    /// \brief 102 - No stack available.
    BTRIEVE_STATUS_CODE_NO_STACK_AVAIL = 102,
    /// \brief 103 - Chunk offset too long.
    BTRIEVE_STATUS_CODE_CHUNK_OFFSET_TOO_LONG = 103,
    /// \brief 104 - Locale error.
    BTRIEVE_STATUS_CODE_LOCALE_ERROR = 104,
    /// \brief 105 - Cannot create with variable tail allocation table.
    BTRIEVE_STATUS_CODE_CANNOT_CREATE_WITH_VAT = 105,
    /// \brief 106 - Chunk cannot get next.
    BTRIEVE_STATUS_CODE_CHUNK_CANNOT_GET_NEXT = 106,
    /// \brief 107 - Chunk incompatible file.
    BTRIEVE_STATUS_CODE_CHUNK_INCOMPATIBLE_FILE = 107,
    /// \brief 109 - Transaction too complex.
    BTRIEVE_STATUS_CODE_TRANSACTION_TOO_COMPLEX = 109,
    /// \brief 110 - Archival BLOG open error.
    BTRIEVE_STATUS_CODE_ARCH_BLOG_OPEN_ERROR = 110,
    /// \brief 111 - Archival file not logged.
    BTRIEVE_STATUS_CODE_ARCH_FILE_NOT_LOGGED = 111,
    /// \brief 112 - Archival file in use.
    BTRIEVE_STATUS_CODE_ARCH_FILE_IN_USE = 112,
    /// \brief 113 - Archival log file not found.
    BTRIEVE_STATUS_CODE_ARCH_LOGFILE_NOT_FOUND = 113,
    /// \brief 114 - Archival log file invalid.
    BTRIEVE_STATUS_CODE_ARCH_LOGFILE_INVALID = 114,
    /// \brief 115 - Archival dump file access error.
    BTRIEVE_STATUS_CODE_ARCH_DUMPFILE_ACCESS_ERROR = 115,
    /// \brief 116 - Locator file indicator.
    BTRIEVE_STATUS_CODE_LOCATOR_FILE_INDICATOR = 116,
    /// \brief 130 - No system locks available.
    BTRIEVE_STATUS_CODE_NO_SYSTEM_LOCKS_AVAILABLE = 130,
    /// \brief 132 - File full.
    BTRIEVE_STATUS_CODE_FILE_FULL = 132,
    /// \brief 133 - More than 5 concurrent users.
    BTRIEVE_STATUS_CODE_MORE_THAN_5_CONCURRENT_USERS = 133,
    /// \brief 134 - International sorting rule not found.
    BTRIEVE_STATUS_CODE_ISR_NOT_FOUND = 134,
    /// \brief 135 - International sorting rule invalid.
    BTRIEVE_STATUS_CODE_ISR_INVALID = 135,
    /// \brief 136 - Alternate collating sequence not found.
    BTRIEVE_STATUS_CODE_ACS_NOT_FOUND = 136,
    /// \brief 137 - Cannot convert RP.
    BTRIEVE_STATUS_CODE_CANNOT_CONVERT_RP = 137,
    /// \brief 138 - Invalid null indicator.
    BTRIEVE_STATUS_CODE_INVALID_NULL_INDICATOR = 138,
    /// \brief 139 - Invalid key option.
    BTRIEVE_STATUS_CODE_INVALID_KEY_OPTION = 139,
    /// \brief 140 - Incompatible close.
    BTRIEVE_STATUS_CODE_INCOMPATIBLE_CLOSE = 140,
    /// \brief 141 - Invalid user name.
    BTRIEVE_STATUS_CODE_INVALID_USERNAME = 141,
    /// \brief 142 - Invalid database.
    BTRIEVE_STATUS_CODE_INVALID_DATABASE = 142,
    /// \brief 143 - No SSQL rights.
    BTRIEVE_STATUS_CODE_NO_SSQL_RIGHTS = 143,
    /// \brief 144 - Already logged in.
    BTRIEVE_STATUS_CODE_ALREADY_LOGGED_IN = 144,
    /// \brief 145 - No database services.
    BTRIEVE_STATUS_CODE_NO_DATABASE_SERVICES = 145,
    /// \brief 146 - Duplicate system key.
    BTRIEVE_STATUS_CODE_DUPLICATE_SYSTEM_KEY = 146,
    /// \brief 147 - Log segment missing.
    BTRIEVE_STATUS_CODE_LOG_SEGMENT_MISSING = 147,
    /// \brief 148 - Roll forward error.
    BTRIEVE_STATUS_CODE_ROLL_FORWARD_ERROR = 148,
    /// \brief 149 - System key internal.
    BTRIEVE_STATUS_CODE_SYSTEM_KEY_INTERNAL = 149,
    /// \brief 150 - DBS internal error.
    BTRIEVE_STATUS_CODE_DBS_INTERNAL_ERROR = 150,
    /// \brief 151 - Chunk offset too small.
    /// \note Renamed from BTRIEVE_STATUS_CODE_NESTING_DEPTH_ERROR in Zen v15 R1.
    BTRIEVE_STATUS_CODE_CHUNK_OFFSET_TOO_SMALL = 151,
    /// \brief 160 - Invalid parameter to MicroKernel.
    BTRIEVE_STATUS_CODE_INVALID_PARAMETER_TO_MKDE = 160,
    /// \brief 161 - User count limit exceeded.
    BTRIEVE_STATUS_CODE_USER_COUNT_LIMIT_EXCEEDED = 161,
    /// \brief 162 - Client table full.
    BTRIEVE_STATUS_CODE_CLIENT_TABLE_FULL = 162,
    /// \brief 163 - Last segment error.
    BTRIEVE_STATUS_CODE_LAST_SEGMENT_ERROR = 163,
    /// \brief 170 - Login failed bad user name.
    BTRIEVE_STATUS_CODE_LOGIN_FAILED_BAD_USERNAME = 170,
    /// \brief 171 - Login failed bad password.
    BTRIEVE_STATUS_CODE_LOGIN_FAILED_BAD_PASSWORD = 171,
    /// \brief 172 - Login failed bad database.
    BTRIEVE_STATUS_CODE_LOGIN_FAILED_BAD_DATABASE = 172,
    /// \brief 173 - Login already logged in.
    BTRIEVE_STATUS_CODE_LOGIN_ALREADY_LOGGED_IN = 173,
    /// \brief 174 - Login logout failed.
    BTRIEVE_STATUS_CODE_LOGIN_LOGOUT_FAILED = 174,
    /// \brief 175 - Login wrong uniform resource indicator format.
    BTRIEVE_STATUS_CODE_LOGIN_WRONG_URI_FORMAT = 175,
    /// \brief 176 - Login file and table not found.
    BTRIEVE_STATUS_CODE_LOGIN_FILE_AND_TABLE_NOT_FOUND = 176,
    /// \brief 177 - Login table not in database.
    BTRIEVE_STATUS_CODE_LOGIN_TABLE_NOT_IN_DATABASE = 177,
    /// \brief 178 - Login directory not in database.
    BTRIEVE_STATUS_CODE_LOGIN_DIRECTORY_NOT_IN_DATABASE = 178,
    /// \brief 1001 - Lock parameter out of range.
    BTRIEVE_STATUS_CODE_LOCK_PARM_OUTOFRANGE = 1001,
    /// \brief 1002 - Memory allocation error.
    BTRIEVE_STATUS_CODE_MEM_ALLOCATION_ERR = 1002,
    /// \brief 1003 - Memory parameter too small.
    BTRIEVE_STATUS_CODE_MEM_PARM_TOO_SMALL = 1003,
    /// \brief 1004 - Page size parameter out of range.
    BTRIEVE_STATUS_CODE_PAGE_SIZE_PARM_OUTOFRANGE = 1004,
    /// \brief 1005 - Invalid preimage parameter.
    BTRIEVE_STATUS_CODE_INVALID_PREIMAGE_PARM = 1005,
    /// \brief 1006 - Preimage buffer parameter out of range.
    BTRIEVE_STATUS_CODE_PREIMAGE_BUF_PARM_OUTOFRANGE = 1006,
    /// \brief 1007 - Files parameter out of range.
    BTRIEVE_STATUS_CODE_FILES_PARM_OUTOFRANGE = 1007,
    /// \brief 1008 - Invalid initialization parameter.
    BTRIEVE_STATUS_CODE_INVALID_INIT_PARM = 1008,
    /// \brief 1009 - Invalid transaction parameter.
    BTRIEVE_STATUS_CODE_INVALID_TRANS_PARM = 1009,
    /// \brief 1010 - Error access transaction control file.
    BTRIEVE_STATUS_CODE_ERROR_ACC_TRANS_CONTROL_FILE = 1010,
    /// \brief 1011 - Compression buffer parameter out of range.
    BTRIEVE_STATUS_CODE_COMPRESSION_BUF_PARM_OUTOFRANGE = 1011,
    /// \brief 1012 - Invalid "Maximum Number Of Files In A Transaction" option.
    BTRIEVE_STATUS_CODE_INV_N_OPTION = 1012,
    /// \brief 1013 - Task list full.
    BTRIEVE_STATUS_CODE_TASK_LIST_FULL = 1013,
    /// \brief 1014 - Stop warning.
    BTRIEVE_STATUS_CODE_STOP_WARNING = 1014,
    /// \brief 1015 - Pointer parameter invalid.
    BTRIEVE_STATUS_CODE_POINTER_PARM_INVALID = 1015,
    /// \brief 1016 - Already initialized.
    BTRIEVE_STATUS_CODE_ALREADY_INITIALIZED = 1016,
    /// \brief 1017 - %Btrieve Requester can't find resource DLL.
    BTRIEVE_STATUS_CODE_REQ_CANT_FIND_RES_DLL = 1017,
    /// \brief 1018 - Already inside %Btrieve function.
    BTRIEVE_STATUS_CODE_ALREADY_INSIDE_BTR_FUNCTION = 1018,
    /// \brief 1019 - Callback abort.
    BTRIEVE_STATUS_CODE_CALLBACK_ABORT = 1019,
    /// \brief 1020 - Interface communications error.
    BTRIEVE_STATUS_CODE_INTF_COMM_ERROR = 1020,
    /// \brief 1021 - Failed to initialize.
    BTRIEVE_STATUS_CODE_FAILED_TO_INITIALIZE = 1021,
    /// \brief 1022 - MicroKernel shutting down.
    BTRIEVE_STATUS_CODE_MKDE_SHUTTING_DOWN = 1022,
    /// \brief 2000 - Internal error.
    BTRIEVE_STATUS_CODE_INTERNAL_ERROR = 2000,
    /// \brief 2001 - Insufficient memory allocation.
    BTRIEVE_STATUS_CODE_INSUFFICIENT_MEM_ALLOC = 2001,
    /// \brief 2002 - Invalid option.
    BTRIEVE_STATUS_CODE_INVALID_OPTION = 2002,
    /// \brief 2003 - No local access allowed.
    BTRIEVE_STATUS_CODE_NO_LOCAL_ACCESS_ALLOWED = 2003,
    /// \brief 2004 - SPX not installed.
    BTRIEVE_STATUS_CODE_SPX_NOT_INSTALLED = 2004,
    /// \brief 2005 - Incorrect SPX version.
    BTRIEVE_STATUS_CODE_INCORRECT_SPX_VERSION = 2005,
    /// \brief 2006 - No available SPX connection.
    BTRIEVE_STATUS_CODE_NO_AVAIL_SPX_CONNECTION = 2006,
    /// \brief 2007 - Invalid pointer parameter.
    BTRIEVE_STATUS_CODE_INVALID_PTR_PARM = 2007,
    /// \brief 2008 - Can't connect to 615.
    BTRIEVE_STATUS_CODE_CANT_CONNECT_TO_615 = 2008,
    /// \brief 2009 - Can't load MicroKernel router.
    BTRIEVE_STATUS_CODE_CANT_LOAD_MKDE_ROUTER = 2009,
    /// \brief 2010 - UT thunk not loaded.
    BTRIEVE_STATUS_CODE_UT_THUNK_NOT_LOADED = 2010,
    /// \brief 2011 - No resource DLL.
    BTRIEVE_STATUS_CODE_NO_RESOURCE_DLL = 2011,
    /// \brief 2012 - Operating system error.
    BTRIEVE_STATUS_CODE_OS_ERROR = 2012,
    /// \brief 3000 - MicroKernel router memory error.
    BTRIEVE_STATUS_CODE_MK_ROUTER_MEM_ERROR = 3000,
    /// \brief 3001 - MicroKernel no local access allowed.
    BTRIEVE_STATUS_CODE_MK_NO_LOCAL_ACCESS_ALLOWED = 3001,
    /// \brief 3002 - MicroKernel no resource DLL.
    BTRIEVE_STATUS_CODE_MK_NO_RESOURCE_DLL = 3002,
    /// \brief 3003 - MicroKernel incompatible component.
    BTRIEVE_STATUS_CODE_MK_INCOMPAT_COMPONENT = 3003,
    /// \brief 3004 - MicroKernel timeout error.
    BTRIEVE_STATUS_CODE_MK_TIMEOUT_ERROR = 3004,
    /// \brief 3005 - MicroKernel operating system error.
    BTRIEVE_STATUS_CODE_MK_OS_ERROR = 3005,
    /// \brief 3006 - MicroKernel invalid session.
    BTRIEVE_STATUS_CODE_MK_INVALID_SESSION = 3006,
    /// \brief 3007 - MicroKernel server not found.
    BTRIEVE_STATUS_CODE_MK_SERVER_NOT_FOUND = 3007,
    /// \brief 3008 - MicroKernel invalid configuration.
    BTRIEVE_STATUS_CODE_MK_INVALID_CONFIG = 3008,
    /// \brief 3009 - MicroKernel NETAPI not loaded.
    BTRIEVE_STATUS_CODE_MK_NETAPI_NOT_LOADED = 3009,
    /// \brief 3010 - MicroKernel NWAPI not loaded.
    BTRIEVE_STATUS_CODE_MK_NWAPI_NOT_LOADED = 3010,
    /// \brief 3011 - MicroKernel thunk not loaded.
    BTRIEVE_STATUS_CODE_MK_THUNK_NOT_LOADED = 3011,
    /// \brief 3012 - MicroKernel local not loaded.
    BTRIEVE_STATUS_CODE_MK_LOCAL_NOT_LOADED = 3012,
    /// \brief 3013 - MicroKernel Network Services Layer not loaded.
    BTRIEVE_STATUS_CODE_MK_PNSL_NOT_LOADED = 3013,
    /// \brief 3014 - MicroKernel can't find engine.
    BTRIEVE_STATUS_CODE_MK_CANT_FIND_ENGINE = 3014,
    /// \brief 3015 - MicroKernel initialization error.
    BTRIEVE_STATUS_CODE_MK_INIT_ERROR = 3015,
    /// \brief 3016 - MicroKernel internal error.
    BTRIEVE_STATUS_CODE_MK_INTERNAL_ERROR = 3016,
    /// \brief 3017 - MicroKernel local MicroKernel data buffer too small.
    BTRIEVE_STATUS_CODE_MK_LOCAL_MKDE_DATABUF_TOO_SMALL = 3017,
    /// \brief 3018 - MicroKernel closed error.
    BTRIEVE_STATUS_CODE_MK_CLOSED_ERROR = 3018,
    /// \brief 3019 - MicroKernel semaphore error.
    BTRIEVE_STATUS_CODE_MK_SEMAPHORE_ERROR = 3019,
    /// \brief 3020 - MicroKernel loading error.
    BTRIEVE_STATUS_CODE_MK_LOADING_ERROR = 3020,
    /// \brief 3021 - MicroKernel bad SRB format.
    BTRIEVE_STATUS_CODE_MK_BAD_SRB_FORMAT = 3021,
    /// \brief 3022 - MicroKernel data buffer length too large.
    BTRIEVE_STATUS_CODE_MK_DATABUF_LEN_TOO_LARGE = 3022,
    /// \brief 3023 - MicroKernel task table full.
    BTRIEVE_STATUS_CODE_MK_TASK_TABLE_FULL = 3023,
    /// \brief 3024 - MicroKernel invalid operation on remote.
    BTRIEVE_STATUS_CODE_MK_INVALID_OP_ON_REMOTE = 3024,
    /// \brief 3025 - MicroKernel PIDs not loaded.
    BTRIEVE_STATUS_CODE_MK_PIDS_NOT_LOADED = 3025,
    /// \brief 3026 - MicroKernel bad PIDs.
    BTRIEVE_STATUS_CODE_MK_BAD_PIDS = 3026,
    /// \brief 3027 - MicroKernel IDS connect failure.
    BTRIEVE_STATUS_CODE_MK_IDS_CONNECT_FAILURE = 3027,
    /// \brief 3028 - MicroKernel IDS login failure.
    BTRIEVE_STATUS_CODE_MK_IDS_LOGIN_FAILURE = 3028,
    /// \brief 3101 - Network layer failure.
    BTRIEVE_STATUS_CODE_NL_FAILURE = 3101,
    /// \brief 3102 - Network layer not initialized.
    BTRIEVE_STATUS_CODE_NL_NOT_INITIALIZED = 3102,
    /// \brief 3103 - Network layer name not found.
    BTRIEVE_STATUS_CODE_NL_NAME_NOT_FOUND = 3103,
    /// \brief 3104 - Network layer permission error.
    BTRIEVE_STATUS_CODE_NL_PERMISSION_ERROR = 3104,
    /// \brief 3105 - Network layer no available transport.
    BTRIEVE_STATUS_CODE_NL_NO_AVAILABLE_TRANSPORT = 3105,
    /// \brief 3106 - Network layer connection failure.
    BTRIEVE_STATUS_CODE_NL_CONNECTION_FAILURE = 3106,
    /// \brief 3107 - Network layer out of memory.
    BTRIEVE_STATUS_CODE_NL_OUT_OF_MEMORY = 3107,
    /// \brief 3108 - Network layer invalid session.
    BTRIEVE_STATUS_CODE_NL_INVALID_SESSION = 3108,
    /// \brief 3109 - Network layer more data.
    BTRIEVE_STATUS_CODE_NL_MORE_DATA = 3109,
    /// \brief 3110 - Network layer not connected.
    BTRIEVE_STATUS_CODE_NL_NOT_CONNECTED = 3110,
    /// \brief 3111 - Network layer send failure.
    BTRIEVE_STATUS_CODE_NL_SEND_FAILURE = 3111,
    /// \brief 3112 - Network layer receive failure.
    BTRIEVE_STATUS_CODE_NL_RECEIVE_FAILURE = 3112,
    /// \brief 3113 - Network layer invalid server type.
    BTRIEVE_STATUS_CODE_NL_INVALID_SERVER_TYPE = 3113,
    /// \brief 3114 - Network layer server routing table full.
    BTRIEVE_STATUS_CODE_NL_SRT_FULL = 3114,
    /// \brief 3115 - Network layer transport failure.
    BTRIEVE_STATUS_CODE_NL_TRANSPORT_FAILURE = 3115,
    /// \brief 3116 - Network layer receive data overflow.
    BTRIEVE_STATUS_CODE_NL_RCV_DATA_OVERFLOW = 3116,
    /// \brief 3117 - Network layer CST full.
    BTRIEVE_STATUS_CODE_NL_CST_FULL = 3117,
    /// \brief 3118 - Network layer invalid address family.
    BTRIEVE_STATUS_CODE_NL_INVALID_ADDRESS_FAMILY = 3118,
    /// \brief 3119 - Network layer no authentication context available.
    BTRIEVE_STATUS_CODE_NL_NO_AUTH_CONTEXT_AVAILABLE = 3119,
    /// \brief 3120 - Network layer invalid authentication type.
    BTRIEVE_STATUS_CODE_NL_INVALID_AUTH_TYPE = 3120,
    /// \brief 3121 - Network layer invalid authentication object.
    BTRIEVE_STATUS_CODE_NL_INVALID_AUTH_OBJECT = 3121,
    /// \brief 3122 - Network layer authentication length too small.
    BTRIEVE_STATUS_CODE_NL_AUTH_LEN_TOO_SMALL = 3122,
    /// \brief 3123 - Network layer invalid session level parameter.
    BTRIEVE_STATUS_CODE_NL_INVALID_SESSION_LEVEL_PARM = 3123,
    /// \brief 3124 - Network layer task table full.
    BTRIEVE_STATUS_CODE_NL_TASK_TABLE_FULL = 3124,
    /// \brief 3125 - Network layer NDS name resolution error.
    BTRIEVE_STATUS_CODE_NL_NDS_NAME_RESOLUTION_ERROR = 3125,
    /// \brief 3126 - Network layer file name resolution error.
    BTRIEVE_STATUS_CODE_NL_FILE_NAME_RESOLUTION_ERROR = 3126,
    /// \brief 3127 - Network layer IDS send failure.
    BTRIEVE_STATUS_CODE_NL_IDS_SEND_FAILURE = 3127,
    /// \brief 3128 - Network layer IDS receive failure.
    BTRIEVE_STATUS_CODE_NL_IDS_RCV_FAILURE = 3128,
    /// \brief -7 - Unknown.
    BTRIEVE_STATUS_CODE_UNKNOWN = -7,
} btrieve_status_code_t;

/// \brief The %Btrieve variable length records modes.
/// \see BtrieveVariableLengthRecordsModeToString
typedef enum {
    /// \brief No.
    BTRIEVE_VARIABLE_LENGTH_RECORDS_MODE_NO = 0,
    /// \brief Yes.
    BTRIEVE_VARIABLE_LENGTH_RECORDS_MODE_YES = 1,
    /// \brief Yes with variable allocation tails.
    BTRIEVE_VARIABLE_LENGTH_RECORDS_MODE_YES_VARIABLE_ALLOCATION_TAILS = 2,
    /// \brief Unknown.
    BTRIEVE_VARIABLE_LENGTH_RECORDS_MODE_UNKNOWN = -7
} btrieve_variable_length_records_mode_t;

/// \brief The %Btrieve alternate collation sequence modes.
/// \see BtrieveACSModeToString
typedef enum {
    /// \brief None.
    BTRIEVE_ACS_MODE_NONE = 0,
    /// \brief Named.
    BTRIEVE_ACS_MODE_NAMED = 1,
    /// \brief Numbered.
    BTRIEVE_ACS_MODE_NUMBERED = 2,
    /// \brief Case insensitive.
    BTRIEVE_ACS_MODE_CASE_INSENSITIVE = 3,
    /// \brief Default.
    BTRIEVE_ACS_MODE_DEFAULT = 4,
    /// \brief Unknown.
    BTRIEVE_ACS_MODE_UNKNOWN = -7
} btrieve_acs_mode_t;

/// \brief The %Btrieve record compression modes.
/// \see BtrieveRecordCompressionModeToString
typedef enum {
    /// \brief None.
    BTRIEVE_RECORD_COMPRESSION_MODE_NONE = 0,
    /// \brief Blank truncation.
    BTRIEVE_RECORD_COMPRESSION_MODE_BLANK_TRUNCATION = 1,
    /// \brief Run length encoding.
    BTRIEVE_RECORD_COMPRESSION_MODE_RUN_LENGTH_ENCODING = 2,
    /// \brief Unknown.
    BTRIEVE_RECORD_COMPRESSION_MODE_UNKNOWN = -7
} btrieve_record_compression_mode_t;

/// \brief The %Btrieve duplicate modes.
/// \see BtrieveDuplicateModeToString
typedef enum {
    /// \brief Duplicates not allowed.
    BTRIEVE_DUPLICATE_MODE_NOT_ALLOWED = 0,
    /// \brief Duplicates allowed, nonrepeating.
    BTRIEVE_DUPLICATE_MODE_ALLOWED_NONREPEATING = 1,
    /// \brief Duplicates allowed, repeating.
    BTRIEVE_DUPLICATE_MODE_ALLOWED_REPEATING = 2,
    /// \brief Unknown.
    BTRIEVE_DUPLICATE_MODE_UNKNOWN = -7
} btrieve_duplicate_mode_t;

/// \brief The %Btrieve comparisons.
/// \par \SNIPPET
/// \snippet btest.c Comparison
typedef enum {
    /// \brief None.
    BTRIEVE_COMPARISON_NONE = 0,
    /// \brief Equal.
    BTRIEVE_COMPARISON_EQUAL = 1,
    /// \brief Greater than.
    BTRIEVE_COMPARISON_GREATER_THAN = 2,
    /// \brief Less than.
    BTRIEVE_COMPARISON_LESS_THAN = 3,
    /// \brief Not equal.
    BTRIEVE_COMPARISON_NOT_EQUAL = 4,
    /// \brief Greater than or equal.
    BTRIEVE_COMPARISON_GREATER_THAN_OR_EQUAL = 5,
    /// \brief Less than or equal.
    BTRIEVE_COMPARISON_LESS_THAN_OR_EQUAL = 6,
    /// \brief Like.
    BTRIEVE_COMPARISON_LIKE = 7,
    /// \brief Not like.
    BTRIEVE_COMPARISON_NOT_LIKE = 8,
    /// \brief JSON filter query match.
    BTRIEVE_COMPARISON_JSON_QUERY_MATCH = 9,
    /// \brief Unknown.
    BTRIEVE_COMPARISON_UNKNOWN = -7
} btrieve_comparison_t;

/// \brief The %Btrieve connectors.
typedef enum {
    /// \brief Last.
    BTRIEVE_CONNECTOR_LAST = 0,
    /// \brief And.
    BTRIEVE_CONNECTOR_AND = 1,
    /// \brief Or.
    BTRIEVE_CONNECTOR_OR = 2,
    /// \brief Unknown.
    BTRIEVE_CONNECTOR_UNKNOWN = -7
} btrieve_connector_t;

/// \brief The %Btrieve disk drives.
typedef enum {
    /// \brief Disk drive A.
    BTRIEVE_DISK_DRIVE_A = 0,
    /// \brief Disk drive B.
    BTRIEVE_DISK_DRIVE_B = 1,
    /// \brief Disk drive C.
    BTRIEVE_DISK_DRIVE_C = 2,
    /// \brief Disk drive D.
    BTRIEVE_DISK_DRIVE_D = 3,
    /// \brief Disk drive E.
    BTRIEVE_DISK_DRIVE_E = 4,
    /// \brief Disk drive F.
    BTRIEVE_DISK_DRIVE_F = 5,
    /// \brief Disk drive G.
    BTRIEVE_DISK_DRIVE_G = 6,
    /// \brief Disk drive H.
    BTRIEVE_DISK_DRIVE_H = 7,
    /// \brief Disk drive I.
    BTRIEVE_DISK_DRIVE_I = 8,
    /// \brief Disk drive J.
    BTRIEVE_DISK_DRIVE_J = 9,
    /// \brief Disk drive K.
    BTRIEVE_DISK_DRIVE_K = 10,
    /// \brief Disk drive L.
    BTRIEVE_DISK_DRIVE_L = 11,
    /// \brief Disk drive M.
    BTRIEVE_DISK_DRIVE_M = 12,
    /// \brief Disk drive N.
    BTRIEVE_DISK_DRIVE_N = 13,
    /// \brief Disk drive O.
    BTRIEVE_DISK_DRIVE_O = 14,
    /// \brief Disk drive P.
    BTRIEVE_DISK_DRIVE_P = 15,
    /// \brief Disk drive Q.
    BTRIEVE_DISK_DRIVE_Q = 16,
    /// \brief Disk drive R.
    BTRIEVE_DISK_DRIVE_R = 17,
    /// \brief Disk drive S.
    BTRIEVE_DISK_DRIVE_S = 18,
    /// \brief Disk drive T.
    BTRIEVE_DISK_DRIVE_T = 19,
    /// \brief Disk drive U.
    BTRIEVE_DISK_DRIVE_U = 20,
    /// \brief Disk drive V.
    BTRIEVE_DISK_DRIVE_V = 21,
    /// \brief Disk drive W.
    BTRIEVE_DISK_DRIVE_W = 22,
    /// \brief Disk drive X.
    BTRIEVE_DISK_DRIVE_X = 23,
    /// \brief Disk drive Y.
    BTRIEVE_DISK_DRIVE_Y = 24,
    /// \brief Disk drive Z.
    BTRIEVE_DISK_DRIVE_Z = 25,
    /// \brief Default.
    BTRIEVE_DISK_DRIVE_DEFAULT = 26,
    /// \brief Unknown.
    BTRIEVE_DISK_DRIVE_UNKNOWN = -7
} btrieve_disk_drive_t;

/// \brief The %Btrieve version types.
/// \see BtrieveVersionTypeToString
typedef enum {
    /// \brief None.
    BTRIEVE_VERSION_TYPE_NONE = 0,
    /// \brief %Btrieve engine.
    BTRIEVE_VERSION_TYPE_BTRIEVE_ENGINE = 1,
    /// \brief Workgroup engine.
    BTRIEVE_VERSION_TYPE_WORKGROUP_ENGINE = 2,
    /// \brief DOS workstation.
    BTRIEVE_VERSION_TYPE_DOS_WORKSTATION = 3,
    /// \brief Client.
    BTRIEVE_VERSION_TYPE_CLIENT_REQUESTOR = 4,
    /// \brief Windows server engine.
    BTRIEVE_VERSION_TYPE_WINDOWS_SERVER_ENGINE = 5,
    /// \brief UNIX.
    BTRIEVE_VERSION_TYPE_UNIX = 6,
    /// \brief Client engine.
    BTRIEVE_VERSION_TYPE_CLIENT_ENGINE = 7,
    /// \brief Unknown.
    BTRIEVE_VERSION_TYPE_UNKNOWN = -7
} btrieve_version_type_t;

/// \brief The %Btrieve create modes.
/// \par \SNIPPET
/// \snippet btest.c CreateMode
typedef enum { 
    /// \brief Overwrite.
    BTRIEVE_CREATE_MODE_OVERWRITE = 0,
    /// \brief No overwrite.
    BTRIEVE_CREATE_MODE_NO_OVERWRITE = 1,
    /// \brief Unknown.
    BTRIEVE_CREATE_MODE_UNKNOWN = -7
} btrieve_create_mode_t;

/// \brief The %Btrieve location modes.
/// \par \SNIPPET
/// \snippet btest.c Location
typedef enum {
    /// \brief No preference.
    BTRIEVE_LOCATION_MODE_NO_PREFERENCE = 0,
    /// \brief Force local.
    BTRIEVE_LOCATION_MODE_FORCE_LOCAL = 1,
    /// \brief Force remote.
    BTRIEVE_LOCATION_MODE_FORCE_REMOTE = 2,
    /// \brief Unknown.
    BTRIEVE_LOCATION_MODE_UNKNOWN = -7
} btrieve_location_mode_t;

/// \brief The %Btrieve lock modes.
/// \par \SNIPPET
/// \snippet btest.c Lock
/// \see BtrieveLockModeToString
typedef enum {
    /// \brief None.
    BTRIEVE_LOCK_MODE_NONE = 0,
    /// \brief Single with wait.
    BTRIEVE_LOCK_MODE_SINGLE_WAIT = 1,
    /// \brief Single with no wait.
    BTRIEVE_LOCK_MODE_SINGLE_NO_WAIT = 2,
    /// \brief Multiple with wait.
    BTRIEVE_LOCK_MODE_MULTIPLE_WAIT = 3,
    /// \brief Multiple with no wait.
    BTRIEVE_LOCK_MODE_MULTIPLE_NO_WAIT = 4,
    /// \brief Unknown.
    BTRIEVE_LOCK_MODE_UNKNOWN = -7
} btrieve_lock_mode_t;

/// \brief The %Btrieve open modes.
/// \par \SNIPPET
/// \snippet btest.c OpenMode
typedef enum { 
    /// \brief Normal.
    BTRIEVE_OPEN_MODE_NORMAL = 0,
    /// \brief Accelerated.
    BTRIEVE_OPEN_MODE_ACCELERATED = 1,
    /// \brief Read only.
    BTRIEVE_OPEN_MODE_READ_ONLY = 2,
    /// \brief Exclusive.
    BTRIEVE_OPEN_MODE_EXCLUSIVE = 3,
    /// \brief Writable.
    BTRIEVE_OPEN_MODE_WRITABLE = 4,
    /// \brief Unknown.
    BTRIEVE_OPEN_MODE_UNKNOWN = -7
} btrieve_open_mode_t;

/// \brief The %Btrieve file owner modes.
/// \see BtrieveOwnerModeToString
typedef enum {
    /// \brief None.
    BTRIEVE_OWNER_MODE_NONE = 0,
    /// \brief No encryption with no read allowed.
    BTRIEVE_OWNER_MODE_NO_ENCRYPTION_NO_READ_ALLOWED = 1,
    /// \brief No encryption with read allowed.
    BTRIEVE_OWNER_MODE_NO_ENCRYPTION_READ_ALLOWED = 2,
    /// \brief Encryption with no read allowed.
    BTRIEVE_OWNER_MODE_ENCRYPTION_NO_READ_ALLOWED = 3,
    /// \brief Encryption with read allowed.
    BTRIEVE_OWNER_MODE_ENCRYPTION_READ_ALLOWED = 4,
    /// \brief Unknown.
    BTRIEVE_OWNER_MODE_UNKNOWN = -7
} btrieve_owner_mode_t;

/// \brief The %Btrieve file encryption types.
/// \see BtrieveOwnerModeToString
typedef enum {
    /// \brief None.
    BTRIEVE_ENCRYPT_TYPE_NONE = 0,
    /// \brief Traditional encryption.
    BTRIEVE_ENCRYPT_TYPE_TRADITIONAL = 1,
    /// \brief 128-bit encryption.
    BTRIEVE_ENCRYPT_TYPE_128_BIT = 2,
    /// \brief AES-192 encryption.
    BTRIEVE_ENCRYPT_TYPE_AES_192 = 3,
    /// \brief AES-256 encryption.
    BTRIEVE_ENCRYPT_TYPE_AES_256 = 4,
    /// \brief Unknown.
    BTRIEVE_ENCRYPT_TYPE_UNKNOWN = -7
} btrieve_encrypt_type_t;

/// \brief The %Btrieve transaction modes.
typedef enum { 
    /// \brief Exclusive.
    BTRIEVE_TRANSACTION_MODE_EXCLUSIVE = 0,
    /// \brief Concurrent with write wait.
    BTRIEVE_TRANSACTION_MODE_CONCURRENT_WRITE_WAIT = 1,
    /// \brief Concurrent with no write wait.
    BTRIEVE_TRANSACTION_MODE_CONCURRENT_NO_WRITE_WAIT = 2,
    /// \brief Unknown.
    BTRIEVE_TRANSACTION_MODE_UNKNOWN = -7
} btrieve_transaction_mode_t;

/// \brief The %Btrieve unlock modes.
typedef enum {
    /// \brief Single.
    BTRIEVE_UNLOCK_MODE_SINGLE = 0,
    /// \brief Multiple.
    BTRIEVE_UNLOCK_MODE_MULTIPLE = 1,
    /// \brief Unknown.
    BTRIEVE_UNLOCK_MODE_UNKNOWN = -7
} btrieve_unlock_mode_t;

/// \brief The %Btrieve indexes.
/// \par \SNIPPET
/// \snippet btest.c Index
/// \see BtrieveIndexToString
typedef enum {
    /// \brief Index 1
    BTRIEVE_INDEX_1 = 0,
    /// \brief Index 2
    BTRIEVE_INDEX_2 = 1,
    /// \brief Index 3
    BTRIEVE_INDEX_3 = 2,
    /// \brief Index 4
    BTRIEVE_INDEX_4 = 3,
    /// \brief Index 5
    BTRIEVE_INDEX_5 = 4,
    /// \brief Index 6
    BTRIEVE_INDEX_6 = 5,
    /// \brief Index 7
    BTRIEVE_INDEX_7 = 6,
    /// \brief Index 8
    BTRIEVE_INDEX_8 = 7,
    /// \brief Index 9
    BTRIEVE_INDEX_9 = 8,
    /// \brief Index 10
    BTRIEVE_INDEX_10 = 9,
    /// \brief Index 11
    BTRIEVE_INDEX_11 = 10,
    /// \brief Index 12
    BTRIEVE_INDEX_12 = 11,
    /// \brief Index 13
    BTRIEVE_INDEX_13 = 12,
    /// \brief Index 14
    BTRIEVE_INDEX_14 = 13,
    /// \brief Index 15
    BTRIEVE_INDEX_15 = 14,
    /// \brief Index 16
    BTRIEVE_INDEX_16 = 15,
    /// \brief Index 17
    BTRIEVE_INDEX_17 = 16,
    /// \brief Index 18
    BTRIEVE_INDEX_18 = 17,
    /// \brief Index 19
    BTRIEVE_INDEX_19 = 18,
    /// \brief Index 20
    BTRIEVE_INDEX_20 = 19,
    /// \brief Index 21
    BTRIEVE_INDEX_21 = 20,
    /// \brief Index 22
    BTRIEVE_INDEX_22 = 21,
    /// \brief Index 23
    BTRIEVE_INDEX_23 = 22,
    /// \brief Index 24
    BTRIEVE_INDEX_24 = 23,
    /// \brief Index 25
    BTRIEVE_INDEX_25 = 24,
    /// \brief Index 26
    BTRIEVE_INDEX_26 = 25,
    /// \brief Index 27
    BTRIEVE_INDEX_27 = 26,
    /// \brief Index 28
    BTRIEVE_INDEX_28 = 27,
    /// \brief Index 29
    BTRIEVE_INDEX_29 = 28,
    /// \brief Index 30
    BTRIEVE_INDEX_30 = 29,
    /// \brief Index 31
    BTRIEVE_INDEX_31 = 30,
    /// \brief Index 32
    BTRIEVE_INDEX_32 = 31,
    /// \brief Index 33
    BTRIEVE_INDEX_33 = 32,
    /// \brief Index 34
    BTRIEVE_INDEX_34 = 33,
    /// \brief Index 35
    BTRIEVE_INDEX_35 = 34,
    /// \brief Index 36
    BTRIEVE_INDEX_36 = 35,
    /// \brief Index 37
    BTRIEVE_INDEX_37 = 36,
    /// \brief Index 38
    BTRIEVE_INDEX_38 = 37,
    /// \brief Index 39
    BTRIEVE_INDEX_39 = 38,
    /// \brief Index 40
    BTRIEVE_INDEX_40 = 39,
    /// \brief Index 41
    BTRIEVE_INDEX_41 = 40,
    /// \brief Index 42
    BTRIEVE_INDEX_42 = 41,
    /// \brief Index 43
    BTRIEVE_INDEX_43 = 42,
    /// \brief Index 44
    BTRIEVE_INDEX_44 = 43,
    /// \brief Index 45
    BTRIEVE_INDEX_45 = 44,
    /// \brief Index 46
    BTRIEVE_INDEX_46 = 45,
    /// \brief Index 47
    BTRIEVE_INDEX_47 = 46,
    /// \brief Index 48
    BTRIEVE_INDEX_48 = 47,
    /// \brief Index 49
    BTRIEVE_INDEX_49 = 48,
    /// \brief Index 50
    BTRIEVE_INDEX_50 = 49,
    /// \brief Index 51
    BTRIEVE_INDEX_51 = 50,
    /// \brief Index 52
    BTRIEVE_INDEX_52 = 51,
    /// \brief Index 53
    BTRIEVE_INDEX_53 = 52,
    /// \brief Index 54
    BTRIEVE_INDEX_54 = 53,
    /// \brief Index 55
    BTRIEVE_INDEX_55 = 54,
    /// \brief Index 56
    BTRIEVE_INDEX_56 = 55,
    /// \brief Index 57
    BTRIEVE_INDEX_57 = 56,
    /// \brief Index 58
    BTRIEVE_INDEX_58 = 57,
    /// \brief Index 59
    BTRIEVE_INDEX_59 = 58,
    /// \brief Index 60
    BTRIEVE_INDEX_60 = 59,
    /// \brief Index 61
    BTRIEVE_INDEX_61 = 60,
    /// \brief Index 62
    BTRIEVE_INDEX_62 = 61,
    /// \brief Index 63
    BTRIEVE_INDEX_63 = 62,
    /// \brief Index 64
    BTRIEVE_INDEX_64 = 63,
    /// \brief Index 65
    BTRIEVE_INDEX_65 = 64,
    /// \brief Index 66
    BTRIEVE_INDEX_66 = 65,
    /// \brief Index 67
    BTRIEVE_INDEX_67 = 66,
    /// \brief Index 68
    BTRIEVE_INDEX_68 = 67,
    /// \brief Index 69
    BTRIEVE_INDEX_69 = 68,
    /// \brief Index 70
    BTRIEVE_INDEX_70 = 69,
    /// \brief Index 71
    BTRIEVE_INDEX_71 = 70,
    /// \brief Index 72
    BTRIEVE_INDEX_72 = 71,
    /// \brief Index 73
    BTRIEVE_INDEX_73 = 72,
    /// \brief Index 74
    BTRIEVE_INDEX_74 = 73,
    /// \brief Index 75
    BTRIEVE_INDEX_75 = 74,
    /// \brief Index 76
    BTRIEVE_INDEX_76 = 75,
    /// \brief Index 77
    BTRIEVE_INDEX_77 = 76,
    /// \brief Index 78
    BTRIEVE_INDEX_78 = 77,
    /// \brief Index 79
    BTRIEVE_INDEX_79 = 78,
    /// \brief Index 80
    BTRIEVE_INDEX_80 = 79,
    /// \brief Index 81
    BTRIEVE_INDEX_81 = 80,
    /// \brief Index 82
    BTRIEVE_INDEX_82 = 81,
    /// \brief Index 83
    BTRIEVE_INDEX_83 = 82,
    /// \brief Index 84
    BTRIEVE_INDEX_84 = 83,
    /// \brief Index 85
    BTRIEVE_INDEX_85 = 84,
    /// \brief Index 86
    BTRIEVE_INDEX_86 = 85,
    /// \brief Index 87
    BTRIEVE_INDEX_87 = 86,
    /// \brief Index 88
    BTRIEVE_INDEX_88 = 87,
    /// \brief Index 89
    BTRIEVE_INDEX_89 = 88,
    /// \brief Index 90
    BTRIEVE_INDEX_90 = 89,
    /// \brief Index 91
    BTRIEVE_INDEX_91 = 90,
    /// \brief Index 92
    BTRIEVE_INDEX_92 = 91,
    /// \brief Index 93
    BTRIEVE_INDEX_93 = 92,
    /// \brief Index 94
    BTRIEVE_INDEX_94 = 93,
    /// \brief Index 95
    BTRIEVE_INDEX_95 = 94,
    /// \brief Index 96
    BTRIEVE_INDEX_96 = 95,
    /// \brief Index 97
    BTRIEVE_INDEX_97 = 96,
    /// \brief Index 98
    BTRIEVE_INDEX_98 = 97,
    /// \brief Index 99
    BTRIEVE_INDEX_99 = 98,
    /// \brief Index 100
    BTRIEVE_INDEX_100 = 99,
    /// \brief Index 101
    BTRIEVE_INDEX_101 = 100,
    /// \brief Index 102
    BTRIEVE_INDEX_102 = 101,
    /// \brief Index 103
    BTRIEVE_INDEX_103 = 102,
    /// \brief Index 104
    BTRIEVE_INDEX_104 = 103,
    /// \brief Index 105
    BTRIEVE_INDEX_105 = 104,
    /// \brief Index 106
    BTRIEVE_INDEX_106 = 105,
    /// \brief Index 107
    BTRIEVE_INDEX_107 = 106,
    /// \brief Index 108
    BTRIEVE_INDEX_108 = 107,
    /// \brief Index 109
    BTRIEVE_INDEX_109 = 108,
    /// \brief Index 110
    BTRIEVE_INDEX_110 = 109,
    /// \brief Index 111
    BTRIEVE_INDEX_111 = 110,
    /// \brief Index 112
    BTRIEVE_INDEX_112 = 111,
    /// \brief Index 113
    BTRIEVE_INDEX_113 = 112,
    /// \brief Index 114
    BTRIEVE_INDEX_114 = 113,
    /// \brief Index 115
    BTRIEVE_INDEX_115 = 114,
    /// \brief Index 116
    BTRIEVE_INDEX_116 = 115,
    /// \brief Index 117
    BTRIEVE_INDEX_117 = 116,
    /// \brief Index 118
    BTRIEVE_INDEX_118 = 117,
    /// \brief Index 119
    BTRIEVE_INDEX_119 = 118,
    /// \brief System Version 2.
    BTRIEVE_INDEX_SYSTEM_VERSION_2 = 124,
    /// \brief System.
    BTRIEVE_INDEX_SYSTEM = 125, 
    /// \brief None.
    BTRIEVE_INDEX_NONE = -1, 
    /// \brief Unknown.
    BTRIEVE_INDEX_UNKNOWN = -7
} btrieve_index_t;

/// \brief The %Btrieve data types.
/// \par \SNIPPET
/// \snippet btest.c DataType
/// \see BtrieveDataTypeToString
typedef enum {
    /// \brief Character.
    BTRIEVE_DATA_TYPE_CHAR = 0,
    /// \brief Null terminated string.
    BTRIEVE_DATA_TYPE_ZSTRING = 1,
    /// \brief Integer.
    BTRIEVE_DATA_TYPE_INTEGER = 2,
    /// \brief Unsigned binary.
    BTRIEVE_DATA_TYPE_UNSIGNED_BINARY = 3,
    /// \brief Float.
    BTRIEVE_DATA_TYPE_FLOAT = 4,
    /// \brief Automatic increment.
    BTRIEVE_DATA_TYPE_AUTOINCREMENT = 5,
    /// \brief Date.
    BTRIEVE_DATA_TYPE_DATE = 6,
    /// \brief Numeric sign trailing separate.
    BTRIEVE_DATA_TYPE_NUMERICSTS = 7,
    /// \brief Time.
    BTRIEVE_DATA_TYPE_TIME = 8,
    /// \brief Numeric signed ASCII.
    BTRIEVE_DATA_TYPE_NUMERICSA = 9,
    /// \brief Decimal.
    BTRIEVE_DATA_TYPE_DECIMAL = 10,
    /// \brief Currency.
    BTRIEVE_DATA_TYPE_CURRENCY = 11,
    /// \brief Money.
    BTRIEVE_DATA_TYPE_MONEY = 12,
    /// \brief Timestamp.
    BTRIEVE_DATA_TYPE_TIMESTAMP = 13,
    /// \brief Logical.
    BTRIEVE_DATA_TYPE_LOGICAL = 14,
    /// \brief Wide string.
    BTRIEVE_DATA_TYPE_WSTRING = 15,
    /// \brief Numeric.
    BTRIEVE_DATA_TYPE_NUMERIC = 16,
    /// \brief Wide null terminated string.
    BTRIEVE_DATA_TYPE_WZSTRING = 17,
    /// \brief BASIC float.
    BTRIEVE_DATA_TYPE_BFLOAT = 18,
    /// \brief Globally unique identifier.
    BTRIEVE_DATA_TYPE_GUID = 19,
    /// \brief Length first string.
    BTRIEVE_DATA_TYPE_LSTRING = 20,
    /// \brief Null indicator segment.
    BTRIEVE_DATA_TYPE_NULL_INDICATOR_SEGMENT = 21,
    /// \brief Legacy string.
    BTRIEVE_DATA_TYPE_LEGACY_STRING = 22,
    /// \brief Legacy binary.
    BTRIEVE_DATA_TYPE_LEGACY_BINARY = 23,
    /// \brief Automatic timestamp.
    BTRIEVE_DATA_TYPE_AUTOTIMESTAMP = 24,
    /// \brief Character large object.
    BTRIEVE_DATA_TYPE_CLOB = 25,
    /// \brief Nanosecond resolution timestamp.
    BTRIEVE_DATA_TYPE_TIMESTAMP2 = 26,
    /// \brief Bit.
    BTRIEVE_DATA_TYPE_BIT = 27,
    /// \brief Numeric sign leading COBOL.
    BTRIEVE_DATA_TYPE_NUMERICSLB = 28,
    /// \brief Numeric sign leading separate.
    BTRIEVE_DATA_TYPE_NUMERICSLS = 29,
    /// \brief Numeric sign trailing COBOL.
    BTRIEVE_DATA_TYPE_NUMERICSTB = 30,
    /// \brief Unknown.
    BTRIEVE_DATA_TYPE_UNKNOWN = -7 
} btrieve_data_type_t;

/// \brief The %Btrieve file versions.
/// \see BtrieveFileVersionToString
typedef enum {
    /// \brief File version 6.0.
    BTRIEVE_FILE_VERSION_6_0 = 0,
    /// \brief File version 6.1.
    BTRIEVE_FILE_VERSION_6_1 = 1,
    /// \brief File version 7.0.
    BTRIEVE_FILE_VERSION_7_0 = 2,
    /// \brief File version 8.0.
    BTRIEVE_FILE_VERSION_8_0 = 3,
    /// \brief File version 9.0.
    BTRIEVE_FILE_VERSION_9_0 = 4,
    /// \brief File version 9.5.
    BTRIEVE_FILE_VERSION_9_5 = 5,
    /// \brief File version 13.0.
    BTRIEVE_FILE_VERSION_13_0 = 7,
    /// \brief File version 16.0.
    BTRIEVE_FILE_VERSION_16_0 = 8,
    /// \brief Default.
    BTRIEVE_FILE_VERSION_DEFAULT = 6,
    /// \brief Unknown.
    BTRIEVE_FILE_VERSION_UNKNOWN = -7
} btrieve_file_version_t;

/// \brief The %Btrieve free space thresholds.
/// \see BtrieveFreeSpaceThresholdToString
typedef enum {
    /// \brief Free space threshold 10%.
    BTRIEVE_FREE_SPACE_THRESHOLD_10_PERCENT = 0,
    /// \brief Free space threshold 20%.
    BTRIEVE_FREE_SPACE_THRESHOLD_20_PERCENT = 1,
    /// \brief Free space threshold 30%.
    BTRIEVE_FREE_SPACE_THRESHOLD_30_PERCENT = 2,
    /// \brief Default.
    BTRIEVE_FREE_SPACE_THRESHOLD_DEFAULT = 3,
    /// \brief Unknown.
    BTRIEVE_FREE_SPACE_THRESHOLD_UNKNOWN = -7
} btrieve_free_space_threshold_t;

/// \brief The %Btrieve null key modes.
/// \see BtrieveNullKeyModeToString
typedef enum {
    /// \brief All segments.
    BTRIEVE_NULL_KEY_MODE_ALL_SEGMENTS = 0,
    /// \brief Any segments.
    BTRIEVE_NULL_KEY_MODE_ANY_SEGMENTS = 1,
    /// \brief None.
    BTRIEVE_NULL_KEY_MODE_NONE = 2,
    /// \brief Unknown.
    BTRIEVE_NULL_KEY_MODE_UNKNOWN = -7
} btrieve_null_key_mode_t;

/// \brief The %Btrieve page lock type.
/// \see BtrievePageLockTypeToString
typedef enum {
    /// \brief None.
    BTRIEVE_PAGE_LOCK_TYPE_NONE = 0,
    /// \brief Data page.
    BTRIEVE_PAGE_LOCK_TYPE_DATA_PAGE = 1,
    /// \brief Index page.
    BTRIEVE_PAGE_LOCK_TYPE_INDEX_PAGE = 2,
    /// \brief Variable page.
    BTRIEVE_PAGE_LOCK_TYPE_VARIABLE_PAGE = 3,
    /// \brief Unknown.
    BTRIEVE_PAGE_LOCK_TYPE_UNKNOWN = -7
} btrieve_page_lock_type_t;

/// \brief The %Btrieve page sizes.
/// \see BtrievePageSizeToString
typedef enum {
    /// \brief Page size 512.
    BTRIEVE_PAGE_SIZE_512 = 0,
    /// \brief Page size 1024.
    BTRIEVE_PAGE_SIZE_1024 = 1,
    /// \brief Page size 1536.
    BTRIEVE_PAGE_SIZE_1536 = 2,
    /// \brief Page size 2048.
    BTRIEVE_PAGE_SIZE_2048 = 3,
    /// \brief Page size 3072.
    BTRIEVE_PAGE_SIZE_3072 = 4,
    /// \brief Page size 3584.
    BTRIEVE_PAGE_SIZE_3584 = 5,
    /// \brief Page size 4096.
    BTRIEVE_PAGE_SIZE_4096 = 6,
    /// \brief Page size 8192.
    BTRIEVE_PAGE_SIZE_8192 = 7,
    /// \brief Page size 16384.
    BTRIEVE_PAGE_SIZE_16384 = 8,
    /// \brief Default.
    BTRIEVE_PAGE_SIZE_DEFAULT = 9,
    /// \brief Unknown.
    BTRIEVE_PAGE_SIZE_UNKNOWN = -7
} btrieve_page_size_t;

/// \brief The %Btrieve system data modes.
/// \see BtrieveSystemDataModeToString
typedef enum {
    /// \brief Yes.
    BTRIEVE_SYSTEM_DATA_MODE_YES = 0,
    /// \brief No.
    BTRIEVE_SYSTEM_DATA_MODE_NO = 1,
    /// \brief Default.
    BTRIEVE_SYSTEM_DATA_MODE_DEFAULT = 2,
    /// \brief Version 2
    BTRIEVE_SYSTEM_DATA_MODE_VERSION_2 = 3,
    /// \brief Unknown.
    BTRIEVE_SYSTEM_DATA_MODE_UNKNOWN = -7
} btrieve_system_data_mode_t;

/// \brief The %Btrieve system fields.
/// \see BtrieveSystemFieldToString
typedef enum {
   /// \brief Create time.
   BTRIEVE_SYSTEM_FIELD_CREATE_TIME = 0,
   /// \brief Update time.
   BTRIEVE_SYSTEM_FIELD_UPDATE_TIME = 1,
   /// \brief Unknown.
   BTRIEVE_SYSTEM_FIELD_UNKNOWN = -7
} btrieve_system_field_t;

/// \brief A %Btrieve bulk create payload.
/// \see BtrieveBulkCreatePayloadAllocate
typedef struct btrieve_bulk_create_payload *btrieve_bulk_create_payload_t;

/// \brief A %Btrieve bulk create result.
/// \see BtrieveBulkCreateResultAllocate
typedef struct btrieve_bulk_create_result *btrieve_bulk_create_result_t;

/// \brief A %Btrieve bulk retrieve attributes.
/// \see BtrieveBulkRetrieveAttributesAllocate
typedef struct btrieve_bulk_retrieve_attributes *btrieve_bulk_retrieve_attributes_t;

/// \brief A %Btrieve bulk delete attributes.
/// \see BtrieveBulkDeleteAttributesAllocate
typedef struct btrieve_bulk_delete_attributes *btrieve_bulk_delete_attributes_t;

/// \brief A %Btrieve bulk retrieve result.
/// \see BtrieveBulkRetrieveResultAllocate
typedef struct btrieve_bulk_retrieve_result *btrieve_bulk_retrieve_result_t;

/// \brief A %Btrieve bulk delete result.
/// \see BtrieveBulkDeleteResultAllocate
typedef struct btrieve_bulk_delete_result *btrieve_bulk_delete_result_t;

/// \brief A %Btrieve client.
/// \par \SNIPPET
/// \snippet btest.c client_t
/// \snippet btest.c ClientAllocate
/// \see BtrieveClientAllocate
typedef struct btrieve_client *btrieve_client_t;

/// \brief A %Btrieve file.
/// \par \SNIPPET
/// \snippet btest.c file_t
/// \see BtrieveClientFileOpen
typedef struct btrieve_file *btrieve_file_t;

/// \brief A %Btrieve file attributes.
/// \par \SNIPPET
/// \snippet btest.c file_attributes_t
/// \snippet btest.c FileAttributesAllocate
/// \see BtrieveFileAttributesAllocate
typedef struct btrieve_file_attributes *btrieve_file_attributes_t;

/// \brief A %Btrieve file information.
/// \see BtrieveFileInformationAllocate
typedef struct btrieve_file_information *btrieve_file_information_t;

/// \brief A %Btrieve filter.
/// \see BtrieveFilterAllocate
typedef struct btrieve_filter *btrieve_filter_t;

/// \brief A %Btrieve index attributes.
/// \par \SNIPPET
/// \snippet btest.c index_attributes_t
/// \snippet btest.c IndexAttributesAllocate
/// \see BtrieveIndexAttributesAllocate
typedef struct btrieve_index_attributes *btrieve_index_attributes_t;

/// \brief A %Btrieve key segment.
/// \par \SNIPPET
/// \snippet btest.c key_segment_t
/// \snippet btest.c KeySegmentAllocate
/// \see BtrieveKeySegmentAllocate
typedef struct btrieve_key_segment *btrieve_key_segment_t;

/// \brief A %Btrieve version.
/// \see BtrieveVersionAllocate
typedef struct btrieve_version *btrieve_version_t;

/// \brief Get ACS mode from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "!= BTRIEVE_ACS_MODE_UNKNOWN" The ACS mode.
/// \retval "= BTRIEVE_ACS_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE btrieve_acs_mode_t BtrieveKeySegmentGetACSMode(btrieve_key_segment_t keySegment);

/// \brief Get data type from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "!= BTRIEVE_DATA_TYPE_UNKNOWN" The data type.
/// \retval "= BTRIEVE_DATA_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE btrieve_data_type_t BtrieveKeySegmentGetDataType(btrieve_key_segment_t keySegment);

/// \brief Get duplicate mode from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "!= BTRIEVE_DUPLICATE_MODE_UNKNOWN" The duplicate mode.
/// \retval "= BTRIEVE_DUPLICATE_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE btrieve_duplicate_mode_t BtrieveKeySegmentGetDuplicateMode(btrieve_key_segment_t keySegment);

/// \brief Get file version from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_FILE_VERSION_UNKNOWN" The file version.
/// \retval "= BTRIEVE_FILE_VERSION_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_file_version_t BtrieveFileInformationGetFileVersion(btrieve_file_information_t fileInformation);

/// \brief Get free space threshold from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_FREE_SPACE_THRESHOLD_UNKNOWN" The free space threshold.
/// \retval "= BTRIEVE_FREE_SPACE_THRESHOLD_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_free_space_threshold_t BtrieveFileInformationGetFreeSpaceThreshold(btrieve_file_information_t fileInformation);

/// \brief Get duplicate record conflict index from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_INDEX_UNKNOWN" The duplicate record conflict index.
/// \retval "= BTRIEVE_INDEX_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_index_t BtrieveFileInformationGetDuplicateRecordConflictIndex(btrieve_file_information_t fileInformation);

/// \brief Get lock owner index from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_INDEX_UNKNOWN" The lock owner index.
/// \retval "= BTRIEVE_INDEX_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_index_t BtrieveFileInformationGetLockOwnerIndex(btrieve_file_information_t fileInformation);

/// \brief Get log index from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_INDEX_UNKNOWN" The log index.
/// \retval "= BTRIEVE_INDEX_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_index_t BtrieveFileInformationGetLogIndex(btrieve_file_information_t fileInformation);

/// \brief Get index from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "!= BTRIEVE_INDEX_UNKNOWN" The index.
/// \retval "= BTRIEVE_INDEX_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE btrieve_index_t BtrieveKeySegmentGetIndex(btrieve_key_segment_t keySegment);

/// \brief Get lock owner explicit lock mode from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_LOCK_MODE_UNKNOWN" The lock owner explicit lock mode.
/// \retval "= BTRIEVE_LOCK_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_lock_mode_t BtrieveFileInformationGetLockOwnerExplicitLockMode(btrieve_file_information_t fileInformation);

/// \brief Get null key mode from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "!= BTRIEVE_NULL_KEY_MODE_UNKNOWN" The null key mode.
/// \retval "= BTRIEVE_NULL_KEY_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE btrieve_null_key_mode_t BtrieveKeySegmentGetNullKeyMode(btrieve_key_segment_t keySegment);

/// \brief Get owner mode from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_OWNER_MODE_UNKNOWN" The owner mode.
/// \retval "= BTRIEVE_OWNER_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_owner_mode_t BtrieveFileInformationGetOwnerMode(btrieve_file_information_t fileInformation);

/// \brief Get encryption type from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_ENCRYPT_TYPE_UNKNOWN" The encryption type.
/// \retval "= BTRIEVE_ENCRYPT_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_encrypt_type_t BtrieveFileInformationGetEncryptionType(btrieve_file_information_t fileInformation);

/// \brief Get lock owner page lock type from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_PAGE_LOCK_TYPE_UNKNOWN" The lock owner page lock type.
/// \retval "= BTRIEVE_PAGE_LOCK_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_page_lock_type_t BtrieveFileInformationGetLockOwnerPageLockType(btrieve_file_information_t fileInformation);

/// \brief Get page size from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_PAGE_SIZE_UNKNOWN" The page size.
/// \retval "= BTRIEVE_PAGE_SIZE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE btrieve_page_size_t BtrieveFileInformationGetPageSize(btrieve_file_information_t fileInformation);

/// \brief Get record compression mode from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_RECORD_COMPRESSION_MODE_UNKNOWN" The record compression mode.
/// \retval "= BTRIEVE_RECORD_COMPRESSION_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
/// \see BtrieveRecordCompressionModeToString
extern LINKAGE btrieve_record_compression_mode_t BtrieveFileInformationGetRecordCompressionMode(btrieve_file_information_t fileInformation);

/// \brief Add a record to a %Btrieve bulk create payload 
/// \param[in] bulkCreatePayload The bulk create payload. See BtrieveBulkCreatePayloadAllocate().
/// \param[in] record The record.
/// \param[in] recordLength The record length. Record length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkCreatePayloadAddRecord(btrieve_bulk_create_payload_t bulkCreatePayload, const char *record, int recordLength);

/// \brief Allocate a %Btrieve bulk create payload 
/// \param[out] btrieveBulkCreatePayloadPtr The %Btrieve bulk create payload pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkCreatePayloadFree
extern LINKAGE btrieve_status_code_t BtrieveBulkCreatePayloadAllocate(btrieve_bulk_create_payload_t *btrieveBulkCreatePayloadPtr);

/// \brief Free a %Btrieve bulk create payload 
/// \param[in] bulkCreatePayload The bulk create payload.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkCreatePayloadAllocate
extern LINKAGE btrieve_status_code_t BtrieveBulkCreatePayloadFree(btrieve_bulk_create_payload_t bulkCreatePayload);

/// \brief Allocate a %Btrieve bulk create result 
/// \param[out] btrieveBulkCreateResultPtr The %Btrieve bulk create result pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkCreateResultFree
extern LINKAGE btrieve_status_code_t BtrieveBulkCreateResultAllocate(btrieve_bulk_create_result_t *btrieveBulkCreateResultPtr);

/// \brief Free a %Btrieve bulk create result 
/// \param[in] bulkCreateResult The bulk create result.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkCreateResultAllocate
extern LINKAGE btrieve_status_code_t BtrieveBulkCreateResultFree(btrieve_bulk_create_result_t bulkCreateResult);

/// \brief Get last status code from a %Btrieve bulk create result  
/// \note \IS_INTENDED_FOR_USE_C
/// \param[in] bulkCreateResult The bulk create result. See BtrieveFileBulkCreate().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkCreateResultGetLastStatusCode(btrieve_bulk_create_result_t bulkCreateResult);

/// \brief Add a cosine similarity.
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] vector The vector. Must point at an array of 32-bit floating point numbers.
/// \param[in] vectorLength The vector length in bytes. Vector length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Computes the similarity between the passed in vector and the vector specified by offset.
/// Bulk retrieve attributes have no similarities initially.
/// \OFFSET_PLUS_LENGTH_C
/// The vectors specified must be in little endian format.
/// Only one similarity may be added to the same bulk retrieve.
/// Aggregates and fields may not be added to a bulk retrieve attributes containing a similarity.
/// The returned similarity is a 32-bit floating point number.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddCosineSimilarity(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, const char *vector, int vectorLength, int offset);

/// \brief Add a dot product similarity.
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] vector The vector. Must point at an array of 32-bit floating point numbers.
/// \param[in] vectorLength The vector length in bytes. Vector length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Computes the similarity between the passed in vector and the vector specified by offset.
/// Bulk retrieve attributes have no similarities initially.
/// \OFFSET_PLUS_LENGTH_C
/// The vectors specified must be in little endian format.
/// Only one similarity may be added to the same bulk retrieve.
/// Aggregates and fields may not be added to a bulk retrieve attributes containing a similarity.
/// The returned similarity is a 32-bit floating point number.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddDotProductSimilarity(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, const char *vector, int vectorLength, int offset);

/// \brief Add an Euclidean distance similarity.
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] vector The vector. Must point at an array of 32-bit floating point numbers.
/// \param[in] vectorLength The vector length in bytes. Vector length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Computes the similarity between the passed in vector and the vector specified by offset.
/// Bulk retrieve attributes have no similarities initially.
/// \OFFSET_PLUS_LENGTH_C
/// The vectors specified must be in little endian format.
/// Only one similarity may be added to the same bulk retrieve.
/// Aggregates and fields may not be added to a bulk retrieve attributes containing a similarity.
/// The returned similarity is a 32-bit floating point number.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddEuclideanDistanceSimilarity(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, const char *vector, int vectorLength, int offset);

/// \brief Add a minimum aggregate to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] dataType The data type. Only BTRIEVE_DATA_TYPE_INTEGER and BTRIEVE_DATA_TYPE_FLOAT are supported.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no aggregates initially.
/// \OFFSET_PLUS_LENGTH_C
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// The field specified by offset and length must be in little endian format.
/// Aggregates and fields may not be added to the same bulk retrieve.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddMinimumAggregate(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_data_type_t dataType, int offset, int length);

/// \brief Add a maximum aggregate to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] dataType The data type. Only BTRIEVE_DATA_TYPE_INTEGER and BTRIEVE_DATA_TYPE_FLOAT are supported.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no aggregates initially.
/// \OFFSET_PLUS_LENGTH_C
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// The field specified by offset and length must be in little endian format.
/// Similarities and fields may not be added to a bulk retrieve attributes containing an aggregate.
/// Aggregates and fields may not be added to the same bulk retrieve.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddMaximumAggregate(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_data_type_t dataType, int offset, int length);

/// \brief Add a field to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no fields initially.
/// \OFFSET_PLUS_LENGTH_C
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddField(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, int offset, int length);

/// \brief Add an absolute value field to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] dataType The data type. Only BTRIEVE_DATA_TYPE_INTEGER and BTRIEVE_DATA_TYPE_FLOAT are supported.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no fields initially.
/// \OFFSET_PLUS_LENGTH_C
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// The field specified by offset and length must be in little endian format.
/// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddAbsoluteValueField(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_data_type_t dataType, int offset, int length);

/// \brief Add a ceiling field to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] dataType  The data type.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no fields initially.
/// \OFFSET_PLUS_LENGTH_C
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// The field specified by offset and length must be in little endian format.
/// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddCeilingField(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_data_type_t dataType, int offset, int length);

/// \brief Add a floor field to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] dataType The data type. Only BTRIEVE_DATA_TYPE_INTEGER and BTRIEVE_DATA_TYPE_FLOAT are supported.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no fields initially.
/// \OFFSET_PLUS_LENGTH_C
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// The field specified by offset and length must be in little endian format.
/// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddFloorField(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_data_type_t dataType, int offset, int length);

/// \brief Add a round field to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] dataType The data type. Only BTRIEVE_DATA_TYPE_INTEGER and BTRIEVE_DATA_TYPE_FLOAT are supported.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] precision The precision. Precision must be in the range INT16_MIN to INT16_MAX, inclusive.
/// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no fields initially.
/// \OFFSET_PLUS_LENGTH_C
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// The field specified by offset and length must be in little endian format.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddRoundField(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_data_type_t dataType, int offset, int length, int precision);

/// \brief Add a truncate field to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] dataType The data type. Only BTRIEVE_DATA_TYPE_INTEGER and BTRIEVE_DATA_TYPE_FLOAT are supported.
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] precision The precision. Precision must be in the range INT16_MIN to INT16_MAX, inclusive.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no fields initially.
/// \OFFSET_PLUS_LENGTH_C
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// The field specified by offset and length must be in little endian format.
/// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddTruncateField(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_data_type_t dataType, int offset, int length, int precision);

/// \brief Add a system field to a %Btrieve bulk retrieve attributes
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] systemField The system field.
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no system fields initially.
/// The sum of the lengths of all added fields is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddSystemField(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_system_field_t systemField);

/// \brief Add a filter to a %Btrieve bulk retrieve attributes 
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \par \DESCRIPTION
/// Bulk retrieve attributes have no filters initially.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAddFilter(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_filter_t filter);

/// \brief Add a filter to a %Btrieve bulk delete attributes 
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteAttributesAddFilter(btrieve_bulk_delete_attributes_t bulkDeleteAttributes, btrieve_filter_t filter);

/// \brief Allocate a %Btrieve bulk retrieve attributes 
/// \param[out] bulkRetrieveAttributesPtr The %Btrieve bulk retrieve attributes pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkRetrieveAttributesFree
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesAllocate(btrieve_bulk_retrieve_attributes_t *bulkRetrieveAttributesPtr);

/// \brief Allocate a %Btrieve bulk delete attributes 
/// \param[out] bulkDeleteAttributesPtr The %Btrieve bulk delete attributes pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkDeleteAttributesFree
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteAttributesAllocate(btrieve_bulk_delete_attributes_t *bulkDeleteAttributesPtr);

/// \brief Free a %Btrieve bulk retrieve attributes 
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkRetrieveAttributesAllocate
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesFree(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes);

/// \brief Free a %Btrieve bulk delete attributes 
/// \param[in] bulkDeleteAttributes The bulk delete attributes.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkDeleteAttributesAllocate
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteAttributesFree(btrieve_bulk_delete_attributes_t bulkDeleteAttributes);

/// \brief Set maximum record count for a %Btrieve bulk retrieve attributes 
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] maximumRecordCount The maximum record count.
/// \par \DESCRIPTION
/// Maximum record count defaults to zero.
/// Aggregates do not honor the maximum record count.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesSetMaximumRecordCount(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, int maximumRecordCount);

/// \brief Set maximum record count for a %Btrieve bulk delete attributes 
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[in] maximumRecordCount The maximum record count.
/// \par \DESCRIPTION
/// Maximum record count defaults to zero.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteAttributesSetMaximumRecordCount(btrieve_bulk_delete_attributes_t bulkDeleteAttributes, int maximumRecordCount);

/// \brief Set maximum reject count for a %Btrieve bulk retrieve attributes 
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] maximumRejectCount The maximum reject count.
/// \par \DESCRIPTION
/// Maximum reject count defaults to the Zen defined value.
/// Similarities and aggregates do not honor the maximum reject count.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesSetMaximumRejectCount(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, int maximumRejectCount);

/// \brief Set maximum reject count for a %Btrieve bulk delete attributes 
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[in] maximumRejectCount The maximum reject count.
/// \par \DESCRIPTION
/// Maximum reject count defaults to the Zen defined value.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteAttributesSetMaximumRejectCount(btrieve_bulk_delete_attributes_t bulkDeleteAttributes, int maximumRejectCount);

/// \brief Set skip current record for a %Btrieve bulk retrieve attributes 
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[in] skipCurrentRecord Skip current record?
/// \par \DESCRIPTION
/// Skip current record is disabled by default.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveAttributesSetSkipCurrentRecord(btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, int skipCurrentRecord);

/// \brief Set skip current record for a %Btrieve bulk delete attributes 
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[in] skipCurrentRecord Skip current record?
/// \par \DESCRIPTION
/// Skip current record is disabled by default.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteAttributesSetSkipCurrentRecord(btrieve_bulk_delete_attributes_t bulkDeleteAttributes, int skipCurrentRecord);

/// \brief Allocate a %Btrieve bulk retrieve result 
/// \param[out] bulkRetrieveResultPtr The %Btrieve bulk retrieve results pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkRetrieveResultFree
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveResultAllocate(btrieve_bulk_retrieve_result_t *bulkRetrieveResultPtr);

/// \brief Allocate a %Btrieve bulk delete result 
/// \param[out] bulkDeleteResultPtr The %Btrieve bulk delete results pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkDeleteResultFree
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteResultAllocate(btrieve_bulk_delete_result_t *bulkDeleteResultPtr);

/// \brief Free a %Btrieve bulk retrieve result 
/// \param[in] bulkRetrieveResult The bulk retrieve result.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkRetrieveResultAllocate
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveResultFree(btrieve_bulk_retrieve_result_t bulkRetrieveResult);

/// \brief Free a %Btrieve bulk delete result 
/// \param[in] bulkDeleteResult The bulk delete result.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveBulkDeleteResultAllocate
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteResultFree(btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Get last status code from a %Btrieve bulk retrieve result  
/// \note \IS_INTENDED_FOR_USE_C
/// \param[in] bulkRetrieveResult The bulk retrieve result. See BtrieveFileBulkRetrieveNext() and BtrieveFileBulkRetrievePrevious().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkRetrieveResultGetLastStatusCode(btrieve_bulk_retrieve_result_t bulkRetrieveResult);

/// \brief Get last status code from a %Btrieve bulk delete result  
/// \note \IS_INTENDED_FOR_USE_C
/// \param[in] bulkDeleteResult The bulk delete result. See BtrieveFileBulkDeleteNext() and BtrieveFileBulkDeletePrevious().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveBulkDeleteResultGetLastStatusCode(btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Allocate a %Btrieve client 
/// \param[out] btrieveClientPtr The %Btrieve client pointer.
/// \param[in] serviceAgentIdentifier The service agent identifier.  Use 0 to automatically generate a service agent identifier based on the caller's process identifier.
/// \param[in] clientIdentifier The client identifier.  Use 0 to automatically generate a client identifier based on the caller's thread identifier.
/// \par \SNIPPET
/// \snippet btest.c ClientAllocate
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveClientFree
extern LINKAGE btrieve_status_code_t BtrieveClientAllocate(btrieve_client_t *btrieveClientPtr, int serviceAgentIdentifier, int clientIdentifier);

/// \brief Continuous operation begin for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] pathNames The path names.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientContinuousOperationBegin(btrieve_client_t client, const char *pathNames);

/// \brief Continuous operation begin for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] pathNames The path names.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientContinuousOperationBeginW(btrieve_client_t client, const wchar_t *pathNames);

/// \brief Continuous operation end for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] pathNames The path names.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientContinuousOperationEnd(btrieve_client_t client, const char *pathNames);

/// \brief Continuous operation end for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] pathNames The path names.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientContinuousOperationEndW(btrieve_client_t client, const wchar_t *pathNames);

/// \brief File close for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \par \SNIPPET
/// \snippet btest.c ClientFileClose
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileClose(btrieve_client_t client, btrieve_file_t file);

/// \brief File create for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] fileName The file name.
/// \param[in] createMode The create mode.
/// \param[in] locationMode The location mode.
/// \par \SNIPPET
/// \snippet btest.c ClientFileCreate
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileCreate(btrieve_client_t client, const btrieve_file_attributes_t fileAttributes, const btrieve_index_attributes_t indexAttributes, const char *fileName, btrieve_create_mode_t createMode, btrieve_location_mode_t locationMode);

/// \brief File create for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] fileName The file name.
/// \param[in] createMode The create mode.
/// \param[in] locationMode The location mode.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileCreateW(btrieve_client_t client, const btrieve_file_attributes_t fileAttributes, const btrieve_index_attributes_t indexAttributes, const wchar_t *fileName, btrieve_create_mode_t createMode, btrieve_location_mode_t locationMode);

/// \cond
/// \brief File create for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] fileInformation The file information. See BtrieveFileInformationAllocate().
/// \param[in] fileName The file name.
/// \param[in] createMode The create mode.
/// \param[in] locationMode The location mode.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileCreateUsingFileInformation(btrieve_client_t client, const btrieve_file_information_t fileInformation, const char *fileName, btrieve_create_mode_t createMode, btrieve_location_mode_t locationMode);

/// \brief File create for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] fileInformation The file information. See BtrieveFileInformationAllocate().
/// \param[in] fileName The file name.
/// \param[in] createMode The create mode.
/// \param[in] locationMode The location mode.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileCreateUsingFileInformationW(btrieve_client_t client, const btrieve_file_information_t fileInformation, const wchar_t *fileName, btrieve_create_mode_t createMode, btrieve_location_mode_t locationMode);
/// \endcond

/// \brief File delete for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] fileName The file name.
/// \par \SNIPPET
/// \snippet btest.c ClientFileDelete
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileDelete(btrieve_client_t client, const char *fileName);

/// \brief File delete for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] fileName The file name.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileDeleteW(btrieve_client_t client, const wchar_t *fileName);

/// \brief File open for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[out] btrieveFilePtr The %Btrieve file pointer.
/// \param[in] fileName The file name.
/// \param[in] ownerName The owner name.
/// \param[in] openMode The open mode.
/// \param[in] locationMode The location mode.
/// \par \SNIPPET
/// \snippet btest.c ClientFileOpen
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveClientFileClose
extern LINKAGE btrieve_status_code_t BtrieveClientFileOpen(btrieve_client_t client, btrieve_file_t *btrieveFilePtr, const char *fileName, const char *ownerName, btrieve_open_mode_t openMode, btrieve_location_mode_t locationMode);

/// \brief File open for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[out] btrieveFilePtr The %Btrieve file pointer.
/// \param[in] fileName The file name.
/// \param[in] ownerName The owner name.
/// \param[in] openMode The open mode.
/// \param[in] locationMode The location mode.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveClientFileClose
extern LINKAGE btrieve_status_code_t BtrieveClientFileOpenW(btrieve_client_t client, btrieve_file_t *btrieveFilePtr, const wchar_t *fileName, const char *ownerName, btrieve_open_mode_t openMode, btrieve_location_mode_t locationMode);

/// \brief File rename for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] existingFileName The existing file name.
/// \param[in] newFileName The new file name.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileRename(btrieve_client_t client, const char *existingFileName, const char *newFileName);

/// \brief File rename for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] existingFileName The existing file name.
/// \param[in] newFileName The new file name.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientFileRenameW(btrieve_client_t client, const wchar_t *existingFileName, const wchar_t *newFileName);

/// \brief Free a %Btrieve client 
/// \param[in] client The client.
/// \par \SNIPPET
/// \snippet btest.c ClientFree
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveClientAllocate
extern LINKAGE btrieve_status_code_t BtrieveClientFree(btrieve_client_t client);

/// \brief Get current directory from a %Btrieve client  
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] diskDrive The disk drive. \MUST_USE_DISK_DRIVE_DEFAULT_C
/// \param[out] currentDirectory The current directory.
/// \param[in] currentDirectorySize The current directory size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientGetCurrentDirectory(btrieve_client_t client, btrieve_disk_drive_t diskDrive, char *currentDirectory, int currentDirectorySize);

/// \brief Get current directory from a %Btrieve client  
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] diskDrive The disk drive. \MUST_USE_DISK_DRIVE_DEFAULT_C
/// \param[in] currentDirectory The current directory.
/// \param[in] currentDirectorySize The current directory size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientGetCurrentDirectoryW(btrieve_client_t client, btrieve_disk_drive_t diskDrive, wchar_t *currentDirectory, int currentDirectorySize);

/// \brief Get last status code from a %Btrieve client  
/// \note \IS_INTENDED_FOR_USE_C
/// \param[in] client The client. See BtrieveClientAllocate().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientGetLastStatusCode(btrieve_client_t client);

/// \brief Get version from a %Btrieve client  
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[out] version The version. See BtrieveVersionAllocate().
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientGetVersion(btrieve_client_t client, btrieve_version_t version, btrieve_file_t file);

/// \brief Login for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] databaseURI The database URI.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientLogin(btrieve_client_t client, const char *databaseURI);

/// \brief Login for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] databaseURI The database URI.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientLoginW(btrieve_client_t client, const wchar_t *databaseURI);

/// \brief Logout for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] databaseURI The database URI.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientLogout(btrieve_client_t client, const char *databaseURI);

/// \brief Logout for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] databaseURI The database URI.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientLogoutW(btrieve_client_t client, const wchar_t *databaseURI);

/// \brief Reset for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientReset(btrieve_client_t client);

/// \brief Set current directory for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] currentDirectory The current directory.
/// \par \DESCRIPTION
/// Current directory defaults to the current working directory.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientSetCurrentDirectory(btrieve_client_t client, const char *currentDirectory);

/// \brief Set current directory for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] currentDirectory The current directory.
/// \par \DESCRIPTION
/// Current directory defaults to the current working directory.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientSetCurrentDirectoryW(btrieve_client_t client, const wchar_t *currentDirectory);

/// \brief Stop for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientStop(btrieve_client_t client);

/// \brief Transaction abort for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientTransactionAbort(btrieve_client_t client);

/// \brief Transaction begin for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \param[in] transactionMode The transaction mode.
/// \param[in] lockMode The lock mode.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientTransactionBegin(btrieve_client_t client, btrieve_transaction_mode_t transactionMode, btrieve_lock_mode_t lockMode);

/// \brief Transaction end for a %Btrieve client 
/// \param[in] client The client. See BtrieveClientAllocate().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveClientTransactionEnd(btrieve_client_t client);

/// \brief Allocate a %Btrieve file attributes 
/// \param[out] btrieveFileAttributesPtr The %Btrieve file attributes pointer.
/// \par \SNIPPET
/// \snippet btest.c FileAttributesAllocate
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveFileAttributesFree
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesAllocate(btrieve_file_attributes_t *btrieveFileAttributesPtr);

/// \brief Free a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes.
/// \par \SNIPPET
/// \snippet btest.c FileAttributesFree
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveFileAttributesAllocate
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesFree(btrieve_file_attributes_t fileAttributes);

/// \brief Set balanced indexes for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] enableBalancedIndexes Enable balanced indexes?
/// \par \DESCRIPTION
/// Balanced indexes are disabled by default.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetBalancedIndexes(btrieve_file_attributes_t fileAttributes, int enableBalancedIndexes);

/// \brief Set file version for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] fileVersion The file version.
/// \par \DESCRIPTION
/// File version defaults to BTRIEVE_FILE_VERSION_DEFAULT. 
/// A file version of BTRIEVE_FILE_VERSION_6_0 is mapped to BTRIEVE_FILE_VERSION_6_1.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetFileVersion(btrieve_file_attributes_t fileAttributes, btrieve_file_version_t fileVersion);

/// \brief Set fixed record length for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \par \SNIPPET
/// \param[in] fixedRecordLength The fixed record length. The fixed record length must be between 1 and BTRIEVE_MAXIMUM_RECORD_LENGTH, inclusive.
/// \par \DESCRIPTION
/// Fixed record length defaults to zero.
/// \snippet btest.c FileAttributesSetFixedRecordLength
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetFixedRecordLength(btrieve_file_attributes_t fileAttributes, int fixedRecordLength);

/// \brief Set free space threshold for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] freeSpaceThreshold The free space threshold.
/// \par \DESCRIPTION
/// Free space threshold defaults to BTRIEVE_FREE_SPACE_THRESHOLD_DEFAULT.
/// The free space threshold setting is ignored unless variable length records are enabled.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetFreeSpaceThreshold(btrieve_file_attributes_t fileAttributes, btrieve_free_space_threshold_t freeSpaceThreshold);

/// \brief Set key only for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] enableKeyOnly Enable key only?
/// \par \DESCRIPTION
/// Key only is disabled by default.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetKeyOnly(btrieve_file_attributes_t fileAttributes, int enableKeyOnly);

/// \brief Set page size for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] pageSize The page size.
/// \param[in] enablePageCompression Enable page compression?
/// \par \DESCRIPTION
/// Page size defaults to BTRIEVE_PAGE_SIZE_4096.
/// Page compression is disabled by default and requires file version BTRIEVE_FILE_VERSION_9_5 or greater.
/// Not all page sizes are valid for all file versions.
/// Pages sizes greater than 4096 require file version BTRIEVE_FILE_VERSION_9_0 or greater.
/// Pages sizes which are unsupported for a particular file version will be rounded up to the next supported size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetPageSize(btrieve_file_attributes_t fileAttributes, btrieve_page_size_t pageSize, int enablePageCompression);

/// \brief Set preallocated page count for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] preallocatedPageCount The preallocated page count.
/// \par \DESCRIPTION
/// Preallocated page count defaults to zero.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetPreallocatedPageCount(btrieve_file_attributes_t fileAttributes, int preallocatedPageCount);

/// \brief Set record compression mode for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] recordCompressionMode The record compression mode.
/// \par \DESCRIPTION
/// Record compression defaults to BTRIEVE_RECORD_COMPRESSION_MODE_NONE.
/// Compression mode blank truncation requires variable length records be enabled.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetRecordCompressionMode(btrieve_file_attributes_t fileAttributes, btrieve_record_compression_mode_t recordCompressionMode);

/// \brief Set reserved duplicate pointer count for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] reservedDuplicatePointerCount The reserved duplicate pointer count.
/// \par \DESCRIPTION
/// Reserved duplicate pointer count defaults to zero.
/// The maximum reserved duplicate pointer count may be less than 119 depending on file version and page size.
/// For example, a file version of BTRIEVE_FILE_VERSION_9_0 and page size of BTRIEVE_PAGE_SIZE_3584 allows for a maximum reserved duplicate pointer count of 54.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetReservedDuplicatePointerCount(btrieve_file_attributes_t fileAttributes, int reservedDuplicatePointerCount);

/// \brief Set system data mode for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] systemDataMode The system data mode.
/// \par \DESCRIPTION
/// System data mode defaults to BTRIEVE_SYSTEM_DATA_MODE_DEFAULT.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetSystemDataMode(btrieve_file_attributes_t fileAttributes, btrieve_system_data_mode_t systemDataMode);

/// \brief Set variable length records mode for a %Btrieve file attributes 
/// \param[in] fileAttributes The file attributes. See BtrieveFileAttributesAllocate().
/// \param[in] variableLengthRecordsMode The variable length records mode.
/// \par \DESCRIPTION
/// Variable length records mode defaults to BTRIEVE_VARIABLE_LENGTH_RECORDS_MODE_NO.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileAttributesSetVariableLengthRecordsMode(btrieve_file_attributes_t fileAttributes, btrieve_variable_length_records_mode_t variableLengthRecordsMode);

/// \brief Bulk create for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] bulkCreatePayload The bulk create payload. See BtrieveBulkCreatePayloadAllocate().
/// \param[out] bulkCreateResult The bulk create result. See BtrieveBulkCreateResultAllocate().
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkCreate(btrieve_file_t file, const btrieve_bulk_create_payload_t bulkCreatePayload, btrieve_bulk_create_result_t bulkCreateResult);

/// \brief Bulk retrieve next for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[out] bulkRetrieveResult The bulk retrieve result. See BtrieveBulkRetrieveResultAllocate().
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor must be established. The cursor is traversed forward to the last record examined.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkRetrieveNext(btrieve_file_t file, const btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_bulk_retrieve_result_t bulkRetrieveResult, btrieve_lock_mode_t lockMode);

/// \brief Bulk retrieve next by comparison for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] comparison The comparison.
/// \param[in] index The index.
/// \param[in] key The key.
/// \param[in] keyLength The key length.
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[out] bulkRetrieveResult The bulk retrieve result. See BtrieveBulkRetrieveResultAllocate().
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkRetrieveNextByComparison(btrieve_file_t file, btrieve_comparison_t comparison, btrieve_index_t index, char *key, int keyLength, const btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_bulk_retrieve_result_t bulkRetrieveResult, btrieve_lock_mode_t lockMode);

/// \brief Bulk retrieve previous for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[out] bulkRetrieveResult The bulk retrieve result. See BtrieveBulkRetrieveResultAllocate().
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor must be established. The cursor is traversed backwards to the last record examined.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkRetrievePrevious(btrieve_file_t file, const btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_bulk_retrieve_result_t bulkRetrieveResult, btrieve_lock_mode_t lockMode);

/// \brief Bulk retrieve previous by comparison for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] comparison The comparison.
/// \param[in] index The index.
/// \param[in] key The key.
/// \param[in] keyLength The key length.
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[out] bulkRetrieveResult The bulk retrieve result. See BtrieveBulkRetrieveResultAllocate().
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkRetrievePreviousByComparison(btrieve_file_t file, btrieve_comparison_t comparison, btrieve_index_t index, char *key, int keyLength, const btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_bulk_retrieve_result_t bulkRetrieveResult, btrieve_lock_mode_t lockMode);

/// \brief Bulk retrieve first for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[out] bulkRetrieveResult The bulk retrieve result. See BtrieveBulkRetrieveResultAllocate().
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkRetrieveFirst(btrieve_file_t file, btrieve_index_t index, const btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_bulk_retrieve_result_t bulkRetrieveResult, btrieve_lock_mode_t lockMode);

/// \brief Bulk retrieve last for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] bulkRetrieveAttributes The bulk retrieve attributes. See BtrieveBulkRetrieveAttributesAllocate().
/// \param[out] bulkRetrieveResult The bulk retrieve result. See BtrieveBulkRetrieveResultAllocate().
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkRetrieveLast(btrieve_file_t file, btrieve_index_t index, const btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes, btrieve_bulk_retrieve_result_t bulkRetrieveResult, btrieve_lock_mode_t lockMode);

/// \brief Bulk delete next for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[out] bulkDeleteResult The bulk delete result. See BtrieveBulkDeleteResultAllocate().
/// \par \CURSOR
/// The cursor must be established. The cursor is traversed forward to the last record examined.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkDeleteNext(btrieve_file_t file, const btrieve_bulk_delete_attributes_t bulkDeleteAttributes, btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Bulk delete next by comparison for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] comparison The comparison.
/// \param[in] index The index.
/// \param[in] key The key.
/// \param[in] keyLength The key length.
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[out] bulkDeleteResult The bulk delete result. See BtrieveBulkDeleteResultAllocate().
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkDeleteNextByComparison(btrieve_file_t file, btrieve_comparison_t comparison, btrieve_index_t index, char *key, int keyLength, btrieve_bulk_delete_attributes_t bulkDeleteAttributes, btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Bulk delete previous for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[out] bulkDeleteResult The bulk delete result. See BtrieveBulkDeleteResultAllocate().
/// \par \CURSOR
/// The cursor must be established. The cursor is traversed backwards to the last record examined.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkDeletePrevious(btrieve_file_t file, const btrieve_bulk_delete_attributes_t bulkDeleteAttributes, btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Bulk delete previous by comparison for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] comparison The comparison.
/// \param[in] index The index.
/// \param[in] key The key.
/// \param[in] keyLength The key length.
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[out] bulkDeleteResult The bulk delete result. See BtrieveBulkDeleteResultAllocate().
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkDeletePreviousByComparison(btrieve_file_t file, btrieve_comparison_t comparison, btrieve_index_t index, char *key, int keyLength, btrieve_bulk_delete_attributes_t bulkDeleteAttributes, btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Bulk delete first for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[out] bulkDeleteResult The bulk delete result. See BtrieveBulkDeleteResultAllocate().
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkDeleteFirst(btrieve_file_t file, btrieve_index_t index, btrieve_bulk_delete_attributes_t bulkDeleteAttributes, btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Bulk delete last for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] bulkDeleteAttributes The bulk delete attributes. See BtrieveBulkDeleteAttributesAllocate().
/// \param[out] bulkDeleteResult The bulk delete result. See BtrieveBulkDeleteResultAllocate().
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileBulkDeleteLast(btrieve_file_t file, btrieve_index_t index, btrieve_bulk_delete_attributes_t bulkDeleteAttributes, btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Get information from a %Btrieve file  
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[out] fileInformation The file information. See BtrieveFileInformationAllocate().
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileGetInformation(btrieve_file_t file, btrieve_file_information_t fileInformation);

/// \brief Get last status code from a %Btrieve file  
/// \note \IS_INTENDED_FOR_USE_C
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \par \SNIPPET
/// \snippet btest.c FileGetLastStatusCode
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileGetLastStatusCode(btrieve_file_t file);

/// \brief Index create for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \par \SNIPPET
/// \snippet btest.c FileIndexCreate
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileIndexCreate(btrieve_file_t file, const btrieve_index_attributes_t indexAttributes);

/// \brief Index drop for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \par \CURSOR
/// If the cursor index is dropped the cursor is destroyed. Otherwise, the cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileIndexDrop(btrieve_file_t file, btrieve_index_t index);

/// \brief Allocate a %Btrieve file information 
/// \param[out] btrieveFileInformationPtr The %Btrieve file information pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveFileInformationFree
extern LINKAGE btrieve_status_code_t BtrieveFileInformationAllocate(btrieve_file_information_t *btrieveFileInformationPtr);

/// \brief Free a %Btrieve file information 
/// \param[in] fileInformation The file information.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveFileInformationAllocate
extern LINKAGE btrieve_status_code_t BtrieveFileInformationFree(btrieve_file_information_t fileInformation);

/// \brief Get gateway name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetGatewayName(btrieve_file_information_t fileInformation, char *name, int nameSize);

/// \brief Get key segment from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] keySegment The key segment. See BtrieveKeySegmentAllocate().
/// \param[in] keySegmentNumber The key segment number.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetKeySegment(btrieve_file_information_t fileInformation, btrieve_key_segment_t keySegment, int keySegmentNumber);

/// \brief Get last status code from a %Btrieve file information  
/// \note \IS_INTENDED_FOR_USE_C
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetLastStatusCode(btrieve_file_information_t fileInformation);

/// \brief Get lock owner name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetLockOwnerName(btrieve_file_information_t fileInformation, char *name, int nameSize);

/// \brief Get referential integrity file name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetReferentialIntegrityFileName(btrieve_file_information_t fileInformation, char *name, int nameSize);

/// \brief Get security current database name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetSecurityCurrentDatabaseName(btrieve_file_information_t fileInformation, char *name, int nameSize);

/// \brief Get security current user name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetSecurityCurrentUserName(btrieve_file_information_t fileInformation, char *name, int nameSize);

/// \brief Get security handle database name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetSecurityHandleDatabaseName(btrieve_file_information_t fileInformation, char *name, int nameSize);

/// \brief Get security handle table name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetSecurityHandleTableName(btrieve_file_information_t fileInformation, char *name, int nameSize);

/// \brief Get security handle user name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetSecurityHandleUserName(btrieve_file_information_t fileInformation, char *name, int nameSize);

/// \brief Get segment file name from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \param[in] segmentFileNumber The segment file number.
/// \param[out] name The name.
/// \param[in] nameSize The name size.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString,
extern LINKAGE btrieve_status_code_t BtrieveFileInformationGetSegmentFileName(btrieve_file_information_t fileInformation, int segmentFileNumber, char *name, int nameSize);

/// \brief Key retrieve for a %Btrieve file 
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] comparison The comparison.
/// \param[in] index The index.
/// \param[out] key The key.
/// \param[in] keyLength The key length.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileKeyRetrieve(btrieve_file_t file, btrieve_comparison_t comparison, btrieve_index_t index, char *key, int keyLength);

/// \brief Key retrieve first for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileKeyRetrieveFirst(btrieve_file_t file, btrieve_index_t index, char *key, int keySize);

/// \brief Key retrieve last for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileKeyRetrieveLast(btrieve_file_t file, btrieve_index_t index, char *key, int keySize);

/// \brief Key retrieve next for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \par \CURSOR
/// The cursor must be established. The cursor is traversed forward one key.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileKeyRetrieveNext(btrieve_file_t file, char *key, int keySize);

/// \brief Key retrieve previous for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \par \CURSOR
/// The cursor must be established. The cursor is traversed backward one key.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileKeyRetrievePrevious(btrieve_file_t file, char *key, int keySize);

/// \brief Record append chunk for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] chunk The chunk.
/// \param[in] chunkLength The chunk length.
/// \par \DESCRIPTION
/// The file must have variable length records. See BtrieveFileAttributesSetVariableLengthRecordsMode.
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is established at the end of the appended chunk.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordAppendChunk(btrieve_file_t file, const char *chunk, int chunkLength);

/// \brief Record create for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in,out] record The record. If the current record contains a BTRIEVE_DATA_TYPE_AUTOINCREMENT or BTRIEVE_DATA_TYPE_AUTOTIMESTAMP key segment then the passed in record may be modifed.
/// \param[in] recordLength The record length. Record length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH. See BtrieveFileRecordAppendChunk() for creating records greater than BTRIEVE_MAXIMUM_RECORD_LENGTH in length.
/// \par \SNIPPET
/// \snippet btest.c FileRecordCreate
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordCreate(btrieve_file_t file, char *record, int recordLength);

/// \brief Record delete for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordDelete(btrieve_file_t file);

/// \brief Record truncate for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] offset The offset. If offset is -1, then the current offset is used.
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is established at the end of the record.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordTruncate(btrieve_file_t file, int offset);

/// \brief Record unlock for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] unlockMode The unlock mode.
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordUnlock(btrieve_file_t file, btrieve_unlock_mode_t unlockMode);

/// \brief Record update for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] record The record.
/// \param[in] recordLength The record length. Record length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH. See BtrieveFileRecordUpdateChunk() for updating records greater than BTRIEVE_MAXIMUM_RECORD_LENGTH in length.
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordUpdate(btrieve_file_t file, const char *record, int recordLength);

/// \brief Record update chunk for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] offset The offset. If offset is -1, then the current offset is used.
/// \param[in] chunk The chunk.
/// \param[in] chunkLength The chunk length.
/// \par \SNIPPET
/// \snippet btestchunk.cpp RecordUpdateChunk
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is established at the end of the updated chunk.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordUpdateChunk(btrieve_file_t file, int offset, const char *chunk, int chunkLength);

/// \brief Set owner for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] ownerMode The owner mode.
/// \param[in] ownerName The owner name.
/// \param[in] ownerNameAgain The owner name again.
/// \param[in] useLongOwnerName Use a long owner name?
/// \par \DESCRIPTION
/// Owner mode defaults to BTRIEVE_OWNER_MODE_NONE.
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileSetOwner(btrieve_file_t file, btrieve_owner_mode_t ownerMode, const char *ownerName, const char *ownerNameAgain, int useLongOwnerName);

/// \brief Record insert chunk for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] offset The offset. If offset is -1, then the current offset is used.
/// \param[in] chunk The chunk.
/// \param[in] chunkLength The chunk length.
/// \par \DESCRIPTION
/// The file must have variable length records. See BtrieveFileAttributesSetVariableLengthRecordsMode.
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is established at the end of the inserted chunk.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordInsertChunk(btrieve_file_t file, int offset, const char *chunk, int chunkLength);

/// \brief Record delete chunk for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] offset The offset. If offset is -1, then the current offset is used.
/// \param[in] chunkLength The chunk length.
/// \par \DESCRIPTION
/// The file must have variable length records. See BtrieveFileAttributesSetVariableLengthRecordsMode.
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is established at the end of the deleted chunk.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileRecordDeleteChunk(btrieve_file_t file, int offset, int chunkLength);

/// \brief Unlock cursor position for a %Btrieve file 
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] cursorPosition The cursor position.
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFileUnlockCursorPosition(btrieve_file_t file, long long cursorPosition);

/// \brief Allocate a %Btrieve filter 
/// \param[out] btrieveFilterPtr The %Btrieve filter pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveFilterFree
extern LINKAGE btrieve_status_code_t BtrieveFilterAllocate(btrieve_filter_t *btrieveFilterPtr);

/// \brief Free a %Btrieve filter 
/// \param[in] filter The filter.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveFilterAllocate
extern LINKAGE btrieve_status_code_t BtrieveFilterFree(btrieve_filter_t filter);

/// \brief Set ACS mode for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] ACSMode The ACS mode.
/// \par \DESCRIPTION
/// Alternate collation sequence mode defaults to BTRIEVE_ACS_MODE_NONE. To set the alternate collation sequence mode to BTRIEVE_ACS_MODE_NAMED use BtrieveFilterSetACSName() or BtrieveFilterSetACSUserDefined(). BTRIEVE_ACS_MODE_NUMBERED is unsupported.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetACSMode(btrieve_filter_t filter, btrieve_acs_mode_t ACSMode);

/// \brief Set ACS name for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] name The name.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetACSName(btrieve_filter_t filter, const char *name);

/// \brief Set ACS user defined for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] name The name.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetACSUserDefined(btrieve_filter_t filter, const char *name);

/// \brief Set comparison for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] comparison The comparison.
/// \par \DESCRIPTION
/// Comparison defaults to BTRIEVE_COMPARISON_NONE.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetComparison(btrieve_filter_t filter, btrieve_comparison_t comparison);

/// \brief Set comparison constant for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] constant The constant.
/// \param[in] constantLength The comparison constant length. Comparison constant length is limited to BTRIEVE_MAXIMUM_KEY_LENGTH.
/// \par \DESCRIPTION
/// Filters have no comparison constant initially.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetComparisonConstant(btrieve_filter_t filter, const char *constant, int constantLength);

/// \brief Set comparison field for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] offset The comparison field offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \par \DESCRIPTION
/// Filters have no comparison field initially.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetComparisonField(btrieve_filter_t filter, int offset);

/// \brief Set connector for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] connector The connector.
/// \par \DESCRIPTION
/// Field connector defaults to BTRIEVE_CONNECTOR_LAST.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetConnector(btrieve_filter_t filter, btrieve_connector_t connector);

/// \brief Set field for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] dataType The data type. BTRIEVE_DATA_TYPE_LEGACY_BINARY and BTRIEVE_DATA_TYPE_LEGACY_STRING are not supported.
/// \par \DESCRIPTION
/// Filters have no field initially.
/// \OFFSET_PLUS_LENGTH_C
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetField(btrieve_filter_t filter, int offset, int length, btrieve_data_type_t dataType);

/// \brief Set system field for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] systemField The system field.
/// \par \DESCRIPTION
/// Filters have no system field initially.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetSystemField(btrieve_filter_t filter, btrieve_system_field_t systemField);

/// \brief Set like code page name for a %Btrieve filter 
/// \param[in] filter The filter. See BtrieveFilterAllocate().
/// \param[in] name The name.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveFilterSetLikeCodePageName(btrieve_filter_t filter, const char *name);

/// \brief segment for a %Btrieve index attributes Addkey 
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] keySegment The key segment. See BtrieveKeySegmentAllocate().
/// \par \SNIPPET
/// \snippet btest.c IndexAttributesAddKeySegment
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesAddKeySegment(btrieve_index_attributes_t indexAttributes, btrieve_key_segment_t keySegment);

/// \brief Allocate a %Btrieve index attributes 
/// \param[out] btrieveIndexAttributesPtr The %Btrieve index attributes pointer.
/// \par \SNIPPET
/// \snippet btest.c IndexAttributesAllocate
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveIndexAttributesFree
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesAllocate(btrieve_index_attributes_t *btrieveIndexAttributesPtr);

/// \brief Free a %Btrieve index attributes 
/// \param[in] indexAttributes The index attributes.
/// \par \SNIPPET
/// \snippet btest.c IndexAttributesFree
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveIndexAttributesAllocate
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesFree(btrieve_index_attributes_t indexAttributes);

/// \brief Set ACS mode for a %Btrieve index attributes 
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] ACSMode The ACS mode.
/// \par \DESCRIPTION
/// Alternate collation sequence mode defaults to BTRIEVE_ACS_MODE_NONE. To set the alternate collation sequence mode to BTRIEVE_ACS_MODE_NAMED use BtrieveIndexAttributesSetACSName() or BtrieveIndexAttributesSetACSUserDefined(). To set the alternate collation sequence mode to BTRIEVE_ACS_MODE_NUMBERED use BtrieveIndexAttributesSetACSNumber().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString,
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesSetACSMode(btrieve_index_attributes_t indexAttributes, btrieve_acs_mode_t ACSMode);

/// \brief Set ACS name for a %Btrieve index attributes 
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] ACSName The ACS name.
/// \par \DESCRIPTION
/// By default there is no alternate collation sequence name. Setting the alternate collation sequence name also sets the alternate collation sequence mode to BTRIEVE_ACS_MODE_NAMED.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesSetACSName(btrieve_index_attributes_t indexAttributes, const char *ACSName);

/// \brief Set ACS number for a %Btrieve index attributes 
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] ACSNumber The ACS number.
/// \par \DESCRIPTION
/// By default there is no alternate collation sequence number. Setting the alternate collation sequence number also sets the alternate collation sequence mode to BTRIEVE_ACS_MODE_NUMBERED.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesSetACSNumber(btrieve_index_attributes_t indexAttributes, int ACSNumber);

/// \brief Set ACS user defined for a %Btrieve index attributes 
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] ACSName The ACS name.
/// \param[in] ACSMap The ACS map.
/// \param[in] ACSMapLength The ACS map length.
/// \par \DESCRIPTION
/// By default there is no user defined alternate collation sequence. Setting the user defined alternate collation sequence also sets the alternate collation sequence mode to BTRIEVE_ACS_MODE_NAMED.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesSetACSUserDefined(btrieve_index_attributes_t indexAttributes, const char *ACSName, const char *ACSMap, int ACSMapLength);

/// \brief Set duplicate mode for a %Btrieve index attributes 
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] duplicateMode The duplicate mode.
/// \par \DESCRIPTION
/// Duplicate mode defaults to BTRIEVE_DUPLICATE_MODE_NOT_ALLOWED.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesSetDuplicateMode(btrieve_index_attributes_t indexAttributes, btrieve_duplicate_mode_t duplicateMode);

/// \brief Set index for a %Btrieve IndexAttributes
/// \param[in] index The index.
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \par \DESCRIPTION
/// Index defaults to the first available index starting at BTRIEVE_INDEX_1.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesSetIndex(btrieve_index_attributes_t indexAttributes, btrieve_index_t index);

/// \brief Set modifiable for a %Btrieve index attributes 
/// \param[in] indexAttributes The index attributes. See BtrieveIndexAttributesAllocate().
/// \param[in] enableModifiable Enable modifiable?
/// \par \DESCRIPTION
/// Modifiable is enabled by default.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveIndexAttributesSetModifiable(btrieve_index_attributes_t indexAttributes, int enableModifiable);

/// \brief Allocate a %Btrieve key segment 
/// \param[out] btrieveKeySegmentPtr The %Btrieve key segment pointer.
/// \par \SNIPPET
/// \snippet btest.c KeySegmentAllocate
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveKeySegmentFree
extern LINKAGE btrieve_status_code_t BtrieveKeySegmentAllocate(btrieve_key_segment_t *btrieveKeySegmentPtr);

/// \brief Free a %Btrieve key segment 
/// \param[in] keySegment The key segment.
/// \par \SNIPPET
/// \snippet btest.c KeySegmentFree
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveKeySegmentAllocate
extern LINKAGE btrieve_status_code_t BtrieveKeySegmentFree(btrieve_key_segment_t keySegment);

/// \brief Get last status code from a %Btrieve key segment  
/// \note \IS_INTENDED_FOR_USE_C
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveKeySegmentGetLastStatusCode(btrieve_key_segment_t keySegment);

/// \brief Set descending sort order for a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveKeySegmentAllocate().
/// \param[in] setDescendingSortOrder Enable descending sort order?
/// \par \DESCRIPTION
/// Descending sort order is disabled by default.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveKeySegmentSetDescendingSortOrder(btrieve_key_segment_t keySegment, int setDescendingSortOrder);

/// \brief Set field for a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveKeySegmentAllocate().
/// \param[in] offset The offset. Offset must be less than BTRIEVE_MAXIMUM_RECORD_LENGTH.
/// \param[in] length The length. Length is limited to BTRIEVE_MAXIMUM_KEY_LENGTH.
/// \param[in] dataType The data type.
/// \par \DESCRIPTION
/// Key segments have no field initially.
/// \OFFSET_PLUS_LENGTH_C
/// \par \SNIPPET
/// \snippet btest.c KeySegmentSetField
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveKeySegmentSetField(btrieve_key_segment_t keySegment, int offset, int length, btrieve_data_type_t dataType);

/// \brief Set null key mode for a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveKeySegmentAllocate().
/// \param[in] nullKeyMode The null key mode.
/// \par \DESCRIPTION
/// Null key mode defaults to BTRIEVE_NULL_KEY_MODE_NONE.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveKeySegmentSetNullKeyMode(btrieve_key_segment_t keySegment, btrieve_null_key_mode_t nullKeyMode);

/// \brief Set null value for a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveKeySegmentAllocate().
/// \param[in] nullValue The null value.
/// \par \DESCRIPTION
/// Null value defaults to zero.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveKeySegmentSetNullValue(btrieve_key_segment_t keySegment, int nullValue);

/// \brief Allocate a %Btrieve version 
/// \param[out] btrieveVersionPtr The %Btrieve version pointer.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveVersionFree
extern LINKAGE btrieve_status_code_t BtrieveVersionAllocate(btrieve_version_t *btrieveVersionPtr);

/// \brief Free a %Btrieve version 
/// \param[in] version The version.
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString, BtrieveFileInformationAllocate
extern LINKAGE btrieve_status_code_t BtrieveVersionFree(btrieve_version_t version);

/// \brief Get last status code from a %Btrieve version 
/// \note \IS_INTENDED_FOR_USE_C
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval "= BTRIEVE_STATUS_CODE_NO_ERROR" \SUCCESS
/// \retval "!= BTRIEVE_STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
/// \see BtrieveStatusCodeToString
extern LINKAGE btrieve_status_code_t BtrieveVersionGetLastStatusCode(btrieve_version_t version);

/// \brief Get system data mode from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_SYSTEM_DATA_MODE_UNKNOWN" The system data mode.
/// \retval "= BTRIEVE_SYSTEM_DATA_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
/// \see BtrieveSystemDataModeToString
extern LINKAGE btrieve_system_data_mode_t BtrieveFileInformationGetSystemDataMode(btrieve_file_information_t fileInformation);

/// \brief Get variable length records mode from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "!= BTRIEVE_VARIABLE_LENGTH_RECORDS_MODE_UNKNOWN" The variable length records mode.
/// \retval "= BTRIEVE_VARIABLE_LENGTH_RECORDS_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
/// \see BtrieveVariableLengthRecordsModeToString
extern LINKAGE btrieve_variable_length_records_mode_t BtrieveFileInformationGetVariableLengthRecordsMode(btrieve_file_information_t fileInformation);

/// \brief Get client version type from a %Btrieve version 
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval "!= BTRIEVE_VERSION_TYPE_UNKNOWN" The client version type.
/// \retval "= BTRIEVE_VERSION_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE btrieve_version_type_t BtrieveVersionGetClientVersionType(btrieve_version_t version);

/// \brief Get local version type from a %Btrieve version 
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval "!= BTRIEVE_VERSION_TYPE_UNKNOWN" The local version type.
/// \retval "= BTRIEVE_VERSION_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE btrieve_version_type_t BtrieveVersionGetLocalVersionType(btrieve_version_t version);

/// \brief Get remote version type from a %Btrieve version 
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval "!= BTRIEVE_VERSION_TYPE_UNKNOWN" The remote version type.
/// \retval "= BTRIEVE_VERSION_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE btrieve_version_type_t BtrieveVersionGetRemoteVersionType(btrieve_version_t version);

/// \brief Return a %Btrieve ACS mode as a string
/// \param[in] ACSMode The ACS mode.
/// \return The string.
extern LINKAGE const char *BtrieveACSModeToString(btrieve_acs_mode_t ACSMode);

/// \brief Return a %Btrieve data type as a string
/// \param[in] dataType The data type.
/// \return The string.
extern LINKAGE const char *BtrieveDataTypeToString(btrieve_data_type_t dataType);

/// \brief Return a %Btrieve duplicate mode as a string
/// \param[in] duplicateMode The duplicate mode.
/// \return The string.
extern LINKAGE const char *BtrieveDuplicateModeToString(btrieve_duplicate_mode_t duplicateMode);

/// \brief Return a %Btrieve file version as a string
/// \param[in] fileVersion The file version.
/// \return The string.
extern LINKAGE const char *BtrieveFileVersionToString(btrieve_file_version_t fileVersion);

/// \brief Return a %Btrieve free space threshold as a string
/// \param[in] freeSpaceThreshold The free space threshold.
/// \return The string.
extern LINKAGE const char *BtrieveFreeSpaceThresholdToString(btrieve_free_space_threshold_t freeSpaceThreshold);

/// \brief Return a %Btrieve index as a string
/// \param[in] index The index.
/// \return The string.
extern LINKAGE const char *BtrieveIndexToString(btrieve_index_t index);

/// \brief Return a %Btrieve lock mode as a string
/// \param[in] lockMode The lock mode.
/// \return The string.
extern LINKAGE const char *BtrieveLockModeToString(btrieve_lock_mode_t lockMode);

/// \brief Return a %Btrieve null key mode as a string
/// \param[in] nullKeyMode The null key mode.
/// \return The string.
extern LINKAGE const char *BtrieveNullKeyModeToString(btrieve_null_key_mode_t nullKeyMode);

/// \brief Return a %Btrieve owner mode as a string
/// \param[in] ownerMode The owner mode.
/// \return The string.
extern LINKAGE const char *BtrieveOwnerModeToString(btrieve_owner_mode_t ownerMode);

/// \brief Return a %Btrieve encryption type as a string
/// \param[in] encryptType The encrypt type.
/// \return The string.
extern LINKAGE const char *BtrieveEncryptionTypeToString(btrieve_encrypt_type_t encryptType);

/// \brief Return a %Btrieve page lock type as a string
/// \param[in] pageLockType The page lock type.
/// \return The string.
extern LINKAGE const char *BtrievePageLockTypeToString(btrieve_page_lock_type_t pageLockType);

/// \brief Return a %Btrieve page size as a string
/// \param[in] pageSize The page size.
/// \return The string.
extern LINKAGE const char *BtrievePageSizeToString(btrieve_page_size_t pageSize);

/// \brief Return a %Btrieve record compression mode as a string
/// \param[in] recordCompressionMode The record compression mode.
/// \return The string.
extern LINKAGE const char *BtrieveRecordCompressionModeToString(btrieve_record_compression_mode_t recordCompressionMode);

/// \brief Return a %Btrieve status code as a string
/// \param[in] statusCode The status code.
/// \return The string.
extern LINKAGE const char *BtrieveStatusCodeToString(btrieve_status_code_t statusCode);

/// \brief Return a %Btrieve system data mode as a string
/// \param[in] systemDataMode The system data mode.
/// \return The string.
extern LINKAGE const char *BtrieveSystemDataModeToString(btrieve_system_data_mode_t systemDataMode);

/// \brief Return a %Btrieve variable length records mode as a string
/// \param[in] variableLengthRecordsMode The variable length records mode.
/// \return The string.
extern LINKAGE const char *BtrieveVariableLengthRecordsModeToString(btrieve_variable_length_records_mode_t variableLengthRecordsMode);

/// \brief Return a %Btrieve version type as a string
/// \param[in] versionType The version type. See BtrieveClientGetVersion().
/// \return The string.
extern LINKAGE const char *BtrieveVersionTypeToString(btrieve_version_type_t versionType);

/// \brief Return a %Btrieve system field as a string
/// \param[in] systemField The system field.
/// \return The string.
extern LINKAGE const char *BtrieveSystemFieldToString(btrieve_system_field_t systemField);

/// \brief Get record count from a %Btrieve bulk create result  
/// \param[in] bulkCreateResult The bulk create result. See BtrieveFileBulkCreate().
/// \retval ">= 0" The record count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveBulkCreateResultGetLastStatusCode()}
extern LINKAGE int BtrieveBulkCreateResultGetRecordCount(btrieve_bulk_create_result_t bulkCreateResult);

/// \brief Get record cursor position  from a %Btrieve bulk create result  
/// \param[in] bulkCreateResult The bulk create result. See BtrieveFileBulkCreate().
/// \param[in] recordNumber The record number.
/// \retval ">= 0" The record cursor position .
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveBulkCreateResultGetLastStatusCode()}
extern LINKAGE long long BtrieveBulkCreateResultGetRecordCursorPosition(btrieve_bulk_create_result_t bulkCreateResult, int recordNumber);

/// \brief Get record from a %Btrieve bulk retrieve result  
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \param[in] bulkRetrieveResult The bulk retrieve result. See BtrieveFileBulkRetrieveNext() and BtrieveFileBulkRetrievePrevious().
/// \param[in] recordNumber The record number.
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveBulkRetrieveResultGetLastStatusCode()}
extern LINKAGE int BtrieveBulkRetrieveResultGetRecord(btrieve_bulk_retrieve_result_t bulkRetrieveResult, int recordNumber, char *record, int recordSize);

/// \brief Get record count from a %Btrieve bulk retrieve result  
/// \param[in] bulkRetrieveResult The bulk retrieve result. See BtrieveFileBulkRetrieveNext() and BtrieveFileBulkRetrievePrevious().
/// \retval ">= 0" The record count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveBulkRetrieveResultGetLastStatusCode()}
extern LINKAGE int BtrieveBulkRetrieveResultGetRecordCount(btrieve_bulk_retrieve_result_t bulkRetrieveResult);

/// \brief Get record count from a %Btrieve bulk delete result  
/// \param[in] bulkDeleteResult The bulk delete result. See BtrieveFileBulkDeleteNext() and BtrieveFileBulkDeletePrevious().
/// \retval ">= 0" The record count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveBulkDeleteResultGetLastStatusCode()}
extern LINKAGE int BtrieveBulkDeleteResultGetRecordCount(btrieve_bulk_delete_result_t bulkDeleteResult);

/// \brief Get record cursor position  from a %Btrieve bulk retrieve result  
/// \param[in] bulkRetrieveResult The bulk retrieve result. See BtrieveFileBulkRetrieveNext() and BtrieveFileBulkRetrievePrevious().
/// \param[in] recordNumber The record number.
/// \retval ">= 0" The record cursor position .
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveBulkRetrieveResultGetLastStatusCode()}
extern LINKAGE long long BtrieveBulkRetrieveResultGetRecordCursorPosition(btrieve_bulk_retrieve_result_t bulkRetrieveResult, int recordNumber);

/// \brief Get record cursor position  from a %Btrieve bulk delete result  
/// \param[in] bulkDeleteResult The bulk delete result. See BtrieveFileBulkDeleteNext() and BtrieveFileBulkDeletePrevious().
/// \param[in] recordNumber The record number.
/// \retval ">= 0" The record cursor position .
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveBulkDeleteResultGetLastStatusCode()}
extern LINKAGE long long BtrieveBulkDeleteResultGetRecordCursorPosition(btrieve_bulk_delete_result_t bulkDeleteResult, int recordNumber);

/// \brief Get record length from a %Btrieve bulk retrieve result  
/// \param[in] bulkRetrieveResult The bulk retrieve result. See BtrieveFileBulkRetrieveNext() and BtrieveFileBulkRetrievePrevious().
/// \param[in] recordNumber The record number.
/// \retval ">= 0" The record length.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveBulkRetrieveResultGetLastStatusCode()}
extern LINKAGE int BtrieveBulkRetrieveResultGetRecordLength(btrieve_bulk_retrieve_result_t bulkRetrieveResult, int recordNumber);

/// \brief Get client identifier from a %Btrieve client  
/// \param[in] client The client. See BtrieveClientAllocate().
/// \retval ">= 0" The client identifier.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveClientGetLastStatusCode()}
extern LINKAGE int BtrieveClientGetClientIdentifier(btrieve_client_t client);

/// \brief Get service agent identifier from a %Btrieve client  
/// \param[in] client The client. See BtrieveClientAllocate().
/// \retval ">= 0" The service agent identifier.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveClientGetLastStatusCode()}
extern LINKAGE int BtrieveClientGetServiceAgentIdentifier(btrieve_client_t client);

/// \brief Get cursor position from a %Btrieve file  
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval ">= 0" The cursor position.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE long long BtrieveFileGetCursorPosition(btrieve_file_t file);

/// \brief Get record create time from a %Btrieve file
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "> 0" The record create time as a %Btrieve timestamp.
/// \retval "= 0" The record create time is unavailable. Ensure the file was created with system data. See BtrieveFileAttributesSetSystemDataMode().
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
/// \see BtrieveTimestampToUnixEpochMicroseconds
extern LINKAGE long long BtrieveFileGetRecordCreateTime(btrieve_file_t file);

/// \brief Get record update time from a %Btrieve file
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval "> 0" The record update time as a %Btrieve timestamp.
/// \retval "= 0" The record update time is unavailable. Ensure the file was created with system data version 2. See BtrieveFileAttributesSetSystemDataMode().
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
/// \see BtrieveTimestampToUnixEpochMicroseconds
extern LINKAGE long long BtrieveFileGetRecordUpdateTime(btrieve_file_t file);

/// \brief Get numerator from a %Btrieve file  
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] key The key.
/// \param[in] keyLength The key length.
/// \param[in] cursorPosition The cursor position.
/// \param[in] denominator The denominator.
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval ">= 0" The numerator.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileGetNumerator(btrieve_file_t file, btrieve_index_t index, const char *key, int keyLength, long long cursorPosition, int denominator);

/// \brief Get percentage from a %Btrieve file  
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] key The key.
/// \param[in] keyLength The key length.
/// \param[in] cursorPosition The cursor position.
/// \par \CURSOR
/// The cursor is unaffected.
/// \par \CURRENT_OFFSET
/// The current offset is unaffected.
/// \retval ">= 0" The percentage.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileGetPercentage(btrieve_file_t file, btrieve_index_t index, const char *key, int keyLength, long long cursorPosition);

/// \brief Get balanced indexes from a %Btrieve file information  
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Balanced indexes enabled.
/// \retval 0 Balanced indexes disabled.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetBalancedIndexes(btrieve_file_information_t fileInformation);

/// \brief Get client transactions from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Client transactions are active.
/// \retval 0 Client transactions are not active.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetClientTransactions(btrieve_file_information_t fileInformation);

/// \brief Get continuous operation from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The file is in continuous operation.
/// \retval 0 The file isn't in continuous operation.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetContinuousOperation(btrieve_file_information_t fileInformation);

/// \brief Get duplicate record conflict cursor position from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The duplicate record conflict cursor position.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE long long BtrieveFileInformationGetDuplicateRecordConflictCursorPosition(btrieve_file_information_t fileInformation);

/// \brief Get explicit locks from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Explicit locks are active.
/// \retval 0 Explicit locks are not active.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetExplicitLocks(btrieve_file_information_t fileInformation);

/// \brief Get fixed record length from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The fixed record length.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetFixedRecordLength(btrieve_file_information_t fileInformation);

/// \brief Get gateway major version from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The gateway major version.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetGatewayMajorVersion(btrieve_file_information_t fileInformation);

/// \brief Get gateway minor version from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The gateway minor version.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetGatewayMinorVersion(btrieve_file_information_t fileInformation);

/// \brief Get gateway patch level from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The gateway patch level.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetGatewayPatchLevel(btrieve_file_information_t fileInformation);

/// \brief Get gateway platform from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The gateway platform.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetGatewayPlatform(btrieve_file_information_t fileInformation);

/// \brief Get handle count from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The handle count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetHandleCount(btrieve_file_information_t fileInformation);

/// \brief Get identifier from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The identifier.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetIdentifier(btrieve_file_information_t fileInformation);

/// \brief Get index count from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The index count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetIndexCount(btrieve_file_information_t fileInformation);

/// \brief Get key only from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" File is key only.
/// \retval 0 File isn't key only.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetKeyOnly(btrieve_file_information_t fileInformation);

/// \brief Get lock owner client identifier from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The lock owner client identifier.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerClientIdentifier(btrieve_file_information_t fileInformation);

/// \brief Get lock owner file lock from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The lock owner has a file lock.
/// \retval 0 The lock owner doesn't have a file lock.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerFileLock(btrieve_file_information_t fileInformation);

/// \brief Get lock owner implicit lock from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The lock owner has an implicit lock.
/// \retval 0 The lock owner doesn't have an implicit lock.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerImplicitLock(btrieve_file_information_t fileInformation);

/// \brief Get lock owner record lock from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The lock owner has a record lock.
/// \retval 0 The lock owner doesn't have a record lock.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerRecordLock(btrieve_file_information_t fileInformation);

/// \brief Get lock owner same process from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The lock owner is the same process.
/// \retval 0 The lock owner isn't the same process.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerSameProcess(btrieve_file_information_t fileInformation);

/// \brief Get lock owner service agent identifier from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The lock owner service agent identifier.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerServiceAgentIdentifier(btrieve_file_information_t fileInformation);

/// \brief Get lock owner time in transaction from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The lock owner time in transaction.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerTimeInTransaction(btrieve_file_information_t fileInformation);

/// \brief Get lock owner transaction level from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The lock owner transaction level.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerTransactionLevel(btrieve_file_information_t fileInformation);

/// \brief Get lock owner write hold from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The lock owner has a write hold.
/// \retval 0 The lock owner doesn't have a write hold.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerWriteHold(btrieve_file_information_t fileInformation);

/// \brief Get lock owner write no wait from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The lock owner is performing a write with no wait.
/// \retval 0 The lock owner isn't performing a write with no wait.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLockOwnerWriteNoWait(btrieve_file_information_t fileInformation);

/// \brief Get loggable from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" File is loggable.
/// \retval 0 File isn't loggable.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetLoggable(btrieve_file_information_t fileInformation);

/// \brief Get open timestamp from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The open timestamp.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetOpenTimestamp(btrieve_file_information_t fileInformation);

/// \brief Get page compression from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Page compression enabled.
/// \retval 0 Page compression disabled.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetPageCompression(btrieve_file_information_t fileInformation);

/// \brief Get page preallocation from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Page preallocation enabled.
/// \retval 0 Page preallocation disabled.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetPagePreallocation(btrieve_file_information_t fileInformation);

/// \brief Get read only from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The file is opened read only.
/// \retval 0 The file isn't opened read only.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetReadOnly(btrieve_file_information_t fileInformation);

/// \brief Get record count from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The record count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE long long BtrieveFileInformationGetRecordCount(btrieve_file_information_t fileInformation);

/// \brief Get referential integrity constraints from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The file has referential integrity constraints.
/// \retval 0 The file doesn't have referential integrity constraints.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetReferentialIntegrityConstraints(btrieve_file_information_t fileInformation);

/// \brief Get referential integrity cursor position from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The referential integrity cursor position.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE long long BtrieveFileInformationGetReferentialIntegrityCursorPosition(btrieve_file_information_t fileInformation);

/// \brief Get referential integrity operation code from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The referential integrity operation code.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetReferentialIntegrityOperationCode(btrieve_file_information_t fileInformation);

/// \brief Get security current database authentication by database from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database authentication by database.
/// \retval 0 Security current database not authentication by database.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseAuthenticationByDatabase(btrieve_file_information_t fileInformation);

/// \brief Get security current database authorization by database from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database authorization by database.
/// \retval 0 Security current database not authorization by database.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseAuthorizationByDatabase(btrieve_file_information_t fileInformation);

/// \brief Get security current database btpasswd from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database Btpasswd.
/// \retval 0 Security current database not Btpasswd.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseBtpasswd(btrieve_file_information_t fileInformation);

/// \brief Get security current database explicit from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database explicit.
/// \retval 0 Security current database not explicit.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseExplicit(btrieve_file_information_t fileInformation);

/// \brief Get security current database implicit from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database implicit.
/// \retval 0 Security current database not implicit.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseImplicit(btrieve_file_information_t fileInformation);

/// \brief Get security current database PAM from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database PAM.
/// \retval 0 Security current database not PAM.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabasePAM(btrieve_file_information_t fileInformation);

/// \brief Get security current database RTSS complete from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database RTSS complete.
/// \retval 0 Security current database not RTSS complete.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseRTSSComplete(btrieve_file_information_t fileInformation);

/// \brief Get security current database RTSS disabled from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database RTSS disabled.
/// \retval 0 Security current database not RTSS disabled.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseRTSSDisabled(btrieve_file_information_t fileInformation);

/// \brief Get security current database RTSS preauthorized from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database RTSS preauthorized.
/// \retval 0 Security current database not RTSS preauthorized.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseRTSSPreauthorized(btrieve_file_information_t fileInformation);

/// \brief Get security current database trusted from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database trusted.
/// \retval 0 Security current database not trusted.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseTrusted(btrieve_file_information_t fileInformation);

/// \brief Get security current database windows named pipe from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database Windows named pipe.
/// \retval 0 Security current database not Windows named pipe.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseWindowsNamedPipe(btrieve_file_information_t fileInformation);

/// \brief Get security current database workgroup from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security current database workgroup.
/// \retval 0 Security current database not workgroup.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityCurrentDatabaseWorkgroup(btrieve_file_information_t fileInformation);

/// \brief Get security handle authentication by database from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle authentication by database.
/// \retval 0 Security handle not authentication by database.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleAuthenticationByDatabase(btrieve_file_information_t fileInformation);

/// \brief Get security handle authorization by database from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle authorization by database.
/// \retval 0 Security handle not authorization by database.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleAuthorizationByDatabase(btrieve_file_information_t fileInformation);

/// \brief Get security handle btpasswd from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle Btpasswd.
/// \retval 0 Security handle not Btpasswd.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleBtpasswd(btrieve_file_information_t fileInformation);

/// \brief Get security handle explicit from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle explicit.
/// \retval 0 Security handle not explicit.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleExplicit(btrieve_file_information_t fileInformation);

/// \brief Get security handle implicit from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle implicit.
/// \retval 0 Security handle not implicit.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleImplicit(btrieve_file_information_t fileInformation);

/// \brief Get security handle PAM from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle PAM.
/// \retval 0 Security handle not PAM.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandlePAM(btrieve_file_information_t fileInformation);

/// \brief Get security handle RTSS complete from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle RTSS complete.
/// \retval 0 Security handle not RTSS complete.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleRTSSComplete(btrieve_file_information_t fileInformation);

/// \brief Get security handle RTSS disabled from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle RTSS disabled.
/// \retval 0 Security handle not RTSS disabled.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleRTSSDisabled(btrieve_file_information_t fileInformation);

/// \brief Get security handle RTSS preauthorized from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle RTSS preautorized.
/// \retval 0 Security handle not RTSS preautorized.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleRTSSPreauthorized(btrieve_file_information_t fileInformation);

/// \brief Get security handle trusted from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle trusted.
/// \retval 0 Security handle not trusted.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleTrusted(btrieve_file_information_t fileInformation);

/// \brief Get security handle windows named pipe from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle Windows named pipe.
/// \retval 0 Security handle not Windows named pipe.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleWindowsNamedPipe(btrieve_file_information_t fileInformation);

/// \brief Get security handle workgroup from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security handle workgroup.
/// \retval 0 Security handle not workgroup.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityHandleWorkgroup(btrieve_file_information_t fileInformation);

/// \brief Get security permission alter from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission alter.
/// \retval 0 Security permission no alter.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionAlter(btrieve_file_information_t fileInformation);

/// \brief Get security permission create file from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission file create.
/// \retval 0 Security permission no file create.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionCreateFile(btrieve_file_information_t fileInformation);

/// \brief Get security permission create record from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission record create.
/// \retval 0 Security permission no record create.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionCreateRecord(btrieve_file_information_t fileInformation);

/// \brief Get security permission create stored procedure from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission create stored procedure.
/// \retval 0 Security permission no create stored procedure.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionCreateStoredProcedure(btrieve_file_information_t fileInformation);

/// \brief Get security permission create view from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission create view.
/// \retval 0 Security permission no create view.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionCreateView(btrieve_file_information_t fileInformation);

/// \brief Get security permission delete from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission delete.
/// \retval 0 Security permission no delete.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionDelete(btrieve_file_information_t fileInformation);

/// \brief Get security permission execute from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission execute.
/// \retval 0 Security permission no execute.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionExecute(btrieve_file_information_t fileInformation);

/// \brief Get security permission no rights from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission no rights.
/// \retval 0 Security permission rights.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionNoRights(btrieve_file_information_t fileInformation);

/// \brief Get security permission open from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission open.
/// \retval 0 Security permission no open.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionOpen(btrieve_file_information_t fileInformation);

/// \brief Get security permission read from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission read.
/// \retval 0 Security permission no read.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionRead(btrieve_file_information_t fileInformation);

/// \brief Get security permission refer from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission refer.
/// \retval 0 Security permission no refer.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionRefer(btrieve_file_information_t fileInformation);

/// \brief Get security permission update from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" Security permission update.
/// \retval 0 Security permission no update.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSecurityPermissionUpdate(btrieve_file_information_t fileInformation);

/// \brief Get segment count from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The segment count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSegmentCount(btrieve_file_information_t fileInformation);

/// \brief Get segmented from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" File is segmented.
/// \retval 0 File isn't segmented.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSegmented(btrieve_file_information_t fileInformation);

/// \brief Get system data from a %Btrieve file information 
/// \note Renamed from BtrieveFileInformationGetSystemIndexPresent() in Zen v15 R1.
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval 2 System data version 2 is present.
/// \retval 1 System data is present.
/// \retval 0 System data isn't present.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSystemData(btrieve_file_information_t fileInformation);

/// \brief Get system index size from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The system index size.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSystemIndexSize(btrieve_file_information_t fileInformation);

/// \brief Get system index from a %Btrieve file information 
/// \note Renamed from BtrieveFileInformationGetSystemIndexUsed() in Zen v15 R1.
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval 3 System indexes BTRIEVE_INDEX_SYSTEM and BTRIEVE_INDEX_SYSTEM_VERSION_2 are present.
/// \retval 2 System index BTRIEVE_INDEX_SYSTEM_VERSION_2 is present.
/// \retval 1 System index BTRIEVE_INDEX_SYSTEM is present.
/// \retval 0 No system index is present.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSystemIndex(btrieve_file_information_t fileInformation);

/// \brief Get system index version from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The system index version.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetSystemIndexVersion(btrieve_file_information_t fileInformation);

/// \brief Get unused duplicate pointer count from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The unused duplicate pointer count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetUnusedDuplicatePointerCount(btrieve_file_information_t fileInformation);

/// \brief Get unused page count from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The unused page count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetUnusedPageCount(btrieve_file_information_t fileInformation);

/// \brief Get usage count from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval ">= 0" The usage count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetUsageCount(btrieve_file_information_t fileInformation);

/// \brief Get wrong owner from a %Btrieve file information 
/// \param[in] fileInformation The file information. See BtrieveFileGetInformation().
/// \retval "> 0" The file was opened with an incorrect owner name.
/// \retval 0 The file wasn't opened with an incorrect owner name.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileInformationGetLastStatusCode()}
extern LINKAGE int BtrieveFileInformationGetWrongOwner(btrieve_file_information_t fileInformation);

/// \brief Record retrieve for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] comparison The comparison.
/// \param[in] index The index.
/// \param[in] key The key.
/// \param[in] keyLength The key length.
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \param[in] lockMode The lock mode.
/// \par \SNIPPET
/// \snippet btest.c FileRecordRetrieve
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C \DATA_LENGTH_ERROR_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrieve(btrieve_file_t file, btrieve_comparison_t comparison, btrieve_index_t index, char *key, int keyLength, char *record, int recordSize, btrieve_lock_mode_t lockMode);

/// \brief Record retrieve by cursor position for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] cursorPosition The cursor position.
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C \DATA_LENGTH_ERROR_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrieveByCursorPosition(btrieve_file_t file, btrieve_index_t index, long long cursorPosition, char *key, int keySize, char *record, int recordSize, btrieve_lock_mode_t lockMode);

/// \brief Record retrieve by fraction for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] numerator The numerator.
/// \param[in] denominator The denominator.
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C \DATA_LENGTH_ERROR_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrieveByFraction(btrieve_file_t file, btrieve_index_t index, int numerator, int denominator, char *key, int keySize, char *record, int recordSize, btrieve_lock_mode_t lockMode);

/// \brief Record retrieve by percentage for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[in] percentage The percentage.
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C \DATA_LENGTH_ERROR_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrieveByPercentage(btrieve_file_t file, btrieve_index_t index, int percentage, char *key, int keySize, char *record, int recordSize, btrieve_lock_mode_t lockMode);

/// \brief Record retrieve chunk for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] offset The offset. If offset is -1, then the current offset is used.
/// \param[in] length The length.
/// \param[in] lockMode The lock mode.
/// \param[out] chunk The chunk.
/// \param[in] chunkSize The chunk size.
/// \par \CURSOR
/// The cursor is unaffected but must be established.
/// \par \CURRENT_OFFSET
/// The current offset is established at the end of the retrieved chunk.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ You may retrieve multiple chunks.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrieveChunk(btrieve_file_t file, int offset, int length, char *chunk, int chunkSize, btrieve_lock_mode_t lockMode);

/// \brief Record retrieve first for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C \DATA_LENGTH_ERROR_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrieveFirst(btrieve_file_t file, btrieve_index_t index, char *key, int keySize, char *record, int recordSize, btrieve_lock_mode_t lockMode); 

/// \brief Record retrieve last for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[in] index The index.
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor is established.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C \DATA_LENGTH_ERROR_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrieveLast(btrieve_file_t file, btrieve_index_t index, char *key, int keySize, char *record, int recordSize, btrieve_lock_mode_t lockMode); 

/// \brief Record retrieve next for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor must be established. The cursor is traversed forward one record.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C \DATA_LENGTH_ERROR_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrieveNext(btrieve_file_t file, char *key, int keySize, char *record, int recordSize, btrieve_lock_mode_t lockMode);

/// \brief Record retrieve previous for a %Btrieve file 
/// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE_C
/// \note The declaration of this function changed in PSQL v15 R0.
/// \param[out] key The key.
/// \param[in] keySize The key size.
/// \param[in] file The file. See BtrieveClientFileOpen().
/// \param[out] record The record.
/// \param[in] recordSize The record size.
/// \param[in] lockMode The lock mode.
/// \par \CURSOR
/// The cursor must be established. The cursor is traversed backward one record.
/// \par \CURRENT_OFFSET
/// The current offset is destroyed.
/// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED_C \RETRIEVING_RECORDS_GREATER_THAN_C \DATA_LENGTH_ERROR_C
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveFileGetLastStatusCode()}
extern LINKAGE int BtrieveFileRecordRetrievePrevious(btrieve_file_t file, char *key, int keySize, char *record, int recordSize, btrieve_lock_mode_t lockMode);

/// \brief Get ACS number from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The ACS number.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetACSNumber(btrieve_key_segment_t keySegment);

/// \brief Get descending sort order from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "> 0" Descending sort order is set.
/// \retval 0 Descending sort order isn't set.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetDescendingSortOrder(btrieve_key_segment_t keySegment);

/// \brief Get key continues from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "> 0" Key continues is set.
/// \retval 0 Key continues isn't set.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetKeyContinues(btrieve_key_segment_t keySegment);

/// \brief Get length from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The length.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetLength(btrieve_key_segment_t keySegment);

/// \brief Get modifiable from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval "> 0" Modifications are allowed.
/// \retval 0 Modifications are not allowed.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetModifiable(btrieve_key_segment_t keySegment);

/// \brief Get null all segments from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The null all segments.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetNullAllSegments(btrieve_key_segment_t keySegment);

/// \brief Get null any segment from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The null any segment.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetNullAnySegment(btrieve_key_segment_t keySegment);

/// \brief Get null value from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The null value.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetNullValue(btrieve_key_segment_t keySegment);

/// \brief Get offset from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The offset.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetOffset(btrieve_key_segment_t keySegment);

/// \brief Get segmented from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The segmented.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetSegmented(btrieve_key_segment_t keySegment);

/// \brief Get unique value count from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The unique value count.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE long long BtrieveKeySegmentGetUniqueValueCount(btrieve_key_segment_t keySegment);

/// \brief Get use numbered ACS from a %Btrieve key segment 
/// \param[in] keySegment The key segment. See BtrieveFileInformationGetKeySegment().
/// \retval ">= 0" The use numbered ACS.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveKeySegmentGetLastStatusCode()}
extern LINKAGE int BtrieveKeySegmentGetUseNumberedACS(btrieve_key_segment_t keySegment);

/// \brief Get client revision number from a %Btrieve version 
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval ">= 0" The client revision number.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE int BtrieveVersionGetClientRevisionNumber(btrieve_version_t version);

/// \brief Get client version number from a %Btrieve version 
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval ">= 0" The client version number.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE int BtrieveVersionGetClientVersionNumber(btrieve_version_t version);

/// \brief Get local revision number from a %Btrieve version 
/// \param[in] version The version.  See BtrieveClientGetVersion().
/// \retval ">= 0" The local revision number.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE int BtrieveVersionGetLocalRevisionNumber(btrieve_version_t version);

/// \brief Get local version number from a %Btrieve version 
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval ">= 0" The local version number.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE int BtrieveVersionGetLocalVersionNumber(btrieve_version_t version);

/// \brief Get remote revision number from a %Btrieve version 
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval ">= 0" The remote revision number.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE int BtrieveVersionGetRemoteRevisionNumber(btrieve_version_t version);

/// \brief Get remote version number from a %Btrieve version 
/// \param[in] version The version. See BtrieveClientGetVersion().
/// \retval ">= 0" The remote version number.
/// \retval -1 \ERROR_HAS_OCCURRED
/// \TO_DETERMINE_STATUS_CODE_C{BtrieveVersionGetLastStatusCode()}
extern LINKAGE int BtrieveVersionGetRemoteVersionNumber(btrieve_version_t version);

/// \brief Convert a Unix Epoch based timestamp in microseconds to a %Btrieve TIMESTAMP.
/// \param[in] microseconds The Unix Epoch timestamp in microseconds granularity.
/// \return The %Btrieve TIMESTAMP.
extern LINKAGE long long BtrieveUnixEpochMicrosecondsToTimestamp(long long microseconds);

/// \brief Convert a %Btrieve TIMESTAMP to a Unix Epoch timestamp in microseconds.
/// \param[in] timestamp The %Btrieve TIMESTAMP.
/// \return The Epoch based timestamp in microseconds.
extern LINKAGE long long BtrieveTimestampToUnixEpochMicroseconds(long long timestamp);

#ifdef __cplusplus
}
#endif

#endif
