////////////////////////////////////////////////////////////
//
// Copyright (C) 2016 - Actian Corp.
//
////////////////////////////////////////////////////////////

// This file exists in both project "zen.core" and "zen.docs".
// Make sure that both files are kept in sync.
//
// When updating this file make the corresponding changes to "btrieveC.h".

#ifndef _BTRIEVECPP_H
#define _BTRIEVECPP_H

#include <cstddef>
#include <deque>

#include "btrieveC.h"

class BRestful;
class BtrieveBulkCreatePayload;
class BtrieveBulkCreateResult;
class BtrieveBulkDeleteAttributes;
class BtrieveBulkDeleteResult;
class BtrieveBulkRetrieveAttributes;
class BtrieveBulkRetrieveResult;
class BtrieveClient;
class BtrieveCollection;
class BtrieveDocumentSet;
class BtrieveFile;
class BtrieveFileAttributes;
class BtrieveFileInformation;
class BtrieveFilter;
class BtrieveIndexAttributes;
class BtrieveKeySegment;
class BtrieveVersion;
class BtrieveSpace;
class BtrieveVectorSet;

/// \brief The %Btrieve class.
class LINKAGE Btrieve
{
public:
   /// \brief The %Btrieve maximum record length.
   static const int MAXIMUM_RECORD_LENGTH = BTRIEVE_MAXIMUM_RECORD_LENGTH;

   /// \brief The %Btrieve maximum key segment length.
   static const int MAXIMUM_KEY_LENGTH = BTRIEVE_MAXIMUM_KEY_LENGTH;

   /// \brief The %Btrieve maximum bulk record count.
   static const int MAXIMUM_BULK_RECORD_COUNT = BTRIEVE_MAXIMUM_BULK_RECORD_COUNT;

   /// \brief The %Btrieve maximum space dimensions.
   static const int MAXIMUM_SPACE_DIMENSIONS = 8192;

   /// \brief The %Btrieve alternate collation sequence modes.
   /// \see Btrieve::ACSModeToString
   enum ACSMode {
      /// \brief None.
      ACS_MODE_NONE = 0,
      /// \brief Named.
      ACS_MODE_NAMED = 1,
      /// \brief Numbered.
      ACS_MODE_NUMBERED = 2,
      /// \brief Case insensitive.
      ACS_MODE_CASE_INSENSITIVE = 3,
      /// \brief Default.
      ACS_MODE_DEFAULT = 4,
      /// \brief Unknown.
      ACS_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve create modes.
   /// \SNIPPET_CPP_PYTHON{CreateMode, btest.cpp, btest.py}
   enum CreateMode {
      /// \brief Overwrite.
      CREATE_MODE_OVERWRITE = 0,
      /// \brief No overwrite.
      CREATE_MODE_NO_OVERWRITE = 1,
      /// \brief Unknown.
      CREATE_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve comparisons.
   /// \SNIPPET_CPP_PYTHON{Comparison, btestbulk.cpp, btestbulk.py}
   enum Comparison {
      /// \brief None.
      COMPARISON_NONE = 0,
      /// \brief Equal.
      COMPARISON_EQUAL = 1,
      /// \brief Greater than.
      COMPARISON_GREATER_THAN = 2,
      /// \brief Less than.
      COMPARISON_LESS_THAN = 3,
      /// \brief Not equal.
      COMPARISON_NOT_EQUAL = 4,
      /// \brief Greater than or equal.
      COMPARISON_GREATER_THAN_OR_EQUAL = 5,
      /// \brief Less than or equal.
      COMPARISON_LESS_THAN_OR_EQUAL = 6,
      /// \brief Like.
      COMPARISON_LIKE = 7,
      /// \brief Not like.
      COMPARISON_NOT_LIKE = 8,
      /// \brief JSON filter query match.
      COMPARISON_JSON_QUERY_MATCH = 9,
      /// \brief Unknown.
      COMPARISON_UNKNOWN = -7
   };

   /// \brief The %Btrieve connectors.
   /// \SNIPPET_CPP_PYTHON{Connector, bfilter.cpp, bfilter.py}
   enum Connector {
      /// \brief Last.
      CONNECTOR_LAST = 0,
      /// \brief And.
      CONNECTOR_AND = 1,
      /// \brief Or.
      CONNECTOR_OR = 2,
      /// \brief Unknown.
      CONNECTOR_UNKNOWN = -7
   };


   /// \brief The %Btrieve disk drives.
   enum DiskDrive {
      /// \brief Disk drive A.
      DISK_DRIVE_A = 0, 
      /// \brief Disk drive B.
      DISK_DRIVE_B = 1, 
      /// \brief Disk drive C.
      DISK_DRIVE_C = 2,
      /// \brief Disk drive D.
      DISK_DRIVE_D = 3, 
      /// \brief Disk drive E.
      DISK_DRIVE_E = 4, 
      /// \brief Disk drive F.
      DISK_DRIVE_F = 5,
      /// \brief Disk drive G.
      DISK_DRIVE_G = 6, 
      /// \brief Disk drive H.
      DISK_DRIVE_H = 7, 
      /// \brief Disk drive I.
      DISK_DRIVE_I = 8,
      /// \brief Disk drive J.
      DISK_DRIVE_J = 9, 
      /// \brief Disk drive K.
      DISK_DRIVE_K = 10, 
      /// \brief Disk drive L.
      DISK_DRIVE_L = 11,
      /// \brief Disk drive M.
      DISK_DRIVE_M = 12, 
      /// \brief Disk drive N.
      DISK_DRIVE_N = 13, 
      /// \brief Disk drive O.
      DISK_DRIVE_O = 14,
      /// \brief Disk drive P.
      DISK_DRIVE_P = 15, 
      /// \brief Disk drive Q.
      DISK_DRIVE_Q = 16, 
      /// \brief Disk drive R.
      DISK_DRIVE_R = 17,
      /// \brief Disk drive S.
      DISK_DRIVE_S = 18, 
      /// \brief Disk drive T.
      DISK_DRIVE_T = 19, 
      /// \brief Disk drive U.
      DISK_DRIVE_U = 20,
      /// \brief Disk drive V.
      DISK_DRIVE_V = 21, 
      /// \brief Disk drive W.
      DISK_DRIVE_W = 22, 
      /// \brief Disk drive X.
      DISK_DRIVE_X = 23,
      /// \brief Disk drive Y.
      DISK_DRIVE_Y = 24, 
      /// \brief Disk drive Z.
      DISK_DRIVE_Z = 25, 
      /// \brief Default.
      DISK_DRIVE_DEFAULT = 26,
      /// \brief Unknown.
      DISK_DRIVE_UNKNOWN = -7
   };

   /// \brief The %Btrieve duplicate modes.
   /// \SNIPPET_CPP_PYTHON{DuplicateMode, bindexattributes.cpp, bindexattributes.py}
   /// \see Btrieve::DuplicateModeToString
   enum DuplicateMode {
      /// \brief Duplicates not allowed.
      DUPLICATE_MODE_NOT_ALLOWED = 0,
      /// \brief Duplicates allowed, nonrepeating.
      DUPLICATE_MODE_ALLOWED_NONREPEATING = 1,
      /// \brief Duplicates allowed, repeating.
      DUPLICATE_MODE_ALLOWED_REPEATING = 2,
      /// \brief Unknown.
      DUPLICATE_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve data types.
   /// \SNIPPET_CPP_PYTHON{DataType, btest.cpp, btest.py}
   /// \see Btrieve::DataTypeToString
   enum DataType {
      /// \brief Character.
      DATA_TYPE_CHAR = 0,
      /// \brief Null terminated string.
      DATA_TYPE_ZSTRING = 1,
      /// \brief Integer.
      DATA_TYPE_INTEGER = 2,
      /// \brief Unsigned binary.
      DATA_TYPE_UNSIGNED_BINARY = 3,
      /// \brief Float.
      DATA_TYPE_FLOAT = 4,
      /// \brief Automatic increment.
      DATA_TYPE_AUTOINCREMENT = 5,
      /// \brief Date.
      DATA_TYPE_DATE = 6,
      /// \brief Numeric sign trailing separate.
      DATA_TYPE_NUMERICSTS = 7,
      /// \brief Time.
      DATA_TYPE_TIME = 8,
      /// \brief Numeric signed ASCII.
      DATA_TYPE_NUMERICSA = 9,
      /// \brief Decimal.
      DATA_TYPE_DECIMAL = 10,
      /// \brief Currency.
      DATA_TYPE_CURRENCY = 11,
      /// \brief Money.
      DATA_TYPE_MONEY = 12,
      /// \brief Timestamp.
      DATA_TYPE_TIMESTAMP = 13,
      /// \brief Logical.
      DATA_TYPE_LOGICAL = 14,
      /// \brief Wide string.
      DATA_TYPE_WSTRING = 15,
      /// \brief Numeric.
      DATA_TYPE_NUMERIC = 16,
      /// \brief Wide null terminated string.
      DATA_TYPE_WZSTRING = 17,
      /// \brief BASIC float.
      DATA_TYPE_BFLOAT = 18,
      /// \brief Globally unique identifier.
      DATA_TYPE_GUID = 19,
      /// \brief Length first string.
      DATA_TYPE_LSTRING = 20,
      /// \brief Null indicator segment.
      DATA_TYPE_NULL_INDICATOR_SEGMENT = 21,
      /// \brief Legacy string.
      DATA_TYPE_LEGACY_STRING = 22,
      /// \brief Legacy binary.
      DATA_TYPE_LEGACY_BINARY = 23,
      /// \brief Automatic timestamp.
      DATA_TYPE_AUTOTIMESTAMP = 24,
      /// \brief Character large object.
      DATA_TYPE_CLOB = 25,
      /// \brief Nanosecond resolution timestamp.
      DATA_TYPE_TIMESTAMP2 = 26,
      /// \brief Bit.
      DATA_TYPE_BIT = 27,
      /// \brief Numeric sign leading COBOL.
      DATA_TYPE_NUMERICSLB = 28,
      /// \brief Numeric sign leading separate.
      DATA_TYPE_NUMERICSLS = 29,
      /// \brief Numeric sign trailing COBOL.
      DATA_TYPE_NUMERICSTB = 30,
      /// \brief Unknown.
      DATA_TYPE_UNKNOWN = -7 
   };

   /// \brief The %Btrieve file versions.
   /// \SNIPPET_CPP_PYTHON{FileVersion, bfileattributes.cpp, bfileattributes.py}
   /// \see Btrieve::FileVersionToString
   enum FileVersion {
      /// \brief File version 6.0.
      FILE_VERSION_6_0 = 0,
      /// \brief File version 6.1.
      FILE_VERSION_6_1 = 1,
      /// \brief File version 7.0.
      FILE_VERSION_7_0 = 2,
      /// \brief File version 8.0.
      FILE_VERSION_8_0 = 3,
      /// \brief File version 9.0.
      FILE_VERSION_9_0 = 4,
      /// \brief File version 9.5.
      FILE_VERSION_9_5 = 5,
      /// \brief File version 13.0.
      FILE_VERSION_13_0 = 7,
      /// \brief File version 16.0.
      FILE_VERSION_16_0 = 8,
      /// \brief Default.
      FILE_VERSION_DEFAULT = 6,
      /// \brief Unknown.
      FILE_VERSION_UNKNOWN = -7
   };

   /// \brief The %Btrieve free space thresholds.
   /// \SNIPPET_CPP_PYTHON{FreeSpaceThreshold, bfileattributes.cpp, bfileattributes.py}
   /// \see Btrieve::FreeSpaceThresholdToString
   enum FreeSpaceThreshold {
      /// \brief Free space threshold 10%.
      FREE_SPACE_THRESHOLD_10_PERCENT = 0,
      /// \brief Free space threshold 20%.
      FREE_SPACE_THRESHOLD_20_PERCENT = 1,
      /// \brief Free space threshold 30%.
      FREE_SPACE_THRESHOLD_30_PERCENT = 2,
      /// \brief Default.
      FREE_SPACE_THRESHOLD_DEFAULT = 3,
      /// \brief Unknown.
      FREE_SPACE_THRESHOLD_UNKNOWN = -7
   };

   /// \brief The %Btrieve indexes.
   /// \SNIPPET_CPP_PYTHON{Index, btest.cpp, btest.py}
   /// \see Btrieve::IndexToString
   enum Index {
      /// \brief Index 1
      INDEX_1 = 0,
      /// \brief Index 2
      INDEX_2 = 1,
      /// \brief Index 3
      INDEX_3 = 2,
      /// \brief Index 4
      INDEX_4 = 3,
      /// \brief Index 5
      INDEX_5 = 4,
      /// \brief Index 6
      INDEX_6 = 5,
      /// \brief Index 7
      INDEX_7 = 6,
      /// \brief Index 8
      INDEX_8 = 7,
      /// \brief Index 9
      INDEX_9 = 8,
      /// \brief Index 10
      INDEX_10 = 9,
      /// \brief Index 11
      INDEX_11 = 10,
      /// \brief Index 12
      INDEX_12 = 11,
      /// \brief Index 13
      INDEX_13 = 12,
      /// \brief Index 14
      INDEX_14 = 13,
      /// \brief Index 15
      INDEX_15 = 14,
      /// \brief Index 16
      INDEX_16 = 15,
      /// \brief Index 17
      INDEX_17 = 16,
      /// \brief Index 18
      INDEX_18 = 17,
      /// \brief Index 19
      INDEX_19 = 18,
      /// \brief Index 20
      INDEX_20 = 19,
      /// \brief Index 21
      INDEX_21 = 20,
      /// \brief Index 22
      INDEX_22 = 21,
      /// \brief Index 23
      INDEX_23 = 22,
      /// \brief Index 24
      INDEX_24 = 23,
      /// \brief Index 25
      INDEX_25 = 24,
      /// \brief Index 26
      INDEX_26 = 25,
      /// \brief Index 27
      INDEX_27 = 26,
      /// \brief Index 28
      INDEX_28 = 27,
      /// \brief Index 29
      INDEX_29 = 28,
      /// \brief Index 30
      INDEX_30 = 29,
      /// \brief Index 31
      INDEX_31 = 30,
      /// \brief Index 32
      INDEX_32 = 31,
      /// \brief Index 33
      INDEX_33 = 32,
      /// \brief Index 34
      INDEX_34 = 33,
      /// \brief Index 35
      INDEX_35 = 34,
      /// \brief Index 36
      INDEX_36 = 35,
      /// \brief Index 37
      INDEX_37 = 36,
      /// \brief Index 38
      INDEX_38 = 37,
      /// \brief Index 39
      INDEX_39 = 38,
      /// \brief Index 40
      INDEX_40 = 39,
      /// \brief Index 41
      INDEX_41 = 40,
      /// \brief Index 42
      INDEX_42 = 41,
      /// \brief Index 43
      INDEX_43 = 42,
      /// \brief Index 44
      INDEX_44 = 43,
      /// \brief Index 45
      INDEX_45 = 44,
      /// \brief Index 46
      INDEX_46 = 45,
      /// \brief Index 47
      INDEX_47 = 46,
      /// \brief Index 48
      INDEX_48 = 47,
      /// \brief Index 49
      INDEX_49 = 48,
      /// \brief Index 50
      INDEX_50 = 49,
      /// \brief Index 51
      INDEX_51 = 50,
      /// \brief Index 52
      INDEX_52 = 51,
      /// \brief Index 53
      INDEX_53 = 52,
      /// \brief Index 54
      INDEX_54 = 53,
      /// \brief Index 55
      INDEX_55 = 54,
      /// \brief Index 56
      INDEX_56 = 55,
      /// \brief Index 57
      INDEX_57 = 56,
      /// \brief Index 58
      INDEX_58 = 57,
      /// \brief Index 59
      INDEX_59 = 58,
      /// \brief Index 60
      INDEX_60 = 59,
      /// \brief Index 61
      INDEX_61 = 60,
      /// \brief Index 62
      INDEX_62 = 61,
      /// \brief Index 63
      INDEX_63 = 62,
      /// \brief Index 64
      INDEX_64 = 63,
      /// \brief Index 65
      INDEX_65 = 64,
      /// \brief Index 66
      INDEX_66 = 65,
      /// \brief Index 67
      INDEX_67 = 66,
      /// \brief Index 68
      INDEX_68 = 67,
      /// \brief Index 69
      INDEX_69 = 68,
      /// \brief Index 70
      INDEX_70 = 69,
      /// \brief Index 71
      INDEX_71 = 70,
      /// \brief Index 72
      INDEX_72 = 71,
      /// \brief Index 73
      INDEX_73 = 72,
      /// \brief Index 74
      INDEX_74 = 73,
      /// \brief Index 75
      INDEX_75 = 74,
      /// \brief Index 76
      INDEX_76 = 75,
      /// \brief Index 77
      INDEX_77 = 76,
      /// \brief Index 78
      INDEX_78 = 77,
      /// \brief Index 79
      INDEX_79 = 78,
      /// \brief Index 80
      INDEX_80 = 79,
      /// \brief Index 81
      INDEX_81 = 80,
      /// \brief Index 82
      INDEX_82 = 81,
      /// \brief Index 83
      INDEX_83 = 82,
      /// \brief Index 84
      INDEX_84 = 83,
      /// \brief Index 85
      INDEX_85 = 84,
      /// \brief Index 86
      INDEX_86 = 85,
      /// \brief Index 87
      INDEX_87 = 86,
      /// \brief Index 88
      INDEX_88 = 87,
      /// \brief Index 89
      INDEX_89 = 88,
      /// \brief Index 90
      INDEX_90 = 89,
      /// \brief Index 91
      INDEX_91 = 90,
      /// \brief Index 92
      INDEX_92 = 91,
      /// \brief Index 93
      INDEX_93 = 92,
      /// \brief Index 94
      INDEX_94 = 93,
      /// \brief Index 95
      INDEX_95 = 94,
      /// \brief Index 96
      INDEX_96 = 95,
      /// \brief Index 97
      INDEX_97 = 96,
      /// \brief Index 98
      INDEX_98 = 97,
      /// \brief Index 99
      INDEX_99 = 98,
      /// \brief Index 100
      INDEX_100 = 99,
      /// \brief Index 101
      INDEX_101 = 100,
      /// \brief Index 102
      INDEX_102 = 101,
      /// \brief Index 103
      INDEX_103 = 102,
      /// \brief Index 104
      INDEX_104 = 103,
      /// \brief Index 105
      INDEX_105 = 104,
      /// \brief Index 106
      INDEX_106 = 105,
      /// \brief Index 107
      INDEX_107 = 106,
      /// \brief Index 108
      INDEX_108 = 107,
      /// \brief Index 109
      INDEX_109 = 108,
      /// \brief Index 110
      INDEX_110 = 109,
      /// \brief Index 111
      INDEX_111 = 110,
      /// \brief Index 112
      INDEX_112 = 111,
      /// \brief Index 113
      INDEX_113 = 112,
      /// \brief Index 114
      INDEX_114 = 113,
      /// \brief Index 115
      INDEX_115 = 114,
      /// \brief Index 116
      INDEX_116 = 115,
      /// \brief Index 117
      INDEX_117 = 116,
      /// \brief Index 118
      INDEX_118 = 117,
      /// \brief Index 119
      INDEX_119 = 118,
      /// \brief System Version 2.
      INDEX_SYSTEM_VERSION_2 = 124,
      /// \brief System.
      INDEX_SYSTEM = 125, 
      /// \brief None.
      INDEX_NONE = -1, 
      /// \brief Unknown.
      INDEX_UNKNOWN = -7
   };

   /// \brief The %Btrieve location modes.
   enum LocationMode {
      /// \brief No preference.
      LOCATION_MODE_NO_PREFERENCE = 0,
      /// \brief Force local.
      LOCATION_MODE_FORCE_LOCAL = 1,
      /// \brief Force remote.
      LOCATION_MODE_FORCE_REMOTE = 2,
      /// \brief Unknown.
      LOCATION_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve lock modes.
   /// \SNIPPET_CPP_PYTHON{LockMode, block.cpp, block.py}
   /// \see Btrieve::LockModeToString
   enum LockMode {
      /// \brief None.
      LOCK_MODE_NONE = 0,
      /// \brief Single with wait.
      LOCK_MODE_SINGLE_WAIT = 1,
      /// \brief Single with no wait.
      LOCK_MODE_SINGLE_NO_WAIT = 2,
      /// \brief Multiple with wait.
      LOCK_MODE_MULTIPLE_WAIT = 3,
      /// \brief Multiple with no wait.
      LOCK_MODE_MULTIPLE_NO_WAIT = 4,
      /// \brief Unknown.
      LOCK_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve null key modes.
   /// \SNIPPET_CPP_PYTHON{NullKeyMode, bindexattributes.cpp, bindexattributes.py}
   /// \see Btrieve::NullKeyModeToString
   enum NullKeyMode {
      /// \brief All segments.
      NULL_KEY_MODE_ALL_SEGMENTS = 0,
      /// \brief Any segments.
      NULL_KEY_MODE_ANY_SEGMENTS = 1,
      /// \brief None.
      NULL_KEY_MODE_NONE = 2,
      /// \brief Unknown.
      NULL_KEY_MODE_UNKNOWN = -7 
   };

   /// \brief The %Btrieve open modes.
   /// \SNIPPET_CPP_PYTHON{OpenMode, btest.cpp, btest.py}
   enum OpenMode {
      /// \brief Normal.
      OPEN_MODE_NORMAL = 0,
      /// \brief Accelerated.
      OPEN_MODE_ACCELERATED = 1,
      /// \brief Read only.
      OPEN_MODE_READ_ONLY = 2,
      /// \brief Exclusive.
      OPEN_MODE_EXCLUSIVE = 3,
      /// \brief Writable.
      OPEN_MODE_WRITABLE = 4,
      /// \brief Unknown.
      OPEN_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve file owner modes.
   /// \see Btrieve::OwnerModeToString
   enum OwnerMode {
      /// \brief None.
      OWNER_MODE_NONE = 0,
      /// \brief No encryption with no read allowed.
      OWNER_MODE_NO_ENCRYPTION_NO_READ_ALLOWED = 1,
      /// \brief No encryption with read allowed.
      OWNER_MODE_NO_ENCRYPTION_READ_ALLOWED = 2,
      /// \brief Encryption with no read allowed.
      OWNER_MODE_ENCRYPTION_NO_READ_ALLOWED = 3,
      /// \brief Encryption with read allowed.
      OWNER_MODE_ENCRYPTION_READ_ALLOWED = 4,
      /// \brief Unknown.
      OWNER_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve file encryption types.
   /// \see Btrieve::OwnerModeToString
   enum EncryptType {
      /// \brief None.
      ENCRYPT_TYPE_NONE = 0,
      /// \brief Traditional encryption.
      ENCRYPT_TYPE_TRADITIONAL = 1,
      /// \brief 128-bit encryption.
      ENCRYPT_TYPE_128_BIT = 2,
      /// \brief AES-192 encryption.
      ENCRYPT_TYPE_AES_192 = 3,
      /// \brief AES-256 encryption.
      ENCRYPT_TYPE_AES_256 = 4,
      /// \brief Unknown.
      ENCRYPT_TYPE_UNKNOWN = -7
   };

   /// \brief The %Btrieve page lock type.
   /// \see Btrieve::PageLockTypeToString
   enum PageLockType {
      /// \brief None.
      PAGE_LOCK_TYPE_NONE = 0,
      /// \brief Data page.
      PAGE_LOCK_TYPE_DATA_PAGE = 1,
      /// \brief Index page.
      PAGE_LOCK_TYPE_INDEX_PAGE = 2,
      /// \brief Variable page.
      PAGE_LOCK_TYPE_VARIABLE_PAGE = 3,
      /// \brief Unknown.
      PAGE_LOCK_TYPE_UNKNOWN = -7
   };

   /// \brief The %Btrieve page sizes.
   /// \SNIPPET_CPP_PYTHON{PageSize, bfileattributes.cpp, bfileattributes.py}
   /// \see Btrieve::PageSizeToString
   enum PageSize {
      /// \brief Page size 512.
      PAGE_SIZE_512 = 0,
      /// \brief Page size 1024.
      PAGE_SIZE_1024 = 1,
      /// \brief Page size 1536.
      PAGE_SIZE_1536 = 2,
      /// \brief Page size 2048.
      PAGE_SIZE_2048 = 3,
      /// \brief Page size 3072.
      PAGE_SIZE_3072 = 4,
      /// \brief Page size 3584.
      PAGE_SIZE_3584 = 5,
      /// \brief Page size 4096.
      PAGE_SIZE_4096 = 6,
      /// \brief Page size 8192.
      PAGE_SIZE_8192 = 7,
      /// \brief Page size 16384.
      PAGE_SIZE_16384 = 8,
      /// \brief Default.
      PAGE_SIZE_DEFAULT = 9,
      /// \brief Unknown.
      PAGE_SIZE_UNKNOWN = -7
   };

   /// \brief The %Btrieve record compression modes.
   /// \SNIPPET_CPP_PYTHON{RecordCompressionMode, bfileattributes.cpp, bfileattributes.py}
   /// \see Btrieve::RecordCompressionModeToString
   enum RecordCompressionMode {
      /// \brief None.
      RECORD_COMPRESSION_MODE_NONE = 0,
      /// \brief Blank truncation.
      RECORD_COMPRESSION_MODE_BLANK_TRUNCATION = 1,
      /// \brief Run length encoding.
      RECORD_COMPRESSION_MODE_RUN_LENGTH_ENCODING = 2,
      /// \brief Unknown.
      RECORD_COMPRESSION_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve status codes.
   /// \SNIPPET_CPP_PYTHON{StatusCode, btest.cpp, btest.py}
   /// \see Btrieve::StatusCodeToString
   enum StatusCode {
		/// \brief 0 - No error.
      STATUS_CODE_NO_ERROR = 0,
		/// \brief 1 - Invalid function.
      STATUS_CODE_INVALID_FUNCTION = 1,
		/// \brief 2 - Input/Output error.
      STATUS_CODE_IO_ERROR = 2,
		/// \brief 3 - File not open.
      STATUS_CODE_FILE_NOT_OPEN = 3,
		/// \brief 4 - Key value not found.
      STATUS_CODE_KEY_VALUE_NOT_FOUND = 4,
		/// \brief 5 - Duplicate key value.
      STATUS_CODE_DUPLICATE_KEY_VALUE = 5,
		/// \brief 6 - Invalid index number.
      STATUS_CODE_INVALID_INDEX_NUMBER = 6,
		/// \brief 7 - Different index number.
      STATUS_CODE_DIFFERENT_INDEX_NUMBER = 7,
		/// \brief 8 - Position not set.
      STATUS_CODE_POSITION_NOT_SET = 8,
		/// \brief 9 - End of file.
      STATUS_CODE_END_OF_FILE = 9,
		/// \brief 10 - Modifiable key value error.
      STATUS_CODE_MODIFIABLE_KEYVALUE_ERROR = 10,
		/// \brief 11 - File name bad.
      STATUS_CODE_FILENAME_BAD = 11,
		/// \brief 12 - File not found.
      STATUS_CODE_FILE_NOT_FOUND = 12,
		/// \brief 13 - Extended file error.
      STATUS_CODE_EXTENDED_FILE_ERROR = 13,
		/// \brief 14 - Preimage open error.
      STATUS_CODE_PREIMAGE_OPEN_ERROR = 14,
		/// \brief 15 - Preimage input/output error.
      STATUS_CODE_PREIMAGE_IO_ERROR = 15,
		/// \brief 16 - Expansion error.
      STATUS_CODE_EXPANSION_ERROR = 16,
		/// \brief 17 - Close error.
      STATUS_CODE_CLOSE_ERROR = 17,
		/// \brief 18 - Disk full.
      STATUS_CODE_DISKFULL = 18,
		/// \brief 19 - Unrecoverable error.
      STATUS_CODE_UNRECOVERABLE_ERROR = 19,
		/// \brief 20 - Record manager inactive.
      STATUS_CODE_RECORD_MANAGER_INACTIVE = 20,
		/// \brief 21 - Key buffer too short.
      STATUS_CODE_KEYBUFFER_TOO_SHORT = 21,
		/// \brief 22 - Data length error.
      STATUS_CODE_DATALENGTH_ERROR = 22,
		/// \brief 23 - Position block length.
      STATUS_CODE_POSITIONBLOCK_LENGTH = 23,
		/// \brief 24 - Page size error.
      STATUS_CODE_PAGE_SIZE_ERROR = 24,
		/// \brief 25 - Create input/output error.
      STATUS_CODE_CREATE_IO_ERROR = 25,
		/// \brief 26 - Number of indexes.
      STATUS_CODE_NUMBER_OF_INDEXES = 26,
		/// \brief 27 - Invalid key position.
      STATUS_CODE_INVALID_KEY_POSITION = 27,
		/// \brief 28 - Invalid record length.
      STATUS_CODE_INVALID_RECORD_LENGTH = 28,
		/// \brief 29 - Invalid keylength.
      STATUS_CODE_INVALID_KEYLENGTH = 29,
		/// \brief 30 - Not a %Btrieve file.
      STATUS_CODE_NOT_A_BTRIEVE_FILE = 30,
		/// \brief 31 - File already extended.
      STATUS_CODE_FILE_ALREADY_EXTENDED = 31,
		/// \brief 32 - Extend input/output error.
      STATUS_CODE_EXTEND_IO_ERROR = 32,
		/// \brief 33 - %Btrieve cannot unload.
      STATUS_CODE_BTR_CANNOT_UNLOAD = 33,
		/// \brief 34 - Invalid extension name.
      STATUS_CODE_INVALID_EXTENSION_NAME = 34,
		/// \brief 35 - Directory error.
      STATUS_CODE_DIRECTORY_ERROR = 35,
		/// \brief 36 - Transaction error.
      STATUS_CODE_TRANSACTION_ERROR = 36,
		/// \brief 37 - Transaction is active.
      STATUS_CODE_TRANSACTION_IS_ACTIVE = 37,
		/// \brief 38 - Transaction file input/output error.
      STATUS_CODE_TRANSACTION_FILE_IO_ERROR = 38,
		/// \brief 39 - End transaction error.
      STATUS_CODE_END_TRANSACTION_ERROR = 39,
		/// \brief 40 - Transaction maximum files.
      STATUS_CODE_TRANSACTION_MAX_FILES = 40,
		/// \brief 41 - Operation not allowed.
      STATUS_CODE_OPERATION_NOT_ALLOWED = 41,
		/// \brief 42 - Incomplete accelerated access.
      STATUS_CODE_INCOMPLETE_ACCEL_ACCESS = 42,
		/// \brief 43 - Invalid record address.
      STATUS_CODE_INVALID_RECORD_ADDRESS = 43,
		/// \brief 44 - Null key path.
      STATUS_CODE_NULL_KEYPATH = 44,
		/// \brief 45 - Inconsistent key flags.
      STATUS_CODE_INCONSISTENT_KEY_FLAGS = 45,
		/// \brief 46 - Access to file denied.
      STATUS_CODE_ACCESS_TO_FILE_DENIED = 46,
		/// \brief 47 - Maximum open files.
      STATUS_CODE_MAXIMUM_OPEN_FILES = 47,
		/// \brief 48 - Invalid alternate collating sequence definition.
      STATUS_CODE_INVALID_ALT_SEQUENCE_DEF = 48,
		/// \brief 49 - Key type error.
      STATUS_CODE_KEY_TYPE_ERROR = 49,
		/// \brief 50 - Owner already set.
      STATUS_CODE_OWNER_ALREADY_SET = 50,
		/// \brief 51 - Invalid owner.
      STATUS_CODE_INVALID_OWNER = 51,
		/// \brief 52 - Error writing cache.
      STATUS_CODE_ERROR_WRITING_CACHE = 52,
		/// \brief 53 - Invalid interface.
      STATUS_CODE_INVALID_INTERFACE = 53,
		/// \brief 54 - Variable page error.
      STATUS_CODE_VARIABLE_PAGE_ERROR = 54,
		/// \brief 55 - Autoincrement error.
      STATUS_CODE_AUTOINCREMENT_ERROR = 55,
		/// \brief 56 - Incomplete index.
      STATUS_CODE_INCOMPLETE_INDEX = 56,
		/// \brief 57 - Expaned memory error.
      STATUS_CODE_EXPANED_MEM_ERROR = 57,
		/// \brief 58 - Compress buffer too short.
      STATUS_CODE_COMPRESS_BUFFER_TOO_SHORT = 58,
		/// \brief 59 - File already exists.
      STATUS_CODE_FILE_ALREADY_EXISTS = 59,
		/// \brief 60 - Reject count reached.
      STATUS_CODE_REJECT_COUNT_REACHED = 60,
		/// \brief 61 - Small extended get buffer error.
      STATUS_CODE_SMALL_EX_GET_BUFFER_ERROR = 61,
		/// \brief 62 - Invalid get expression.
      STATUS_CODE_INVALID_GET_EXPRESSION = 62,
		/// \brief 63 - Invalid extended insert buffer.
      STATUS_CODE_INVALID_EXT_INSERT_BUFF = 63,
		/// \brief 64 - Optimize limit reached.
      STATUS_CODE_OPTIMIZE_LIMIT_REACHED = 64,
		/// \brief 65 - Invalid extractor.
      STATUS_CODE_INVALID_EXTRACTOR = 65,
		/// \brief 66 - Referential integrity too many databases.
      STATUS_CODE_RI_TOO_MANY_DATABASES = 66,
		/// \brief 67 - Referential integrity SQL data dictionary files cannot open.
      STATUS_CODE_RIDDF_CANNOT_OPEN = 67,
		/// \brief 68 - Referential integrity cascade too deep.
      STATUS_CODE_RI_CASCADE_TOO_DEEP = 68,
		/// \brief 69 - Referential integrity cascade error.
      STATUS_CODE_RI_CASCADE_ERROR = 69,
		/// \brief 71 - Referential integrity violation.
      STATUS_CODE_RI_VIOLATION = 71,
		/// \brief 72 - Referential integrity referenced file cannot open.
      STATUS_CODE_RI_REFERENCED_FILE_CANNOT_OPEN = 72,
		/// \brief 73 - Referential integrity out of sync.
      STATUS_CODE_RI_OUT_OF_SYNC = 73,
		/// \brief 74 - End changed to abort.
      STATUS_CODE_END_CHANGED_TO_ABORT = 74,
		/// \brief 76 - Referential integrity conflict.
      STATUS_CODE_RI_CONFLICT = 76,
		/// \brief 77 - Can't loop in server.
      STATUS_CODE_CANT_LOOP_IN_SERVER = 77,
		/// \brief 78 - Dead lock.
      STATUS_CODE_DEAD_LOCK = 78,
		/// \brief 79 - Programming error.
      STATUS_CODE_PROGRAMMING_ERROR = 79,
		/// \brief 80 - Conflict.
      STATUS_CODE_CONFLICT = 80,
		/// \brief 81 - Lock error.
      STATUS_CODE_LOCKERROR = 81,
		/// \brief 82 - Lost position.
      STATUS_CODE_LOST_POSITION = 82,
		/// \brief 83 - Read outside transaction.
      STATUS_CODE_READ_OUTSIDE_TRANSACTION = 83,
		/// \brief 84 - Record in use.
      STATUS_CODE_RECORD_INUSE = 84,
		/// \brief 85 - File in use.
      STATUS_CODE_FILE_INUSE = 85,
		/// \brief 86 - File table full.
      STATUS_CODE_FILE_TABLE_FULL = 86,
		/// \brief 87 - No handles available.
      STATUS_CODE_NOHANDLES_AVAILABLE = 87,
		/// \brief 88 - Incompatible mode error.
      STATUS_CODE_INCOMPATIBLE_MODE_ERROR = 88,
		/// \brief 90 - Device table full.
      STATUS_CODE_DEVICE_TABLE_FULL = 90,
		/// \brief 91 - Server error.
      STATUS_CODE_SERVER_ERROR = 91,
		/// \brief 92 - Transaction table full.
      STATUS_CODE_TRANSACTION_TABLE_FULL = 92,
		/// \brief 93 - Incompatible lock type.
      STATUS_CODE_INCOMPATIBLE_LOCK_TYPE = 93,
		/// \brief 94 - Permission error.
      STATUS_CODE_PERMISSION_ERROR = 94,
		/// \brief 95 - Session no longer valid.
      STATUS_CODE_SESSION_NO_LONGER_VALID = 95,
		/// \brief 96 - Communications error.
      STATUS_CODE_COMMUNICATIONS_ERROR = 96,
		/// \brief 97 - Data message too small.
      STATUS_CODE_DATA_MESSAGE_TOO_SMALL = 97,
		/// \brief 98 - Internal transaction error.
      STATUS_CODE_INTERNAL_TRANSACTION_ERROR = 98,
		/// \brief 99 - Requester can't access runtime.
      STATUS_CODE_REQUESTER_CANT_ACCESS_RUNTIME = 99,
		/// \brief 100 - No cache buffers available.
      STATUS_CODE_NO_CACHE_BUFFERS_AVAIL = 100,
		/// \brief 101 - No operating system memory available.
      STATUS_CODE_NO_OS_MEMORY_AVAIL = 101,
		/// \brief 102 - No stack available.
      STATUS_CODE_NO_STACK_AVAIL = 102,
		/// \brief 103 - Chunk offset too long.
      STATUS_CODE_CHUNK_OFFSET_TOO_LONG = 103,
		/// \brief 104 - Locale error.
      STATUS_CODE_LOCALE_ERROR = 104,
		/// \brief 105 - Cannot create with variable tail allocation table.
      STATUS_CODE_CANNOT_CREATE_WITH_VAT = 105,
		/// \brief 106 - Chunk cannot get next.
      STATUS_CODE_CHUNK_CANNOT_GET_NEXT = 106,
		/// \brief 107 - Chunk incompatible file.
      STATUS_CODE_CHUNK_INCOMPATIBLE_FILE = 107,
		/// \brief 109 - Transaction too complex.
      STATUS_CODE_TRANSACTION_TOO_COMPLEX = 109,
		/// \brief 110 - Archival BLOG open error.
      STATUS_CODE_ARCH_BLOG_OPEN_ERROR = 110,
		/// \brief 111 - Archival file not logged.
      STATUS_CODE_ARCH_FILE_NOT_LOGGED = 111,
		/// \brief 112 - Archival file in use.
      STATUS_CODE_ARCH_FILE_IN_USE = 112,
		/// \brief 113 - Archival log file not found.
      STATUS_CODE_ARCH_LOGFILE_NOT_FOUND = 113,
		/// \brief 114 - Archival log file invalid.
      STATUS_CODE_ARCH_LOGFILE_INVALID = 114,
		/// \brief 115 - Archival dump file access error.
      STATUS_CODE_ARCH_DUMPFILE_ACCESS_ERROR = 115,
		/// \brief 116 - Locator file indicator.
      STATUS_CODE_LOCATOR_FILE_INDICATOR = 116,
		/// \brief 130 - No system locks available.
      STATUS_CODE_NO_SYSTEM_LOCKS_AVAILABLE = 130,
		/// \brief 132 - File full.
      STATUS_CODE_FILE_FULL = 132,
		/// \brief 133 - More than 5 concurrent users.
      STATUS_CODE_MORE_THAN_5_CONCURRENT_USERS = 133,
		/// \brief 134 - International sorting rule not found.
      STATUS_CODE_ISR_NOT_FOUND = 134,
		/// \brief 135 - International sorting rule invalid.
      STATUS_CODE_ISR_INVALID = 135,
		/// \brief 136 - Alternate collating sequence not found.
      STATUS_CODE_ACS_NOT_FOUND = 136,
		/// \brief 137 - Cannot convert RP.
      STATUS_CODE_CANNOT_CONVERT_RP = 137,
		/// \brief 138 - Invalid null indicator.
      STATUS_CODE_INVALID_NULL_INDICATOR = 138,
		/// \brief 139 - Invalid key option.
      STATUS_CODE_INVALID_KEY_OPTION = 139,
		/// \brief 140 - Incompatible close.
      STATUS_CODE_INCOMPATIBLE_CLOSE = 140,
		/// \brief 141 - Invalid user name.
      STATUS_CODE_INVALID_USERNAME = 141,
		/// \brief 142 - Invalid database.
      STATUS_CODE_INVALID_DATABASE = 142,
		/// \brief 143 - No SSQL rights.
      STATUS_CODE_NO_SSQL_RIGHTS = 143,
		/// \brief 144 - Already logged in.
      STATUS_CODE_ALREADY_LOGGED_IN = 144,
		/// \brief 145 - No database services.
      STATUS_CODE_NO_DATABASE_SERVICES = 145,
		/// \brief 146 - Duplicate system key.
      STATUS_CODE_DUPLICATE_SYSTEM_KEY = 146,
		/// \brief 147 - Log segment missing.
      STATUS_CODE_LOG_SEGMENT_MISSING = 147,
		/// \brief 148 - Roll forward error.
      STATUS_CODE_ROLL_FORWARD_ERROR = 148,
		/// \brief 149 - System key internal.
      STATUS_CODE_SYSTEM_KEY_INTERNAL = 149,
		/// \brief 150 - DBS internal error.
      STATUS_CODE_DBS_INTERNAL_ERROR = 150,
		/// \brief 151 - Chunk offset too small.
      /// \note Renamed from STATUS_CODE_NESTING_DEPTH_ERROR in Zen v15 R1.
      STATUS_CODE_CHUNK_OFFSET_TOO_SMALL = 151,
		/// \brief 160 - Invalid parameter to MicroKernel.
      STATUS_CODE_INVALID_PARAMETER_TO_MKDE = 160,
		/// \brief 161 - User count limit exceeded.
      STATUS_CODE_USER_COUNT_LIMIT_EXCEEDED = 161,
		/// \brief 162 - Client table full.
      STATUS_CODE_CLIENT_TABLE_FULL = 162,
		/// \brief 163 - Last segment error.
      STATUS_CODE_LAST_SEGMENT_ERROR = 163,
		/// \brief 170 - Login failed bad user name.
      STATUS_CODE_LOGIN_FAILED_BAD_USERNAME = 170,
		/// \brief 171 - Login failed bad password.
      STATUS_CODE_LOGIN_FAILED_BAD_PASSWORD = 171,
		/// \brief 172 - Login failed bad database.
      STATUS_CODE_LOGIN_FAILED_BAD_DATABASE = 172,
		/// \brief 173 - Login already logged in.
      STATUS_CODE_LOGIN_ALREADY_LOGGED_IN = 173,
		/// \brief 174 - Login logout failed.
      STATUS_CODE_LOGIN_LOGOUT_FAILED = 174,
		/// \brief 175 - Login wrong uniform resource indicator format.
      STATUS_CODE_LOGIN_WRONG_URI_FORMAT = 175,
		/// \brief 176 - Login file and table not found.
      STATUS_CODE_LOGIN_FILE_AND_TABLE_NOT_FOUND = 176,
		/// \brief 177 - Login table not in database.
      STATUS_CODE_LOGIN_TABLE_NOT_IN_DATABASE = 177,
		/// \brief 178 - Login directory not in database.
      STATUS_CODE_LOGIN_DIRECTORY_NOT_IN_DATABASE = 178,
		/// \brief 1001 - Lock parameter out of range.
      STATUS_CODE_LOCK_PARM_OUTOFRANGE = 1001,
		/// \brief 1002 - Memory allocation error.
      STATUS_CODE_MEM_ALLOCATION_ERR = 1002,
		/// \brief 1003 - Memory parameter too small.
      STATUS_CODE_MEM_PARM_TOO_SMALL = 1003,
		/// \brief 1004 - Page size parameter out of range.
      STATUS_CODE_PAGE_SIZE_PARM_OUTOFRANGE = 1004,
		/// \brief 1005 - Invalid preimage parameter.
      STATUS_CODE_INVALID_PREIMAGE_PARM = 1005,
		/// \brief 1006 - Preimage buffer parameter out of range.
      STATUS_CODE_PREIMAGE_BUF_PARM_OUTOFRANGE = 1006,
		/// \brief 1007 - Files parameter out of range.
      STATUS_CODE_FILES_PARM_OUTOFRANGE = 1007,
		/// \brief 1008 - Invalid initialization parameter.
      STATUS_CODE_INVALID_INIT_PARM = 1008,
		/// \brief 1009 - Invalid transaction parameter.
      STATUS_CODE_INVALID_TRANS_PARM = 1009,
		/// \brief 1010 - Error access transaction control file.
      STATUS_CODE_ERROR_ACC_TRANS_CONTROL_FILE = 1010,
		/// \brief 1011 - Compression buffer parameter out of range.
      STATUS_CODE_COMPRESSION_BUF_PARM_OUTOFRANGE = 1011,
		/// \brief 1012 - Invalid "Maximum Number Of Files In A Transaction" option.
      STATUS_CODE_INV_N_OPTION = 1012,
		/// \brief 1013 - Task list full.
      STATUS_CODE_TASK_LIST_FULL = 1013,
		/// \brief 1014 - Stop warning.
      STATUS_CODE_STOP_WARNING = 1014,
		/// \brief 1015 - Pointer parameter invalid.
      STATUS_CODE_POINTER_PARM_INVALID = 1015,
		/// \brief 1016 - Already initialized.
      STATUS_CODE_ALREADY_INITIALIZED = 1016,
		/// \brief 1017 - %Btrieve Requester can't find resource DLL.
      STATUS_CODE_REQ_CANT_FIND_RES_DLL = 1017,
		/// \brief 1018 - Already inside %Btrieve function.
      STATUS_CODE_ALREADY_INSIDE_BTR_FUNCTION = 1018,
		/// \brief 1019 - Callback abort.
      STATUS_CODE_CALLBACK_ABORT = 1019,
		/// \brief 1020 - Interface communications error.
      STATUS_CODE_INTF_COMM_ERROR = 1020,
		/// \brief 1021 - Failed to initialize.
      STATUS_CODE_FAILED_TO_INITIALIZE = 1021,
		/// \brief 1022 - MicroKernel shutting down.
      STATUS_CODE_MKDE_SHUTTING_DOWN = 1022,
		/// \brief 2000 - Internal error.
      STATUS_CODE_INTERNAL_ERROR = 2000,
		/// \brief 2001 - Insufficient memory allocation.
      STATUS_CODE_INSUFFICIENT_MEM_ALLOC = 2001,
		/// \brief 2002 - Invalid option.
      STATUS_CODE_INVALID_OPTION = 2002,
		/// \brief 2003 - No local access allowed.
      STATUS_CODE_NO_LOCAL_ACCESS_ALLOWED = 2003,
		/// \brief 2004 - SPX not installed.
      STATUS_CODE_SPX_NOT_INSTALLED = 2004,
		/// \brief 2005 - Incorrect SPX version.
      STATUS_CODE_INCORRECT_SPX_VERSION = 2005,
		/// \brief 2006 - No available SPX connection.
      STATUS_CODE_NO_AVAIL_SPX_CONNECTION = 2006,
		/// \brief 2007 - Invalid pointer parameter.
      STATUS_CODE_INVALID_PTR_PARM = 2007,
		/// \brief 2008 - Can't connect to 615.
      STATUS_CODE_CANT_CONNECT_TO_615 = 2008,
		/// \brief 2009 - Can't load MicroKernel router.
      STATUS_CODE_CANT_LOAD_MKDE_ROUTER = 2009,
		/// \brief 2010 - UT thunk not loaded.
      STATUS_CODE_UT_THUNK_NOT_LOADED = 2010,
		/// \brief 2011 - No resource DLL.
      STATUS_CODE_NO_RESOURCE_DLL = 2011,
		/// \brief 2012 - Operating system error.
      STATUS_CODE_OS_ERROR = 2012,
		/// \brief 3000 - MicroKernel router memory error.
      STATUS_CODE_MK_ROUTER_MEM_ERROR = 3000,
		/// \brief 3001 - MicroKernel no local access allowed.
      STATUS_CODE_MK_NO_LOCAL_ACCESS_ALLOWED = 3001,
		/// \brief 3002 - MicroKernel no resource DLL.
      STATUS_CODE_MK_NO_RESOURCE_DLL = 3002,
		/// \brief 3003 - MicroKernel incompatible component.
      STATUS_CODE_MK_INCOMPAT_COMPONENT = 3003,
		/// \brief 3004 - MicroKernel timeout error.
      STATUS_CODE_MK_TIMEOUT_ERROR = 3004,
		/// \brief 3005 - MicroKernel operating system error.
      STATUS_CODE_MK_OS_ERROR = 3005,
		/// \brief 3006 - MicroKernel invalid session.
      STATUS_CODE_MK_INVALID_SESSION = 3006,
		/// \brief 3007 - MicroKernel server not found.
      STATUS_CODE_MK_SERVER_NOT_FOUND = 3007,
		/// \brief 3008 - MicroKernel invalid configuration.
      STATUS_CODE_MK_INVALID_CONFIG = 3008,
		/// \brief 3009 - MicroKernel NETAPI not loaded.
      STATUS_CODE_MK_NETAPI_NOT_LOADED = 3009,
		/// \brief 3010 - MicroKernel NWAPI not loaded.
      STATUS_CODE_MK_NWAPI_NOT_LOADED = 3010,
		/// \brief 3011 - MicroKernel thunk not loaded.
      STATUS_CODE_MK_THUNK_NOT_LOADED = 3011,
		/// \brief 3012 - MicroKernel local not loaded.
      STATUS_CODE_MK_LOCAL_NOT_LOADED = 3012,
		/// \brief 3013 - MicroKernel Network Services Layer not loaded.
      STATUS_CODE_MK_PNSL_NOT_LOADED = 3013,
		/// \brief 3014 - MicroKernel can't find engine.
      STATUS_CODE_MK_CANT_FIND_ENGINE = 3014,
		/// \brief 3015 - MicroKernel initialization error.
      STATUS_CODE_MK_INIT_ERROR = 3015,
		/// \brief 3016 - MicroKernel internal error.
      STATUS_CODE_MK_INTERNAL_ERROR = 3016,
		/// \brief 3017 - MicroKernel local MicroKernel data buffer too small.
      STATUS_CODE_MK_LOCAL_MKDE_DATABUF_TOO_SMALL = 3017,
		/// \brief 3018 - MicroKernel closed error.
      STATUS_CODE_MK_CLOSED_ERROR = 3018,
		/// \brief 3019 - MicroKernel semaphore error.
      STATUS_CODE_MK_SEMAPHORE_ERROR = 3019,
		/// \brief 3020 - MicroKernel loading error.
      STATUS_CODE_MK_LOADING_ERROR = 3020,
		/// \brief 3021 - MicroKernel bad SRB format.
      STATUS_CODE_MK_BAD_SRB_FORMAT = 3021,
		/// \brief 3022 - MicroKernel data buffer length too large.
      STATUS_CODE_MK_DATABUF_LEN_TOO_LARGE = 3022,
		/// \brief 3023 - MicroKernel task table full.
      STATUS_CODE_MK_TASK_TABLE_FULL = 3023,
		/// \brief 3024 - MicroKernel invalid operation on remote.
      STATUS_CODE_MK_INVALID_OP_ON_REMOTE = 3024,
		/// \brief 3025 - MicroKernel PIDs not loaded.
      STATUS_CODE_MK_PIDS_NOT_LOADED = 3025,
		/// \brief 3026 - MicroKernel bad PIDs.
      STATUS_CODE_MK_BAD_PIDS = 3026,
		/// \brief 3027 - MicroKernel IDS connect failure.
      STATUS_CODE_MK_IDS_CONNECT_FAILURE = 3027,
		/// \brief 3028 - MicroKernel IDS login failure.
      STATUS_CODE_MK_IDS_LOGIN_FAILURE = 3028,
		/// \brief 3101 - Network layer failure.
      STATUS_CODE_NL_FAILURE = 3101,
		/// \brief 3102 - Network layer not initialized.
      STATUS_CODE_NL_NOT_INITIALIZED = 3102,
		/// \brief 3103 - Network layer name not found.
      STATUS_CODE_NL_NAME_NOT_FOUND = 3103,
		/// \brief 3104 - Network layer permission error.
      STATUS_CODE_NL_PERMISSION_ERROR = 3104,
		/// \brief 3105 - Network layer no available transport.
      STATUS_CODE_NL_NO_AVAILABLE_TRANSPORT = 3105,
		/// \brief 3106 - Network layer connection failure.
      STATUS_CODE_NL_CONNECTION_FAILURE = 3106,
		/// \brief 3107 - Network layer out of memory.
      STATUS_CODE_NL_OUT_OF_MEMORY = 3107,
		/// \brief 3108 - Network layer invalid session.
      STATUS_CODE_NL_INVALID_SESSION = 3108,
		/// \brief 3109 - Network layer more data.
      STATUS_CODE_NL_MORE_DATA = 3109,
		/// \brief 3110 - Network layer not connected.
      STATUS_CODE_NL_NOT_CONNECTED = 3110,
		/// \brief 3111 - Network layer send failure.
      STATUS_CODE_NL_SEND_FAILURE = 3111,
		/// \brief 3112 - Network layer receive failure.
      STATUS_CODE_NL_RECEIVE_FAILURE = 3112,
		/// \brief 3113 - Network layer invalid server type.
      STATUS_CODE_NL_INVALID_SERVER_TYPE = 3113,
		/// \brief 3114 - Network layer server routing table full.
      STATUS_CODE_NL_SRT_FULL = 3114,
		/// \brief 3115 - Network layer transport failure.
      STATUS_CODE_NL_TRANSPORT_FAILURE = 3115,
		/// \brief 3116 - Network layer receive data overflow.
      STATUS_CODE_NL_RCV_DATA_OVERFLOW = 3116,
		/// \brief 3117 - Network layer CST full.
      STATUS_CODE_NL_CST_FULL = 3117,
		/// \brief 3118 - Network layer invalid address family.
      STATUS_CODE_NL_INVALID_ADDRESS_FAMILY = 3118,
		/// \brief 3119 - Network layer no authentication context available.
      STATUS_CODE_NL_NO_AUTH_CONTEXT_AVAILABLE = 3119,
		/// \brief 3120 - Network layer invalid authentication type.
      STATUS_CODE_NL_INVALID_AUTH_TYPE = 3120,
		/// \brief 3121 - Network layer invalid authentication object.
      STATUS_CODE_NL_INVALID_AUTH_OBJECT = 3121,
		/// \brief 3122 - Network layer authentication length too small.
      STATUS_CODE_NL_AUTH_LEN_TOO_SMALL = 3122,
		/// \brief 3123 - Network layer invalid session level parameter.
      STATUS_CODE_NL_INVALID_SESSION_LEVEL_PARM = 3123,
		/// \brief 3124 - Network layer task table full.
      STATUS_CODE_NL_TASK_TABLE_FULL = 3124,
		/// \brief 3125 - Network layer NDS name resolution error.
      STATUS_CODE_NL_NDS_NAME_RESOLUTION_ERROR = 3125,
		/// \brief 3126 - Network layer file name resolution error.
      STATUS_CODE_NL_FILE_NAME_RESOLUTION_ERROR = 3126,
		/// \brief 3127 - Network layer IDS send failure.
      STATUS_CODE_NL_IDS_SEND_FAILURE = 3127,
		/// \brief 3128 - Network layer IDS receive failure.
      STATUS_CODE_NL_IDS_RCV_FAILURE = 3128,
		/// \brief -7 - Unknown.
      STATUS_CODE_UNKNOWN = -7,
   };

   /// \brief The %Btrieve system data modes.
   /// \SNIPPET_CPP_PYTHON{SystemDataMode, bfileattributes.cpp, bfileattributes.py}
   /// \see Btrieve::SystemDataModeToString
   enum SystemDataMode {
      /// \brief Yes.
      SYSTEM_DATA_MODE_YES = 0, 
      /// \brief No.
      SYSTEM_DATA_MODE_NO = 1,
      /// \brief Default.
      SYSTEM_DATA_MODE_DEFAULT = 2,
      /// \brief Version 2
      SYSTEM_DATA_MODE_VERSION_2 = 3,
      /// \brief Unknown.
      SYSTEM_DATA_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve transaction modes.
   enum TransactionMode {
      /// \brief Exclusive.
      TRANSACTION_MODE_EXCLUSIVE = 0,
      /// \brief Concurrent with write wait.
      TRANSACTION_MODE_CONCURRENT_WRITE_WAIT = 1,
      /// \brief Concurrent with no write wait.
      TRANSACTION_MODE_CONCURRENT_NO_WRITE_WAIT = 2,
      /// \brief Unknown.
      TRANSACTION_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve unlock modes.
   /// \SNIPPET_CPP_PYTHON{UnlockMode, block.cpp, block.py}
   enum UnlockMode {
      /// \brief Single.
      UNLOCK_MODE_SINGLE = 0,
      /// \brief Multiple.
      UNLOCK_MODE_MULTIPLE = 1,
      /// \brief Unknown.
      UNLOCK_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve variable length records modes.
   /// \SNIPPET_CPP_PYTHON{VariableLengthRecordsMode, bfileattributes.cpp, bfileattributes.py}
   /// \see Btrieve::VariableLengthRecordsModeToString
   enum VariableLengthRecordsMode {
      /// \brief No.
      VARIABLE_LENGTH_RECORDS_MODE_NO = 0, 
      /// \brief Yes.
      VARIABLE_LENGTH_RECORDS_MODE_YES = 1,
      /// \brief Yes with variable allocation tails.
      VARIABLE_LENGTH_RECORDS_MODE_YES_VARIABLE_ALLOCATION_TAILS = 2,
      /// \brief Unknown.
      VARIABLE_LENGTH_RECORDS_MODE_UNKNOWN = -7
   };

   /// \brief The %Btrieve version types.
   /// \see Btrieve::VersionTypeToString
   enum VersionType {
      /// \brief None.
      VERSION_TYPE_NONE = 0, 
      /// \brief %Btrieve engine.
      VERSION_TYPE_BTRIEVE_ENGINE = 1, 
      /// \brief Workgroup engine.
      VERSION_TYPE_WORKGROUP_ENGINE = 2,
      /// \brief DOS workstation.
      VERSION_TYPE_DOS_WORKSTATION = 3, 
      /// \brief Client.
      VERSION_TYPE_CLIENT_REQUESTOR = 4,
      /// \brief Windows server engine.
      VERSION_TYPE_WINDOWS_SERVER_ENGINE = 5, 
      /// \brief UNIX.
      VERSION_TYPE_UNIX = 6, 
      /// \brief Client engine.
      VERSION_TYPE_CLIENT_ENGINE = 7, 
      /// \brief Unknown.
      VERSION_TYPE_UNKNOWN = -7
   };

   /// \brief The %Btrieve system fields.
   /// \see Btrieve::SystemFieldToString
   enum SystemField {
      /// \brief Create time.
      SYSTEM_FIELD_CREATE_TIME = 0,
      /// \brief Update time.
      SYSTEM_FIELD_UPDATE_TIME = 1,
      /// \brief Unknown.
      SYSTEM_FIELD_UNKNOWN = -7 
   };

   /// \brief Return an alternate collation sequence mode as a string.
   /// \param[in] ACSMode The alternate collation sequence mode.
   /// \SNIPPET_CPP_PYTHON{ACSModeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *ACSModeToString(ACSMode ACSMode);
   /// \brief Return a duplicate mode as a string.
   /// \param[in] duplicateMode The duplicate mode.
   /// \SNIPPET_CPP_PYTHON{DuplicateModeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *DuplicateModeToString(DuplicateMode duplicateMode);
   /// \brief Return a data type as a string.
   /// \param[in] dataType The data type.
   /// \SNIPPET_CPP_PYTHON{DataTypeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *DataTypeToString(DataType dataType);
   /// \brief Return a file version as a string.
   /// \param[in] fileVersion The file version.
   /// \SNIPPET_CPP_PYTHON{FileVersionToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *FileVersionToString(FileVersion fileVersion);
   /// \brief Return a free space threshold as a string.
   /// \param[in] freeSpaceThreshold The free space threshold.
   /// \SNIPPET_CPP_PYTHON{FreeSpaceThresholdToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *FreeSpaceThresholdToString(FreeSpaceThreshold freeSpaceThreshold);
   /// \brief Return an index as a string.
   /// \param[in] index The index.
   /// \SNIPPET_CPP_PYTHON{IndexToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *IndexToString(Index index);
   /// \brief Return a null key mode as a string.
   /// \param[in] nullKeyMode The null key mode.
   /// \SNIPPET_CPP_PYTHON{NullKeyModeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *NullKeyModeToString(NullKeyMode nullKeyMode);
   /// \brief Return a page size as a string.
   /// \param[in] pageSize The page size.
   /// \SNIPPET_CPP_PYTHON{PageSizeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *PageSizeToString(PageSize pageSize);
   /// \brief Return an record compression mode as a string.
   /// \param[in] recordCompressionMode The record compression mode.
   /// \SNIPPET_CPP_PYTHON{RecordCompressionModeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *RecordCompressionModeToString(RecordCompressionMode recordCompressionMode);
   /// \brief Return a status code as a string.
   /// \param[in] statusCode The status code.
   /// \SNIPPET_CPP_PYTHON{StatusCodeToString, btest.cpp, btest.py}
   /// \return The string.
   static const char *StatusCodeToString(StatusCode statusCode);
   /// \brief Return a system data mode as a string.
   /// \param[in] systemDataMode The system data mode.
   /// \SNIPPET_CPP_PYTHON{SystemDataModeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *SystemDataModeToString(SystemDataMode systemDataMode);
   /// \brief Return a version type as a string.
   /// \param[in] versionType The version type.
   /// \SNIPPET_CPP_PYTHON{VersionTypeToString, bversion.cpp, bversion.py}
   /// \return The string.
   static const char *VersionTypeToString(VersionType versionType);
   /// \brief Return a page lock type as a string.
   /// \param[in] pageLockType The page lock type.
   /// \SNIPPET_CPP_PYTHON{PageLockTypeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *PageLockTypeToString(PageLockType pageLockType);
   /// \brief Return a lock mode as a string.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{LockModeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *LockModeToString(LockMode lockMode);
   /// \brief Return an owner mode as a string.
   /// \param[in] ownerMode The owner mode.
   /// \SNIPPET_CPP_PYTHON{OwnerModeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *OwnerModeToString(OwnerMode ownerMode);
   /// \brief Return the encryption type as a string.
   /// \param[in] encryptType The encryption type.
   /// \return The string.
   static const char *EncryptTypeToString(EncryptType encryptType);
   /// \brief Return a variable length records mode as a string.
   /// \param[in] variableLengthRecordsMode The variable length records mode.
   /// \SNIPPET_CPP_PYTHON{VariableLengthRecordsModeToString, bfileinformation.cpp, bfileinformation.py}
   /// \return The string.
   static const char *VariableLengthRecordsModeToString(VariableLengthRecordsMode variableLengthRecordsMode);
   /// \brief Return a system field as a string.
   /// \param[in] systemField The system field.
   /// \return The string.
   static const char *SystemFieldToString(SystemField systemField);
   /// \brief Convert a Unix Epoch based timestamp in microseconds to a TIMESTAMP.
   /// \param[in] microseconds The Unix Epoch timestamp in microseconds granularity.
   /// \return The %Btrieve TIMESTAMP.
   static long long UnixEpochMicrosecondsToTimestamp(long long microseconds);
   /// \brief Convert a %Btrieve TIMESTAMP to a Unix Epoch timestamp in microseconds.
   /// \param[in] timestamp The %Btrieve TIMESTAMP.
   /// \return The Epoch based timestamp in microseconds.
   static long long TimestampToUnixEpochMicroseconds(long long timestamp);

private:
   Btrieve();
   ~Btrieve();
};

/// \brief The %Btrieve client class.
class LINKAGE BtrieveClient
{
public:
   /// \par
   /// The service agent and client identifier will be automatically generated based on the caller's process and thread identifiers.
   BtrieveClient();
   /// \param[in] serviceAgentIdentifier The service agent identifier. Use 0 to automatically generate a service agent identifier based on the caller's process identifier.
   /// \param[in] clientIdentifier The client identifier. Use 0 to automatically generate a client identifier based on the caller's thread identifier.
   /// \SNIPPET_CPP_PYTHON{BtrieveClient, btest.cpp, btest.py}
   BtrieveClient(int serviceAgentIdentifier, int clientIdentifier);
   ~BtrieveClient();

   /// \brief Get the version.
   /// \param[out] btrieveVersion The version.
   /// \param[in] btrieveFile The %Btrieve file.
   /// \SNIPPET_CPP_PYTHON{GetVersion, bversion.cpp, bversion.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetVersion(BtrieveVersion *btrieveVersion, BtrieveFile *btrieveFile = NULL);
   /// \brief Open a file.
   /// \param[out] btrieveFile The %Btrieve file.
   /// \param[in] fileName The file name.
   /// \param[in] ownerName The owner name.
   /// \param[in] openMode The open mode.
   /// \param[in] locationMode The location mode.
   /// \SNIPPET_CPP_PYTHON{FileOpen, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileOpen(BtrieveFile *btrieveFile, const char *fileName, const char *ownerName, Btrieve::OpenMode openMode, Btrieve::LocationMode locationMode = Btrieve::LOCATION_MODE_NO_PREFERENCE);
   /// \brief Open a file.
   /// \param[out] btrieveFile The %Btrieve file.
   /// \param[in] fileName The file name.
   /// \param[in] ownerName The owner name.
   /// \param[in] openMode The open mode.
   /// \param[in] locationMode The location mode.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileOpen(BtrieveFile *btrieveFile, const wchar_t *fileName, const char *ownerName, Btrieve::OpenMode openMode, Btrieve::LocationMode locationMode = Btrieve::LOCATION_MODE_NO_PREFERENCE);

// Avoid Windows renaming of GetCurrentDirectory.
#ifdef GetCurrentDirectory
   #undef GetCurrentDirectory
#endif

   /// \brief Get the current directory.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] diskDrive The disk drive. \MUST_USE_DISK_DRIVE_DEFAULT
   /// \retval "!= NULL" The current directory.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveClient::GetLastStatusCode}
   const char *GetCurrentDirectory(Btrieve::DiskDrive diskDrive);
#ifndef SWIG
   /// \brief Get the current directory.
   /// \note \METHOD_ONLY_CPP
   /// \param[in] diskDrive The disk drive. \MUST_USE_DISK_DRIVE_DEFAULT
   /// \param[out] currentDirectory The current directory.
   /// \param[in] currentDirectorySize The current directory size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetCurrentDirectory(Btrieve::DiskDrive diskDrive, char *currentDirectory, int currentDirectorySize);
#endif
   /// \brief Get the current directory.
   /// \param[in] diskDrive The disk drive. \MUST_USE_DISK_DRIVE_DEFAULT
   /// \param[out] currentDirectory The current directory.
   /// \param[in] currentDirectorySize The current directory size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetCurrentDirectory(Btrieve::DiskDrive diskDrive, wchar_t *currentDirectory, int currentDirectorySize);
   /// \brief Begin continuous operation.
   /// \param[in] pathNames The path names.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode ContinuousOperationBegin(const char *pathNames);
   /// \brief Begin continuous operation.
   /// \param[in] pathNames The path names.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode ContinuousOperationBegin(const wchar_t *pathNames);
   /// \brief End continuous operation.
   /// \param[in] pathNames The path names.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode ContinuousOperationEnd(const char *pathNames);
   /// \brief End continuous operation.
   /// \param[in] pathNames The path names.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode ContinuousOperationEnd(const wchar_t *pathNames);
   /// \brief Close a file.
   /// \param[in] btrieveFile The %Btrieve file.
   /// \SNIPPET_CPP_PYTHON{FileClose, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileClose(BtrieveFile *btrieveFile);  
   /// \brief Create a file.
   /// \param[in] btrieveFileAttributes The file attributes.
   /// \param[in] fileName The file name.
   /// \param[in] createMode The create mode.
   /// \param[in] locationMode The location mode.
   /// \SNIPPET_CPP_PYTHON{FileCreate, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileCreate(BtrieveFileAttributes *btrieveFileAttributes, const char *fileName, Btrieve::CreateMode createMode, Btrieve::LocationMode locationMode = Btrieve::LOCATION_MODE_NO_PREFERENCE);
   /// \brief Create a file.
   /// \param[in] btrieveFileAttributes The file attributes.
   /// \param[in] btrieveIndexAttributes The index attributes.
   /// \param[in] fileName The file name.
   /// \param[in] createMode The create mode.
   /// \param[in] locationMode The location mode.
   /// \SNIPPET_CPP_PYTHON{FileCreate, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileCreate(BtrieveFileAttributes *btrieveFileAttributes, BtrieveIndexAttributes *btrieveIndexAttributes, const char *fileName, Btrieve::CreateMode createMode, Btrieve::LocationMode locationMode = Btrieve::LOCATION_MODE_NO_PREFERENCE);
/// \cond
   /// \brief Create a file.
   /// \param[in] btrieveFileInformation The file information.
   /// \param[in] fileName The file name.
   /// \param[in] createMode The create mode.
   /// \param[in] locationMode The location mode.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileCreate(BtrieveFileInformation *btrieveFileInformation, const char *fileName, Btrieve::CreateMode createMode, Btrieve::LocationMode locationMode = Btrieve::LOCATION_MODE_NO_PREFERENCE);
/// \endcond
   /// \brief Create a file.
   /// \param[in] btrieveFileAttributes The file attributes.
   /// \param[in] fileName The file name.
   /// \param[in] createMode The create mode.
   /// \param[in] locationMode The location mode.
   /// \SNIPPET_CPP_PYTHON{FileCreate, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileCreate(BtrieveFileAttributes *btrieveFileAttributes, const wchar_t *fileName, Btrieve::CreateMode createMode, Btrieve::LocationMode locationMode = Btrieve::LOCATION_MODE_NO_PREFERENCE);
   /// \brief Create a file.
   /// \param[in] btrieveFileAttributes The file attributes.
   /// \param[in] btrieveIndexAttributes The index attributes.
   /// \param[in] fileName The file name.
   /// \param[in] createMode The create mode.
   /// \param[in] locationMode The location mode.
   /// \SNIPPET_CPP_PYTHON{FileCreate, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileCreate(BtrieveFileAttributes *btrieveFileAttributes, BtrieveIndexAttributes *btrieveIndexAttributes, const wchar_t *fileName, Btrieve::CreateMode createMode, Btrieve::LocationMode locationMode = Btrieve::LOCATION_MODE_NO_PREFERENCE);
/// \cond
   /// \brief Create a file.
   /// \param[in] btrieveFileInformation The file information.
   /// \param[in] fileName The file name.
   /// \param[in] createMode The create mode.
   /// \param[in] locationMode The location mode.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileCreate(BtrieveFileInformation *btrieveFileInformation, const wchar_t *fileName, Btrieve::CreateMode createMode, Btrieve::LocationMode locationMode = Btrieve::LOCATION_MODE_NO_PREFERENCE);
/// \endcond
   /// \brief Delete a file.
   /// \param[in] fileName The file name.
   /// \SNIPPET_CPP_PYTHON{FileDelete, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileDelete(const char *fileName);
   /// \brief Delete a file.
   /// \param[in] fileName The file name.
   /// \SNIPPET_CPP_PYTHON{FileDelete, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileDelete(const wchar_t *fileName);
   /// \brief Rename a file.
   /// \param[in] existingFileName The existing file name.
   /// \param[in] newFileName The new file name.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileRename(const char *existingFileName, const char *newFileName);
   /// \brief Rename a file.
   /// \param[in] existingFileName The existing file name.
   /// \param[in] newFileName The new file name.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode FileRename(const wchar_t *existingFileName, const wchar_t *newFileName);
   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
   /// \brief Login.
   /// \param[in] databaseURI The database uniform resource indicator.
   /// \SNIPPET_CPP_PYTHON{Login, bversion.cpp, bversion.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Login(const char *databaseURI);
   /// \brief Login.
   /// \param[in] databaseURI The database uniform resource indicator.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Login(const wchar_t *databaseURI);
   /// \brief Logout.
   /// \param[in] databaseURI The database uniform resource indicator.
   /// \SNIPPET_CPP_PYTHON{Logout, bversion.cpp, bversion.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Logout(const char *databaseURI);
   /// \brief Logout.
   /// \param[in] databaseURI The database uniform resource indicator.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Logout(const wchar_t *databaseURI);
   /// \brief Reset.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Reset();

// Avoid Windows renaming of SetCurrentDirectory.
#ifdef SetCurrentDirectory
   #undef SetCurrentDirectory
#endif

   /// \brief Set the current directory.
   /// \param[in] currentDirectory The current directory.
   /// \par \DESCRIPTION
   /// Current directory defaults to the current working directory.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetCurrentDirectory(const char *currentDirectory);
   /// \brief Set the current directory.
   /// \param[in] currentDirectory The current directory.
   /// \par \DESCRIPTION
   /// Current directory defaults to the current working directory.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetCurrentDirectory(const wchar_t *currentDirectory);
   /// \brief Stop.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Stop();
   /// \brief Abort a transaction.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode TransactionAbort();
   /// \brief Begin a transaction.
   /// \param[in] transactionMode The transaction mode.
   /// \param[in] lockMode The lock mode.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode TransactionBegin(Btrieve::TransactionMode transactionMode, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief End a transaction.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode TransactionEnd();
   /// \brief Get the service agent identifier.
   /// \retval ">= 0" The service agent identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveClient::GetLastStatusCode}
   int GetServiceAgentIdentifier();
   /// \brief Get the client identifier.
   /// \retval ">= 0" The client identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveClient::GetLastStatusCode}
   int GetClientIdentifier();
   /// \brief Create a collection.
   /// \param[in] collectionName The collection name.
   /// \SNIPPET_CPP_PYTHON{CollectionCreate, bcollection.cpp, bcollection.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode CollectionCreate(const char *collectionName);
   /// \brief Delete a collection.
   /// \param[in] collectionName The collection name.
   /// \SNIPPET_CPP_PYTHON{CollectionDelete, bcollection.cpp, bcollection.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode CollectionDelete(const char *collectionName);
   /// \brief Rename a collection.
   /// \param[in] existingCollectionName The existing collection name.
   /// \param[in] newCollectionName The new collection name.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode CollectionRename(const char *existingCollectionName, const char *newCollectionName);
   /// \brief Open a collection.
   /// \param[out] btrieveCollection The %Btrieve collection.
   /// \param[in] collectionName The collection name.
   /// \param[in] password The password.
   /// \SNIPPET_CPP_PYTHON{CollectionOpen, bcollection.cpp, bcollection.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode CollectionOpen(BtrieveCollection *btrieveCollection, const char *collectionName, const char *password = NULL);
   /// \brief Close a collection.
   /// \param[in] btrieveCollection The %Btrieve collection.
   /// \SNIPPET_CPP_PYTHON{CollectionClose, bcollection.cpp, bcollection.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode CollectionClose(BtrieveCollection *btrieveCollection);  
   /// \brief Create a space.
   /// \param[in] spaceName The space name.
   /// \param[in] dimensions The dimensions. The dimensions must be between 1 and Btrieve::MAXIMUM_SPACE_DIMENSIONS, inclusive.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SpaceCreate(const char *spaceName, int dimensions);
   /// \brief Delete a space.
   /// \param[in] spaceName The space name.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SpaceDelete(const char *spaceName);
   /// \brief Rename a space.
   /// \param[in] existingSpaceName The existing space name.
   /// \param[in] newSpaceName The new space name.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SpaceRename(const char *existingSpaceName, const char *newSpaceName);
   /// \brief Open a space.
   /// \param[out] btrieveSpace The %Btrieve space.
   /// \param[in] spaceName The space name.
   /// \param[in] password The password.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SpaceOpen(BtrieveSpace *btrieveSpace, const char *spaceName, const char *password = NULL);
   /// \brief Close a space.
   /// \param[in] btrieveSpace The %Btrieve space.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SpaceClose(BtrieveSpace *btrieveSpace);  

private:
   btrieve_client_t btrieveClient = NULL;
};

/// \brief The %Btrieve file class.
///
/// \details In order to retrieve, update, and delete records in a %Btrieve file the cursor must be established. The following methods are among those used to establish the cursor:
/// - BtrieveFile::RecordRetrieve
/// - BtrieveFile::RecordRetrieveFirst
/// - BtrieveFile::RecordRetrieveLast
///
/// \details Once established, the cursor enables traversal over the records in the file. The following methods are among those used to traverse over the file:
/// - BtrieveFile::RecordRetrieveNext
/// - BtrieveFile::RecordRetrievePrevious
///
/// \details The cursor may be re-established repeatedly.
class LINKAGE BtrieveFile
{
   friend class BtrieveClient;

protected:
   btrieve_file_t GetBtrieveFile();
   btrieve_file_t *GetBtrieveFilePtr();
   BtrieveFile(btrieve_file_t btrieveFileIn);

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveFile, btest.cpp, btest.py}
   /// \see BtrieveClient::FileOpen
   BtrieveFile();
   /// \see BtrieveClient::FileClose
   ~BtrieveFile();

   /// \brief Delete a record.
   /// \SNIPPET_CPP_PYTHON{RecordDelete, bkeyonly.cpp, bkeyonly.py}
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode RecordDelete();
   /// \brief Retrieve a record by fraction.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[in] numerator The numerator given as 0 through denominator.
   /// \param[in] denominator The denominator.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieveByFraction, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveByFraction(Btrieve::Index index, int numerator, int denominator, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a record by fraction.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[in] numerator The numerator given as 0 through denominator.
   /// \param[in] denominator The denominator.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveByFraction(Btrieve::Index index, int numerator, int denominator, char *key, int keySize, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a record by percentage.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[in] percentage The percentage given as 0 through 10,000, corresponding to 0% through 100.00%.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieveByPercentage, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveByPercentage(Btrieve::Index index, int percentage, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a record by percentage.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[in] percentage The percentage given as 0 through 10,000, corresponding to 0% through 100.00%.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveByPercentage(Btrieve::Index index, int percentage, char *key, int keySize, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a record by cursor position.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[in] cursorPosition The cursor position.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieveByCursorPosition, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveByCursorPosition(Btrieve::Index index, long long cursorPosition, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a record by cursor position.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[in] cursorPosition The cursor position.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveByCursorPosition(Btrieve::Index index, long long cursorPosition, char *key, int keySize, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a record using the given comparison, index, and key.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \note The signature for this method changed in PSQL v15 R0.
   /// \param[in] comparison The comparison. Btrieve::COMPARISON_NOT_EQUAL isn't supported.
   /// \param[in] index The index.
   /// \param[inout] key The key.
   /// \param[in] keyLength The key length.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieve, btest.cpp, btest.py}
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieve(Btrieve::Comparison comparison, Btrieve::Index index, char *key, int keyLength, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a key using the given comparison, index, and key.
   /// \note The signature for this method changed in PSQL v15 R0.
   /// \param[in] comparison The comparison. Btrieve::COMPARISON_NOT_EQUAL isn't supported.
   /// \param[in] index The index.
   /// \param[inout] key The key.
   /// \param[in] keyLength The key length.
   /// \SNIPPET_CPP_PYTHON{KeyRetrieve, bkeyonly.cpp, bkeyonly.py}
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode KeyRetrieve(Btrieve::Comparison comparison, Btrieve::Index index, char *key, int keyLength);
   /// \brief Retrieve bulk next.
   /// \param[in] bulkRetrieveAttributes The bulk retrieve attributes.
   /// \param[out] bulkRetrieveResult The bulk retrieve result.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{BulkRetrieveNext, btestbulk.cpp, btestbulk.py}
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed forward to the last record examined.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were retrieved. There are no more records matching the given bulk retrieve attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were retrieved. There may be more records matching the given bulk retrieve attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkRetrieveNext(BtrieveBulkRetrieveAttributes *bulkRetrieveAttributes, BtrieveBulkRetrieveResult *bulkRetrieveResult, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve bulk next.
   /// \param[in] comparison The comparison. Btrieve::COMPARISON_NOT_EQUAL isn't supported.
   /// \param[in] index The index.
   /// \param[inout] key The key.
   /// \param[in] keyLength The key length.
   /// \param[in] bulkRetrieveAttributes The bulk retrieve attributes.
   /// \param[out] bulkRetrieveResult The bulk retrieve result.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were retrieved. There are no more records matching the given bulk retrieve attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were retrieved. There may be more records matching the given bulk retrieve attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkRetrieveNext(Btrieve::Comparison comparison, Btrieve::Index index, char *key, int keyLength, BtrieveBulkRetrieveAttributes *bulkRetrieveAttributes, BtrieveBulkRetrieveResult *bulkRetrieveResult, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve bulk previous.
   /// \param[in] bulkRetrieveAttributes The bulk retrieve attributes.
   /// \param[out] bulkRetrieveResult The bulk retrieve result.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed backwards to the last record examined.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were retrieved. There are no more records matching the given bulk retrieve attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were retrieved. There may be more records matching the given bulk retrieve attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkRetrievePrevious(BtrieveBulkRetrieveAttributes *bulkRetrieveAttributes, BtrieveBulkRetrieveResult *bulkRetrieveResult, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve bulk previous.
   /// \param[in] comparison The comparison. Btrieve::COMPARISON_NOT_EQUAL isn't supported.
   /// \param[in] index The index.
   /// \param[inout] key The key.
   /// \param[in] keyLength The key length.
   /// \param[in] bulkRetrieveAttributes The bulk retrieve attributes.
   /// \param[out] bulkRetrieveResult The bulk retrieve result.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were retrieved. There are no more records matching the given bulk retrieve attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were retrieved. There may be more records matching the given bulk retrieve attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkRetrievePrevious(Btrieve::Comparison comparison, Btrieve::Index index, char *key, int keyLength, BtrieveBulkRetrieveAttributes *bulkRetrieveAttributes, BtrieveBulkRetrieveResult *bulkRetrieveResult, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve bulk first.
   /// \param[in] index The index.
   /// \param[in] bulkRetrieveAttributes The bulk retrieve attributes.
   /// \param[out] bulkRetrieveResult The bulk retrieve result.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were retrieved. There are no more records matching the given bulk retrieve attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were retrieved. There may be more records matching the given bulk retrieve attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkRetrieveFirst(Btrieve::Index index, BtrieveBulkRetrieveAttributes *bulkRetrieveAttributes, BtrieveBulkRetrieveResult *bulkRetrieveResult, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve bulk last.
   /// \param[in] index The index.
   /// \param[in] bulkRetrieveAttributes The bulk retrieve attributes.
   /// \param[out] bulkRetrieveResult The bulk retrieve result.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were retrieved. There are no more records matching the given bulk retrieve attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were retrieved. There may be more records matching the given bulk retrieve attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkRetrieveLast(Btrieve::Index index, BtrieveBulkRetrieveAttributes *bulkRetrieveAttributes, BtrieveBulkRetrieveResult *bulkRetrieveResult, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Delete bulk next.
   /// \param[in] bulkDeleteAttributes The bulk delete attributes.
   /// \param[out] bulkDeleteResult The bulk delete result.
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed forward to the last record examined.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were deleted. There are no more records matching the given bulk delete attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were deleted. There may be more records matching the given bulk delete attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkDeleteNext(BtrieveBulkDeleteAttributes *bulkDeleteAttributes, BtrieveBulkDeleteResult *bulkDeleteResult);
   /// \brief Delete bulk next.
   /// \param[in] comparison The comparison. Btrieve::COMPARISON_NOT_EQUAL isn't supported.
   /// \param[in] index The index.
   /// \param[inout] key The key.
   /// \param[in] keyLength The key length.
   /// \param[in] bulkDeleteAttributes The bulk delete attributes.
   /// \param[out] bulkDeleteResult The bulk delete result.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were deleted. There are no more records matching the given bulk delete attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were deleted. There may be more records matching the given bulk delete attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkDeleteNext(Btrieve::Comparison comparison, Btrieve::Index index, char *key, int keyLength, BtrieveBulkDeleteAttributes *bulkDeleteAttributes, BtrieveBulkDeleteResult *bulkDeleteResult);
   /// \brief Delete bulk previous.
   /// \param[in] bulkDeleteAttributes The bulk delete attributes.
   /// \param[out] bulkDeleteResult The bulk delete result.
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed backwards to the last record examined.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were deleted. There are no more records matching the given bulk delete attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were deleted. There may be more records matching the given bulk delete attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkDeletePrevious(BtrieveBulkDeleteAttributes *bulkDeleteAttributes, BtrieveBulkDeleteResult *bulkDeleteResult);
   /// \brief Delete bulk previous.
   /// \param[in] comparison The comparison. Btrieve::COMPARISON_NOT_EQUAL isn't supported.
   /// \param[in] index The index.
   /// \param[inout] key The key.
   /// \param[in] keyLength The key length.
   /// \param[in] bulkDeleteAttributes The bulk delete attributes.
   /// \param[out] bulkDeleteResult The bulk delete result.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were deleted. There are no more records matching the given bulk delete attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were deleted. There may be more records matching the given bulk delete attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkDeletePrevious(Btrieve::Comparison comparison, Btrieve::Index index, char *key, int keyLength, BtrieveBulkDeleteAttributes *bulkDeleteAttributes, BtrieveBulkDeleteResult *bulkDeleteResult);
   /// \brief Delete bulk first.
   /// \param[in] index The index.
   /// \param[in] bulkDeleteAttributes The bulk delete attributes.
   /// \param[out] bulkDeleteResult The bulk delete result.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were deleted. There are no more records matching the given bulk delete attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were deleted. There may be more records matching the given bulk delete attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkDeleteFirst(Btrieve::Index index, BtrieveBulkDeleteAttributes *bulkDeleteAttributes, BtrieveBulkDeleteResult *bulkDeleteResult);
   /// \brief Delete bulk last.
   /// \param[in] index The index.
   /// \param[in] bulkDeleteAttributes The bulk delete attributes.
   /// \param[out] bulkDeleteResult The bulk delete result.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_END_OF_FILE" \SUCCESS Zero or more records were deleted. There are no more records matching the given bulk delete attributes.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS One or more records were deleted. There may be more records matching the given bulk delete attributes.
   /// \retval "Otherwise" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkDeleteLast(Btrieve::Index index, BtrieveBulkDeleteAttributes *bulkDeleteAttributes, BtrieveBulkDeleteResult *bulkDeleteResult);
   /// \brief Get the file information.
   /// \param[out] btrieveFileInformation The file information.
   /// \SNIPPET_CPP_PYTHON{GetInformation, bfileinformation.cpp, bfileinformation.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetInformation(BtrieveFileInformation *btrieveFileInformation);
   /// \brief Retrieve the first record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieveFirst, btestbulk.cpp, btestbulk.py}
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveFirst(Btrieve::Index index, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve the first record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveFirst(Btrieve::Index index, char *key, int keySize, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve the first key.
   /// \param[in] index The index.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode KeyRetrieveFirst(Btrieve::Index index, char *key, int keySize);
   /// \brief Get the numerator which best approximates the location of the cursor position within the file relative to the denominator.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \param[in] cursorPosition The cursor position.
   /// \param[in] denominator The denominator.
   /// \SNIPPET_CPP_PYTHON{GetNumeratorCursorPosition, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval ">= 0" The numerator returned as 0 through denominator.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int GetNumerator(long long cursorPosition, int denominator);
   /// \brief Get the numerator which best approximates the location of the key within the index relative to the denominator.
   /// \param[in] index The index.
   /// \param[in] key The key.
   /// \param[in] keyLength The key length.
   /// \param[in] denominator The denominator.
   /// \SNIPPET_CPP_PYTHON{GetNumeratorIndex, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval ">= 0" The numerator returned as 0 through denominator.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int GetNumerator(Btrieve::Index index, const char *key, int keyLength, int denominator);
   /// \brief Retrieve the last record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieveLast, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveLast(Btrieve::Index index, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve the last record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] index An index or Btrieve::INDEX_NONE.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveLast(Btrieve::Index index, char *key, int keySize, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve the last key.
   /// \param[in] index The index.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode KeyRetrieveLast(Btrieve::Index index, char *key, int keySize);
   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \SNIPPET_CPP_PYTHON{GetLastStatusCode, btest.cpp, btest.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
   /// \brief Retrieve the next key.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed forward one key.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode KeyRetrieveNext(char *key, int keySize);
   /// \brief Retrieve the next record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieveNext, bindexattributes.cpp, bindexattributes.py}
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed forward one record.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveNext(char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve the next record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed forward one record.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveNext(char *key, int keySize, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Get the percentage which best approximates the location of the cursor position within the file.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \param[in] cursorPosition The cursor position.
   /// \SNIPPET_CPP_PYTHON{GetPercentageCursorPosition, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval ">= 0" The percentage returned as 0 through 10,000, corresponding to 0% through 100.00%.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int GetPercentage(long long cursorPosition);
   /// \brief Get the percentage which best approximates the location of the key within the given index.
   /// \param[in] index The index.
   /// \param[in] key The key.
   /// \param[in] keyLength The key length.
   /// \SNIPPET_CPP_PYTHON{GetPercentageIndex, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval ">= 0" The percentage returned as 0 through 10,000, corresponding to 0% through 100.00%.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int GetPercentage(Btrieve::Index index, const char *key, int keyLength);
   /// \brief Get the cursor position.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \SNIPPET_CPP_PYTHON{GetCursorPosition, bpercentage.cpp, bpercentage.py}
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval ">= 0" The cursor position.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   long long GetCursorPosition();
   /// \brief Get the record create time.
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "> 0" The record create time as a %Btrieve timestamp.
   /// \retval "= 0" The record create time is unavailable. Ensure the file was created with system data. See BtrieveFileAttributes::SetSystemDataMode.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   /// \see Btrieve::TimestampToUnixEpochMicroseconds
   long long GetRecordCreateTime();
   /// \brief Get the record update time.
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "> 0" The record update time as a %Btrieve timestamp.
   /// \retval "= 0" The record update time is unavailable. Ensure the file was created with system data version 2. See BtrieveFileAttributes::SetSystemDataMode.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   /// \see Btrieve::TimestampToUnixEpochMicroseconds
   long long GetRecordUpdateTime();
   /// \brief Retrieve the previous key.
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed backward one key.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode KeyRetrievePrevious(char *key, int keySize);
   /// \brief Retrieve the previous record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed backward one record.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrievePrevious(char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve the previous record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[out] key The key.
   /// \param[in] keySize The key size.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \param[in] lockMode The lock mode.
   /// \par \CURSOR
   /// The cursor must be established. The cursor is traversed backward one record.
   /// \par \CURRENT_OFFSET
   /// The current offset is destroyed.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN \DATA_LENGTH_ERROR
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrievePrevious(char *key, int keySize, char *record, int recordSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a record chunk.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] offset The offset. If offset is -1, then the current offset is used.
   /// \param[in] length The length.
   /// \param[out] chunk The chunk.
   /// \param[in] chunkSize The chunk size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieveChunk, btestchunk.cpp, btestchunk.py}
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is established at the end of the retrieved chunk.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ You may retrieve multiple chunks.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveChunk(int offset, int length, char *chunk, int chunkSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Retrieve a record chunk at the current offset.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] length The length.
   /// \param[out] chunk The chunk.
   /// \param[in] chunkSize The chunk size.
   /// \param[in] lockMode The lock mode.
   /// \SNIPPET_CPP_PYTHON{RecordRetrieveChunkNoOffset, btestchunk.cpp, btestchunk.py}
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset must be established and is then traversed forward to the end of the retrieved chunk.
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ You may retrieve multiple chunks.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFile::GetLastStatusCode}
   int RecordRetrieveChunk(int length, char *chunk, int chunkSize, Btrieve::LockMode lockMode = Btrieve::LOCK_MODE_NONE);
   /// \brief Create an index.
   /// \param[in] btrieveIndexAttributes The index attributes.
   /// \SNIPPET_CPP_PYTHON{IndexCreate, btest.cpp, btest.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode IndexCreate(BtrieveIndexAttributes *btrieveIndexAttributes);
   /// \brief Drop an index.
   /// \param[in] index The index.
   /// \par \CURSOR
   /// If the cursor index is dropped the cursor is destroyed. Otherwise, the cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode IndexDrop(Btrieve::Index index);
   /// \brief Create a record.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \param[in,out] record The record. If the current record contains a Btrieve::DATA_TYPE_AUTOINCREMENT or Btrieve::DATA_TYPE_AUTOTIMESTAMP key segment then the passed in record may be modifed.
   /// \param[in] recordLength The record length. Record length is limited to Btrieve::MAXIMUM_RECORD_LENGTH bytes. See BtrieveFile::RecordAppendChunk for creating records greater than Btrieve::MAXIMUM_RECORD_LENGTH in length.
   /// \SNIPPET_CPP_PYTHON{RecordCreate, btest.cpp, btest.py}
   /// \par \CURSOR
   /// The cursor is established.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode RecordCreate(char *record, int recordLength);
   /// \brief Bulk create.
   /// \param[in] btrieveBulkCreatePayload The bulk create payload.
   /// \param[out] btrieveBulkCreateResult The bulk create result.
   /// \SNIPPET_CPP_PYTHON{BulkCreate, btestbulk.cpp, btestbulk.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode BulkCreate(BtrieveBulkCreatePayload *btrieveBulkCreatePayload, BtrieveBulkCreateResult *btrieveBulkCreateResult);
   /// \brief Set the owner.
   /// \param[in] ownerMode The owner mode.
   /// \param[in] ownerName The owner name.
   /// \param[in] ownerNameAgain The owner name (again).
   /// \param[in] useLongOwnerName Use a long owner name?
   /// \par \DESCRIPTION
   /// Owner mode defaults to Btrieve::OWNER_MODE_NONE.
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetOwner(Btrieve::OwnerMode ownerMode, const char *ownerName = NULL, const char *ownerNameAgain = NULL, bool useLongOwnerName = true);
   /// \brief Truncate a record.
   /// \param[in] offset The offset. If offset is -1, then the current offset is used.
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is established at the end of the record.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   Btrieve::StatusCode RecordTruncate(int offset);
   /// \brief Truncate a record at the current offset.
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset must be established and remains unchanged.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   Btrieve::StatusCode RecordTruncate();
   /// \brief Unlock a record or records.
   /// \param[in] unlockMode The unlock mode.
   /// \SNIPPET_CPP_PYTHON{RecordUnlock, block.cpp, block.py}
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode RecordUnlock(Btrieve::UnlockMode unlockMode);
   /// \brief Unlock the cursor position.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \param[in] cursorPosition The cursor position.
   /// \par \CURSOR
   /// The cursor is unaffected.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode UnlockCursorPosition(long long cursorPosition);
   /// \brief Update a record.
   /// \param[in] record The record.
   /// \param[in] recordLength The record length. Record length is limited to Btrieve::MAXIMUM_RECORD_LENGTH bytes. See BtrieveFile::RecordUpdateChunk for updating records greater than Btrieve::MAXIMUM_RECORD_LENGTH in length.
   /// \SNIPPET_CPP_PYTHON{RecordUpdate, block.cpp, block.py}
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is unaffected.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode RecordUpdate(const char *record, int recordLength);
   /// \brief Append a record chunk.
   /// \param[in] chunk The chunk.
   /// \param[in] chunkLength The chunk length.
   /// \par \DESCRIPTION
   /// The file must have variable length records. See BtrieveFileAttributes::SetVariableLengthRecordsMode.
   /// \SNIPPET_CPP_PYTHON{RecordAppendChunk, btestappend.cpp, btestappend.py}
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is established at the end of the appended chunk.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode RecordAppendChunk(const char *chunk, int chunkLength);
   /// \brief Delete a record chunk.
   /// \param[in] offset The offset. If offset is -1, then the current offset is used. Offset must be within the variable length portion of the record.
   /// \param[in] chunkLength The chunk length.
   /// \par \DESCRIPTION
   /// The file must have variable length records. See BtrieveFileAttributes::SetVariableLengthRecordsMode.
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is established at the end of the deleted chunk.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   Btrieve::StatusCode RecordDeleteChunk(int offset, int chunkLength);
   /// \brief Delete a record chunk at the current offset.
   /// \param[in] chunkLength The chunk length.
   /// \par \DESCRIPTION
   /// The file must have variable length records. See BtrieveFileAttributes::SetVariableLengthRecordsMode.
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset must be established and is then traversed forward to the end of the deleted chunk.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   Btrieve::StatusCode RecordDeleteChunk(int chunkLength);
   /// \brief Insert a record chunk.
   /// \param[in] offset The offset. If offset is -1, then the current offset is used. Offset must be within the variable length portion of the record.
   /// \param[in] chunk The chunk.
   /// \param[in] chunkLength The chunk length.
   /// \par \DESCRIPTION
   /// The file must have variable length records. See BtrieveFileAttributes::SetVariableLengthRecordsMode.
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is established at the end of the inserted chunk.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   Btrieve::StatusCode RecordInsertChunk(int offset, const char *chunk, int chunkLength);
   /// \brief Insert a record chunk at the current offset.
   /// \param[in] chunk The chunk.
   /// \param[in] chunkLength The chunk length.
   /// \par \DESCRIPTION
   /// The file must have variable length records. See BtrieveFileAttributes::SetVariableLengthRecordsMode.
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset must be established and is then traversed forward to the end of the inserted chunk.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   Btrieve::StatusCode RecordInsertChunk(const char *chunk, int chunkLength);
   /// \brief Update a record chunk.
   /// \param[in] offset The offset. If offset is -1, then the current offset is used.
   /// \param[in] chunk The chunk.
   /// \param[in] chunkLength The chunk length.
   /// \SNIPPET_CPP_PYTHON{RecordUpdateChunk, btestchunk.cpp, btestchunk.py}
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset is established at the end of the updated chunk.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   Btrieve::StatusCode RecordUpdateChunk(int offset, const char *chunk, int chunkLength);
   /// \brief Update a record chunk at the current offset.
   /// \param[in] chunk The chunk.
   /// \param[in] chunkLength The chunk length.
   /// \SNIPPET_CPP_PYTHON{RecordUpdateChunkNoOffset, btestchunk.cpp, btestchunk.py}
   /// \par \CURSOR
   /// The cursor is unaffected but must be established.
   /// \par \CURRENT_OFFSET
   /// The current offset must be established and is then traversed forward to the end of the updated chunk.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   Btrieve::StatusCode RecordUpdateChunk(const char *chunk, int chunkLength);

private:
   btrieve_file_t btrieveFile = NULL;
};

/// \brief The %Btrieve file attributes class.
class LINKAGE BtrieveFileAttributes
{
   friend class BtrieveClient;

protected:
   btrieve_file_attributes_t GetBtrieveFileAttributes();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveFileAttributes, btest.cpp, btest.py}
   BtrieveFileAttributes();
   ~BtrieveFileAttributes();

   /// \brief Set balanced indexes.
   /// \param[in] enableBalancedIndexes Enable balanced indexes?
   /// \par \DESCRIPTION
   /// Balanced indexes are disabled by default.
   /// \SNIPPET_CPP_PYTHON{SetBalancedIndexes, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetBalancedIndexes(bool enableBalancedIndexes);
   /// \brief Set record compression mode.
   /// \param[in] recordCompressionMode The record compression mode.
   /// \par \DESCRIPTION
   /// Record compression defaults to Btrieve::RECORD_COMPRESSION_MODE_NONE.
   /// Compression mode blank truncation requires variable length records be enabled.
   /// \SNIPPET_CPP_PYTHON{SetRecordCompressionMode, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetRecordCompressionMode(Btrieve::RecordCompressionMode recordCompressionMode);
   /// \brief Set the reserved duplicate pointer count.
   /// \param[in] reservedDuplicatePointerCount The reserved duplicate pointer count. The reserved duplicate pointer count must be between 0 and 119, inclusive.
   /// \par \DESCRIPTION
   /// Reserved duplicate pointer count defaults to zero.
   /// The maximum reserved duplicate pointer count may be less than 119 depending on file version and page size.
   /// For example, a file version of Btrieve::FILE_VERSION_9_0 and page size of Btrieve::PAGE_SIZE_3584 allows for a maximum reserved duplicate pointer count of 54.
   /// \SNIPPET_CPP_PYTHON{SetReservedDuplicatePointerCount, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetReservedDuplicatePointerCount(int reservedDuplicatePointerCount);
   /// \brief Set the file version.
   /// \param[in] fileVersion The file version.
   /// \par \DESCRIPTION
   /// File version defaults to Btrieve::FILE_VERSION_DEFAULT. 
   /// A file version of Btrieve::FILE_VERSION_6_0 is mapped to Btrieve::FILE_VERSION_6_1.
   /// \SNIPPET_CPP_PYTHON{SetFileVersion, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetFileVersion(Btrieve::FileVersion fileVersion);
   /// \brief Set the free space threshold.
   /// \param[in] freeSpaceThreshold The free space threshold.
   /// \par \DESCRIPTION
   /// Free space threshold defaults to Btrieve::FREE_SPACE_THRESHOLD_DEFAULT.
   /// The free space threshold setting is ignored unless variable length records are enabled.
   /// \SNIPPET_CPP_PYTHON{SetFreeSpaceThreshold, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetFreeSpaceThreshold(Btrieve::FreeSpaceThreshold freeSpaceThreshold);
   /// \brief Set the page size.
   /// \param[in] pageSize The page size.
   /// \param[in] enablePageCompression Enable page compression?
   /// \par \DESCRIPTION
   /// Page size defaults to Btrieve::PAGE_SIZE_4096.
   /// Page compression is disabled by default and requires file version Btrieve::FILE_VERSION_9_5 or greater.
   /// Not all page sizes are valid for all file versions.
   /// Pages sizes greater than 4096 require file version Btrieve::FILE_VERSION_9_0 or greater.
   /// Pages sizes which are unsupported for a particular file version will be rounded up to the next supported size.
   /// \SNIPPET_CPP_PYTHON{SetPageSize, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetPageSize(Btrieve::PageSize pageSize, bool enablePageCompression = false);
   /// \brief Set the fixed record length.
   /// \param[in] fixedRecordLength The fixed record length. The fixed record length must be between 1 and Btrieve::MAXIMUM_RECORD_LENGTH, inclusive.
   /// \par \DESCRIPTION
   /// Fixed record length defaults to zero.
   /// \SNIPPET_CPP_PYTHON{SetFixedRecordLength, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetFixedRecordLength(int fixedRecordLength);
   /// \brief Set the preallocated page count.
   /// \param[in] preallocatedPageCount The preallocated page count. The preallocated page count must be between 0 and 65535, inclusive.
   /// \par \DESCRIPTION
   /// Preallocated page count defaults to zero.
   /// \SNIPPET_CPP_PYTHON{SetPreallocatedPageCount, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetPreallocatedPageCount(int preallocatedPageCount);
   /// \brief Set the system data mode.
   /// \param[in] systemDataMode The system data mode.
   /// \par \DESCRIPTION
   /// System data mode defaults to Btrieve::SYSTEM_DATA_MODE_DEFAULT.
   /// \SNIPPET_CPP_PYTHON{SetSystemDataMode, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetSystemDataMode(Btrieve::SystemDataMode systemDataMode);
   /// \brief Set the variable length records mode.
   /// \param[in] variableLengthRecordsMode The variable length records mode.
   /// \par \DESCRIPTION
   /// Variable length records mode defaults to Btrieve::VARIABLE_LENGTH_RECORDS_MODE_NO.
   /// \SNIPPET_CPP_PYTHON{SetVariableLengthRecordsMode, bfileattributes.cpp, bfileattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetVariableLengthRecordsMode(Btrieve::VariableLengthRecordsMode variableLengthRecordsMode);
   /// \brief Set key only.
   /// \param[in] enableKeyOnly Enable key only?
   /// \par \DESCRIPTION
   /// Key only is disabled by default.
   /// Enabling key only will cause the system data mode to be ignored.
   /// \SNIPPET_CPP_PYTHON{SetKeyOnly, bkeyonly.cpp, bkeyonly.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetKeyOnly(bool enableKeyOnly);

private:
   btrieve_file_attributes_t btrieveFileAttributes = NULL;
};

/// \brief The %Btrieve file information class.
class LINKAGE BtrieveFileInformation
{
   friend class BtrieveClient;
   friend class BtrieveFile;

protected:
   btrieve_file_information_t GetBtrieveFileInformation();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveFileInformation, bfileinformation.cpp, bfileinformation.py}
   /// \see BtrieveFile::GetInformation
   BtrieveFileInformation();
   ~BtrieveFileInformation();

   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \SNIPPET_CPP_PYTHON{FileInformationGetLastStatusCode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
   /// \brief Get the file version.
   /// \SNIPPET_CPP_PYTHON{GetFileVersion, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::FILE_VERSION_UNKNOWN" The file version.
   /// \retval "= Btrieve::FILE_VERSION_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::FileVersionToString
   Btrieve::FileVersion GetFileVersion();
   /// \brief Get the free space threshold.
   /// \SNIPPET_CPP_PYTHON{GetFreeSpaceThreshold, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::FREE_SPACE_THRESHOLD_UNKNOWN" The free space threshold.
   /// \retval "= Btrieve::FREE_SPACE_THRESHOLD_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::FreeSpaceThresholdToString
   Btrieve::FreeSpaceThreshold GetFreeSpaceThreshold();
   /// \brief Get the page size.
   /// \SNIPPET_CPP_PYTHON{GetPageSize, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::PAGE_SIZE_UNKNOWN" The page size.
   /// \retval "= Btrieve::PAGE_SIZE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::PageSizeToString
   Btrieve::PageSize GetPageSize();
   /// \brief Get the system data mode.
   /// \SNIPPET_CPP_PYTHON{GetSystemDataMode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::SYSTEM_DATA_MODE_UNKNOWN" \SUCCESS
   /// \retval "= Btrieve::SYSTEM_DATA_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::SystemDataModeToString
   Btrieve::SystemDataMode GetSystemDataMode();
   /// \brief Get a key segment.
   /// \param[in] btrieveKeySegment The key segment.
   /// \param[in] keySegmentNumber The key segment number. The number is one-based.
   /// \SNIPPET_CPP_PYTHON{GetKeySegment, bfileinformation.cpp, bfileinformation.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetKeySegment(BtrieveKeySegment *btrieveKeySegment, int keySegmentNumber);
   /// \brief Get balanced indexes.
   /// \SNIPPET_CPP_PYTHON{GetBalancedIndexes, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Balanced indexes enabled.
   /// \retval 0 Balanced indexes disabled.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetBalancedIndexes();
   /// \brief Get the record compression mode.
   /// \SNIPPET_CPP_PYTHON{GetRecordCompressionMode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::RECORD_COMPRESSION_MODE_UNKNOWN" \SUCCESS
   /// \retval "= Btrieve::RECORD_COMPRESSION_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::RecordCompressionModeToString
   Btrieve::RecordCompressionMode GetRecordCompressionMode();
   /// \brief Get key only.
   /// \SNIPPET_CPP_PYTHON{GetKeyOnly, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" File is key only.
   /// \retval 0 File isn't key only.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetKeyOnly();
   /// \brief Get page compression.
   /// \SNIPPET_CPP_PYTHON{GetPageCompression, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Page compression enabled.
   /// \retval 0 Page compression disabled.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetPageCompression();
   /// \brief Get page preallocation.
   /// \SNIPPET_CPP_PYTHON{GetPagePreallocation, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Page preallocation enabled.
   /// \retval 0 Page preallocation disabled.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetPagePreallocation();
   /// \brief Get the variable length records mode.
   /// \SNIPPET_CPP_PYTHON{GetVariableLengthRecordsMode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::SYSTEM_DATA_MODE_UNKNOWN" \SUCCESS
   /// \retval "= Btrieve::SYSTEM_DATA_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::VariableLengthRecordsModeToString
   Btrieve::VariableLengthRecordsMode GetVariableLengthRecordsMode();
   /// \brief Get the unused duplicate pointer count.
   /// \SNIPPET_CPP_PYTHON{GetUnusedDuplicatePointerCount, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The unused duplicate pointer count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetUnusedDuplicatePointerCount();
   /// \brief Get the index count.
   /// \SNIPPET_CPP_PYTHON{GetIndexCount, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The index count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetIndexCount();
   /// \brief Get the fixed record length.
   /// \SNIPPET_CPP_PYTHON{GetFixedRecordLength, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The fixed record length.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetFixedRecordLength();
   /// \brief Get the record count.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \SNIPPET_CPP_PYTHON{GetRecordCount, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The record count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   long long GetRecordCount();
   /// \brief Get segmented.
   /// \SNIPPET_CPP_PYTHON{GetSegmented, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" File is segmented.
   /// \retval 0 File isn't segmented.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetSegmented();
   /// \brief Get the unused page count.
   /// \SNIPPET_CPP_PYTHON{GetUnusedPageCount, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The unused page count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetUnusedPageCount();
   /// \brief Get the log index.
   /// \SNIPPET_CPP_PYTHON{GetLogIndex, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::INDEX_UNKNOWN" The index.
   /// \retval "= Btrieve::INDEX_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::IndexToString
   Btrieve::Index GetLogIndex();
   /// \brief Get loggable.
   /// \SNIPPET_CPP_PYTHON{GetLoggable, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" File is loggable.
   /// \retval 0 File isn't loggable.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLoggable();
   /// \brief Get system data.
   /// \note Renamed from BtrieveFileInformation::GetSystemIndexPresent in Zen v15 R1.
   /// \SNIPPET_CPP_PYTHON{GetSystemData, bfileinformation.cpp, bfileinformation.py}
   /// \retval 2 System data version 2 is present.
   /// \retval 1 System data is present.
   /// \retval 0 System data isn't present.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetSystemData();
   /// \brief Get the system index size.
   /// \SNIPPET_CPP_PYTHON{GetSystemIndexSize, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The system index size.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetSystemIndexSize();
   /// \brief Get system index.
   /// \note Renamed from BtrieveFileInformation::GetSystemIndexUsed in Zen v15 R1.
   /// \SNIPPET_CPP_PYTHON{GetSystemIndex, bfileinformation.cpp, bfileinformation.py}
   /// \retval 3 System indexes Btrieve::INDEX_SYSTEM and Btrieve::INDEX_SYSTEM_VERSION_2 are present.
   /// \retval 2 System index Btrieve::INDEX_SYSTEM_VERSION_2 is present.
   /// \retval 1 System index Btrieve::INDEX_SYSTEM is present.
   /// \retval 0 No system index is present.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetSystemIndex();
   /// \brief Get the system index version.
   /// \SNIPPET_CPP_PYTHON{GetSystemIndexVersion, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The system index version.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetSystemIndexVersion();
   /// \brief Get the duplicate record conflict cursor position.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \SNIPPET_CPP_PYTHON{GetDuplicateRecordConflictCursorPosition, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The duplicate record conflict cursor position.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   long long GetDuplicateRecordConflictCursorPosition();
   /// \brief Get the duplicate record conflict index.
   /// \SNIPPET_CPP_PYTHON{GetDuplicateRecordConflictIndex, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::INDEX_UNKNOWN" The duplicate record conflict index.
   /// \retval "= Btrieve::INDEX_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::IndexToString
   Btrieve::Index GetDuplicateRecordConflictIndex();
   /// \brief Get the identifier.
   /// \SNIPPET_CPP_PYTHON{GetIdentifier, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetIdentifier();
   /// \brief Get the handle count.
   /// \SNIPPET_CPP_PYTHON{GetHandleCount, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The handle count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetHandleCount();
   /// \brief Get the open timestamp.
   /// \SNIPPET_CPP_PYTHON{GetOpenTimestamp, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The open timestamp.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetOpenTimestamp();
   /// \brief Get the usage count.
   /// \SNIPPET_CPP_PYTHON{GetUsageCount, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The usage count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetUsageCount();
   /// \brief Get explicit locks.
   /// \SNIPPET_CPP_PYTHON{GetExplicitLocks, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Explicit locks are active.
   /// \retval 0 Explicit locks are not active.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetExplicitLocks();
   /// \brief Get client transactions.
   /// \SNIPPET_CPP_PYTHON{GetClientTransactions, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Client transactions are active.
   /// \retval 0 Client transactions are not active.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetClientTransactions();
   /// \brief Get read only.
   /// \SNIPPET_CPP_PYTHON{GetReadOnly, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The file is opened read only.
   /// \retval 0 The file isn't opened read only.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetReadOnly();
   /// \brief Get continuous operation.
   /// \SNIPPET_CPP_PYTHON{GetContinuousOperation, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The file is in continuous operation.
   /// \retval 0 The file isn't in continuous operation.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetContinuousOperation();
   /// \brief Get referential integrity constraints.
   /// \SNIPPET_CPP_PYTHON{GetReferentialIntegrityConstraints, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The file has referential integrity constraints.
   /// \retval 0 The file doesn't have referential integrity constraints.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetReferentialIntegrityConstraints();
   /// \brief Get the owner mode.
   /// \SNIPPET_CPP_PYTHON{GetOwnerMode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::OWNER_MODE_UNKNOWN" The owner mode.
   /// \retval "= Btrieve::OWNER_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::OwnerModeToString
   Btrieve::OwnerMode GetOwnerMode();
   /// \brief Get the file encryption type.
   /// \retval "!= Btrieve::ENCRYPT_TYPE_UNKNOWN" The owner mode.
   /// \retval "= Btrieve::ENCRYPT_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::EncryptTypeToString
   Btrieve::EncryptType GetEncryptionType();
   /// \brief Get wrong owner.
   /// \SNIPPET_CPP_PYTHON{GetWrongOwner, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The file was opened with an incorrect owner name.
   /// \retval 0 The file wasn't opened with an incorrect owner name.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetWrongOwner();
   /// \brief Get the gateway major version.
   /// \SNIPPET_CPP_PYTHON{GetGatewayMajorVersion, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The gateway major version.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetGatewayMajorVersion();
   /// \brief Get the gateway minor version.
   /// \SNIPPET_CPP_PYTHON{GetGatewayMinorVersion, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The gateway minor version.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetGatewayMinorVersion();
   /// \brief Get the gateway patch level.
   /// \SNIPPET_CPP_PYTHON{GetGatewayPatchLevel, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The gateway patch version.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetGatewayPatchLevel();
   /// \brief Get the gateway platform.
   /// \SNIPPET_CPP_PYTHON{GetGatewayPlatform, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The gateway platform.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetGatewayPlatform();
   /// \brief Get the gateway name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \SNIPPET_CPP_PYTHON{GetGatewayName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   const char *GetGatewayName();
#ifndef SWIG
   /// \brief Get the gateway name.
   /// \note \METHOD_ONLY_CPP
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetGatewayName(char *name, int nameSize);
#endif
   /// \brief Get lock owner implicit lock.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerImplicitLock, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The lock owner has an implicit lock.
   /// \retval 0 The lock owner doesn't have an implicit lock.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerImplicitLock();
   /// \brief Get lock owner file lock.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerFileLock, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The lock owner has a file lock.
   /// \retval 0 The lock owner doesn't have a file lock.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerFileLock();
   /// \brief Get lock owner record lock.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerRecordLock, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The lock owner has a record lock.
   /// \retval 0 The lock owner doesn't have a record lock.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerRecordLock();
   /// \brief Get lock owner same process.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerSameProcess, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The lock owner is the same process.
   /// \retval 0 The lock owner isn't the same process.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerSameProcess();
   /// \brief Get lock owner write no wait.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerWriteNoWait, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The lock owner is performing a write with no wait.
   /// \retval 0 The lock owner isn't performing a write with no wait.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerWriteNoWait();
   /// \brief Get lock owner write hold.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerWriteHold, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" The lock owner has a write hold.
   /// \retval 0 The lock owner doesn't have a write hold.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerWriteHold();
   /// \brief Get the lock owner time in transaction.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerTimeInTransaction, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The lock owner time in transaction.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerTimeInTransaction();
   /// \brief Get the lock owner transaction level.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerTransactionLevel, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The lock owner transaction level.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerTransactionLevel();
   /// \brief Get the lock owner name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   const char *GetLockOwnerName();
#ifndef SWIG
   /// \brief Get the lock owner name.
   /// \note \METHOD_ONLY_CPP
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLockOwnerName(char *name, int nameSize);
#endif
   /// \brief Get the lock owner service agent identifier.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerServiceAgentIdentifier, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The lock owner service agent identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerServiceAgentIdentifier();
   /// \brief Get the lock owner client identifier.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerClientIdentifier, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The lock owner client identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetLockOwnerClientIdentifier();
   /// \brief Get the lock owner explcit lock mode.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerExplicitLockMode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::LOCK_MODE_UNKNOWN" The lock owner explcit lock mode.
   /// \retval "= Btrieve::LOCK_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::LockModeToString
   Btrieve::LockMode GetLockOwnerExplicitLockMode();
   /// \brief Get the lock owner page lock type.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerPageLockType, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::PAGE_LOCK_TYPE_UNKNOWN" The lock owner page lock type.
   /// \retval "= Btrieve::PAGE_LOCK_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::PageLockTypeToString
   Btrieve::PageLockType GetLockOwnerPageLockType();
   /// \brief Get the lock owner index.
   /// \SNIPPET_CPP_PYTHON{GetLockOwnerIndex, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::INDEX_UNKNOWN" The lock owner index.
   /// \retval "= Btrieve::INDEX_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   /// \see Btrieve::IndexToString
   Btrieve::Index GetLockOwnerIndex();
   /// \brief Get the referential integrity file name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \SNIPPET_CPP_PYTHON{GetReferentialIntegrityFileName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   const char *GetReferentialIntegrityFileName();
#ifndef SWIG
   /// \brief Get the referential integrity file name.
   /// \note \METHOD_ONLY_CPP
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetReferentialIntegrityFileName(char *name, int nameSize);
#endif
   /// \brief Get the referential integrity operation code.
   /// \SNIPPET_CPP_PYTHON{GetReferentialIntegrityOperationCode, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The operation code.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetReferentialIntegrityOperationCode();
   /// \brief Get the referential integrity cursor position.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \SNIPPET_CPP_PYTHON{GetReferentialIntegrityCursorPosition, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The cursor position.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   long long GetReferentialIntegrityCursorPosition();
   /// \brief Get the segment count.
   /// \SNIPPET_CPP_PYTHON{GetSegmentCount, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The exists count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
   int GetSegmentCount();
   /// \brief Get the segment file name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] segmentFileNumber The segment file number.
   /// \SNIPPET_CPP_PYTHON{GetSegmentFileName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   const char *GetSegmentFileName(int segmentFileNumber);
#ifndef SWIG
   /// \brief Get the segment file name.
   /// \note \METHOD_ONLY_CPP
   /// \param[in] segmentFileNumber The segment file number.
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetSegmentFileName(int segmentFileNumber, char *name, int nameSize);
#endif

   // TODO: Coalesce some of the following.
   /// \brief Get security handle trusted.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleTrusted, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle trusted.
   /// \retval 0 Security handle not trusted.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleTrusted();
   /// \brief Get security handle implicit.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleImplicit, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle implicit.
   /// \retval 0 Security handle not implicit.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleImplicit();
   /// \brief Get security handle explicit.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleExplicit, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle explicit.
   /// \retval 0 Security handle not explicit.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleExplicit();
   /// \brief Get security handle authentication by database.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleAuthenticationByDatabase, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle authentication by database.
   /// \retval 0 Security handle not authentication by database.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleAuthenticationByDatabase();
   /// \brief Get security handle authorization by database.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleAuthorizationByDatabase, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle authorization by database.
   /// \retval 0 Security handle not authorization by database.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleAuthorizationByDatabase();
   /// \brief Get security handle Windows named pipe.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleWindowsNamedPipe, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle Windows named pipe.
   /// \retval 0 Security handle not Windows named pipe.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleWindowsNamedPipe();
   /// \brief Get security handle workgroup.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleWorkgroup, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle workgroup.
   /// \retval 0 Security handle not workgroup.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleWorkgroup();
   /// \brief Get security handle Btpasswd.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleBtpasswd, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle Btpasswd.
   /// \retval 0 Security handle not Btpasswd.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleBtpasswd();
   /// \brief Get security handle PAM (Pluggable Authentication Module).
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandlePAM, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle PAM.
   /// \retval 0 Security handle not PAM.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandlePAM();
   /// \brief Get security handle RTSS  (Runtime Server Support) complete.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleRTSSComplete, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle RTSS complete.
   /// \retval 0 Security handle not RTSS complete.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleRTSSComplete();
   /// \brief Get security handle RTSS  (Runtime Server Support) preautorized.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleRTSSPreauthorized, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle RTSS preautorized.
   /// \retval 0 Security handle not RTSS preautorized.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleRTSSPreauthorized();
   /// \brief Get security handle RTSS  (Runtime Server Support) disabled.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleRTSSDisabled, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security handle RTSS disabled.
   /// \retval 0 Security handle not RTSS disabled.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityHandleRTSSDisabled();
   /// \brief Get security current database trusted.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseTrusted, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database trusted.
   /// \retval 0 Security current database not trusted.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseTrusted();
   /// \brief Get security current database implicit.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseImplicit, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database implicit.
   /// \retval 0 Security current database not implicit.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseImplicit();
   /// \brief Get security current database explicit.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseExplicit, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database explicit.
   /// \retval 0 Security current database not explicit.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseExplicit();
   /// \brief Get security current database authentication by database.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseAuthenticationByDatabase, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database authentication by database.
   /// \retval 0 Security current database not authentication by database.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseAuthenticationByDatabase();
   /// \brief Get security current database authorization by database.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseAuthorizationByDatabase, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database authorization by database.
   /// \retval 0 Security current database not authorization by database.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseAuthorizationByDatabase();
   /// \brief Get security current database Windows named pipe.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseWindowsNamedPipe, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database Windows named pipe.
   /// \retval 0 Security current database not Windows named pipe.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseWindowsNamedPipe();
   /// \brief Get security current database workgroup.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseWorkgroup, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database workgroup.
   /// \retval 0 Security current database not workgroup.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseWorkgroup();
   /// \brief Get security current database Btpasswd.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseBtpasswd, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database Btpasswd.
   /// \retval 0 Security current database not Btpasswd.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseBtpasswd();
   /// \brief Get security current database PAM (Pluggable Authentication Module).
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabasePAM, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database PAM.
   /// \retval 0 Security current database not PAM.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabasePAM();
   /// \brief Get security current database RTSS (Runtime Server Support) complete.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseRTSSComplete, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database RTSS complete.
   /// \retval 0 Security current database not RTSS complete.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseRTSSComplete();
   /// \brief Get security current database RTSS  (Runtime Server Support) preauthorized.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseRTSSPreauthorized, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database RTSS preauthorized.
   /// \retval 0 Security current database not RTSS preauthorized.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseRTSSPreauthorized();
   /// \brief Get security current database RTSS  (Runtime Server Support) disabled.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseRTSSDisabled, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security current database RTSS disabled.
   /// \retval 0 Security current database not RTSS disabled.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityCurrentDatabaseRTSSDisabled();
   /// \brief Get security permission no rights.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionNoRights, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission no rights.
   /// \retval 0 Security permission rights.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionNoRights();
   /// \brief Get security permission open.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionOpen, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission open.
   /// \retval 0 Security permission no open.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionOpen();
   /// \brief Get security permission read.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionRead, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission read.
   /// \retval 0 Security permission no read.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionRead();
   /// \brief Get security permission file create.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionCreateFile, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission file create.
   /// \retval 0 Security permission no file create.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionCreateFile();
   /// \brief Get security permission update.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionUpdate, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission update.
   /// \retval 0 Security permission no update.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionUpdate();
   /// \brief Get security permission record create.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionCreateRecord, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission record create.
   /// \retval 0 Security permission no record create.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionCreateRecord();
   /// \brief Get security permission delete.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionDelete, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission delete.
   /// \retval 0 Security permission no delete.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionDelete();
   /// \brief Get security permission execute.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionExecute, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission execute.
   /// \retval 0 Security permission no execute.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionExecute();
   /// \brief Get security permission alter.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionAlter, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission alter.
   /// \retval 0 Security permission no alter.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionAlter();
   /// \brief Get security permission refer.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionRefer, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission refer.
   /// \retval 0 Security permission no refer.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionRefer();
   /// \brief Get security permission create view.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionCreateView, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission create view.
   /// \retval 0 Security permission no create view.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionCreateView();
   /// \brief Get security permission create stored procedure.
   /// \SNIPPET_CPP_PYTHON{GetSecurityPermissionCreateStoredProcedure, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Security permission create stored procedure.
   /// \retval 0 Security permission no create stored procedure.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	int GetSecurityPermissionCreateStoredProcedure();
   /// \brief Get the security handle database name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleDatabaseName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	const char *GetSecurityHandleDatabaseName();
#ifndef SWIG
   /// \brief Get the security handle database name.
   /// \note \METHOD_ONLY_CPP
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
	Btrieve::StatusCode GetSecurityHandleDatabaseName(char *name, int nameSize);
#endif
   /// \brief Get the security handle table name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleTableName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	const char *GetSecurityHandleTableName();
#ifndef SWIG
   /// \brief Get the security handle table name.
   /// \note \METHOD_ONLY_CPP
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
	Btrieve::StatusCode GetSecurityHandleTableName(char *name, int nameSize);
#endif
   /// \brief Get the security handle user name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \SNIPPET_CPP_PYTHON{GetSecurityHandleUserName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	const char *GetSecurityHandleUserName();
#ifndef SWIG
   /// \brief Get the security handle user name.
   /// \note \METHOD_ONLY_CPP
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
	Btrieve::StatusCode GetSecurityHandleUserName(char *name, int nameSize);
#endif
   /// \brief Get the security current database name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentDatabaseName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	const char *GetSecurityCurrentDatabaseName();
#ifndef SWIG
   /// \brief Get the security current database name.
   /// \note \METHOD_ONLY_CPP
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
	Btrieve::StatusCode GetSecurityCurrentDatabaseName(char *name, int nameSize);
#endif
   /// \brief Get the security current user name.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \SNIPPET_CPP_PYTHON{GetSecurityCurrentUserName, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= NULL" The name.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveFileInformation::GetLastStatusCode}
	const char *GetSecurityCurrentUserName();
#ifndef SWIG
   /// \brief Get the security current user name.
   /// \note \METHOD_ONLY_CPP
   /// \param[out] name The name.
   /// \param[in] nameSize The name size.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
	Btrieve::StatusCode GetSecurityCurrentUserName(char *name, int nameSize);
#endif

private:
   btrieve_file_information_t btrieveFileInformation = NULL;
};

/// \brief The %Btrieve filter class.
class LINKAGE BtrieveFilter
{
   friend class BtrieveBulkRetrieveAttributes;
   friend class BtrieveBulkDeleteAttributes;

protected:
   btrieve_filter_t GetBtrieveFilter();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveFilter, btestbulk.cpp, btestbulk.py}
   BtrieveFilter();
   ~BtrieveFilter();

   /// \brief Set the comparison.
   /// \param[in] comparison The comparison.
   /// \par \DESCRIPTION
   /// Comparison defaults to Btrieve::COMPARISON_NONE.
   /// \SNIPPET_CPP_PYTHON{SetComparison, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetComparison(Btrieve::Comparison comparison);
   /// \brief Set the comparison constant.
   /// \param[in] constant The comparison constant.
   /// \param[in] constantLength The comparison constant length. Comparison constant length is limited to Btrieve::MAXIMUM_KEY_LENGTH.
   /// \par \DESCRIPTION
   /// Filters have no comparison constant initially.
   /// \SNIPPET_CPP_PYTHON{SetComparisonConstant, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetComparisonConstant(const char *constant, int constantLength);
   /// \brief Set the comparison field.
   /// \param[in] offset The comparison field offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Filters have no comparison field initially.
   /// \SNIPPET_CPP_PYTHON{SetComparisonField, bfilter.cpp, bfilter.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetComparisonField(int offset);
   /// \brief Set the field.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] dataType The data type. Btrieve::DATA_TYPE_LEGACY_BINARY and Btrieve::DATA_TYPE_LEGACY_STRING are not supported.
   /// \par \DESCRIPTION
   /// Filters have no field initially.
   /// \OFFSET_PLUS_LENGTH
   /// \SNIPPET_CPP_PYTHON{SetField, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetField(int offset, int length, Btrieve::DataType dataType);
   /// \brief Set the system field.
   /// \param[in] systemField The system field.
   /// \par \DESCRIPTION
   /// Filters have no system field initially.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetSystemField(Btrieve::SystemField systemField);
   /// \brief Set the code page name to be used for the comparison types Like or Not Like.
   /// \param[in] name The name of the code page.
   /// \par \DESCRIPTION
   /// Supply the name of the code page to be used for string comparisons for the Like/Not Like comparisons. The comparison constant and data must use the same code page for accurate comparison.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetLikeCodePageName(const char *name);
   /// \brief Set the field connector.
   /// \param[in] connector The connector.
   /// \par \DESCRIPTION
   /// Field connector defaults to Btrieve::CONNECTOR_LAST.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetConnector(Btrieve::Connector connector);
   /// \brief Set the alternate collation sequence mode.
   /// \note This method replaced BtrieveFilter::SetCaseSensitivity in PSQL v13 R2.
   /// \param[in] ACSMode The alternate collation sequence mode.
   /// \par \DESCRIPTION
   /// Alternate collation sequence mode defaults to Btrieve::ACS_MODE_NONE. To set the alternate collation sequence mode to Btrieve::ACS_MODE_NAMED use BtrieveFilter::SetACSName or BtrieveFilter::SetACSUserDefined. Btrieve::ACS_MODE_NUMBERED is unsupported.
   /// \SNIPPET_CPP_PYTHON{SetACSMode, bfilter.cpp, bfilter.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetACSMode(Btrieve::ACSMode ACSMode);
   /// \brief Set the user defined alternate collation sequence name.
   /// \param[in] name The name.
   /// \par \DESCRIPTION
   /// Filters have no user defined alternate collation sequence name initially. To reference a user defined alternate collation sequence it must exist in the file. A user defined alternate collation sequence may be added to a file when an index is created. See BtrieveFile::IndexCreate.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetACSUserDefined(const char *name);
   /// \brief Set the alternate collation sequence name.
   /// \param[in] name The name.
   /// \par \DESCRIPTION
   /// Filters have no alternate collation sequence name initially. To reference an alternate collation sequence name it must exist in the file. An alternate collation sequence name may be added to a file when an index is created. See BtrieveFile::IndexCreate.
   /// \SNIPPET_CPP_PYTHON{SetACSName, bfilter.cpp, bfilter.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetACSName(const char *name);

private:
   btrieve_filter_t btrieveFilter = NULL;
};

/// \brief The %Btrieve bulk retrieve attributes class.
class LINKAGE BtrieveBulkRetrieveAttributes
{
   friend class BtrieveFile;

protected:
   btrieve_bulk_retrieve_attributes_t GetBtrieveBulkRetrieveAttributes();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveBulkRetrieveAttributes, btestbulk.cpp, btestbulk.py}
   BtrieveBulkRetrieveAttributes();
   ~BtrieveBulkRetrieveAttributes();

   /// \brief Add a cosine similarity.
   /// \param[in] vector The vector. Must point at an array of 32-bit floating point numbers.
   /// \param[in] vectorLength The vector length in bytes. Vector length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Computes the similarity between the passed in vector and the vector specified by offset.
   /// Bulk retrieve attributes have no similarities initially.
   /// \OFFSET_PLUS_LENGTH
   /// The vectors specified must be in little endian format.
   /// Only one similarity may be added to the same bulk retrieve.
   /// Aggregates and fields may not be added to a bulk retrieve attributes containing a similarity.
   /// The returned similarity is a 32-bit floating point number.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddCosineSimilarity(const char *vector, int vectorLength, int offset);
   /// \brief Add a dot product similarity.
   /// \param[in] vector The vector. Must point at an array of 32-bit floating point numbers.
   /// \param[in] vectorLength The vector length in bytes. Vector length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Computes the similarity between the passed in vector and the vector specified by offset.
   /// Bulk retrieve attributes have no similarities initially.
   /// \OFFSET_PLUS_LENGTH
   /// The vectors specified must be in little endian format.
   /// Only one similarity may be added to the same bulk retrieve.
   /// Aggregates and fields may not be added to a bulk retrieve attributes containing a similarity.
   /// The returned similarity is a 32-bit floating point number.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddDotProductSimilarity(const char *vector, int vectorLength, int offset);
   /// \brief Add a Euclidean distance similarity.
   /// \param[in] vector The vector. Must point at an array of 32-bit floating point numbers.
   /// \param[in] vectorLength The vector length in bytes. Vector length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Computes the similarity between the passed in vector and the vector specified by offset.
   /// Bulk retrieve attributes have no similarities initially.
   /// \OFFSET_PLUS_LENGTH
   /// The vectors specified must be in little endian format.
   /// Only one similarity may be added to the same bulk retrieve.
   /// Aggregates and fields may not be added to a bulk retrieve attributes containing a similarity.
   /// The returned similarity is a 32-bit floating point number.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddEuclideanDistanceSimilarity(const char *vector, int vectorLength, int offset);
   /// \brief Add a minimum aggregate.
   /// \param[in] dataType The data type. Only Btrieve::DATA_TYPE_INTEGER and Btrieve::DATA_TYPE_FLOAT are supported.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no aggregates initially.
   /// \OFFSET_PLUS_LENGTH
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// The field specified by offset and length must be in little endian format.
   /// Similarities and fields may not be added to a bulk retrieve attributes containing an aggregate.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddMinimumAggregate(Btrieve::DataType dataType, int offset, int length);
   /// \brief Add a maximum aggregate.
   /// \param[in] dataType The data type. Only Btrieve::DATA_TYPE_INTEGER and Btrieve::DATA_TYPE_FLOAT are supported.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no aggregates initially.
   /// \OFFSET_PLUS_LENGTH
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// The field specified by offset and length must be in little endian format.
   /// Similarities and fields may not be added to a bulk retrieve attributes containing an aggregate.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddMaximumAggregate(Btrieve::DataType dataType, int offset, int length);
   /// \brief Add a field.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no fields initially.
   /// \OFFSET_PLUS_LENGTH
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
   /// \SNIPPET_CPP_PYTHON{AddField, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddField(int offset, int length);
   /// \brief Add an absolute value field.
   /// \param[in] dataType  The data type. Only Btrieve::DATA_TYPE_INTEGER and Btrieve::DATA_TYPE_FLOAT are supported.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no fields initially.
   /// \OFFSET_PLUS_LENGTH
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// The field specified by offset and length must be in little endian format.
   /// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddAbsoluteValueField(Btrieve::DataType dataType, int offset, int length);
   /// \brief Add a ceiling field.
   /// \param[in] dataType  The data type. Only Btrieve::DATA_TYPE_INTEGER and Btrieve::DATA_TYPE_FLOAT are supported.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no fields initially.
   /// \OFFSET_PLUS_LENGTH
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// The field specified by offset and length must be in little endian format.
   /// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddCeilingField(Btrieve::DataType dataType, int offset, int length);
   /// \brief Add a floor field.
   /// \param[in] dataType  The data type. Only Btrieve::DATA_TYPE_INTEGER and Btrieve::DATA_TYPE_FLOAT are supported.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no fields initially.
   /// \OFFSET_PLUS_LENGTH
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// The field specified by offset and length must be in little endian format.
   /// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddFloorField(Btrieve::DataType dataType, int offset, int length);
   /// \brief Add a round field.
   /// \param[in] dataType  The data type. Only Btrieve::DATA_TYPE_INTEGER and Btrieve::DATA_TYPE_FLOAT are supported.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] precision The precision. Precision must be in the range INT16_MIN to INT16_MAX, inclusive.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no fields initially.
   /// \OFFSET_PLUS_LENGTH
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// The field specified by offset and length must be in little endian format.
   /// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddRoundField(Btrieve::DataType dataType, int offset, int length, int precision = 0);
   /// \brief Add a truncate field.
   /// \param[in] dataType  The data type. Only Btrieve::DATA_TYPE_INTEGER and Btrieve::DATA_TYPE_FLOAT are supported.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] precision The precision. Precision must be in the range INT16_MIN to INT16_MAX, inclusive.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no fields initially.
   /// \OFFSET_PLUS_LENGTH
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// The field specified by offset and length must be in little endian format.
   /// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddTruncateField(Btrieve::DataType dataType, int offset, int length, int precision = 0);
   /// \brief Add a system field.
   /// \param[in] systemField The system field.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no system fields initially.
   /// The sum of the lengths of all added fields is limited to Btrieve::MAXIMUM_RECORD_LENGTH.
   /// Similarities and aggregates may not be added to a bulk retrieve attributes containing a field.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddSystemField(Btrieve::SystemField systemField);
   /// \brief Add a filter.
   /// \param[in] btrieveFilter The filter.
   /// \par \DESCRIPTION
   /// Bulk retrieve attributes have no filters initially.
   /// \SNIPPET_CPP_PYTHON{AddFilter, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddFilter(BtrieveFilter *btrieveFilter);
   /// \brief Set the maximum record count.
   /// \param[in] maximumRecordCount The maximum record count. The maximum record count must be greater than 0 and less than or equal to Btrieve::MAXIMUM_BULK_RECORD_COUNT.
   /// \par \DESCRIPTION
   /// Maximum record count defaults to zero.
   /// Aggregates do not honor the maximum record count.
   /// \SNIPPET_CPP_PYTHON{SetMaximumRecordCount, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetMaximumRecordCount(int maximumRecordCount);
   /// \brief Set the maximum reject count.
   /// \param[in] maximumRejectCount The maximum reject count. The maximum reject count must be greater than or equal to 0 and less than 65536. A value of zero sets the maximum reject count to the Zen defined value.
   /// \par \DESCRIPTION
   /// Maximum reject count defaults to the Zen defined value.
   /// Similarities and aggregates do not honor the maximum reject count.
   /// \SNIPPET_CPP_PYTHON{SetMaximumRejectCount, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetMaximumRejectCount(int maximumRejectCount);
   /// \brief Set skip current record.
   /// \param[in] skipCurrentRecord Skip current record?
   /// \par \DESCRIPTION
   /// Skip current record is disabled by default.
   /// \SNIPPET_CPP_PYTHON{SetSkipCurrentRecord, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetSkipCurrentRecord(bool skipCurrentRecord);

private:
   btrieve_bulk_retrieve_attributes_t bulkRetrieveAttributes = NULL;
};

/// \brief The %Btrieve bulk delete attributes class.
class LINKAGE BtrieveBulkDeleteAttributes
{
   friend class BtrieveFile;

protected:
   btrieve_bulk_delete_attributes_t GetBtrieveBulkDeleteAttributes();

public:
   BtrieveBulkDeleteAttributes();
   ~BtrieveBulkDeleteAttributes();

   /// \brief Add a filter.
   /// \param[in] btrieveFilter The filter.
   /// \par \DESCRIPTION
   /// Delete bulk attributes have no filters initially.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddFilter(BtrieveFilter *btrieveFilter);
   /// \brief Set the maximum record count.
   /// \param[in] maximumRecordCount The maximum record count. The maximum record count must be greater than 0 and less than or equal to Btrieve::MAXIMUM_BULK_RECORD_COUNT.
   /// \par \DESCRIPTION
   /// Maximum record count defaults to zero.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetMaximumRecordCount(int maximumRecordCount);
   /// \brief Set the maximum reject count.
   /// \param[in] maximumRejectCount The maximum reject count. The maximum reject count must be greater than or equal to 0 and less than 65536. A value of zero sets the maximum reject count to the Zen defined value.
   /// \par \DESCRIPTION
   /// Maximum reject count defaults to the Zen defined value.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetMaximumRejectCount(int maximumRejectCount);
   /// \brief Set skip current record.
   /// \param[in] skipCurrentRecord Skip current record?
   /// \par \DESCRIPTION
   /// Skip current record is disabled by default.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetSkipCurrentRecord(bool skipCurrentRecord);

private:
   btrieve_bulk_delete_attributes_t bulkDeleteAttributes = NULL;
};

/// \brief The %Btrieve bulk retrieve result class.
class LINKAGE BtrieveBulkRetrieveResult
{
   friend class BtrieveFile;

protected:
   btrieve_bulk_retrieve_result_t GetBtrieveBulkRetrieveResult();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveBulkRetrieveResult, btestbulk.cpp, btestbulk.py}
   BtrieveBulkRetrieveResult();
   ~BtrieveBulkRetrieveResult();

   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \SNIPPET_CPP_PYTHON{BulkRetrieveResultGetLastStatusCode, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
   /// \brief Get the record.
   /// \note \DOES_NOT_RETURN_A_BTRIEVE_STATUS_CODE
   /// \param[in] recordNumber The record number.
   /// \param[out] record The record.
   /// \param[in] recordSize The record size.
   /// \SNIPPET_CPP_PYTHON{GetRecord, btestbulk.cpp, btestbulk.py}
   /// \retval ">= 0" \NUMBER_OF_BYTES_READ_LIMITED \RETRIEVING_RECORDS_GREATER_THAN
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveBulkRetrieveResult::GetLastStatusCode}
   int GetRecord(int recordNumber, char *record, int recordSize);
   /// \brief Get the record count.
   /// \SNIPPET_CPP_PYTHON{BulkRetrieveResultGetRecordCount, btestbulk.cpp, btestbulk.py}
   /// \retval ">= 0" The record count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveBulkRetrieveResult::GetLastStatusCode}
   int GetRecordCount();
   /// \brief Get the record length.
   /// \param[in] recordNumber The record number.
   /// \SNIPPET_CPP_PYTHON{GetRecordLength, btestbulk.cpp, btestbulk.py}
   /// \retval ">= 0" The record length.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveBulkRetrieveResult::GetLastStatusCode}
   int GetRecordLength(int recordNumber);
   /// \brief Get the record cursor position.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \param[in] recordNumber The record number.
   /// \SNIPPET_CPP_PYTHON{BulkRetrieveResultGetRecordCursorPosition, btestbulk.cpp, btestbulk.py}
   /// \retval ">= 0" The record cursor position.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveBulkRetrieveResult::GetLastStatusCode}
   long long GetRecordCursorPosition(int recordNumber);

private:
   btrieve_bulk_retrieve_result_t bulkRetrieveResult = NULL;
};

/// \brief The %Btrieve bulk delete result class.
class LINKAGE BtrieveBulkDeleteResult
{
   friend class BtrieveFile;

protected:
   btrieve_bulk_delete_result_t GetBtrieveBulkDeleteResult();

public:
   BtrieveBulkDeleteResult();
   ~BtrieveBulkDeleteResult();

   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
   /// \brief Get the record count.
   /// \retval ">= 0" The record count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveBulkDeleteResult::GetLastStatusCode}
   int GetRecordCount();
   /// \brief Get the record cursor position.
   /// \param[in] recordNumber The record number.
   /// \retval ">= 0" The record cursor position.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveBulkDeleteResult::GetLastStatusCode}
   long long GetRecordCursorPosition(int recordNumber);

private:
   btrieve_bulk_delete_result_t bulkDeleteResult = NULL;
};

/// \brief The %Btrieve index attributes class.
class LINKAGE BtrieveIndexAttributes
{
   friend class BtrieveClient;
   friend class BtrieveFile;

protected:
   btrieve_index_attributes_t GetBtrieveIndexAttributes();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveIndexAttributes, btest.cpp, btest.py}
   BtrieveIndexAttributes();
   ~BtrieveIndexAttributes();

   /// \brief Add a key segment.
   /// \param[in] btrieveKeySegment The key segment.
   /// \par \DESCRIPTION
   /// Index attributes have no key segments initially.
   /// \SNIPPET_CPP_PYTHON{AddKeySegment, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddKeySegment(BtrieveKeySegment *btrieveKeySegment);
   /// \brief Set the index.
   /// \param[in] index The index.
   /// \par \DESCRIPTION
   /// Index defaults to the first available index starting at Btrieve::INDEX_1.
   /// \SNIPPET_CPP_PYTHON{SetIndex, bindexattributes.cpp, bindexattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetIndex(Btrieve::Index index);
   /// \brief Set the duplicate mode.
   /// \param[in] duplicateMode The duplicate mode.
   /// \par \DESCRIPTION
   /// Duplicate mode defaults to Btrieve::DUPLICATE_MODE_NOT_ALLOWED.
   /// \SNIPPET_CPP_PYTHON{SetDuplicateMode, bindexattributes.cpp, bindexattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetDuplicateMode(Btrieve::DuplicateMode duplicateMode);
   /// \brief Set modifiable.
   /// \param[in] enableModifiable Enable modifiable?
   /// \par \DESCRIPTION
   /// Modifiable is enabled by default.
   /// \SNIPPET_CPP_PYTHON{SetModifiable, bindexattributes.cpp, bindexattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetModifiable(bool enableModifiable);
   /// \brief Set the alternate collation sequence mode.
   /// \param[in] ACSMode The alternate collation sequence mode.
   /// \par \DESCRIPTION
   /// Alternate collation sequence mode defaults to Btrieve::ACS_MODE_NONE. To set the alternate collation sequence mode to Btrieve::ACS_MODE_NAMED use BtrieveIndexAttributes::SetACSName or BtrieveIndexAttributes::SetACSUserDefined. To set the alternate collation sequence mode to Btrieve::ACS_MODE_NUMBERED use BtrieveIndexAttributes::SetACSNumber.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetACSMode(Btrieve::ACSMode ACSMode);
   /// \brief Set the alternate collation sequence name.
   /// \note This method accepts international sort rule and unicode alternate collation sequence names only. Use BtrieveIndexAttributes::SetACSUserDefined to specify a user defined alternate collation sequence.
   /// \param[in] ACSName The alternate collation sequence name.
   /// \par \DESCRIPTION
   /// By default there is no alternate collation sequence name. Setting the alternate collation sequence name also sets the alternate collation sequence mode to Btrieve::ACS_MODE_NAMED.
   /// \SNIPPET_CPP_PYTHON{SetACSName, bindexattributes.cpp, bindexattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetACSName(const char *ACSName);
   /// \brief Set the alternate collation sequence number.
   /// \param[in] ACSNumber The alternate collation sequence number.
   /// \par \DESCRIPTION
   /// By default there is no alternate collation sequence number. Setting the alternate collation sequence number also sets the alternate collation sequence mode to Btrieve::ACS_MODE_NUMBERED.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetACSNumber(int ACSNumber);
   /// \brief Set the user defined alternate collation sequence.
   /// \param[in] ACSName The alternate collation sequence name.
   /// \param[in] ACSMap The alternate collation sequence map.
   /// \param[in] ACSMapLength The alternate collation sequence map length.
   /// \par \DESCRIPTION
   /// By default there is no user defined alternate collation sequence. Setting the user defined alternate collation sequence also sets the alternate collation sequence mode to Btrieve::ACS_MODE_NAMED.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetACSUserDefined(const char *ACSName, const char *ACSMap, int ACSMapLength);

private:
   btrieve_index_attributes_t btrieveIndexAttributes = NULL;
};

/// \brief The %Btrieve bulk create payload class.
class LINKAGE BtrieveBulkCreatePayload
{
   friend class BtrieveFile;

protected:
   btrieve_bulk_create_payload_t GetBtrieveBulkCreatePayload();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveBulkCreatePayload, btestbulk.cpp, btestbulk.py}
   BtrieveBulkCreatePayload();
   ~BtrieveBulkCreatePayload();

   /// \brief Add a record.
   /// \param[in] record The record.
   /// \param[in] recordLength The record length. Record length is limited to Btrieve::MAXIMUM_RECORD_LENGTH bytes.
   /// \par \DESCRIPTION
   /// Bulk create attributes have no records initially.
   /// \SNIPPET_CPP_PYTHON{AddRecord, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode AddRecord(const char *record, int recordLength);

private:
   btrieve_bulk_create_payload_t btrieveBulkCreatePayload = NULL;
};

/// \brief The %Btrieve bulk create result class.
class LINKAGE BtrieveBulkCreateResult
{
   friend class BtrieveFile;

protected:
   btrieve_bulk_create_result_t GetBtrieveBulkCreateResult();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveBulkCreateResult, btestbulk.cpp, btestbulk.py}
   BtrieveBulkCreateResult();
   ~BtrieveBulkCreateResult();

   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \SNIPPET_CPP_PYTHON{BulkCreateResultGetLastStatusCode, btestbulk.cpp, btestbulk.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
   /// \brief Get the record count.
   /// \SNIPPET_CPP_PYTHON{BulkCreateResultGetRecordCount, btestbulk.cpp, btestbulk.py}
   /// \retval ">= 0" The record count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveBulkCreateResult::GetLastStatusCode}
   int GetRecordCount();
   /// \brief Get the record cursor position.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \param[in] recordNumber The record number.
   /// \SNIPPET_CPP_PYTHON{BulkCreateResultGetRecordCursorPosition, btestbulk.cpp, btestbulk.py}
   /// \retval ">= 0" The record cursor position.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveBulkCreateResult::GetLastStatusCode}
   long long GetRecordCursorPosition(int recordNumber);

private:
   btrieve_bulk_create_result_t btrieveBulkCreateResult = NULL;
};

/// \brief The %Btrieve key segment class.
class LINKAGE BtrieveKeySegment
{
   friend class BtrieveIndexAttributes;
   friend class BtrieveFileInformation;

protected:
   btrieve_key_segment_t GetBtrieveKeySegment();
   BtrieveKeySegment(btrieve_key_segment_t btrieveKeySegmentIn);

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveKeySegment, btest.cpp, btest.py}
   BtrieveKeySegment();
   ~BtrieveKeySegment();

   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \SNIPPET_CPP_PYTHON{KeySegmentGetLastStatusCode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
   /// \brief Get the alternate collation sequence mode.
   /// \SNIPPET_CPP_PYTHON{GetACSMode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::ACS_MODE_UNKNOWN" The alternate collation sequence mode.
   /// \retval "= Btrieve::ACS_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   /// \see Btrieve::ACSModeToString
   Btrieve::ACSMode GetACSMode();
   /// \brief Get modifiable.
   /// \SNIPPET_CPP_PYTHON{GetModifiable, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Modifications are allowed.
   /// \retval 0 Modifications are not allowed.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   int GetModifiable();
   /// \brief Get key continues.
   /// \SNIPPET_CPP_PYTHON{GetKeyContinues, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Key continues is set.
   /// \retval 0 Key continues isn't set.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   int GetKeyContinues();
   /// \brief Get the data type.
   /// \SNIPPET_CPP_PYTHON{GetDataType, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::DATA_TYPE_UNKNOWN" The data type.
   /// \retval "= Btrieve::DATA_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   /// \see Btrieve::DataTypeToString
   Btrieve::DataType GetDataType();
   /// \brief Get the alternate collation sequence number.
   /// \SNIPPET_CPP_PYTHON{GetACSNumber, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The alternate collation sequence number.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   int GetACSNumber();
   /// \brief Get descending sort order.
   /// \SNIPPET_CPP_PYTHON{GetDescendingSortOrder, bfileinformation.cpp, bfileinformation.py}
   /// \retval "> 0" Descending sort order is set.
   /// \retval 0 Descending sort order isn't set.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   int GetDescendingSortOrder();
   /// \brief Get the duplicate mode.
   /// \SNIPPET_CPP_PYTHON{GetDuplicateMode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::DUPLICATE_MODE_UNKNOWN" The duplicate mode.
   /// \retval "= Btrieve::DUPLICATE_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   /// \see Btrieve::DuplicateModeToString
   Btrieve::DuplicateMode GetDuplicateMode();
   /// \brief Get the index.
   /// \SNIPPET_CPP_PYTHON{GetIndex, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::INDEX_UNKNOWN" The index.
   /// \retval "= Btrieve::INDEX_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   /// \see Btrieve::IndexToString
   Btrieve::Index GetIndex();
   /// \brief Get the length.
   /// \SNIPPET_CPP_PYTHON{GetLength, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The length.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   int GetLength();
   /// \brief Get the null key mode.
   /// \SNIPPET_CPP_PYTHON{GetNullKeyMode, bfileinformation.cpp, bfileinformation.py}
   /// \retval "!= Btrieve::NULL_KEY_MODE_UNKNOWN" The null key mode.
   /// \retval "= Btrieve::NULL_KEY_MODE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   /// \see Btrieve::NullKeyModeToString
   Btrieve::NullKeyMode GetNullKeyMode();
   /// \brief Get the null value.
   /// \SNIPPET_CPP_PYTHON{GetNullValue, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The null value.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   int GetNullValue();
   /// \brief Get the offset.
   /// \SNIPPET_CPP_PYTHON{GetOffset, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The offset.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   int GetOffset();
   /// \brief Get the unique value count.
   /// \note The signature for this method changed in PSQL v13 R2.
   /// \SNIPPET_CPP_PYTHON{GetUniqueValueCount, bfileinformation.cpp, bfileinformation.py}
   /// \retval ">= 0" The unique value count.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveKeySegment::GetLastStatusCode}
   long long GetUniqueValueCount();
   /// \brief Set descending sort order.
   /// \param[in] setDescendingSortOrder Enable descending sort order?
   /// \par \DESCRIPTION
   /// Descending sort order is disabled by default.
   /// \SNIPPET_CPP_PYTHON{SetDescendingSortOrder, bindexattributes.cpp, bindexattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetDescendingSortOrder(bool setDescendingSortOrder);
   /// \brief Set the field.
   /// \param[in] offset The offset. Offset must be less than Btrieve::MAXIMUM_RECORD_LENGTH.
   /// \param[in] length The length. Length is limited to Btrieve::MAXIMUM_KEY_LENGTH.
   /// \param[in] dataType The data type.
   /// \par \DESCRIPTION
   /// Key segments have no field initially.
   /// \OFFSET_PLUS_LENGTH
   /// \SNIPPET_CPP_PYTHON{SetField, btest.cpp, btest.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetField(int offset, int length, Btrieve::DataType dataType);
   /// \brief Set the null key mode.
   /// \param[in] nullKeyMode The null key mode.
   /// \par \DESCRIPTION
   /// Null key mode defaults to Btrieve::NULL_KEY_MODE_NONE.
   /// \SNIPPET_CPP_PYTHON{SetNullKeyMode, bindexattributes.cpp, bindexattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetNullKeyMode(Btrieve::NullKeyMode nullKeyMode);
   /// \brief Set the null value.
   /// \param[in] nullValue The null value.
   /// \par \DESCRIPTION
   /// Null value defaults to zero.
   /// \SNIPPET_CPP_PYTHON{SetNullValue, bindexattributes.cpp, bindexattributes.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetNullValue(int nullValue);

private:
   btrieve_key_segment_t btrieveKeySegment = NULL;
};

/// \brief The %Btrieve version class.
class LINKAGE BtrieveVersion
{
   friend class BtrieveClient;

protected:
   btrieve_version_t GetBtrieveVersion();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveVersion, bversion.cpp, bversion.py}
   /// \see BtrieveClient::GetVersion
   BtrieveVersion();
   ~BtrieveVersion();

   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \SNIPPET_CPP_PYTHON{GetLastStatusCode, bversion.cpp, bversion.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
   /// \brief Get the client version type.
   /// \par \DESCRIPTION
   /// Client refers to the instance of Zen used by the caller.
   /// \SNIPPET_CPP_PYTHON{GetClientVersionType, bversion.cpp, bversion.py}
   /// \retval "!= Btrieve::VERSION_TYPE_UNKNOWN" The client version type.
   /// \retval "= Btrieve::VERSION_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   /// \see Btrieve::VersionTypeToString
   Btrieve::VersionType GetClientVersionType();
   /// \brief Get the remote version type.
   /// \par \DESCRIPTION
   /// Remote refers to the instance of Zen used to access the optionally supplied %Btrieve file.
   /// \SNIPPET_CPP_PYTHON{GetRemoteVersionType, bversion.cpp, bversion.py}
   /// \retval "!= Btrieve::VERSION_TYPE_UNKNOWN" The remote version type.
   /// \retval "= Btrieve::VERSION_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   /// \see Btrieve::VersionTypeToString
   Btrieve::VersionType GetRemoteVersionType();
   /// \brief Get the local version type.
   /// \par \DESCRIPTION
   /// Local refers to the instance of Zen running locally with the caller.
   /// \SNIPPET_CPP_PYTHON{GetLocalVersionType, bversion.cpp, bversion.py}
   /// \retval "!= Btrieve::VERSION_TYPE_UNKNOWN" The local version type.
   /// \retval "= Btrieve::VERSION_TYPE_UNKNOWN" \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   /// \see Btrieve::VersionTypeToString
   Btrieve::VersionType GetLocalVersionType();
   /// \brief Get the client revision number.
   /// \par \DESCRIPTION
   /// Client refers to the instance of Zen used by the caller.
   /// \SNIPPET_CPP_PYTHON{GetClientRevisionNumber, bversion.cpp, bversion.py}
   /// \retval ">= 0" The revision number.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   int GetClientRevisionNumber();
   /// \brief Get the client version number.
   /// \par \DESCRIPTION
   /// Client refers to the instance of Zen used by the caller.
   /// \SNIPPET_CPP_PYTHON{GetClientVersionNumber, bversion.cpp, bversion.py}
   /// \retval ">= 0" The version number.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   int GetClientVersionNumber();
   /// \brief Get the remote revision number.
   /// \par \DESCRIPTION
   /// Remote refers to the instance of Zen used to access the optionally supplied %Btrieve file.
   /// \SNIPPET_CPP_PYTHON{GetRemoteRevisionNumber, bversion.cpp, bversion.py}
   /// \retval ">= 0" The revision number.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   int GetRemoteRevisionNumber();
   /// \brief Get the remote version number.
   /// \par \DESCRIPTION
   /// Remote refers to the instance of Zen used to access the optionally supplied %Btrieve file.
   /// \SNIPPET_CPP_PYTHON{GetRemoteVersionNumber, bversion.cpp, bversion.py}
   /// \retval ">= 0" The version number.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   int GetRemoteVersionNumber();
   /// \brief Get the local revision number.
   /// \par \DESCRIPTION
   /// Local refers to the instance of Zen running locally with the caller.
   /// \SNIPPET_CPP_PYTHON{GetLocalRevisionNumber, bversion.cpp, bversion.py}
   /// \retval ">= 0" The revision number.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   int GetLocalRevisionNumber();
   /// \brief Get the local version number.
   /// \par \DESCRIPTION
   /// Local refers to the instance of Zen running locally with the caller.
   /// \SNIPPET_CPP_PYTHON{GetLocalVersionNumber, bversion.cpp, bversion.py}
   /// \retval ">= 0" The version number.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVersion::GetLastStatusCode}
   int GetLocalVersionNumber();

private:
   btrieve_version_t btrieveVersion = NULL;
};

/// \brief The %Btrieve collection class.
///
/// \note This class is currently a beta release; feedback about usage and suggestions for improvements are requested, and can be submitted at https://communities.actian.com/s/actian-zen.
///
/// \details 
/// A %Btrieve collection consists of zero or more documents.
/// Each document consists of a label, JSON and an optional BLOB.
/// The maximum label length is Btrieve::MAXIMUM_KEY_LENGTH bytes, including the terminating NULL byte.
/// The maximum JSON length is Btrieve::MAXIMUM_RECORD_LENGTH bytes.
/// The maximum BLOB length is 2147483647 bytes (2 GiB - 1).
/// \details
/// JSON field names which begin with "_" or "$" are reserved.
/// The following reserved JSON field names are currently defined:
/// Field Name      | User Settable | System Generated | Indexed | Duplicates Allowed | Type   | Description
/// ----------      | ------------- | ---------------- | ------- | ------------------ | ----   | -----------
/// _blobLength     | no            | yes              | yes     | yes                | number | The document blob length.
/// _createTime     | no            | yes              | yes     | yes                | number | The document create time in microseconds since the Unix Epoch.
/// _id             | no            | yes              | yes     | no                 | number | The document identifier.
/// _jsonLength     | no            | yes              | yes     | yes                | number | The document json length.
/// _label          | yes           | no               | yes     | yes                | string | The document label.
/// _updateTime     | no            | yes              | yes     | yes                | number | The document update time in microseconds since the Unix Epoch.
/// \details
/// Non-user settable JSON field names are silently ignored by BtrieveCollection::DocumentCreate and BtrieveCollection::DocumentUpdate.
/// Use of reserved JSON field names other than those listed above will generate an error.
/// Reserved JSON field names are added to the document JSON retrieved by BtrieveCollection::DocumentRetrieveJson and BtrieveCollection::DocumentRetrieve.
///
/// The document identifier is an autoincrement positive integer with a maximum value of 2147483647 (2^31 - 1).
class LINKAGE BtrieveCollection
{
   friend class BtrieveClient;

protected:
   void SetBtrieveClient(BtrieveClient *btrieveClient);
   BtrieveFile *GetBtrieveFile();

public:
   /// \SNIPPET_CPP_PYTHON{BtrieveCollection, bcollection.cpp, bcollection.py}
   /// \see BtrieveClient::CollectionOpen
   BtrieveCollection();
   /// \see BtrieveClient::CollectionClose
   ~BtrieveCollection();

   /// \brief Create a document.
   /// \param[in] json The json.
   /// \SNIPPET_CPP_PYTHON{DocumentCreate, bcollection.cpp, bcollection.py}
   /// \retval ">= 0" The document identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveCollection::GetLastStatusCode}
   int DocumentCreate(const char *json);
   /// \brief Create a document.
   /// \param[in] json The json.
   /// \param[in] blob The blob.
   /// \param[in] blobLength The blob length. If blob is NULL, then blobLength must be 0.
   /// \retval ">= 0" The document identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveCollection::GetLastStatusCode}
   int DocumentCreate(const char *json, const char *blob, int blobLength);
   /// \brief Retrieve a document blob.
   /// \param[in] id The identifier.
   /// \param[out] blob The blob. 
   /// \param[in] blobSize The blob size.
   /// \param[in] documentBlobOffset The document blob offset.
   /// \param[in] documentBlobLength The document blob length. If document blob length is 0, then the entire document blob will be retrieved beginning at document blob offset.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode DocumentRetrieveBlob(int id, char *blob, int blobSize, int documentBlobOffset = 0, int documentBlobLength = 0);
   /// \brief Retrieve a document json.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] id The identifier.
   /// \SNIPPET_CPP_PYTHON{DocumentRetrieveJson, bcollection.cpp, bcollection.py}
   /// \retval "!= NULL" The json.
   /// \retval NULL \ERROR_HAS_OCCURRED
   const char *DocumentRetrieveJson(int id);
   /// \brief Retrieve a document json.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] id The identifier.
   /// \param[out] blob The blob.
   /// \param[in] blobSize The blob size.
   /// \param[in] documentBlobOffset The document blob offset.
   /// \param[in] documentBlobLength The document blob length. If document blob length is 0, then the entire document blob will be retrieved beginning at document blob offset.
   /// \retval "!= NULL" The json.
   /// \retval NULL \ERROR_HAS_OCCURRED
   const char *DocumentRetrieveJson(int id, char *blob, int blobSize, int documentBlobOffset = 0, int documentBlobLength = 0);
#ifndef SWIG
   /// \brief Retrieve a document.
   /// \note \METHOD_ONLY_CPP
   /// \param[in] id The identifier.
   /// \param[out] json The json.
   /// \param[in] jsonSize The json size.
   /// \param[out] blob The blob.
   /// \param[in] blobSize The blob size.
   /// \param[in] documentBlobOffset The document blob offset.
   /// \param[in] documentBlobLength The document blob length. If document blob length is 0, then the entire document blob will be retrieved beginning at document blob offset.
   /// \par \DESCRIPTION
   /// Remember to allocate enough space to retrieve not only the document JSON but also the reserved JSON field names and values, e.g. "_bloblength" : 1234567890.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode DocumentRetrieve(int id, char *json = NULL, int jsonSize = 0, char *blob = NULL, int blobSize = 0, int documentBlobOffset = 0, int documentBlobLength = 0);
#endif
   /// \brief Retrieve a document entity tag.
   /// \param[in] id The identifier.
   /// \retval ">= 0" The document entity tag.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveCollection::GetLastStatusCode}
   int DocumentRetrieveEntityTag(int id);
   /// \brief Update a document.
   /// \param[in] id The identifier.
   /// \param[in] json The json. If json is NULL, then the document json will not be updated.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode DocumentUpdate(int id, const char *json);
   /// \brief Update a document.
   /// \param[in] id The identifier.
   /// \param[in] json The json. If json is NULL, then the document json will not be updated.
   /// \param[in] blob The blob. If blob is NULL, then the document blob will not be updated.
   /// \param[in] blobLength The blob length. If blob is not NULL and blobLength is 0, then the document blob will be deleted.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode DocumentUpdate(int id, const char *json, const char *blob, int blobLength);
   /// \brief Update a document.
   /// \param[in] id The identifier.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the document or -1.
   /// \param[in] json The json. If json is NULL, then the document json will not be updated.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode DocumentUpdate(int id, int entityTag, const char *json);
   /// \brief Update a document.
   /// \param[in] id The identifier.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the document or -1.
   /// \param[in] json The json. If json is NULL, then the document json will not be updated.
   /// \param[in] blob The blob. If blob is NULL, then the document blob will not be updated.
   /// \param[in] blobLength The blob length. If blob is not NULL and blobLength is 0, then the document blob will be deleted.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode DocumentUpdate(int id, int entityTag, const char *json, const char *blob, int blobLength);
   /// \brief Delete a document.
   /// \param[in] id The identifier.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the document or -1.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode DocumentDelete(int id, int entityTag = -1);
   /// \brief Query the collection.
   /// \param[out] btrieveDocumentSet The %Btrieve document set.
   /// \param[in] query The query. The maximum query length is Btrieve::MAXIMUM_RECORD_LENGTH bytes.
   /// \par \DESCRIPTION
   /// \code{.bnf}
   ///   query ::= { } | { <expressions> }
   ///
   ///   <expressions> ::= <expression> | <expression> , <expressions>
   ///
   ///   <expression> ::= <json_expression> | <reserved_expression>
   ///
   ///   <json_expression> ::= 
   ///      "$and" : [ <braced_json_expression> , <braced_json_expressions> ] |
   ///      "$not" : <braced_json_expression> |
   ///      "$or" : [ <braced_json_expression> , <braced_json_expressions> ] |
   ///      <field_expression>
   ///
   ///   <braced_json_expressions> ::= 
   ///      <braced_json_expression> | 
   ///      <braced_json_expression> , <braced_json_expressions>
   ///
   ///   <braced_json_expression> ::= { <json_expression> }
   ///
   ///   <field_expression> ::=
   ///      <field> : <value> |
   ///      <field> : { "$exists" : false | true } |
   ///      <field> : { "$in" : [ <value> , <values> ] } |
   ///      <field> : { "$nin" : [ <value> , <values> ] } |
   ///      <field> : { "$type" : <type> } |
   ///      <field> : { <operator> : <value> } |
   ///      <field> : { <field_expression> }
   ///
   ///   <field> ::= <string>
   ///
   ///   <values> ::= <value> | <value> , <values>
   ///
   ///   <value> ::= false | true | NULL | <string> | <number> | <array>
   ///
   ///   <type> ::= "array" | "boolean" | "null" | "number" | "object" | "string"
   ///
   ///   <operator> ::= "$eq" | "$gt" | "$gte" | "$lt" | "$lte" | "$ne"
   ///
   ///   <reserved_expression> ::= 
   ///      "_blobLength" : <number> | 
   ///      "_blobLength" : { <operator> : <number> } |
   ///      "_createTime" : <number> | 
   ///      "_createTime" : { <time_operator> : <number> } | 
   ///      "_id" : <number> | 
   ///      "_id" : { <operator> : <number> } |
   ///      "_jsonLength" : <number> | 
   ///      "_jsonLength" : { <operator> : <number> } |
   ///      "_label" : <string> | 
   ///      "_label" : { <string_operator> : <string> } |
   ///      "_sortField" : <sort_field> |
   ///      "_sortOrder" : "ascending" | 
   ///      "_sortOrder" : "descending" |
   ///      "_updateTime" : <number> | 
   ///      "_updateTime" : { <time_operator> : <number> }
   ///
   ///   <sort_field> := "_blobLength" | "_createTime" | "_id" | "_jsonLength" | "_label" | "_updateTime"
   ///
   ///   <time_operator> ::= "$eq" | "$gt" | "$gte" | "$lt" | "$lte"
   ///
   ///   <string_operator> ::= "$eq" | "$gt" | "$gte" | "$like" | "$lt" | "$lte" | "$ne" | "$notlike"
   /// \endcode
   /// \par
   /// In a JSON expression, if the query value is not an array and the document value is an array then each element in the array is evaluated. If any evaluation returns true then the JSON expression returns true.
   /// \par
   /// Only one each of "_sortField" and "_sortOrder" may appear in a query.
   /// The default sort field is "_id".
   /// The default sort order is "ascending".
   /// String values are sorted in code point order (simple byte-by-byte comparison).
   /// \SNIPPET_CPP_PYTHON{Query, bcollection.cpp, bcollection.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// A more detailed description of the error may be available in the Zen event log.
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Query(BtrieveDocumentSet *btrieveDocumentSet, const char *query = NULL);
   /// \brief Encrypt a collection.
   /// \param[in] password The password. Password is a null terminated string limited to 32 bytes in length.
   /// \param[in] passwordAgain The password (again).
   /// \par \DESCRIPTION
   /// An encrypted file cannot be opened without providing the correct password. See BtrieveClient::CollectionOpen.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Encrypt(const char *password, const char *passwordAgain);
   /// \brief Decrypt a collection.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Decrypt();
   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \SNIPPET_CPP_PYTHON{CollectionGetLastStatusCode, bcollection.cpp, bcollection.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
/// \cond
   /// \brief Set the last status code.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetLastStatusCode(Btrieve::StatusCode statusCode);
/// \endcond

private:
   BtrieveClient *btrieveClient = NULL;
   BtrieveFile *btrieveFile = NULL;
   Btrieve::StatusCode lastStatusCode = Btrieve::STATUS_CODE_NO_ERROR;
};

/// \brief The %Btrieve document set class.
///
/// \note This class is currently a beta release; feedback about usage and suggestions for improvements are requested, and can be submitted at https://communities.actian.com/s/actian-zen.
///
/// \details The document set is an ordered set.
class LINKAGE BtrieveDocumentSet
{
   friend class BtrieveCollection;

protected:
   std::deque<int> *GetStdDeque();
   
public:
   /// \SNIPPET_CPP_PYTHON{BtrieveDocumentSet, bcollection.cpp, bcollection.py}
   /// \see BtrieveCollection::Query
   BtrieveDocumentSet();
   ~BtrieveDocumentSet();

   /// \brief Return the number of document identifiers in the set.
   /// \SNIPPET_CPP_PYTHON{Size, bcollection.cpp, bcollection.py}
   /// \retval ">= 0" The number of document identifiers in the set.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveDocumentSet::GetLastStatusCode}
   int Size();
   /// \brief Remove and return the first document identifier from the set.
   /// \SNIPPET_CPP_PYTHON{PopFirst, bcollection.cpp, bcollection.py}
   /// \retval ">= 0" The popped document identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveDocumentSet::GetLastStatusCode}
   int PopFirst();
   /// \brief Remove and return the last document identifier from the set.
   /// \retval ">= 0" The popped document identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveDocumentSet::GetLastStatusCode}
   int PopLast();
   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \SNIPPET_CPP_PYTHON{DocumentSetGetLastStatusCode, bcollection.cpp, bcollection.py}
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();

private:
   std::deque<int> *stdDeque = NULL;
   Btrieve::StatusCode lastStatusCode = Btrieve::STATUS_CODE_NO_ERROR;
};

/// \brief The %BRestful class.
///
/// \note This class is currently a beta release; feedback about usage and suggestions for improvements are requested, and can be submitted at https://communities.actian.com/s/actian-zen.
///
/// \details 
/// The %BRestful class is targeted at RESTful HTTP API developers. To that end a subset of the HTTP methods is supported. These methods are DELETE, GET, POST and PUT.
/// HTTP Method | %BRestful Method | Description         | Required Parameters                       | Optional Parameters                        | Returns
/// ----------- | ---------------- | -----------         | -------------------                       | -------------------                        | -------
/// DELETE      | BRestful::DELETE | Delete a collection | Collection name                           |                                            | Status
/// DELETE      | BRestful::DELETE | Delete a document   | Collection name and document identifier   | Entity tag                                 | Status
/// GET         | BRestful::GET    | Query a collection  | Collection name                           | Query                                      | Status and array of document identifiers
/// GET         | BRestful::GET    | Retrieve a document | Collection name and document identifier   | User BLOB buffer                           | Status, entity tag, document JSON, and optional document BLOB
/// POST        | BRestful::POST   | Create a collection | Collection name                           |                                            | Status
/// POST        | BRestful::POST   | Create a document   | Collection name and document JSON content | Document BLOB content                      | Status, document identifier and entity tag
/// PUT         | BRestful::PUT    | Update a document   | Collection name and document identifier   | Entity tag, Document JSON and BLOB content | Status and entity tag
/// \details 
/// All methods return JSON. In addition to the reserved JSON field names listed in BtrieveCollection the following reserved JSON field names are defined:
/// Field Name      | User Settable | Type             | Description
/// ----------      | ------------- | ----             | -----------
/// _collectionName | no            | string           | The collection name.
/// _entityTag      | no            | number           | The entity tag.
/// _ids            | no            | array of numbers | An array of document identifiers.
/// _statusCode     | no            | number           | The status code.
/// _statusString   | no            | string           | The status string.
/// \details
/// Non-user settable JSON field names are silently ignored by BRestful::GET, BRestful::POST, and BRestful::PUT.
/// Use of reserved JSON field names other than those listed both above and in BtrieveCollection will generate an error.
/// \see BtrieveCollection
class LINKAGE BRestful
{
public:
   /// \brief Create a collection or document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] json The json. If json is not given or is NULL, then a collection will be created.
   /// \par \DESCRIPTION
   /// Creating a collection will not overwrite an existing collection. The existing collection must be deleted first. See BRestful::DELETE.
   /// \par
   /// Creating a document requires that the containing collection exist first.
   /// \SNIPPET_PYTHON{POSTcollection, brestful.py}
   /// \SNIPPET_PYTHON{POSTdocument, brestful.py}
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error"
   ///   } 
   /// \endcode \SUCCESS A collection was created.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_id" : <id>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   } 
   /// \endcode \SUCCESS A document was created.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveClient::CollectionCreate BtrieveCollection::DocumentCreate Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *POST(const char *collectionName, const char *json = NULL);
   /// \brief Create a collection or document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] json The json. If json is NULL, then a collection will be created.
   /// \param[in] blob The blob. If json is NULL, then blob and blobLength are ignored.
   /// \param[in] blobLength The blob length.
   /// \par \DESCRIPTION
   /// Creating a collection will not overwrite an existing collection. The existing collection must be deleted first. See BRestful::DELETE.
   /// \par
   /// Creating a document requires that the containing collection exist first.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error"
   ///   } 
   /// \endcode \SUCCESS A collection was created.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_id" : <id>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   } 
   /// \endcode \SUCCESS A document was created.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveClient::CollectionCreate BtrieveCollection::DocumentCreate Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *POST(const char *collectionName, const char *json, const char *blob, int blobLength);

// Avoid Windows defining of DELETE.
#ifdef DELETE
   #undef DELETE
#endif

   /// \brief Delete a collection or document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] id The identifier. If identifier is not given or is -1, then a collection will be deleted.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the document or -1.
   /// \SNIPPET_PYTHON{DELETE, brestful.py}
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   }
   /// \endcode \SUCCESS A collection was deleted.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_id" : <id>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   }
   /// \endcode \SUCCESS A document was deleted.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveClient::CollectionDelete BtrieveCollection::DocumentDelete Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *DELETE(const char *collectionName, int id = -1, int entityTag = -1);

   /// \brief Retrieve a document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] id The identifier.
   /// \SNIPPET_PYTHON{GETdocument, brestful.py}
   /// \return 
   /// \code{.json}
   ///   { 
   ///      <documentJson>, 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_statusCode" : 0, 
   ///      "_statusString" : No error,
   ///   }
   /// \endcode \SUCCESS
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveCollection::DocumentRetrieve Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *GET(const char *collectionName, int id);
   /// \brief Retrieve a document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] id The identifier.
   /// \param[out] blob The blob.
   /// \param[in] blobSize The blob size.
   /// \param[in] documentBlobOffset The document blob offset.
   /// \param[in] documentBlobLength The document blob length.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      <documentJson>, 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_statusCode" : 0, 
   ///      "_statusString" : No error,
   ///   }
   /// \endcode \SUCCESS
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveCollection::DocumentRetrieve Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *GET(const char *collectionName, int id, char *blob, int blobSize, int documentBlobOffset = 0, int documentBlobLength = 0);
 
   /// \brief Retrieve document identifiers.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] query The query.
   /// \SNIPPET_PYTHON{GETquery, brestful.py}
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_ids" : [ <id> ... ], 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   }
   /// \endcode \SUCCESS
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveCollection::Query Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *GET(const char *collectionName, const char *query = NULL);

   /// \brief Update a document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] id The identifier.
   /// \param[in] json The json.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_id" : <id>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   }
   /// \endcode \SUCCESS
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveCollection::DocumentUpdate Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *PUT(const char *collectionName, int id, const char *json);
   /// \brief Update a document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] id The identifier.
   /// \param[in] json The json.
   /// \param[in] blob The blob.
   /// \param[in] blobLength The blob length.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_id" : <id>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   }
   /// \endcode \SUCCESS
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveCollection::DocumentUpdate Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *PUT(const char *collectionName, int id, const char *json, const char *blob, int blobLength);
   /// \brief Update a document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] id The identifier.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the document or -1.
   /// \param[in] json The json.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_id" : <id>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   }
   /// \endcode \SUCCESS
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveCollection::DocumentUpdate Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *PUT(const char *collectionName, int id, int entityTag, const char *json);
   /// \brief Update a document.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] collectionName The collection name.
   /// \param[in] id The identifier.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the document or -1.
   /// \param[in] json The json.
   /// \param[in] blob The blob.
   /// \param[in] blobLength The blob length.
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_collectionName" : <collectionName>, 
   ///      "_entityTag": <entityTag>,
   ///      "_id" : <id>, 
   ///      "_statusCode" : 0, 
   ///      "_statusString" : "No error" 
   ///   }
   /// \endcode \SUCCESS
   /// \return 
   /// \code{.json}
   ///   { 
   ///      "_statusCode" : <statusCode>, 
   ///      "_statusString" : <statusString>
   ///   } 
   /// \endcode \ERROR_HAS_OCCURRED
   /// \see BtrieveCollection::DocumentUpdate Btrieve::StatusCode Btrieve::StatusCodeToString
   static const char *PUT(const char *collectionName, int id, int entityTag, const char *json, const char *blob, int blobLength);

private:
   BRestful();
   ~BRestful();
};

/// \brief The %Btrieve space class.
///
/// \note This class is currently a beta release; feedback about usage and suggestions for improvements are requested, and can be submitted at https://communities.actian.com/s/actian-zen.
///
/// \details 
/// A %Btrieve space consists of zero or more vectors.
/// Each vector consists of a label, JSON and an optional BLOB.
/// The maximum label length is Btrieve::MAXIMUM_KEY_LENGTH bytes, including the terminating NULL byte.
/// The maximum JSON length is Btrieve::MAXIMUM_RECORD_LENGTH bytes.
/// The maximum BLOB length is 2147483647 bytes (2 GiB - 1).
/// \details
/// JSON field names which begin with "_" or "$" are reserved.
/// The following reserved JSON field names are currently defined:
/// Field Name      | User Settable | System Generated | Indexed | Duplicates Allowed | Type   | Description
/// ----------      | ------------- | ---------------- | ------- | ------------------ | ----   | -----------
/// _blobLength     | no            | yes              | yes     | yes                | number | The vector blob length.
/// _createTime     | no            | yes              | yes     | yes                | number | The vector create time in microseconds since the Unix Epoch.
/// _id             | no            | yes              | yes     | no                 | number | The vector identifier.
/// _jsonLength     | no            | yes              | yes     | yes                | number | The vector json length.
/// _label          | yes           | no               | yes     | yes                | string | The vector label.
/// _updateTime     | no            | yes              | yes     | yes                | number | The vector update time in microseconds since the Unix Epoch.
/// \details
/// Non-user settable JSON field names are silently ignored by BtrieveSpace::VectorCreate and BtrieveSpace::VectorUpdate.
/// Use of reserved JSON field names other than those listed above will generate an error.
/// Reserved JSON field names are added to the vector JSON retrieved by BtrieveSpace::VectorRetrieveJson and BtrieveSpace::VectorRetrieve.
///
/// The vector identifier is an autoincrement positive integer with a maximum value of 2147483647 (2^31 - 1).
class LINKAGE BtrieveSpace
{
   friend class BtrieveClient;

protected:
   void SetBtrieveClient(BtrieveClient *btrieveClient);
   BtrieveFile *GetBtrieveFile();

public:
   /// \see BtrieveClient::SpaceOpen
   BtrieveSpace();
   /// \see BtrieveClient::SpaceClose
   ~BtrieveSpace();

   /// \brief Create a vector.
   /// \param[in] json The json.
   /// \retval ">= 0" The vector identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveSpace::GetLastStatusCode}
   int VectorCreate(const char *json);
   /// \brief Create a vector.
   /// \param[in] json The json.
   /// \param[in] blob The blob.
   /// \param[in] blobLength The blob length. If blob is NULL, then blobLength must be 0.
   /// \retval ">= 0" The vector identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveSpace::GetLastStatusCode}
   int VectorCreate(const char *json, const char *blob, int blobLength);
   /// \brief Retrieve a vector blob.
   /// \param[in] id The identifier.
   /// \param[out] blob The blob. 
   /// \param[in] blobSize The blob size.
   /// \param[in] vectorBlobOffset The vector blob offset.
   /// \param[in] vectorBlobLength The vector blob length. If vector blob length is 0, then the entire vector blob will be retrieved beginning at vector blob offset.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode VectorRetrieveBlob(int id, char *blob, int blobSize, int vectorBlobOffset = 0, int vectorBlobLength = 0);
   /// \brief Retrieve a vector json.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] id The identifier.
   /// \retval "!= NULL" The json.
   /// \retval NULL \ERROR_HAS_OCCURRED
   const char *VectorRetrieveJson(int id);
   /// \brief Retrieve a vector json.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] id The identifier.
   /// \param[out] blob The blob.
   /// \param[in] blobSize The blob size.
   /// \param[in] vectorBlobOffset The vector blob offset.
   /// \param[in] vectorBlobLength The vector blob length. If vector blob length is 0, then the entire vector blob will be retrieved beginning at vector blob offset.
   /// \retval "!= NULL" The json.
   /// \retval NULL \ERROR_HAS_OCCURRED
   const char *VectorRetrieveJson(int id, char *blob, int blobSize, int vectorBlobOffset = 0, int vectorBlobLength = 0);
#ifndef SWIG
   /// \brief Retrieve a vector.
   /// \note \METHOD_ONLY_CPP
   /// \param[in] id The identifier.
   /// \param[out] json The json.
   /// \param[in] jsonSize The json size.
   /// \param[out] blob The blob.
   /// \param[in] blobSize The blob size.
   /// \param[in] vectorBlobOffset The vector blob offset.
   /// \param[in] vectorBlobLength The vector blob length. If vector blob length is 0, then the entire vector blob will be retrieved beginning at vector blob offset.
   /// \par \DESCRIPTION
   /// Remember to allocate enough space to retrieve not only the vector JSON but also the reserved JSON field names and values, e.g. "_bloblength" : 1234567890.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode VectorRetrieve(int id, char *json = NULL, int jsonSize = 0, char *blob = NULL, int blobSize = 0, int vectorBlobOffset = 0, int vectorBlobLength = 0);
#endif
   /// \brief Retrieve a vector entity tag.
   /// \param[in] id The identifier.
   /// \retval ">= 0" The vector entity tag.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveSpace::GetLastStatusCode}
   int VectorRetrieveEntityTag(int id);
   /// \brief Update a vector.
   /// \param[in] id The identifier.
   /// \param[in] json The json. If json is NULL, then the vector json will not be updated.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode VectorUpdate(int id, const char *json);
   /// \brief Update a vector.
   /// \param[in] id The identifier.
   /// \param[in] json The json. If json is NULL, then the vector json will not be updated.
   /// \param[in] blob The blob. If blob is NULL, then the vector blob will not be updated.
   /// \param[in] blobLength The blob length. If blob is not NULL and blobLength is 0, then the vector blob will be deleted.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode VectorUpdate(int id, const char *json, const char *blob, int blobLength);
   /// \brief Update a vector.
   /// \param[in] id The identifier.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the vector or -1.
   /// \param[in] json The json. If json is NULL, then the vector json will not be updated.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode VectorUpdate(int id, int entityTag, const char *json);
   /// \brief Update a vector.
   /// \param[in] id The identifier.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the vector or -1.
   /// \param[in] json The json. If json is NULL, then the vector json will not be updated.
   /// \param[in] blob The blob. If blob is NULL, then the vector blob will not be updated.
   /// \param[in] blobLength The blob length. If blob is not NULL and blobLength is 0, then the vector blob will be deleted.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode VectorUpdate(int id, int entityTag, const char *json, const char *blob, int blobLength);
   /// \brief Delete a vector.
   /// \param[in] id The identifier.
   /// \param[in] entityTag The entity tag. The entity tag must either equal the entity tag of the vector or -1.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode VectorDelete(int id, int entityTag = -1);
   /// \brief Query the space.
   /// \param[out] btrieveVectorSet The %Btrieve vector set.
   /// \param[in] query The query. The maximum query length is Btrieve::MAXIMUM_RECORD_LENGTH bytes.
   /// \par \DESCRIPTION
   /// \code{.bnf}
   ///   query ::= { } | { <expressions> }
   ///
   ///   <expressions> ::= <expression> | <expression> , <expressions>
   ///
   ///   <expression> ::= <json_expression> | <reserved_expression>
   ///
   ///   <json_expression> ::= 
   ///      "$and" : [ <braced_json_expression> , <braced_json_expressions> ] |
   ///      "$not" : <braced_json_expression> |
   ///      "$or" : [ <braced_json_expression> , <braced_json_expressions> ] |
   ///      <field_expression>
   ///
   ///   <braced_json_expressions> ::= 
   ///      <braced_json_expression> | 
   ///      <braced_json_expression> , <braced_json_expressions>
   ///
   ///   <braced_json_expression> ::= { <json_expression> }
   ///
   ///   <field_expression> ::=
   ///      <field> : <value> |
   ///      <field> : { "$exists" : false | true } |
   ///      <field> : { "$in" : [ <value> , <values> ] } |
   ///      <field> : { "$nin" : [ <value> , <values> ] } |
   ///      <field> : { "$type" : <type> } |
   ///      <field> : { <operator> : <value> } |
   ///      <field> : { <field_expression> }
   ///
   ///   <field> ::= <string>
   ///
   ///   <values> ::= <value> | <value> , <values>
   ///
   ///   <value> ::= false | true | NULL | <string> | <number> | <array>
   ///
   ///   <type> ::= "array" | "boolean" | "null" | "number" | "object" | "string"
   ///
   ///   <operator> ::= "$eq" | "$gt" | "$gte" | "$lt" | "$lte" | "$ne"
   ///
   ///   <reserved_expression> ::= 
   ///      "_blobLength" : <number> | 
   ///      "_blobLength" : { <operator> : <number> } |
   ///      "_createTime" : <number> | 
   ///      "_createTime" : { <time_operator> : <number> } | 
   ///      "_id" : <number> | 
   ///      "_id" : { <operator> : <number> } |
   ///      "_jsonLength" : <number> | 
   ///      "_jsonLength" : { <operator> : <number> } |
   ///      "_label" : <string> | 
   ///      "_label" : { <string_operator> : <string> } |
   ///      "_sortField" : <sort_field> |
   ///      "_sortOrder" : "ascending" | 
   ///      "_sortOrder" : "descending" |
   ///      "_updateTime" : <number> | 
   ///      "_updateTime" : { <time_operator> : <number> }
   ///
   ///   <sort_field> := "_blobLength" | "_createTime" | "_id" | "_jsonLength" | "_label" | "_updateTime"
   ///
   ///   <time_operator> ::= "$eq" | "$gt" | "$gte" | "$lt" | "$lte"
   ///
   ///   <string_operator> ::= "$eq" | "$gt" | "$gte" | "$like" | "$lt" | "$lte" | "$ne" | "$notlike"
   /// \endcode
   /// \par
   /// In a JSON expression, if the query value is not an array and the vector value is an array then each element in the array is evaluated. If any evaluation returns true then the JSON expression returns true.
   /// \par
   /// Only one each of "_sortField" and "_sortOrder" may appear in a query.
   /// The default sort field is "_id".
   /// The default sort order is "ascending".
   /// String values are sorted in code point order (simple byte-by-byte comparison).
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// A more detailed description of the error may be available in the Zen event log.
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Query(BtrieveVectorSet *btrieveVectorSet, const char *query = NULL);
   /// \brief Encrypt a space.
   /// \param[in] password The password. Password is a null terminated string limited to 32 bytes in length.
   /// \param[in] passwordAgain The password (again).
   /// \par \DESCRIPTION
   /// An encrypted file cannot be opened without providing the correct password. See BtrieveClient::SpaceOpen.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Encrypt(const char *password, const char *passwordAgain);
   /// \brief Decrypt a space.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode Decrypt();
   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();
/// \cond
   /// \brief Set the last status code.
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode SetLastStatusCode(Btrieve::StatusCode statusCode);
/// \endcond
   /// \brief Calculate the Euclidean distance similarity.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] vector The vector.
   /// \param[in] limit The limit.
   /// \param[in] query The query. See BtrieveSpace::Query.
   /// \retval "!= NULL" The similarity.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveSpace::GetLastStatusCode}
   const char *EuclideanDistanceSimilarity(const char *vector, int limit, const char *query = NULL);
   /// \brief Calculate the cosine similarity.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] vector The vector.
   /// \param[in] limit The limit.
   /// \param[in] query The query. See BtrieveSpace::Query.
   /// \retval "!= NULL" The similarity.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveSpace::GetLastStatusCode}
   const char *CosineSimilarity(const char *vector, int limit, const char *query = NULL);
   /// \brief Calculate the dot product similarity.
   /// \note This method is not thread safe. The string returned will be overwritten on subsequent calls.
   /// \param[in] vector The vector.
   /// \param[in] limit The limit.
   /// \param[in] query The query. See BtrieveSpace::Query.
   /// \retval "!= NULL" The similarity.
   /// \retval NULL \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveSpace::GetLastStatusCode}
   const char *DotProductSimilarity(const char *vector, int limit, const char *query = NULL);

private:
   BtrieveClient *btrieveClient = NULL;
   BtrieveFile *btrieveFile = NULL;
   Btrieve::StatusCode lastStatusCode = Btrieve::STATUS_CODE_NO_ERROR;
   int vectorLength = 0;
   const char *ProcessSimilarity(const char *vector, int limit, const char *query,
        Btrieve::StatusCode (BtrieveBulkRetrieveAttributes::*addSimilarity)(const char *, int, int), 
        const char *similarityKey);
};

/// \brief The %Btrieve vector set class.
///
/// \note This class is currently a beta release; feedback about usage and suggestions for improvements are requested, and can be submitted at https://communities.actian.com/s/actian-zen.
///
/// \details The vector set is an ordered set.
class LINKAGE BtrieveVectorSet
{
   friend class BtrieveSpace;

protected:
   std::deque<int> *GetStdDeque();
   
public:
   /// \see BtrieveSpace::Query
   BtrieveVectorSet();
   ~BtrieveVectorSet();

   /// \brief Return the number of vector identifiers in the set.
   /// \retval ">= 0" The number of vector identifiers in the set.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVectorSet::GetLastStatusCode}
   int Size();
   /// \brief Remove and return the first vector identifier from the set.
   /// \retval ">= 0" The popped vector identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVectorSet::GetLastStatusCode}
   int PopFirst();
   /// \brief Remove and return the last vector identifier from the set.
   /// \retval ">= 0" The popped vector identifier.
   /// \retval -1 \ERROR_HAS_OCCURRED
   /// \TO_DETERMINE_STATUS_CODE{BtrieveVectorSet::GetLastStatusCode}
   int PopLast();
   /// \brief Get the last status code.
   /// \note \IS_INTENDED_FOR_USE
   /// \retval "= Btrieve::STATUS_CODE_NO_ERROR" \SUCCESS
   /// \retval "!= Btrieve::STATUS_CODE_NO_ERROR" \ERROR_HAS_OCCURRED
   /// \see Btrieve::StatusCodeToString
   Btrieve::StatusCode GetLastStatusCode();

private:
   std::deque<int> *stdDeque = NULL;
   Btrieve::StatusCode lastStatusCode = Btrieve::STATUS_CODE_NO_ERROR;
};

#endif
