"""
Repository functions for managing projects.

This module provides the core database operations for projects, including:
- Creating, retrieving, updating, and deleting projects.
- Listing projects with optional pagination.
- Ensuring name uniqueness and handling project-specific exceptions.
"""

from sqlalchemy.orm import Session
from core.models import Project
from core.schemas import ProjectCreate, ProjectUpdate
from core import models
from .exceptions import AlreadyExists, NotFound
from core.validation import validate_project_name
from sqlalchemy import func

#CREATE PROJECT
def create_project(db: Session, data: ProjectCreate) -> Project:
    """
    Create a new project in the database.

    Args:
        db (Session): Database session.
        data (ProjectCreate): Data for the new project.

    Returns:
        Project: The created project.

    Raises:
        AlreadyExists: If a project with the same name already exists.
    """
    
    existing = db.query(Project).filter(func.lower(Project.name) == data.name.lower()).first()
    if existing:
        raise AlreadyExists(f"Project '{data.name}' already exists")
    
    project = Project(name=data.name)
    db.add(project)
    db.commit()
    db.refresh(project)
    return project
    
    
#DELETE PROJECT
def delete_project(db:Session, project_id: int) -> bool:
    """
    Delete a project by its ID.

    Args:
        db (Session): Database session.
        project_id (int): ID of the project to delete.

    Returns:
        bool: True if the project was successfully deleted.

    Raises:
        NotFound: If the project does not exist.
    """
    project = get_project(db, project_id)
    if not project:
        raise NotFound(f"Project not found")
    
    db.delete(project)
    db.commit()
    return True

#GET PROJECT
def get_project(db: Session, project_id: int) -> models.Project | None:
    """
    Retrieve a project by its ID.

    Args:
        db (Session): Database session.
        project_id (int): ID of the project to retrieve.

    Returns:
        Project: The retrieved project.

    Raises:
        NotFound: If the project does not exist.
    """
    project = db.query(models.Project).filter(models.Project.project_id == project_id).first()
    if not project:
        raise NotFound(f"Project not found")
    return project
    
#GET PROJECT BY NAME
def get_project_by_name(db: Session, name: str) -> models.Project:
    """
    Retrieve a project by its name.

    Args:
        db (Session): Database session.
        name (str): Name of the project to retrieve.

    Returns:
        Project: The retrieved project.

    Raises:
        NotFound: If the project does not exist.
    """
    validated_name = validate_project_name(name)
    project = db.query(models.Project).filter(models.Project.name == validated_name).first()
    if not project:
        raise NotFound(f"Project with name '{name}' not found")
    return project

#name -> id and then id -> project would be slower.
#name uniqueness makes the lookup "equivalent" to ID lookup
#indexing on the name column in the model gives faster lookup

#UPDATE
def update_project(db: Session, project_id: int, project_in: ProjectUpdate) -> models.Project:
    """
    Update an existing project.

    Args:
        db (Session): Database session.
        project_id (int): ID of the project to update.
        project_in (ProjectUpdate): Data to update the project with.

    Returns:
        Project: The updated project.

    Raises:
        AlreadyExists: If another project with the same name already exists.
    """
    project = get_project(db, project_id)
    
    if project_in.name is not None:
        exists = db.query(Project).filter(func.lower(Project.name) == project_in.name.lower(), Project.project_id != project_id).first()
        if exists:
            raise AlreadyExists(f"Another project already uses the name '{project_in.name}'")
        project.name = project_in.name
    
    db.commit()
    db.refresh(project)
    return project

#LIST
def list_projects(db: Session, skip: int = 0, limit: int = 100) -> list[models.Project]:
    """
    List all projects with optional pagination.

    Args:
        db (Session): Database session.
        skip (int): Number of projects to skip.
        limit (int): Maximum number of projects to return.

    Returns:
        list[Project]: List of projects.
    """
    # Validate pagination parameters
    if skip < 0:
        raise ValueError("Skip must be non-negative")
    if limit <= 0 or limit > 100:
        raise ValueError("Limit must be between 1 and 100")
    
    return db.query(models.Project).order_by(models.Project.created_at.desc()).offset(skip).limit(limit).all()