# Cardinity Python SDK

[![PyPI version](https://img.shields.io/pypi/v/cardinity-python.svg)](https://pypi.org/project/cardinity-python/)
[![Python versions](https://img.shields.io/pypi/pyversions/cardinity-python.svg)](https://pypi.org/project/cardinity-python/)
[![License](https://img.shields.io/badge/license-MIT-blue.svg)](https://github.com/trendpro/cardinity-python/blob/main/LICENSE)
[![Coverage](https://codecov.io/gh/trendpro/cardinity-python/branch/main/graph/badge.svg)](https://codecov.io/gh/trendpro/cardinity-python)

The Python SDK for the [Cardinity Payment Gateway](https://cardinity.com/). This library provides a simple and intuitive way to integrate Cardinity's payment processing capabilities into your Python applications.

## 🚀 Features

- **Complete API Coverage**: Full support for all Cardinity API operations
- **3D Secure v2**: Built-in support for strong customer authentication  
- **Type Safety**: Full type hints for better development experience
- **Comprehensive Testing**: Extensive test suite with 92% code coverage
- **Production Ready**: Built for scalability and reliability
- **Easy Integration**: Simple, intuitive API design
- **Python 3.8+**: Support for modern Python versions

## 📦 Installation

Install the Cardinity Python SDK using pip:

```bash
pip install cardinity-python
```

Or using [uv](https://github.com/astral-sh/uv) (recommended):

```bash
uv add cardinity-python
```

## 🏁 Quick Start

### Basic Setup

```python
from cardinity import Cardinity

# Initialize the client
cardinity = Cardinity(
    consumer_key="your_consumer_key",
    consumer_secret="your_consumer_secret"
)
```

### Create a Payment

```python
# Create a simple payment
payment = cardinity.create_payment(
    amount="10.00",
    currency="EUR",
    description="Test payment",
    country="LT",
    payment_instrument={
        "pan": "4111111111111111",
        "exp_month": 12,
        "exp_year": 2025,
        "cvc": "123",
        "holder": "John Doe"
    }
)

print(f"Payment created: {payment['id']}")
print(f"Status: {payment['status']}")
```

### Handle 3D Secure Authentication

```python
# Create payment that may require 3DS
payment = cardinity.create_payment(**payment_data)

if payment['status'] == 'pending':
    # 3DS authentication required
    auth_url = payment['authorization_information']['url']
    print(f"Please complete 3DS authentication: {auth_url}")
    
    # After customer completes 3DS, finalize the payment
    finalized = cardinity.finalize_payment(
        payment['id'],
        authorize_data="auth_data_from_3ds_callback"
    )
    print(f"Final status: {finalized['status']}")
```

## 📚 Documentation

- **[Installation Guide](docs/installation.rst)** - Detailed installation instructions
- **[Quick Start Guide](docs/quickstart.rst)** - Get started quickly
- **[Authentication](docs/authentication.rst)** - OAuth setup and security
- **[Examples](examples/)** - Complete code examples
- **[API Reference](docs/api.rst)** - Full API documentation
- **[Migration Guide](docs/migration.rst)** - Migrate from Node.js SDK

## 💡 Examples

### Basic Payment Processing

```python
from cardinity import Cardinity, CardinityError

cardinity = Cardinity(
    consumer_key="your_key",
    consumer_secret="your_secret"
)

try:
    payment = cardinity.create_payment(
        amount="25.00",
        currency="EUR",
        description="Product purchase",
        country="LT",
        payment_instrument={
            "pan": "4111111111111111",
            "exp_month": 12,
            "exp_year": 2025,
            "cvc": "123",
            "holder": "Jane Smith"
        }
    )
    
    if payment['status'] == 'approved':
        print("✅ Payment successful!")
    elif payment['status'] == 'pending':
        print("⏳ Awaiting 3DS authentication")
    
except CardinityError as e:
    print(f"❌ Payment failed: {e}")
```

### Recurring Payments

```python
# Create initial payment for future recurring use
initial_payment = cardinity.create_payment(
    amount="0.00",  # Authorization only
    currency="EUR",
    description="Setup recurring payment",
    country="LT",
    payment_instrument={
        "pan": "4111111111111111",
        "exp_month": 12,
        "exp_year": 2025,
        "cvc": "123",
        "holder": "John Doe"
    }
)

# Create recurring payment using the initial payment
recurring_payment = cardinity.create_recurring_payment(
    amount="29.99",
    currency="EUR", 
    description="Monthly subscription",
    payment_id=initial_payment['id']
)
```

### Refund Processing

```python
# Create a refund
refund = cardinity.create_refund(
    payment_id="payment_id_here",
    amount="10.00",
    description="Customer requested refund"
)

print(f"Refund created: {refund['id']}")
print(f"Status: {refund['status']}")
```

## 🧪 Testing

The SDK includes comprehensive test coverage:

```bash
# Run all tests
pytest

# Run with coverage
pytest --cov=cardinity

# Run integration tests (requires API credentials)
pytest -m integration

# Run performance tests
pytest -m performance
```

### Test Credentials

To get test credentials for development and testing:

1. Log in to your [Cardinity account](https://cardinity.com/)
2. Navigate to the API settings or developer section
3. Generate or retrieve your test/sandbox credentials
4. Use these credentials for testing:

```python
CONSUMER_KEY = "your_test_consumer_key"
CONSUMER_SECRET = "your_test_consumer_secret"
```

### Test Cards

- **Visa Success**: 4111111111111111
- **MasterCard Success**: 5555555555554444  
- **3DS Required**: 4444333322221111
- **American Express**: 378282246310005

**Test Amounts:**
- Success: Any amount < 150.00
- Failure: Any amount >= 150.00

## 🔐 Security & Authentication

The SDK uses OAuth 1.0 with HMAC-SHA1 signatures for secure API communication:

```python
# Production setup with environment variables
import os

cardinity = Cardinity(
    consumer_key=os.getenv('CARDINITY_CONSUMER_KEY'),
    consumer_secret=os.getenv('CARDINITY_CONSUMER_SECRET')
)
```

**Security Best Practices:**
- Never commit API credentials to version control
- Use environment variables for credentials
- Validate all payment data before processing
- Implement proper error handling
- Use HTTPS in production

## 🌍 Environment Configuration

### Production Environment

```python
cardinity = Cardinity(
    consumer_key="your_live_consumer_key",
    consumer_secret="your_live_consumer_secret",
    base_url="https://api.cardinity.com/v1"  # Default
)
```

### Sandbox/Test Environment

```python
cardinity = Cardinity(
    consumer_key="your_test_consumer_key",
    consumer_secret="your_test_consumer_secret"
)
```

## 📋 API Operations

The SDK supports all Cardinity API operations:

### Payments
- `create_payment()` - Create new payments
- `get_payment()` - Retrieve payment information
- `finalize_payment()` - Complete 3DS authentication
- `create_recurring_payment()` - Create recurring payments

### Refunds
- `create_refund()` - Process refunds
- `get_refund()` - Retrieve refund information

### Additional Operations
- `create_settlement()` - Create settlements
- `create_void()` - Void payments
- `create_payment_link()` - Create payment links
- `get_chargeback()` - Retrieve chargeback information

## ⚡ Performance

The SDK is optimized for performance:

- **Async Support**: Built on modern Python async patterns
- **Connection Pooling**: Efficient HTTP connection management
- **Request Optimization**: Minimal overhead per API call
- **Memory Efficient**: Low memory footprint

Performance benchmarks (on average hardware):
- Payment creation: ~200ms
- Payment retrieval: ~150ms
- Concurrent requests: 50+ req/sec

## 🛠️ Development

### Setup Development Environment

```bash
# Clone the repository
git clone https://github.com/trendpro/cardinity-python.git
cd cardinity-python

# Install with development dependencies
uv sync --all-extras

# Run tests
uv run pytest

# Build documentation
uv run sphinx-build docs docs/_build
```

### Contributing

We welcome contributions! Please see our [Contributing Guide](CONTRIBUTING.md) for:

- Development setup instructions
- Code style guidelines
- Testing requirements
- Pull request process

## 📖 API Reference

For complete API documentation, see:

- **[Online Documentation](https://cardinity-python.readthedocs.io/)**
- **[API Reference](docs/api.rst)**
- **[Cardinity API Docs](https://developers.cardinity.com/)**

## 🆘 Support

- **Documentation**: [Read the Docs](https://cardinity-python.readthedocs.io/)
- **Issues**: [GitHub Issues](https://github.com/trendpro/cardinity-python/issues)
- **Support**: kyalo@trendpro.co.ke
- **API Documentation**: [developers.cardinity.com](https://developers.cardinity.com/)

## 📄 License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## 🏷️ Versioning

We use [Semantic Versioning](https://semver.org/) for version management. For available versions, see the [tags on this repository](https://github.com/trendpro/cardinity-python/tags).

## 🎯 Roadmap

- [ ] Enhanced logging and debugging
- [ ] Performance monitoring integration

## 🙏 Acknowledgments

- Built with ❤️ by [Kyalo Kitili](https://github.com/trendpro)
- Inspired by the Node.js SDK
- Thanks to all contributors and users

---

**Ready to start processing payments?** 

1. [Sign up for a Cardinity account](https://cardinity.com/)
2. Get your API credentials
3. Install the SDK: `pip install cardinity-python`
4. Follow the [Quick Start Guide](docs/quickstart.rst)

For questions or support, don't hesitate to [contact me](mailto:kyalo@trendpro.co.ke)!
