"""
Tool om highcharts html files vanaf de command line te openen
"""
import argparse
import logging
import sys
import time
import webbrowser
from pathlib import Path
from typing import Union

from cbsplotlib import __version__
from cbsplotlib.highcharts import HC_DEFAULTS_DIRECTORY

HC_HTML_VIEW_TEMPLATE = "highcharts_view.html"

_logger = logging.getLogger(__name__)


class HtmlViewer:
    """
    Class to view HTML files generated by highcharts
    """
    def __init__(
        self,
        filename: Union[str, Path],
        output_html_file: Union[str, Path] = None,
        output_directory: Union[str, Path] = None,
        view_template: Union[str, Path] = None,
        view_template_directory: Union[str, Path] = None,
        show: bool = False,
        keep: bool = None,
        overwrite: bool = False,
    ):
        """
        Parameters
        ----------
        filename : str or Path
            The file that should be rendered as an HTML page.
        output_html_file : str or Path, optional
            The name of the output file. If not provided, the base name of the filename
            will be used with "_rendered" appended to it.
        output_directory : str or Path, optional
            The directory where the output file should be written. If not provided, the
            directory of the filename will be used.
        view_template : str or Path, optional
            The name of a template file in the view_template_directory. If not provided,
            the default template will be used.
        view_template_directory : str or Path, optional
            The directory where the view_template is located. If not provided, the default
            directory will be used.
        show : bool, optional
            If True, the HTML page will be opened in a browser. If False, the page will not
            be opened.
        keep : bool, optional
            If True, the output file will be kept even if it already exists. If False, the
            file will be overwritten if it already exists. If None, the default is False.
        overwrite : bool, optional
            If True, the output file will be overwritten if it already exists. If False,
            the file will not be overwritten. If None, the default is False.

        Notes
        -----
        The default template directory is $PROJECT_DIR/cbs_hc_defaults.
        """
        self.filename = Path(filename)

        if output_directory is None:
            self.output_directory = self.filename.parent
        else:
            self.output_directory = Path(output_directory)

        self.keep = keep

        if output_html_file is None:
            out_file_base = (
                "_".join([self.filename.stem, "rendered"]) + self.filename.suffix
            )
            if keep is None:
                self.keep = False
        else:
            out_file_base = Path(output_html_file)
            if keep is None:
                self.keep = True

        self.output_html_file = self.output_directory / out_file_base

        if self.keep and self.output_html_file.exists() and not overwrite:
            question = f"File {self.output_html_file} already exists.  Overwrite it? (Enter y/n)"
            answer = None
            while answer not in {"y", "n"}:
                answer = input(question).lower()
                if answer == "n":
                    sys.exit()
                elif answer == "y":
                    pass
                else:
                    print("Please enter 'y' or 'n'")

        self.view_template = view_template
        self.view_template_directory = view_template_directory

        html_contents = self.make_html_template()

        _logger.info(f"Writing to {self.output_html_file}")
        try:
            with open(self.output_html_file, "w", encoding="utf-8") as fp:
                fp.write(html_contents)
        except FileNotFoundError as err:
            _logger.warning(err)
            raise FileNotFoundError(
                "Run eerst de script highchart_example_line_with_datetimes.py "
                "om de html file te maken!"
            )

        if show:
            self.show()
        else:
            self.keep = True

    def make_html_template(self):
        """
        Maak een html file door een template te lezen en de html script van
        de highcharts te vervangen door de inhoud van het bestand dat is meegegeven
        bij het instellen van de klasse.

        Parameters
        ----------
        self: het object zelf

        Returns
        -------
        str
            de html tekst van de highcharts
        """
        if self.view_template_directory is not None:
            defaults_directory = self.view_template_directory
        else:
            defaults_directory = Path(__file__).parent / Path(HC_DEFAULTS_DIRECTORY)

        if self.view_template is not None:
            html_view_template = self.view_template
        else:
            html_view_template = HC_HTML_VIEW_TEMPLATE

        html_template_file = defaults_directory / html_view_template

        with open(html_template_file) as fp:
            html_template = fp.read()

        _logger.info(f"Reading  {self.filename}")
        with open(self.filename, encoding="utf-8") as stream:
            highchart_script = stream.read()

        new_html = html_template.replace("HIGHCHARTS_SCRIPT", highchart_script)

        return new_html

    def show(self):

        """
        Open the generated HTML file in a web browser.

        Parameters
        ----------
        self: the object itself

        Returns
        -------
        None
        """
        url = "file://" + self.output_html_file.absolute().as_posix()

        _logger.info(f"Showing {url}")
        webbrowser.open(url)

    def clean(self):

        """Remove the generated HTML file.

        This method is typically used to clean up after a test.

        Returns
        -------
        None
        """
        _logger.info(f"Remove {self.output_html_file}")
        self.output_html_file.unlink()


def parse_args(args):
    """Parse command line parameters

    Args:
      args (List[str]): command line parameters as a list of strings, for example, ``["--help"]``.

    Returns:
      `argparse.Namespace`: command line parameters namespace
    """
    parser = argparse.ArgumentParser(
        description="Tool om latex tabulars in xls files om te zetten"
    )
    parser.add_argument(
        "--version",
        action="version",
        version="highcharts_html_viewer {ver}".format(ver=__version__),
    )
    parser.add_argument("filename", help="Tabular file name", metavar="FILENAME")
    parser.add_argument(
        "--output_filename",
        help="Naam van de html output file. Moet extensie .html " "hebben",
        metavar="OUTPUT_FILENAME",
    )
    parser.add_argument(
        "--output_directory",
        help="Naam van de output directory. Als niet gegeven wordt het"
        "door de input filenaam bepaald",
        metavar="OUTPUT_DIRECTORY",
    )
    parser.add_argument(
        "--show_html",
        help="Open een browser en laat de html zien",
        action="store_true",
        default=True,
    )
    parser.add_argument(
        "--no_show_html",
        help="Laat de html niet zien. Impliceert keep is true",
        action="store_false",
        dest="show_html",
    )
    parser.add_argument(
        "--keep",
        help="Schrijf alleen de html maar laat nog niet zien",
        action="store_true",
    )
    parser.add_argument(
        "--overwrite", help="Forceer overschrijven", action="store_true"
    )
    parser.add_argument(
        "-v",
        "--verbose",
        dest="loglevel",
        help="set loglevel to INFO",
        action="store_const",
        const=logging.INFO,
        default=logging.INFO,
    )
    parser.add_argument(
        "-vv",
        "--debug",
        dest="loglevel",
        help="set loglevel to DEBUG",
        action="store_const",
        const=logging.DEBUG,
    )
    return parser.parse_args(args)


def setup_logging(loglevel):
    """Setup basic logging

    Args:
      loglevel (int): minimum loglevel for emitting messages
    """
    log_format = "[%(asctime)s] %(levelname)s:%(name)s:%(message)s"
    logging.basicConfig(
        level=loglevel,
        stream=sys.stdout,
        format=log_format,
        datefmt="%Y-%m-%d %H:%M:%S",
    )


def main(args):
    """Wrapper allowing :func:`fib` to be called with string arguments in a CLI fashion

    Instead of returning the value from :func:`fib`, it prints the result to the
    ``stdout`` in a nicely formatted message.

    Args:
      args (List[str]): command line parameters as a list of strings
          (for example, ``["--verbose", "42"]``).
    """
    args = parse_args(args)
    setup_logging(args.loglevel)

    hc_view = HtmlViewer(
        filename=args.filename,
        output_html_file=args.output_filename,
        output_directory=args.output_directory,
        show=args.show_html,
        keep=args.keep,
        overwrite=args.overwrite,
    )

    if not hc_view.keep:
        time.sleep(10)
        hc_view.clean()


def run():
    """Calls :func:`main` passing the CLI arguments extracted from :obj:`sys.argv`

    This function can be used as an entry point to create console scripts with setuptools.
    """
    main(sys.argv[1:])


if __name__ == "__main__":
    run()
