"""
Prometheus metrics collector for the metrics plugin.

This module provides direct prometheus_client integration for fast operational metrics
like request counts, response times, and resource usage. These metrics are optimized
for real-time monitoring and alerting.

Key features:
- Thread-safe metric operations using prometheus_client
- Minimal overhead for high-frequency operations
- Standard Prometheus metric types (Counter, Histogram, Gauge)
- Automatic label management and validation
- Integration with hook events for metric updates
"""

from __future__ import annotations

from typing import Any


try:
    from prometheus_client import CollectorRegistry, Counter, Gauge, Histogram, Info

    PROMETHEUS_AVAILABLE = True
except ImportError:
    PROMETHEUS_AVAILABLE = False

    # Create dummy classes for graceful degradation
    class _DummyCounter:
        def __init__(self, *args: Any, **kwargs: Any) -> None:
            pass

        def labels(self, **kwargs: Any) -> _DummyCounter:
            return self

        def inc(self, value: float = 1) -> None:
            pass

    class _DummyHistogram:
        def __init__(self, *args: Any, **kwargs: Any) -> None:
            pass

        def labels(self, **kwargs: Any) -> _DummyHistogram:
            return self

        def observe(self, value: float) -> None:
            pass

        def time(self) -> _DummyHistogram:
            return self

    class _DummyGauge:
        def __init__(self, *args: Any, **kwargs: Any) -> None:
            pass

        def labels(self, **kwargs: Any) -> _DummyGauge:
            return self

        def set(self, value: float) -> None:
            pass

        def inc(self, value: float = 1) -> None:
            pass

        def dec(self, value: float = 1) -> None:
            pass

    class _DummyInfo:
        def __init__(self, *args: Any, **kwargs: Any) -> None:
            pass

        def info(self, labels: dict[str, str]) -> None:
            pass

    class _DummyCollectorRegistry:
        def __init__(self, *args: Any, **kwargs: Any) -> None:
            pass

    # Assign dummy classes to the expected names
    Counter = _DummyCounter  # type: ignore[misc,assignment]
    Histogram = _DummyHistogram  # type: ignore[misc,assignment]
    Gauge = _DummyGauge  # type: ignore[misc,assignment]
    Info = _DummyInfo  # type: ignore[misc,assignment]
    CollectorRegistry = _DummyCollectorRegistry  # type: ignore[misc,assignment]


from ccproxy.core.logging import get_plugin_logger


logger = get_plugin_logger(__name__)


class PrometheusMetrics:
    """
    Prometheus metrics collector for operational monitoring.

    Provides thread-safe, high-performance metrics collection using prometheus_client.
    Designed for minimal overhead in request processing hot paths.
    """

    def __init__(
        self,
        namespace: str = "ccproxy",
        registry: CollectorRegistry | None = None,
        histogram_buckets: list[float] | None = None,
    ):
        """
        Initialize Prometheus metrics.

        Args:
            namespace: Metric name prefix
            registry: Custom Prometheus registry (uses default if None)
            histogram_buckets: Custom histogram bucket boundaries
        """
        if not PROMETHEUS_AVAILABLE:
            logger.warning(
                "prometheus_client not available. Metrics will be disabled. "
                "Install with: pip install prometheus-client"
            )

        self.namespace = namespace
        # Use default registry if None is passed
        if registry is None and PROMETHEUS_AVAILABLE:
            from prometheus_client import REGISTRY

            self.registry: CollectorRegistry | None = REGISTRY
        else:
            self.registry = registry
        self._enabled = PROMETHEUS_AVAILABLE
        self._histogram_buckets = histogram_buckets or [
            0.01,
            0.05,
            0.1,
            0.25,
            0.5,
            1.0,
            2.5,
            5.0,
            10.0,
            25.0,
        ]

        if self._enabled:
            self._init_metrics()

    def _init_metrics(self) -> None:
        """Initialize all Prometheus metric objects."""
        # Request metrics
        self.request_counter = Counter(
            f"{self.namespace}_requests_total",
            "Total number of requests processed",
            labelnames=["method", "endpoint", "model", "status", "service_type"],
            registry=self.registry,
        )

        self.response_time = Histogram(
            f"{self.namespace}_response_duration_seconds",
            "Response time in seconds",
            labelnames=["model", "endpoint", "service_type"],
            buckets=self._histogram_buckets,
            registry=self.registry,
        )

        # Token metrics
        self.token_counter = Counter(
            f"{self.namespace}_tokens_total",
            "Total tokens processed",
            labelnames=[
                "type",
                "model",
                "service_type",
            ],  # _type: input, output, cache_read, cache_write
            registry=self.registry,
        )

        # Cost metrics
        self.cost_counter = Counter(
            f"{self.namespace}_cost_usd_total",
            "Total cost in USD",
            labelnames=[
                "model",
                "cost_type",
                "service_type",
            ],  # cost_type: input, output, cache, total
            registry=self.registry,
        )

        # Error metrics
        self.error_counter = Counter(
            f"{self.namespace}_errors_total",
            "Total number of errors",
            labelnames=["error_type", "endpoint", "model", "service_type"],
            registry=self.registry,
        )

        # Active requests gauge
        self.active_requests = Gauge(
            f"{self.namespace}_active_requests",
            "Number of currently active requests",
            registry=self.registry,
        )

        # System info
        self.system_info = Info(
            f"{self.namespace}_info", "System information", registry=self.registry
        )

        # Service up metric (for Grafana service health)
        self.up = Gauge(
            "up",
            "Service is up and running",
            labelnames=["job"],
            registry=self.registry,
        )

        # Claude SDK Pool metrics
        self.pool_clients_total = Gauge(
            f"{self.namespace}_pool_clients_total",
            "Total number of clients in the pool",
            registry=self.registry,
        )

        self.pool_clients_available = Gauge(
            f"{self.namespace}_pool_clients_available",
            "Number of available clients in the pool",
            registry=self.registry,
        )

        self.pool_clients_active = Gauge(
            f"{self.namespace}_pool_clients_active",
            "Number of active clients currently processing requests",
            registry=self.registry,
        )

        self.pool_connections_created_total = Counter(
            f"{self.namespace}_pool_connections_created_total",
            "Total number of pool connections created",
            registry=self.registry,
        )

        self.pool_connections_closed_total = Counter(
            f"{self.namespace}_pool_connections_closed_total",
            "Total number of pool connections closed",
            registry=self.registry,
        )

        self.pool_acquisitions_total = Counter(
            f"{self.namespace}_pool_acquisitions_total",
            "Total number of client acquisitions from pool",
            registry=self.registry,
        )

        self.pool_releases_total = Counter(
            f"{self.namespace}_pool_releases_total",
            "Total number of client releases to pool",
            registry=self.registry,
        )

        self.pool_health_check_failures_total = Counter(
            f"{self.namespace}_pool_health_check_failures_total",
            "Total number of pool health check failures",
            registry=self.registry,
        )

        self.pool_acquisition_duration = Histogram(
            f"{self.namespace}_pool_acquisition_duration_seconds",
            "Time taken to acquire a client from the pool",
            buckets=[0.001, 0.005, 0.01, 0.05, 0.1, 0.25, 0.5, 1.0, 2.5, 5.0],
            registry=self.registry,
        )

        # Set initial system info
        try:
            from ccproxy.core import __version__

            version = __version__
        except ImportError:
            version = "unknown"

        self.system_info.info(
            {
                "version": version,
                "metrics_enabled": "true",
            }
        )

        # Set service as up
        self.up.labels(job="ccproxy").set(1)

    def record_request(
        self,
        method: str,
        endpoint: str,
        model: str | None = None,
        status: str | int = "unknown",
        service_type: str | None = None,
    ) -> None:
        """
        Record a request event.

        Args:
            method: HTTP method (GET, POST, etc.)
            endpoint: API endpoint path
            model: Model name used
            status: Response status code or status string
            service_type: Service type (claude_sdk_service, proxy_service)
        """
        if not self._enabled:
            return

        self.request_counter.labels(
            method=method,
            endpoint=endpoint,
            model=model or "unknown",
            status=str(status),
            service_type=service_type or "unknown",
        ).inc()

    def record_response_time(
        self,
        duration_seconds: float,
        model: str | None = None,
        endpoint: str = "unknown",
        service_type: str | None = None,
    ) -> None:
        """
        Record response time.

        Args:
            duration_seconds: Response time in seconds
            model: Model name used
            endpoint: API endpoint
            service_type: Service type (claude_sdk_service, proxy_service)
        """
        if not self._enabled:
            return

        self.response_time.labels(
            model=model or "unknown",
            endpoint=endpoint,
            service_type=service_type or "unknown",
        ).observe(duration_seconds)

    def record_tokens(
        self,
        token_count: int,
        token_type: str,
        model: str | None = None,
        service_type: str | None = None,
    ) -> None:
        """
        Record token usage.

        Args:
            token_count: Number of tokens
            token_type: Type of tokens (input, output, cache_read, cache_write)
            model: Model name
            service_type: Service type (claude_sdk_service, proxy_service)
        """
        if not self._enabled or token_count <= 0:
            return

        self.token_counter.labels(
            type=token_type,
            model=model or "unknown",
            service_type=service_type or "unknown",
        ).inc(token_count)

    def record_cost(
        self,
        cost_usd: float,
        model: str | None = None,
        cost_type: str = "total",
        service_type: str | None = None,
    ) -> None:
        """
        Record cost.

        Args:
            cost_usd: Cost in USD
            model: Model name
            cost_type: Type of cost (input, output, cache, total)
            service_type: Service type (claude_sdk_service, proxy_service)
        """
        if not self._enabled or cost_usd <= 0:
            return

        self.cost_counter.labels(
            model=model or "unknown",
            cost_type=cost_type,
            service_type=service_type or "unknown",
        ).inc(cost_usd)

    def record_error(
        self,
        error_type: str,
        endpoint: str = "unknown",
        model: str | None = None,
        service_type: str | None = None,
    ) -> None:
        """
        Record an error event.

        Args:
            error_type: Type/name of error
            endpoint: API endpoint where error occurred
            model: Model name if applicable
            service_type: Service type (claude_sdk_service, proxy_service)
        """
        if not self._enabled:
            return

        self.error_counter.labels(
            error_type=error_type,
            endpoint=endpoint,
            model=model or "unknown",
            service_type=service_type or "unknown",
        ).inc()

    def set_active_requests(self, count: int) -> None:
        """
        Set the current number of active requests.

        Args:
            count: Number of active requests
        """
        if not self._enabled:
            return

        self.active_requests.set(count)

    def inc_active_requests(self) -> None:
        """Increment active request counter."""
        if not self._enabled:
            return

        self.active_requests.inc()

    def dec_active_requests(self) -> None:
        """Decrement active request counter."""
        if not self._enabled:
            return

        self.active_requests.dec()

    def update_system_info(self, info: dict[str, str]) -> None:
        """
        Update system information.

        Args:
            info: Dictionary of system information key-value pairs
        """
        if not self._enabled:
            return

        self.system_info.info(info)

    def is_enabled(self) -> bool:
        """Check if metrics collection is enabled."""
        return self._enabled

    # Claude SDK Pool metrics methods

    def update_pool_gauges(
        self,
        total_clients: int,
        available_clients: int,
        active_clients: int,
    ) -> None:
        """
        Update pool gauge metrics (current state).

        Args:
            total_clients: Total number of clients in pool
            available_clients: Number of available clients
            active_clients: Number of active clients
        """
        if not self._enabled:
            return

        # Update gauges
        self.pool_clients_total.set(total_clients)
        self.pool_clients_available.set(available_clients)
        self.pool_clients_active.set(active_clients)

        # Note: Counters are managed directly by the pool operations
        # This method only updates the current gauges

    def record_pool_acquisition_time(self, duration_seconds: float) -> None:
        """
        Record the time taken to acquire a client from the pool.

        Args:
            duration_seconds: Time in seconds to acquire client
        """
        if not self._enabled:
            return

        self.pool_acquisition_duration.observe(duration_seconds)

    def inc_pool_connections_created(self) -> None:
        """Increment the pool connections created counter."""
        if not self._enabled:
            return

        self.pool_connections_created_total.inc()

    def inc_pool_connections_closed(self) -> None:
        """Increment the pool connections closed counter."""
        if not self._enabled:
            return

        self.pool_connections_closed_total.inc()

    def inc_pool_acquisitions(self) -> None:
        """Increment the pool acquisitions counter."""
        if not self._enabled:
            return

        self.pool_acquisitions_total.inc()

    def inc_pool_releases(self) -> None:
        """Increment the pool releases counter."""
        if not self._enabled:
            return

        self.pool_releases_total.inc()

    def inc_pool_health_check_failures(self) -> None:
        """Increment the pool health check failures counter."""
        if not self._enabled:
            return

        self.pool_health_check_failures_total.inc()

    def set_pool_clients_total(self, count: int) -> None:
        """Set the total number of clients in the pool."""
        if not self._enabled:
            return

        self.pool_clients_total.set(count)

    def set_pool_clients_available(self, count: int) -> None:
        """Set the number of available clients in the pool."""
        if not self._enabled:
            return

        self.pool_clients_available.set(count)

    def set_pool_clients_active(self, count: int) -> None:
        """Set the number of active clients in the pool."""
        if not self._enabled:
            return

        self.pool_clients_active.set(count)
