"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabContainerRunner = exports.InstanceInterruptionBehavior = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const ssm = require("aws-cdk-lib/aws-ssm");
const cr = require("aws-cdk-lib/custom-resources");
// eslint-disable-next-line import/no-extraneous-dependencies
const compare_versions_1 = require("compare-versions");
const constructs_1 = require("constructs");
/**
 * InstanceInterruptionBehavior enum.
 */
var InstanceInterruptionBehavior;
(function (InstanceInterruptionBehavior) {
    /**
     * hibernate
     */
    InstanceInterruptionBehavior["HIBERNATE"] = "hibernate";
    /**
     * stop
     */
    InstanceInterruptionBehavior["STOP"] = "stop";
    /**
     * terminate
     */
    InstanceInterruptionBehavior["TERMINATE"] = "terminate";
})(InstanceInterruptionBehavior || (exports.InstanceInterruptionBehavior = InstanceInterruptionBehavior = {}));
/**
 * GitlabContainerRunner Construct for create a Gitlab Runner.
 */
class GitlabContainerRunner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.gitlabRunnerImage) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Default Gitlab Runner Image use public.ecr.aws/gitlab/gitlab-runner:latest GitlabRunner Version Maybe > 16.0');
        }
        const spotFleetId = id;
        const defaultProps = {
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:latest',
            gitlaburl: 'https://gitlab.com/',
            ec2type: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            concurrentJobs: 1,
            runnerDescription: 'Docker Runner',
        };
        const runnerProps = { ...defaultProps, ...props };
        if (aws_cdk_lib_1.Token.isUnresolved(props.gitlabtoken)) {
            aws_cdk_lib_1.Annotations.of(this).addWarning(`
      Your Gitlab Token pass by AWS Systems Manager Parameter Store or AWS Secret Manager can not check gitlab token. 
      @see https://docs.gitlab.com/ee/ci/runners/new_creation_workflow.html`);
        }
        else {
            if ((0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') && props.gitlabtoken.includes('glrt-') === false) {
                throw new Error('If gitlabRunnerVersion >= 15.10, gitlabtoken please give glrt-xxxxxxx @see https://docs.gitlab.com/ee/ci/runners/new_creation_workflow.html');
            }
        }
        const tokenParameterStore = new ssm.StringParameter(this, 'GitlabTokenParameter', {
            stringValue: 'GITLAB_TOKEN',
        });
        const shell = aws_ec2_1.UserData.forLinux();
        shell.addCommands(...this.createUserData(runnerProps, tokenParameterStore.parameterName));
        this.runnerRole =
            runnerProps.ec2iamrole ??
                new aws_iam_1.Role(this, 'runner-role', {
                    assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                    description: 'For Gitlab EC2 Runner Role',
                });
        this.validUntil = runnerProps.validUntil;
        tokenParameterStore.grantWrite(this.runnerRole);
        tokenParameterStore.grantRead(this.runnerRole);
        this.vpc =
            runnerProps.selfvpc ??
                new aws_ec2_1.Vpc(this, 'VPC', {
                    ipAddresses: aws_ec2_1.IpAddresses.cidr('10.0.0.0/16'),
                    maxAzs: 2,
                    subnetConfiguration: [
                        {
                            cidrMask: 26,
                            name: 'RunnerVPC',
                            subnetType: aws_ec2_1.SubnetType.PUBLIC,
                        },
                    ],
                    natGateways: 0,
                });
        this.defaultRunnerSG = new aws_ec2_1.SecurityGroup(this, 'SpotFleetSg', {
            vpc: this.vpc,
        });
        const spotOrOnDemand = runnerProps.spotFleet ?? false;
        if (spotOrOnDemand) {
            //--token Error('yes new spotfleet');
            const instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
                roles: [this.runnerRole.roleName],
            });
            const imageId = aws_ec2_1.MachineImage.latestAmazonLinux2().getImage(this).imageId;
            const lt = new aws_ec2_1.CfnLaunchTemplate(this, 'LaunchTemplate', {
                launchTemplateData: {
                    imageId,
                    instanceType: runnerProps.ec2type,
                    blockDeviceMappings: [
                        {
                            deviceName: '/dev/xvda',
                            ebs: {
                                volumeSize: runnerProps.ebsSize ?? 60,
                            },
                        },
                    ],
                    userData: aws_cdk_lib_1.Fn.base64(shell.render()),
                    keyName: runnerProps.keyName,
                    tagSpecifications: [
                        {
                            resourceType: 'instance',
                            tags: [
                                {
                                    key: 'Name',
                                    value: `${aws_cdk_lib_1.Stack.of(this).stackName}/spotFleetGitlabRunner/${spotFleetId}`,
                                },
                            ],
                        },
                    ],
                    instanceMarketOptions: {
                        marketType: 'spot',
                        spotOptions: {
                            instanceInterruptionBehavior: runnerProps.instanceInterruptionBehavior ??
                                InstanceInterruptionBehavior.TERMINATE,
                        },
                    },
                    securityGroupIds: this.defaultRunnerSG.connections.securityGroups.map((m) => m.securityGroupId),
                    iamInstanceProfile: {
                        arn: instanceProfile.attrArn,
                    },
                    metadataOptions: runnerProps.enabledIMDSv2 ? {
                        httpTokens: 'required',
                        httpPutResponseHopLimit: 2,
                    } : undefined,
                },
            });
            const spotFleetRole = new aws_iam_1.Role(this, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            const vpcSubnetSelection = runnerProps.vpcSubnet ?? {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            };
            const subnetConfig = this.vpc
                .selectSubnets(vpcSubnetSelection)
                .subnets.map((s) => ({
                subnetId: s.subnetId,
            }));
            this.cfnSpotFleet = new aws_ec2_1.CfnSpotFleet(this, id, {
                spotFleetRequestConfigData: {
                    launchTemplateConfigs: [
                        {
                            launchTemplateSpecification: {
                                launchTemplateId: lt.ref,
                                version: lt.attrLatestVersionNumber,
                            },
                            overrides: subnetConfig,
                        },
                    ],
                    iamFleetRole: spotFleetRole.roleArn,
                    targetCapacity: 1,
                    validUntil: aws_cdk_lib_1.Lazy.string({ produce: () => this.validUntil }),
                    terminateInstancesWithExpiration: true,
                },
            });
            const onEvent = new lambda.Function(this, 'OnEvent', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.on_event',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: aws_cdk_lib_1.Duration.seconds(60),
            });
            const isComplete = new lambda.Function(this, 'IsComplete', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.is_complete',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: aws_cdk_lib_1.Duration.seconds(60),
                role: onEvent.role,
            });
            const myProvider = new cr.Provider(this, 'MyProvider', {
                onEventHandler: onEvent,
                isCompleteHandler: isComplete,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            onEvent.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['ec2:DescribeSpotFleetInstances'],
                resources: ['*'],
            }));
            const fleetInstancesId = new aws_cdk_lib_1.CustomResource(this, 'GetInstanceId', {
                serviceToken: myProvider.serviceToken,
                properties: {
                    SpotFleetRequestId: this.cfnSpotFleet.ref,
                },
            });
            fleetInstancesId.node.addDependency(this.cfnSpotFleet);
            this.spotFleetInstanceId = aws_cdk_lib_1.Token.asString(fleetInstancesId.getAtt('InstanceId'));
            this.spotFleetRequestId = aws_cdk_lib_1.Token.asString(fleetInstancesId.getAtt('SpotInstanceRequestId'));
            new aws_cdk_lib_1.CfnOutput(this, 'InstanceId', { value: this.spotFleetInstanceId });
            new aws_cdk_lib_1.CfnOutput(this, 'SpotFleetId', { value: this.cfnSpotFleet.ref });
        }
        else {
            this.runnerEc2 = new aws_ec2_1.Instance(this, 'GitlabRunner', {
                instanceType: new aws_ec2_1.InstanceType(runnerProps.ec2type),
                instanceName: 'Gitlab-Runner',
                vpc: this.vpc,
                vpcSubnets: runnerProps.vpcSubnet ?? {
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                },
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux2(),
                role: this.runnerRole,
                userData: shell,
                securityGroup: this.defaultRunnerSG,
                blockDevices: [
                    {
                        deviceName: '/dev/xvda',
                        volume: aws_ec2_1.BlockDeviceVolume.ebs(runnerProps.ebsSize ?? 60),
                    },
                ],
            });
            if (runnerProps.enabledIMDSv2) {
                const template = new aws_ec2_1.LaunchTemplate(this, 'Template', {
                    httpTokens: aws_ec2_1.LaunchTemplateHttpTokens.REQUIRED,
                    httpPutResponseHopLimit: 2,
                });
                const cfnInstance = this.runnerEc2.node.defaultChild;
                cfnInstance.launchTemplate = {
                    launchTemplateId: template.launchTemplateId,
                    version: template.latestVersionNumber,
                };
            }
            new aws_cdk_lib_1.CfnOutput(this, 'Runner-Instance-ID', {
                value: this.runnerEc2.instanceId,
            });
        }
        const unregisterRunnerOnEvent = new lambda.Function(this, 'unregisterRunnerOnEvent', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'unregister_runner.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: aws_cdk_lib_1.Duration.seconds(60),
        });
        const unregisterRunnerProvider = new cr.Provider(this, 'unregisterRunnerProvider', {
            onEventHandler: unregisterRunnerOnEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const unregisterRunnerCR = new aws_cdk_lib_1.CustomResource(this, 'unregisterRunnerCR', {
            resourceType: 'Custom::unregisterRunnerProvider',
            serviceToken: unregisterRunnerProvider.serviceToken,
            properties: {
                TokenParameterStoreName: tokenParameterStore.parameterName,
                GitlabUrl: runnerProps.gitlaburl,
            },
        });
        tokenParameterStore.grantRead(unregisterRunnerOnEvent);
        unregisterRunnerCR.node.addDependency(tokenParameterStore);
        this.runnerRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        new aws_cdk_lib_1.CfnOutput(this, 'Runner-Role-Arn', {
            value: this.runnerRole.roleArn,
        });
    }
    /**
     * Add expire time function for spotfleet runner !!! .
     *
     * @param duration - Block duration.
     */
    expireAfter(duration) {
        const date = new Date();
        date.setSeconds(date.getSeconds() + duration.toSeconds());
        this.validUntil = date.toISOString();
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @param tokenParameterStoreName - the tokenParameterStoreName to put gitlab runner token.
     * @returns Array.
     */
    createUserData(props, tokenParameterStoreName) {
        return [
            'yum update -y ',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlaburl} ${(0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') ? '--token' : '--registration-token'} ${props.gitlabtoken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props?.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "${props.runnerDescription}" \
      ${(0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') ? undefined : `--tag-list "${props.tags?.join(',')}" `} --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
            `sed -i 's/concurrent = .*/concurrent = ${props.concurrentJobs}/g' /home/ec2-user/.gitlab-runner/config.toml`,
            'TOKEN=$(cat /home/ec2-user/.gitlab-runner/config.toml | grep \'token \' | awk \'{print $3}\'| tr -d \'"\')',
            `aws ssm put-parameter --name ${tokenParameterStoreName} --value $TOKEN --overwrite --region ${aws_cdk_lib_1.Stack.of(this).region}`,
        ];
    }
}
exports.GitlabContainerRunner = GitlabContainerRunner;
_a = JSII_RTTI_SYMBOL_1;
GitlabContainerRunner[_a] = { fqn: "cdk-gitlab-runner.GitlabContainerRunner", version: "2.2.310" };
//# sourceMappingURL=data:application/json;base64,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