"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const __1 = require("..");
const deps_1 = require("../deps");
const logging = require("../logging");
const node_package_1 = require("../node-package");
const util_1 = require("./util");
logging.disable();
test('license file is added by default', () => {
    // WHEN
    const project = new TestNodeProject();
    // THEN
    expect(util_1.synthSnapshot(project).LICENSE).toContain('Apache License');
});
test('license file is not added if licensed is false', () => {
    // WHEN
    const project = new TestNodeProject({
        licensed: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    expect(snapshot.LICENSE).toBeUndefined();
    expect(snapshot['.gitignore']).not.toContain('LICENSE');
    expect(snapshot['package.json'].license).toEqual('UNLICENSED');
});
describe('deps', () => {
    test('runtime deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            deps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.RUNTIME);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        expect(pkgjson.peerDependencies).toStrictEqual({});
    });
    test('dev dependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            devDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDevDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.TEST);
        project.deps.addDependency('eee@^1', deps_1.DependencyType.DEVENV);
        project.deps.addDependency('fff@^2', deps_1.DependencyType.BUILD);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.devDependencies.aaa).toStrictEqual('^1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('~4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.devDependencies.eee).toStrictEqual('^1');
        expect(pkgjson.devDependencies.fff).toStrictEqual('^2');
        expect(pkgjson.peerDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // devDependencies are added with pinned versions
        expect(pkgjson.devDependencies.aaa).toStrictEqual('1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies without pinnedDevDep', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDependencyOptions: {
                pinnedDevDependency: false,
            },
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // sanitize
        ['jest', 'jest-junit', 'projen', 'standard-version'].forEach(d => delete pkgjson.devDependencies[d]);
        expect(pkgjson.devDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('bundled deps are automatically added as normal deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            bundledDeps: ['hey@2.1.1'],
        });
        // WHEN
        project.addBundledDeps('foo@^1.2.3');
        project.deps.addDependency('bar@~1.0.0', deps_1.DependencyType.BUNDLED);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            hey: '2.1.1',
            foo: '^1.2.3',
            bar: '~1.0.0',
        });
        expect(pkgjson.bundledDependencies).toStrictEqual([
            'bar',
            'foo',
            'hey',
        ]);
    });
});
describe('npm publishing options', () => {
    test('defaults', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmDistTag).toStrictEqual('latest');
        expect(npm.npmRegistry).toStrictEqual('registry.npmjs.org');
        expect(npm.npmRegistryUrl).toStrictEqual('https://registry.npmjs.org/');
        // since these are all defaults, publishConfig is not defined.
        expect(util_1.synthSnapshot(project)['package.json'].publishConfig).toBeUndefined();
    });
    test('scoped packages default to RESTRICTED access', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.RESTRICTED);
        // since these are all defaults, publishConfig is not defined.
        expect(packageJson(project).publishConfig).toBeUndefined();
    });
    test('non-scoped package cannot be RESTRICTED', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // THEN
        expect(() => new node_package_1.NodePackage(project, {
            packageName: 'my-package',
            npmAccess: node_package_1.NpmAccess.RESTRICTED,
        })).toThrow(/"npmAccess" cannot be RESTRICTED for non-scoped npm package/);
    });
    test('custom settings', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmDistTag: 'next',
            npmRegistryUrl: 'https://foo.bar',
            npmAccess: node_package_1.NpmAccess.PUBLIC,
        });
        // THEN
        expect(npm.npmDistTag).toStrictEqual('next');
        expect(npm.npmRegistry).toStrictEqual('foo.bar');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar/');
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(packageJson(project).publishConfig).toStrictEqual({
            access: 'public',
            registry: 'https://foo.bar/',
            tag: 'next',
        });
    });
    test('deprecated npmRegistry can be used instead of npmRegistryUrl and then https:// is assumed', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmRegistry: 'foo.bar.com',
        });
        // THEN
        expect(npm.npmRegistry).toStrictEqual('foo.bar.com');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar.com/');
        expect(packageJson(project).publishConfig).toStrictEqual({
            registry: 'https://foo.bar.com/',
        });
    });
});
test('extend github release workflow', () => {
    var _a;
    const project = new TestNodeProject();
    (_a = project.releaseWorkflow) === null || _a === void 0 ? void 0 : _a.addJobs({
        publish_docker_hub: {
            'runs-on': 'ubuntu-latest',
            'env': {
                CI: 'true',
            },
            'steps': [
                {
                    name: 'Check out the repo',
                    uses: 'actions/checkout@v2',
                },
                {
                    name: 'Push to Docker Hub',
                    uses: 'docker/build-push-action@v1',
                    with: {
                        username: '${{ secrets.DOCKER_USERNAME }}',
                        password: '${{ secrets.DOCKER_PASSWORD }}',
                        repository: 'projen/projen-docker',
                        tag_with_ref: true,
                    },
                },
            ],
        },
    });
    const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
    expect(workflow).toContain('publish_docker_hub:\n    runs-on: ubuntu-latest\n');
    expect(workflow).toContain('username: ${{ secrets.DOCKER_USERNAME }}\n          password: ${{ secrets.DOCKER_PASSWORD }}');
});
function packageJson(project) {
    return util_1.synthSnapshot(project)['package.json'];
}
class TestNodeProject extends __1.NodeProject {
    constructor(options = {}) {
        super({
            outdir: util_1.mkdtemp(),
            name: 'test-node-project',
            logging: {
                level: __1.LogLevel.OFF,
            },
            defaultReleaseBranch: 'master',
            ...options,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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