"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const os_1 = require("os");
const tasks_1 = require("../../tasks");
const util_1 = require("../util");
test('no tasks, no tasks file', () => {
    const p = new util_1.TestProject();
    expect(synthTasksManifest(p)).toBeUndefined();
});
test('empty task', () => {
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('empty');
    // THEN
    expectManifest(p, {
        tasks: {
            empty: {
                name: 'empty',
            },
        },
    });
});
test('multiple "exec" commands', () => {
    const p = new util_1.TestProject();
    // WHEN
    const task = p.addTask('hello', {
        description: 'hello, world',
        exec: 'echo hello',
        env: {
            FOO: 'bar',
        },
    });
    task.exec('echo world');
    task.exec('echo "with quotes"');
    task.env('BAR', 'baz');
    // THEN
    expectManifest(p, {
        tasks: {
            hello: {
                name: 'hello',
                description: 'hello, world',
                env: {
                    FOO: 'bar',
                    BAR: 'baz',
                },
                steps: [
                    { exec: 'echo hello' },
                    { exec: 'echo world' },
                    { exec: 'echo "with quotes"' },
                ],
            },
        },
    });
});
test('subtasks', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const hello = p.addTask('hello', { exec: 'echo hello' });
    const world = p.addTask('world');
    // WHEN
    world.exec('echo "running hello"');
    world.spawn(hello);
    // THEN
    expectManifest(p, {
        tasks: {
            hello: {
                name: 'hello',
                steps: [{ exec: 'echo hello' }],
            },
            world: {
                name: 'world',
                steps: [
                    { exec: 'echo "running hello"' },
                    { spawn: 'hello' },
                ],
            },
        },
    });
});
test('reset() can be used to reset task steps', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t0 = p.addTask('your-task');
    const t = p.addTask('my-task');
    t.exec('line1');
    t.spawn(t0);
    t.exec('line2');
    // WHEN
    t.reset('line3');
    t.exec('line4');
    // THEN
    expectManifest(p, {
        tasks: {
            'your-task': {
                name: 'your-task',
            },
            'my-task': {
                name: 'my-task',
                steps: [
                    { exec: 'line3' },
                    { exec: 'line4' },
                ],
            },
        },
    });
});
test('prependXXX() can be used to add steps from the top', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const sub = p.addTask('my-sub-task', { exec: 'subexec' });
    const t = p.addTask('my-task');
    t.exec('line1');
    // WHEN
    t.prependExec('line2');
    t.prependSpawn(sub);
    t.prependSay('message');
    // THEN
    expectManifest(p, {
        tasks: {
            'my-sub-task': {
                name: 'my-sub-task',
                steps: [
                    { exec: 'subexec' },
                ],
            },
            'my-task': {
                name: 'my-task',
                steps: [
                    { say: 'message' },
                    { spawn: 'my-sub-task' },
                    { exec: 'line2' },
                    { exec: 'line1' },
                ],
            },
        },
    });
});
test('env() can be used to add environment variables', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t = p.addTask('my-task', {
        env: {
            INITIAL: '123',
            ENV: '456',
        },
    });
    // WHEN
    t.env('FOO', 'BAR');
    t.env('HELLO', 'world');
    // THEN
    expectManifest(p, {
        tasks: {
            'my-task': {
                name: 'my-task',
                env: {
                    INITIAL: '123',
                    ENV: '456',
                    FOO: 'BAR',
                    HELLO: 'world',
                },
            },
        },
    });
});
test('.steps can be used to list all steps in the current task', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t0 = p.addTask('your');
    const t = p.addTask('my');
    t.exec('step1');
    t.exec('step2');
    t.exec('step3');
    t.spawn(t0);
    t.exec('step4');
    // WHEN
    const steps = t.steps;
    // THEN
    expect(steps).toStrictEqual([
        { exec: 'step1' },
        { exec: 'step2' },
        { exec: 'step3' },
        { spawn: 'your' },
        { exec: 'step4' },
    ]);
});
test('"condition" can be used to define a command that will determine if a task should be skipped', () => {
    // GIVEN
    const p = new util_1.TestProject();
    p.addTask('foo', {
        condition: 'false',
        exec: 'foo bar',
    });
    // THEN
    expectManifest(p, {
        tasks: {
            foo: {
                name: 'foo',
                condition: 'false',
                steps: [
                    { exec: 'foo bar' },
                ],
            },
        },
    });
});
describe('toShellCommand()', () => {
    test('single step', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        const t = p.addTask('foo', { exec: 'echo hi there' });
        // THEN
        expect(t.toShellCommand()).toMatchSnapshot();
        expect(shell(t)).toStrictEqual(['hi there']);
    });
    test('with a name', () => {
        // GIVEN
        const p = new util_1.TestProject();
        const t = p.addTask('foo');
        // WHEN
        t.exec('echo step 1', { name: 'STEP I' });
        // THEN
        expect(t.toShellCommand()).toMatchSnapshot();
        expect(shell(t)).toStrictEqual([
            'STEP I',
            'step 1',
        ]);
    });
    test('with a condition', () => {
        // GIVEN
        const p = new util_1.TestProject();
        const t = p.addTask('foo', {
            condition: '[ "${RUNME}" = "1" ]',
        });
        // WHEN
        t.exec('echo hello, world', { name: 'STEP I' });
        // THEN
        expect(t.toShellCommand()).toMatchSnapshot();
        expect(shell(t)).toStrictEqual([]);
        expect(shell(t, { RUNME: '1' })).toStrictEqual([
            'STEP I',
            'hello, world',
        ]);
    });
    test('multiple steps', () => {
        // GIVEN
        const p = new util_1.TestProject();
        const t = p.addTask('foo');
        // WHEN
        t.exec('echo step 1', { name: 'STEP I' });
        t.exec('echo step 2');
        t.exec('echo step 3', { name: 'STEP III' });
        // THEN
        expect(t.toShellCommand()).toMatchSnapshot();
        expect(shell(t)).toStrictEqual([
            'STEP I',
            'step 1',
            'step 2',
            'STEP III',
            'step 3',
        ]);
    });
    test('subtasks', () => {
        // GIVEN
        const p = new util_1.TestProject();
        const t1 = p.addTask('t1');
        const t2 = p.addTask('t2');
        // WHEN
        t1.exec('echo task1-step1');
        t1.exec('echo task1-step2');
        t2.exec('echo task2-step1');
        t2.spawn(t1, { name: 'spawning t1' });
        t2.exec('echo task2-step3');
        // THEN
        expect(t2.toShellCommand()).toMatchSnapshot();
        expect(shell(t2)).toStrictEqual([
            'task2-step1',
            'spawning t1',
            'task1-step1',
            'task1-step2',
            'task2-step3',
        ]);
    });
    test('with environment', () => {
        // GIVEN
        const p = new util_1.TestProject();
        p.tasks.addEnvironment('FOO', 'hello'); // global environment
        // WHEN
        const t1 = p.addTask('t1', {
            env: {
                BAR: '$(echo world)',
            },
            exec: 'echo $FOO, $BAR !',
        });
        // THEN
        expect(t1.toShellCommand()).toMatchSnapshot();
        expect(shell(t1)).toStrictEqual([
            'hello, world !',
        ]);
    });
});
function shell(t, env = {}) {
    const result = child_process_1.spawnSync(t.toShellCommand(), { shell: true, env: { ...process.env, ...env } });
    if (result.status !== 0) {
        throw new Error(`non-zero exit code ${result.status}: ${result.stderr.toString('utf-8')}`);
    }
    return result.stdout.toString('utf-8').trim().split(os_1.EOL).filter(x => x).map(x => x.trim());
}
function expectManifest(p, toStrictEqual) {
    const manifest = synthTasksManifest(p);
    delete manifest['//'];
    expect(manifest).toStrictEqual(toStrictEqual);
}
function synthTasksManifest(p) {
    return util_1.synthSnapshot(p)[tasks_1.Tasks.MANIFEST_FILE];
    ;
}
//# sourceMappingURL=data:application/json;base64,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