"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const _resolve_1 = require("../_resolve");
test('null', () => {
    expect(_resolve_1.resolve(null)).toStrictEqual(null);
});
test('undefined', () => {
    expect(_resolve_1.resolve(undefined)).toStrictEqual(undefined);
});
test('string', () => {
    expect(_resolve_1.resolve('')).toStrictEqual('');
    expect(_resolve_1.resolve('hello')).toStrictEqual('hello');
    expect(_resolve_1.resolve('0')).toStrictEqual('0');
    expect(_resolve_1.resolve('hello world')).toStrictEqual('hello world');
    expect(_resolve_1.resolve('  ')).toStrictEqual('  ');
});
test('number', () => {
    expect(_resolve_1.resolve(0)).toStrictEqual(0);
    expect(_resolve_1.resolve(10)).toStrictEqual(10);
    expect(_resolve_1.resolve(-102)).toStrictEqual(-102);
});
test('boolean', () => {
    expect(_resolve_1.resolve(true)).toStrictEqual(true);
    expect(_resolve_1.resolve(false)).toStrictEqual(false);
});
test('array', () => {
    expect(_resolve_1.resolve([])).toStrictEqual([]);
    expect(_resolve_1.resolve([1, 2, 3])).toStrictEqual([1, 2, 3]);
});
test('object', () => {
    expect(_resolve_1.resolve({})).toStrictEqual({});
    expect(_resolve_1.resolve({ foo: 123, hello: [1, 2, { bar: 3 }] })).toStrictEqual({
        foo: 123,
        hello: [1, 2, { bar: 3 }],
    });
});
test('functions are resolved', () => {
    expect(_resolve_1.resolve(() => 123)).toStrictEqual(123);
    expect(_resolve_1.resolve(() => 'hello')).toStrictEqual('hello');
    expect(_resolve_1.resolve(() => undefined)).toStrictEqual(undefined);
});
test('recursive resolve', () => {
    expect(_resolve_1.resolve(() => [1, 2, () => 30])).toStrictEqual([1, 2, 30]);
    expect(_resolve_1.resolve(() => ({ foo: 123, bar: () => 'bar' }))).toStrictEqual({ foo: 123, bar: 'bar' });
    expect(_resolve_1.resolve(() => ({ foo: 123, bar: () => ['bar', 'baz', { hello: () => 'world' }] }))).toStrictEqual({
        foo: 123,
        bar: ['bar', 'baz', { hello: 'world' }],
    });
});
test('context is passed to functions', () => {
    expect(_resolve_1.resolve((x) => x + 1, { args: [10] })).toStrictEqual(11);
    const calculator = {
        add: (a, b) => a + b,
        sub: (a, b) => a - b,
        mul: (a, b) => a * b,
        div: (a, b) => a / b,
        fun: (a1, b1, text) => [
            `this is ${text}`,
            (a2) => a1 * a2,
            (_, b2) => b1 + b2,
            {
                another: {
                    level: (a3, b3, text3) => `hello ${text3} ${b3} ${a3}`,
                },
            },
        ],
    };
    expect(_resolve_1.resolve(calculator, { args: [10, 2, 'fun'] })).toStrictEqual({
        add: 12,
        sub: 8,
        mul: 20,
        div: 5,
        fun: [
            'this is fun',
            100,
            4,
            { another: { level: 'hello fun 2 10' } },
        ],
    });
});
test('"undefined" values are omitted', () => {
    const r = (o) => _resolve_1.resolve(o);
    expect(r({ foo: undefined })).toStrictEqual({});
    expect(r({ foo: { bar: undefined } })).toStrictEqual({ foo: {} });
    expect(r({ foo: { bar: undefined, hello: 123 } })).toStrictEqual({ foo: { hello: 123 } });
    expect(r({ foo: [undefined] })).toStrictEqual({ foo: [] });
});
test('omitEmpty', () => {
    const r = (o) => _resolve_1.resolve(o, { omitEmpty: true });
    expect(r({ foo: {} })).toStrictEqual(undefined);
    expect(r({ foo: { bar: {} } })).toStrictEqual(undefined);
    expect(r({ foo: [] })).toStrictEqual(undefined);
    expect(r({ bar: { foo: [] } })).toStrictEqual(undefined);
    expect(r({ foo: { hello: 'world', boom: { bar: [] } } })).toStrictEqual({ foo: { hello: 'world' } });
});
//# sourceMappingURL=data:application/json;base64,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