"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const dev_env_1 = require("../dev-env");
const gitpod_1 = require("../gitpod");
const logging = require("../logging");
const util_1 = require("./util");
// This is duplicated vs exported
const GITPOD_FILE = '.gitpod.yml';
const DEVCONTAINER_FILE = '.devcontainer.json';
logging.disable();
describe('dev environment enable/disable', () => {
    test('given gitpod and devContainer are false', () => {
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: false,
            devContainer: false,
        });
        // WHEN
        project.synth();
        // THEN
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeFalsy();
        expect(fs.existsSync(devContainerFilePath)).toBeFalsy();
    });
    test('no gitpod/devcontainer files if they are empty', () => {
        // WHEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // THEN
        project.synth();
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeFalsy();
        expect(fs.existsSync(devContainerFilePath)).toBeFalsy();
    });
    test('given gitpod and devContainer are true', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage({ image: 'foo' });
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addPorts('1234');
        // THEN
        project.synth();
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeTruthy();
        expect(fs.existsSync(devContainerFilePath)).toBeTruthy();
    });
});
describe('dev environment docker options', () => {
    test('given an image', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromImage('jsii/superchain'));
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromImage('jsii/uberchain'));
        // THEN
        const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('image: jsii/superchain');
        const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
        expect(devContainerSnapshot).toStrictEqual({ image: 'jsii/uberchain' });
    });
    test('given a docker file dep', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromFile('.gitpod.Dockerfile'));
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromFile('Dockerfile'));
        // THEN
        const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('image:');
        expect(gitpodSnapshot).toContain('file: .gitpod.Dockerfile');
        const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
        expect(devContainerSnapshot).toStrictEqual({ build: { dockerfile: 'Dockerfile' } });
    });
});
describe('dev environment tasks', () => {
    test('given custom task', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        const task = project.addTask('gitpod-test', { exec: 'text' });
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addTasks(task);
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addTasks(task);
        // THEN
        const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('command');
        expect(gitpodSnapshot).toContain('gitpod-test');
        const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
        expect(devContainerSnapshot.postCreateCommand).toContain('gitpod-test');
    });
    test('given gitpod task options', () => {
        var _a;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        const task = project.addTask('gitpod-test', { exec: 'text' });
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addCustomTask({
            init: 'echo Initializing',
            openIn: gitpod_1.GitpodOpenIn.LEFT,
            openMode: gitpod_1.GitpodOpenMode.SPLIT_BOTTOM,
            command: `npx projen ${task.name}`,
        });
        // THEN
        const snapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(snapshot).toContain('init: echo Initializing');
        expect(snapshot).toContain('openIn: left');
        expect(snapshot).toContain('openMode: split-bottom');
        expect(snapshot).toContain('command: npx projen gitpod-test');
    });
});
test('dev environment ports', () => {
    var _a, _b;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: true,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addPorts('8080', '3000-3999');
    (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addPorts('8080', '3000');
    // THEN
    const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('port: "8080"');
    expect(gitpodSnapshot).toContain('port: 3000-3999');
    const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
    expect(devContainerSnapshot).toStrictEqual({ forwardPorts: ['8080', '3000'] });
});
test('gitpod prebuilds config', () => {
    var _a;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: false,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addPrebuilds({
        master: true,
        branches: true,
        pullRequestsFromForks: true,
        addBadge: false,
    });
    // THEN
    const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('github');
    expect(gitpodSnapshot).toContain('prebuilds');
    expect(gitpodSnapshot).toContain('master');
    expect(gitpodSnapshot).toContain('branches');
    expect(gitpodSnapshot).toContain('pullRequestsFromForks');
    expect(gitpodSnapshot).toContain('addBadge');
});
test('dev environment vscode extensions', () => {
    var _a, _b;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: true,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addVscodeExtensions('dbaeumer.vscode-eslint@2.1.13:5sYlSD6wJi5s3xqD8hupUw==');
    (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addVscodeExtensions('dbaeumer.vscode-eslint');
    // THEN
    const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('extensions:');
    expect(gitpodSnapshot).toContain('dbaeumer.vscode-eslint@2.1.13:5sYlSD6wJi5s3xqD8hupUw==');
    const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
    expect(devContainerSnapshot).toStrictEqual({ extensions: ['dbaeumer.vscode-eslint'] });
});
//# sourceMappingURL=data:application/json;base64,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