"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const json_1 = require("../json");
const util_1 = require("../util");
const util_2 = require("./util");
describe('decamelizeRecursively', () => {
    test('decamel recurses an object structure', () => {
        // GIVEN
        const input = {
            dependsOn: ['a', 'b', 'c'],
            volumes: [
                {
                    driver: 'tmpfs',
                    driverOpts: {
                        type: 'nfs',
                        o: 'addr=...',
                        device: ':/docker/example',
                    },
                },
            ],
        };
        // WHEN
        const output = util_1.decamelizeKeysRecursively(input);
        // THEN
        expect(output).toEqual({
            depends_on: ['a', 'b', 'c'],
            volumes: [
                {
                    driver: 'tmpfs',
                    driver_opts: {
                        type: 'nfs',
                        o: 'addr=...',
                        device: ':/docker/example',
                    },
                },
            ],
        });
    });
    test('decamel quits when it recurses too deeply', () => {
        // GIVEN
        const circle = {};
        circle.circle = circle;
        // WHEN
        expect(() => util_1.decamelizeKeysRecursively(circle)).toThrow(/circular reference/);
    });
    test('decamel can know when not to decamelize a key', () => {
        // GIVEN
        const input = {
            dependsOn: ['a', 'b'],
            environment: {
                leaveThisAlone: true,
                LEAVE_CASE_ALONE: true,
            },
        };
        // WHEN
        const output = util_1.decamelizeKeysRecursively(input, {
            shouldDecamelize(path, _value) {
                return !/^environment\./.test(path.join('.'));
            },
        });
        // THEN
        expect(output).toEqual({
            depends_on: ['a', 'b'],
            environment: {
                leaveThisAlone: true,
                LEAVE_CASE_ALONE: true,
            },
        });
    });
});
test('isTruthy', () => {
    expect(util_1.isTruthy(undefined)).toEqual(false);
    expect(util_1.isTruthy('false')).toEqual(false);
    expect(util_1.isTruthy('0')).toEqual(false);
    expect(util_1.isTruthy('null')).toEqual(false);
    expect(util_1.isTruthy('')).toEqual(false);
    expect(util_1.isTruthy('true')).toEqual(true);
    expect(util_1.isTruthy('1')).toEqual(true);
    expect(util_1.isTruthy('enabled')).toEqual(true);
});
describe('deepMerge (destructive: false)', () => {
    test('merges objects', () => {
        // GIVEN
        const original = { a: { b: 3 } };
        // WHEN
        util_1.deepMerge([original, { a: { c: 4 } }]);
        // THEN
        expect(original).toEqual({ a: { b: 3, c: 4 } });
    });
    test('overwrites non-objects', () => {
        // GIVEN
        const original = { a: [] };
        // WHEN
        util_1.deepMerge([original, { a: { b: 3 } }]);
        // THEN
        expect(original).toEqual({ a: { b: 3 } });
    });
    test('does not overwrite if rightmost is "undefined"', () => {
        // GIVEN
        const original = { a: 1 };
        // WHEN
        util_1.deepMerge([original, { a: undefined }]);
        // THEN
        expect(original).toEqual({ a: 1 });
    });
    test('does not recurse on projects', () => {
        // GIVEN
        const proj1 = new util_2.TestProject();
        const proj2 = new util_2.TestProject();
        const objA = { a: proj1 };
        const objB = { a: proj2 };
        // WHEN
        util_1.deepMerge([objA, objB]);
        // THEN
        expect(objA).toEqual(objB);
    });
    test('does not recurse on components', () => {
        // GIVEN
        const proj = new util_2.TestProject();
        const comp1 = new json_1.JsonFile(proj, 'foo', { obj: 3 });
        const comp2 = new json_1.JsonFile(proj, 'bar', { obj: 5 });
        const objA = { a: comp1 };
        const objB = { a: comp2 };
        // WHEN
        util_1.deepMerge([objA, objB]);
        // THEN
        expect(objA).toEqual(objB);
    });
});
describe('deepMerge (destructive: true)', () => {
    test('merges objects', () => {
        // GIVEN
        const original = { a: { b: 3 } };
        // WHEN
        util_1.deepMerge([original, { a: { c: 4 } }], true);
        // THEN
        expect(original).toEqual({ a: { b: 3, c: 4 } });
    });
    test('overwrites non-objects', () => {
        // GIVEN
        const original = { a: [] };
        // WHEN
        util_1.deepMerge([original, { a: { b: 3 } }], true);
        // THEN
        expect(original).toEqual({ a: { b: 3 } });
    });
    test('does overwrite if rightmost is "undefined"', () => {
        // GIVEN
        const original = { a: 1 };
        // WHEN
        util_1.deepMerge([original, { a: undefined }], true);
        // THEN
        expect(original).toEqual({}); // ! different from the non-destructive case
    });
    test('does not recurse on projects', () => {
        // GIVEN
        const proj1 = new util_2.TestProject();
        const proj2 = new util_2.TestProject();
        const objA = { a: proj1 };
        const objB = { a: proj2 };
        // WHEN
        util_1.deepMerge([objA, objB], true);
        // THEN
        expect(objA).toEqual(objB);
    });
    test('does not recurse on components', () => {
        // GIVEN
        const proj = new util_2.TestProject();
        const comp1 = new json_1.JsonFile(proj, 'foo', { obj: 3 });
        const comp2 = new json_1.JsonFile(proj, 'bar', { obj: 5 });
        const objA = { a: comp1 };
        const objB = { a: comp2 };
        // WHEN
        util_1.deepMerge([objA, objB], true);
        // THEN
        expect(objA).toEqual(objB);
    });
});
test('dedupArray', () => {
    expect(util_1.dedupArray(['a', 'b', 'c'])).toEqual(['a', 'b', 'c']);
    expect(util_1.dedupArray(['a', 'a', 'b', 'a'])).toEqual(['a', 'b']);
});
test('getFilePermissions', () => {
    expect(util_1.getFilePermissions({})).toEqual('644');
    expect(util_1.getFilePermissions({ readonly: true, executable: true })).toEqual('500');
    expect(util_1.getFilePermissions({ readonly: true, executable: false })).toEqual('400');
    expect(util_1.getFilePermissions({ readonly: false, executable: true })).toEqual('755');
    expect(util_1.getFilePermissions({ readonly: false, executable: false })).toEqual('644');
    expect(util_1.getFilePermissions({ readonly: false })).toEqual('644');
    expect(util_1.getFilePermissions({ executable: true })).toEqual('755');
});
test('formatAsPythonModule', () => {
    expect(util_1.formatAsPythonModule('foo-bar-baz')).toEqual('foo_bar_baz');
    expect(util_1.formatAsPythonModule('foo.bar.baz')).toEqual('foo_bar_baz');
});
//# sourceMappingURL=data:application/json;base64,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