"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const core_1 = require("aws-cdk-lib/core");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
/**
 * Bundling with esbuild
 */
class Bundling {
    /**
     * Cargo bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(path.dirname(options.entry), {
            assetHash: options.assetHash,
            assetHashType: options.assetHash
                ? core_1.AssetHashType.CUSTOM
                : core_1.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    constructor(props) {
        this.props = props;
        const packageManagerType = props.packageManagerType ?? types_1.PackageManagerType.CARGO_ZIGBUILD;
        this.packageManager = package_manager_1.PackageManager.fromType(packageManagerType);
        this.target =
            this.props.target && isValidTarget(this.props.target)
                ? this.props.target
                : toTarget(this.props.architecture);
        const buildArgs = ['--release', '--color', 'always'];
        if (props.extraBuildArgs) {
            buildArgs.push(...props.extraBuildArgs);
        }
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling ||
            !this.packageManager.runLocally ||
            !(0, util_1.checkInstalledTarget)(toTarget(this.props.architecture));
        this.image = shouldBuildImage
            ? props.dockerImage ??
                core_1.DockerImage.fromBuild(path.join(__dirname, '../'), {
                    buildArgs: {
                        ...(props.buildArgs ?? {}),
                        // If runtime isn't passed use regional default, lowest common denominator is node18
                        IMAGE: props.runtime.bundlingImage.image,
                    },
                    platform: props.architecture.dockerPlatform,
                })
            : core_1.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: core_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: core_1.AssetStaging.BUNDLING_OUTPUT_DIR,
            buildArgs: buildArgs,
            buildRunner: this.packageManager.runBuildCommand(),
            osPlatform: 'linux', // linux docker image
        });
        this.command = props.command ?? ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        this.workingDirectory = props.workingDirectory;
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
        // Local bundling
        if (!props.forceDockerBundling) {
            // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    createBundlingCommand(options) {
        const pathJoin = osPathJoin(options.osPlatform);
        const osPlatform = os.platform();
        let relativeManifestPath = options.inputDir
            ? pathJoin(options.inputDir, path.relative(path.dirname(this.props.entry), this.props.entry))
            : this.props.entry;
        let binaryName;
        const buildCommand = [
            options.buildRunner,
            `--manifest-path=${relativeManifestPath}`,
            `--target ${this.target}`,
            ...options.buildArgs,
        ];
        if (this.props.binaryName) {
            binaryName = this.props.binaryName;
        }
        else if ((0, util_1.isWorkspace)(this.props.entry) ||
            (0, util_1.hasMultipleBinaries)(this.props.entry)) {
            throw new Error('Your Cargo project is a workspace or contains multiple binaries, use the property `binaryName` to specify the binary to use.');
        }
        else {
            binaryName = (0, util_1.getBinaryName)(this.props.entry);
        }
        if (!binaryName) {
            throw new Error('Your Cargo project is missing the package name or a [[bin]] section, use the property `binaryName` to specify the binary to use');
        }
        buildCommand.push('--bin', binaryName);
        // Features
        if (this.props.features) {
            buildCommand.push('--features', this.props.features.join(','));
        }
        // Log level
        if (this.props.logLevel && this.props.logLevel == types_1.LogLevel.SILENT) {
            buildCommand.push('--silent');
        }
        else if (this.props.logLevel && this.props.logLevel == types_1.LogLevel.VERBOSE) {
            buildCommand.push('--verbose');
        }
        // Move target file to destination
        const sourceBootstrap = pathJoin(path.dirname(relativeManifestPath), 'target', this.target, 'release', binaryName);
        const targetBootstrap = pathJoin(options.outputDir, 'bootstrap');
        const moveCommand = osPlatform === 'win32'
            ? [
                'powershell',
                '-command',
                'Move-Item',
                '-Path',
                sourceBootstrap,
                '-Destination',
                targetBootstrap,
            ]
                .filter((c) => !!c)
                .join(' ')
            : ['mv', sourceBootstrap, targetBootstrap].filter((c) => !!c).join(' ');
        return chain([
            ...(this.props.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []),
            buildCommand.filter((c) => !!c).join(' '),
            moveCommand,
            ...(this.props.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []),
        ]);
    }
    getLocalBundlingProvider() {
        const osPlatform = os.platform();
        const projectRoot = path.dirname(this.props.entry);
        const createLocalCommand = (outputDir, buildArgs) => this.createBundlingCommand({
            inputDir: '',
            outputDir,
            buildRunner: this.packageManager.runBuildCommand(),
            osPlatform,
            buildArgs,
        });
        const environment = this.props.environment ?? {};
        const cwd = projectRoot;
        return {
            tryBundle(outputDir) {
                if (!package_manager_1.PackageManager.fromType(types_1.PackageManagerType.CARGO_ZIGBUILD)
                    .runLocally &&
                    !package_manager_1.PackageManager.fromType(types_1.PackageManagerType.CROSS).runLocally) {
                    process.stderr.write('No package manager cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                const buildArgs = ['--release', '--color', 'always'];
                const localCommand = createLocalCommand(outputDir, buildArgs);
                (0, util_1.exec)(osPlatform === 'win32' ? 'cmd' : 'bash', [osPlatform === 'win32' ? '/c' : '-c', localCommand], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        // show output
                        'ignore', // ignore stdio
                        process.stderr, // redirect stdout to stderr
                        'inherit', // inherit stderr
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
Bundling.X86_64__TARGET = 'x86_64-unknown-linux-gnu';
Bundling.ARM_TARGET = 'aarch64-unknown-linux-gnu';
function chain(commands) {
    return commands.filter((c) => !!c).join(' && ');
}
/**
 * Validates a target against a list of allowed targets.
 * @param target the target to validate.
 */
function isValidTarget(target) {
    return (target.startsWith(Bundling.X86_64__TARGET) ||
        target.startsWith(Bundling.ARM_TARGET));
}
/**
 * Converts an Architecture to a bundling target
 */
function toTarget(architecture) {
    switch (architecture) {
        case aws_lambda_1.Architecture.ARM_64:
            return Bundling.ARM_TARGET;
        default:
            return Bundling.X86_64__TARGET;
    }
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
//# sourceMappingURL=data:application/json;base64,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