"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.callsites = exports.canRunLocally = exports.checkInstalledTarget = exports.exec = exports.isWorkspace = exports.hasMultipleBinaries = exports.getBinaryName = void 0;
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const toml = require("toml");
/**
 * Get Manifest binary name
 * @param entry the Manifest file path
 */
function getBinaryName(entry) {
    try {
        const contents = (0, fs_1.readFileSync)(entry, 'utf8');
        let data = toml.parse(contents);
        if (data.bin && data.bin.length == 1) {
            return data.bin[0].name;
        }
        return data.package?.name;
    }
    catch (err) {
        throw new Error(`Unable to parse Manifest file \`${entry}\`\n` + `${err}\n`);
    }
}
exports.getBinaryName = getBinaryName;
function hasMultipleBinaries(entry) {
    try {
        const contents = (0, fs_1.readFileSync)(entry, 'utf8');
        let data = toml.parse(contents);
        return data.bin && data.bin.length > 1;
    }
    catch (err) {
        throw new Error(`Unable to parse Manifest file \`${entry}\`\n` + `${err}\n`);
    }
}
exports.hasMultipleBinaries = hasMultipleBinaries;
/**
 * Check if the given Manifest contains multiple workspaces
 * @param entry the Manifest file path
 */
function isWorkspace(entry) {
    try {
        const contents = (0, fs_1.readFileSync)(entry, 'utf8');
        let data = toml.parse(contents);
        return (data.workspace &&
            data.workspace.members &&
            data.workspace.members.length > 0);
    }
    catch (err) {
        throw new Error(`Unable to parse Manifest file \`${entry}\`\n` + `${err}\n`);
    }
}
exports.isWorkspace = isWorkspace;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    const proc = (0, child_process_1.spawnSync)(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${proc.stdout
                ?.toString()
                .trim()}\n\n\nstderr: ${proc.stderr?.toString().trim()}`);
        }
        throw new Error(`${cmd} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Check if an architecture target is installed.
 * @param target The target to test
 */
function checkInstalledTarget(target) {
    const proc = (0, child_process_1.spawnSync)('rustup', ['target', 'list', '--installed']);
    if (proc &&
        proc.status === 0 &&
        !proc.error &&
        proc.stdout.toString().includes(target)) {
        return true;
    }
    return false;
}
exports.checkInstalledTarget = checkInstalledTarget;
/**
 * Check if a command can be run locally.
 * @param cmd The command to execute
 */
function canRunLocally(cmd) {
    try {
        const proc = (0, child_process_1.spawnSync)(cmd, ['--version']);
        if (proc.status === 0 && !proc.error) {
            return true;
        }
        return false;
    }
    catch (err) {
        return false;
    }
}
exports.canRunLocally = canRunLocally;
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = new Error().stack?.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
//# sourceMappingURL=data:application/json;base64,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