"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const core_1 = require("aws-cdk-lib/core");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
/**
 * Bundling with esbuild
 */
class Bundling {
    /**
     * Cargo bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(path.dirname(options.entry), {
            assetHash: options.assetHash,
            assetHashType: options.assetHash
                ? core_1.AssetHashType.CUSTOM
                : core_1.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    constructor(props) {
        this.props = props;
        const packageManagerType = props.packageManagerType ?? types_1.PackageManagerType.CARGO_ZIGBUILD;
        this.packageManager = package_manager_1.PackageManager.fromType(packageManagerType);
        this.target =
            this.props.target && isValidTarget(this.props.target)
                ? this.props.target
                : toTarget(this.props.architecture);
        const buildArgs = ['--release', '--color', 'always'];
        if (props.extraBuildArgs) {
            buildArgs.push(...props.extraBuildArgs);
        }
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling ||
            !this.packageManager.runLocally ||
            !(0, util_1.checkInstalledTarget)(toTarget(this.props.architecture));
        this.image = shouldBuildImage
            ? (props.dockerImage ??
                core_1.DockerImage.fromBuild(path.join(__dirname, '../'), {
                    buildArgs: {
                        ...(props.buildArgs ?? {}),
                        // If runtime isn't passed use regional default, lowest common denominator is node18
                        IMAGE: props.runtime.bundlingImage.image,
                    },
                    platform: props.architecture.dockerPlatform,
                }))
            : core_1.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: core_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: core_1.AssetStaging.BUNDLING_OUTPUT_DIR,
            buildArgs: buildArgs,
            buildRunner: this.packageManager.runBuildCommand(),
            osPlatform: 'linux', // linux docker image
        });
        this.command = props.command ?? ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        this.workingDirectory = props.workingDirectory;
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
        // Local bundling
        if (!props.forceDockerBundling) {
            // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    createBundlingCommand(options) {
        const pathJoin = osPathJoin(options.osPlatform);
        const osPlatform = os.platform();
        let relativeManifestPath = options.inputDir
            ? pathJoin(options.inputDir, path.relative(path.dirname(this.props.entry), this.props.entry))
            : this.props.entry;
        let binaryName;
        const buildCommand = [
            options.buildRunner,
            `--manifest-path=${relativeManifestPath}`,
            `--target ${this.target}`,
            ...options.buildArgs,
        ];
        if (this.props.binaryName) {
            binaryName = this.props.binaryName;
        }
        else if ((0, util_1.isWorkspace)(this.props.entry) ||
            (0, util_1.hasMultipleBinaries)(this.props.entry)) {
            throw new Error('Your Cargo project is a workspace or contains multiple binaries, use the property `binaryName` to specify the binary to use.');
        }
        else {
            binaryName = (0, util_1.getBinaryName)(this.props.entry);
        }
        if (!binaryName) {
            throw new Error('Your Cargo project is missing the package name or a [[bin]] section, use the property `binaryName` to specify the binary to use');
        }
        buildCommand.push('--bin', binaryName);
        // Features
        if (this.props.features) {
            buildCommand.push('--features', this.props.features.join(','));
        }
        // Log level
        if (this.props.logLevel && this.props.logLevel == types_1.LogLevel.SILENT) {
            buildCommand.push('--silent');
        }
        else if (this.props.logLevel && this.props.logLevel == types_1.LogLevel.VERBOSE) {
            buildCommand.push('--verbose');
        }
        // Move target file to destination
        const sourceBootstrap = pathJoin(path.dirname(relativeManifestPath), 'target', this.target, 'release', binaryName);
        const targetBootstrap = pathJoin(options.outputDir, 'bootstrap');
        const moveCommand = osPlatform === 'win32'
            ? [
                'powershell',
                '-command',
                'Move-Item',
                '-Path',
                sourceBootstrap,
                '-Destination',
                targetBootstrap,
            ]
                .filter((c) => !!c)
                .join(' ')
            : ['mv', sourceBootstrap, targetBootstrap].filter((c) => !!c).join(' ');
        return chain([
            ...(this.props.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []),
            buildCommand.filter((c) => !!c).join(' '),
            moveCommand,
            ...(this.props.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []),
        ]);
    }
    getLocalBundlingProvider() {
        const osPlatform = os.platform();
        const projectRoot = path.dirname(this.props.entry);
        const createLocalCommand = (outputDir, buildArgs) => this.createBundlingCommand({
            inputDir: '',
            outputDir,
            buildRunner: this.packageManager.runBuildCommand(),
            osPlatform,
            buildArgs,
        });
        const environment = this.props.environment ?? {};
        const cwd = projectRoot;
        return {
            tryBundle(outputDir) {
                if (!package_manager_1.PackageManager.fromType(types_1.PackageManagerType.CARGO_ZIGBUILD)
                    .runLocally &&
                    !package_manager_1.PackageManager.fromType(types_1.PackageManagerType.CROSS).runLocally) {
                    process.stderr.write('No package manager cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                const buildArgs = ['--release', '--color', 'always'];
                const localCommand = createLocalCommand(outputDir, buildArgs);
                (0, util_1.exec)(osPlatform === 'win32' ? 'cmd' : 'bash', [osPlatform === 'win32' ? '/c' : '-c', localCommand], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        // show output
                        'ignore', // ignore stdio
                        process.stderr, // redirect stdout to stderr
                        'inherit', // inherit stderr
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
Bundling.X86_64__TARGET = 'x86_64-unknown-linux-gnu';
Bundling.ARM_TARGET = 'aarch64-unknown-linux-gnu';
function chain(commands) {
    return commands.filter((c) => !!c).join(' && ');
}
/**
 * Validates a target against a list of allowed targets.
 * @param target the target to validate.
 */
function isValidTarget(target) {
    return (target.startsWith(Bundling.X86_64__TARGET) ||
        target.startsWith(Bundling.ARM_TARGET));
}
/**
 * Converts an Architecture to a bundling target
 */
function toTarget(architecture) {
    switch (architecture) {
        case aws_lambda_1.Architecture.ARM_64:
            return Bundling.ARM_TARGET;
        default:
            return Bundling.X86_64__TARGET;
    }
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
//# sourceMappingURL=data:application/json;base64,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