"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RustFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const bundling_1 = require("./bundling");
const util_1 = require("./util");
/**
 * A Rust Lambda function
 */
class RustFunction extends aws_lambda_1.Function {
    constructor(scope, id, props = {}) {
        if (props.runtime && props.runtime.family !== aws_lambda_1.RuntimeFamily.OTHER) {
            throw new Error('Only `Custom` runtimes are supported.');
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        // Handler is not important for Rust function
        const handler = 'rust.handler';
        const architecture = props.architecture ?? aws_lambda_1.Architecture.X86_64;
        const runtime = props.runtime ?? aws_lambda_1.Runtime.PROVIDED_AL2023;
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.bundle({
                ...(props.bundling ?? {}),
                entry,
                runtime,
                architecture,
                binaryName: props.binaryName,
            }),
            handler: handler,
        });
    }
}
exports.RustFunction = RustFunction;
_a = JSII_RTTI_SYMBOL_1;
RustFunction[_a] = { fqn: "@cdklabs/aws-lambda-rust.RustFunction", version: "0.0.9" };
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A Cargo.toml file named as the defining file with id as u (defining-file.id/Cargo.toml)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\Cargo.toml$/.test(entry)) {
            // Only Cargo.toml files are supported.
            // If the user gave a path to a directory, we'll look for a Cargo.toml file in it.
            // If the user gave a path to a file, we'll use it as-is.
            entry = path.join(entry, 'Cargo.toml');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find manifest file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const manifestFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}/Cargo.toml`);
    if (fs.existsSync(manifestFile)) {
        return manifestFile;
    }
    throw new Error('Cannot find manifest file - Cargo.toml');
}
/**
 * Finds the name of the file where the `RustFunction` is defined
 */
function findDefiningFile() {
    let definingIndex;
    const sites = (0, util_1.callsites)();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'RustFunction') {
            // The next site is the site where the RustFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZnVuY3Rpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvZnVuY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLHVEQU1nQztBQUVoQyx5Q0FBc0M7QUFFdEMsaUNBQW1DO0FBNkNuQzs7R0FFRztBQUNILE1BQWEsWUFBYSxTQUFRLHFCQUFRO0lBQ3hDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBMkIsRUFBRTtRQUNyRSxJQUFJLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEtBQUssMEJBQWEsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUNsRSxNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7UUFDM0QsQ0FBQztRQUVELHFCQUFxQjtRQUNyQixNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUFFLEVBQUUsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDdkQsNkNBQTZDO1FBQzdDLE1BQU0sT0FBTyxHQUFHLGNBQWMsQ0FBQztRQUMvQixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLHlCQUFZLENBQUMsTUFBTSxDQUFDO1FBQy9ELE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksb0JBQU8sQ0FBQyxlQUFlLENBQUM7UUFFekQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixHQUFHLEtBQUs7WUFDUixPQUFPO1lBQ1AsSUFBSSxFQUFFLG1CQUFRLENBQUMsTUFBTSxDQUFDO2dCQUNwQixHQUFHLENBQUMsS0FBSyxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUM7Z0JBQ3pCLEtBQUs7Z0JBQ0wsT0FBTztnQkFDUCxZQUFZO2dCQUNaLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTthQUM3QixDQUFDO1lBQ0YsT0FBTyxFQUFFLE9BQU87U0FDakIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUF6Qkgsb0NBMEJDOzs7QUFFRDs7OztHQUlHO0FBQ0gsU0FBUyxTQUFTLENBQUMsRUFBVSxFQUFFLEtBQWM7SUFDM0MsSUFBSSxLQUFLLEVBQUUsQ0FBQztRQUNWLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDaEMsdUNBQXVDO1lBQ3ZDLGtGQUFrRjtZQUNsRix5REFBeUQ7WUFDekQsS0FBSyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ3pDLENBQUM7UUFFRCxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQzFCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0NBQWdDLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDM0QsQ0FBQztRQUVELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVELE1BQU0sWUFBWSxHQUFHLGdCQUFnQixFQUFFLENBQUM7SUFDeEMsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUUzQyxNQUFNLFlBQVksR0FBRyxZQUFZLENBQUMsT0FBTyxDQUN2QyxJQUFJLE1BQU0sQ0FBQyxHQUFHLE9BQU8sR0FBRyxDQUFDLEVBQ3pCLElBQUksRUFBRSxhQUFhLENBQ3BCLENBQUM7SUFDRixJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQztRQUNoQyxPQUFPLFlBQVksQ0FBQztJQUN0QixDQUFDO0lBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO0FBQzVELENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsZ0JBQWdCO0lBQ3ZCLElBQUksYUFBYSxDQUFDO0lBQ2xCLE1BQU0sS0FBSyxHQUFHLElBQUEsZ0JBQVMsR0FBRSxDQUFDO0lBQzFCLEtBQUssTUFBTSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQztRQUM1QyxJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUUsS0FBSyxjQUFjLEVBQUUsQ0FBQztZQUM5QywrREFBK0Q7WUFDL0QsYUFBYSxHQUFHLEtBQUssR0FBRyxDQUFDLENBQUM7WUFDMUIsTUFBTTtRQUNSLENBQUM7SUFDSCxDQUFDO0lBRUQsSUFBSSxDQUFDLGFBQWEsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDO1FBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQsT0FBTyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7QUFDNUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQge1xuICBBcmNoaXRlY3R1cmUsXG4gIEZ1bmN0aW9uT3B0aW9ucyxcbiAgUnVudGltZSxcbiAgUnVudGltZUZhbWlseSxcbiAgRnVuY3Rpb24sXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBCdW5kbGluZyB9IGZyb20gJy4vYnVuZGxpbmcnO1xuaW1wb3J0IHsgQnVuZGxpbmdPcHRpb25zIH0gZnJvbSAnLi90eXBlcyc7XG5pbXBvcnQgeyBjYWxsc2l0ZXMgfSBmcm9tICcuL3V0aWwnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgUnVzdEZ1bmN0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUnVzdEZ1bmN0aW9uUHJvcHMgZXh0ZW5kcyBGdW5jdGlvbk9wdGlvbnMge1xuICAvKipcbiAgICogUGF0aCB0byB0aGUgZW50cnkgQ2FyZ28udG9tbCBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlcml2ZWQgZnJvbSB0aGUgbmFtZSBvZiB0aGUgZGVmaW5pbmcgZmlsZSBhbmQgdGhlIGNvbnN0cnVjdCdzIGlkLlxuICAgKiBJZiB0aGUgYFJ1c3RzRnVuY3Rpb25gIGlzIGRlZmluZWQgaW4gYHN0YWNrLnRzYCB3aXRoIGBteS1iaW5hcnlgIGFzIGlkXG4gICAqIChgbmV3IFJ1c3RGdW5jdGlvbih0aGlzLCAnbXktYmluYXJ5JylgKSwgdGhlIGNvbnN0cnVjdCB3aWxsIGxvb2sgYXQgYHN0YWNrLm15LWJpbmFyeS9DYXJnby50b21sYFxuICAgKi9cbiAgcmVhZG9ubHkgZW50cnk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBiaW5hcnkgdG8gYnVpbGQsIGluIGNhc2UgdGhhdCBpdCdzIGRpZmZlcmVudCB0aGF0IHRoZSBwYWNrYWdlJ3MgbmFtZS5cbiAgICpcbiAgICogQGRlZmF1bHQgQnVpbGQgYWxsIGJpbmFyaWVzXG4gICAqL1xuICByZWFkb25seSBiaW5hcnlOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcnVudGltZSBlbnZpcm9ubWVudC4gT25seSBPUy1vbmx5IHJ1bnRpbWVzIGFyZSBzdXBwb3J0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGBSdW50aW1lLlBST1ZJREVEX0FMMjAyM2BcbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWU/OiBSdW50aW1lO1xuXG4gIC8qKlxuICAgKiBCdW5kbGluZyBvcHRpb25zXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdXNlIGRlZmF1bHQgYnVuZGxpbmcgb3B0aW9uczogYWxsXG4gICAqIGJpbmFyaWVzIGFuZCBwYWNrYWdlcyBhcmUgYnVuZGxlZC5cbiAgICovXG4gIHJlYWRvbmx5IGJ1bmRsaW5nPzogQnVuZGxpbmdPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBUaGUgcGF0aCB0byB0aGUgZGlyZWN0b3J5IGNvbnRhaW5pbmcgcHJvamVjdCBNYW5pZmVzdCBmaWxlXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRpcmVjdG9yeSBjb250YWluaW5nIHRoZSBgZGVwc0xvY2tGaWxlUGF0aGBcbiAgICovXG4gIHJlYWRvbmx5IHByb2plY3RSb290Pzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEEgUnVzdCBMYW1iZGEgZnVuY3Rpb25cbiAqL1xuZXhwb3J0IGNsYXNzIFJ1c3RGdW5jdGlvbiBleHRlbmRzIEZ1bmN0aW9uIHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFJ1c3RGdW5jdGlvblByb3BzID0ge30pIHtcbiAgICBpZiAocHJvcHMucnVudGltZSAmJiBwcm9wcy5ydW50aW1lLmZhbWlseSAhPT0gUnVudGltZUZhbWlseS5PVEhFUikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdPbmx5IGBDdXN0b21gIHJ1bnRpbWVzIGFyZSBzdXBwb3J0ZWQuJyk7XG4gICAgfVxuXG4gICAgLy8gRW50cnkgYW5kIGRlZmF1bHRzXG4gICAgY29uc3QgZW50cnkgPSBwYXRoLnJlc29sdmUoZmluZEVudHJ5KGlkLCBwcm9wcy5lbnRyeSkpO1xuICAgIC8vIEhhbmRsZXIgaXMgbm90IGltcG9ydGFudCBmb3IgUnVzdCBmdW5jdGlvblxuICAgIGNvbnN0IGhhbmRsZXIgPSAncnVzdC5oYW5kbGVyJztcbiAgICBjb25zdCBhcmNoaXRlY3R1cmUgPSBwcm9wcy5hcmNoaXRlY3R1cmUgPz8gQXJjaGl0ZWN0dXJlLlg4Nl82NDtcbiAgICBjb25zdCBydW50aW1lID0gcHJvcHMucnVudGltZSA/PyBSdW50aW1lLlBST1ZJREVEX0FMMjAyMztcblxuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgLi4ucHJvcHMsXG4gICAgICBydW50aW1lLFxuICAgICAgY29kZTogQnVuZGxpbmcuYnVuZGxlKHtcbiAgICAgICAgLi4uKHByb3BzLmJ1bmRsaW5nID8/IHt9KSxcbiAgICAgICAgZW50cnksXG4gICAgICAgIHJ1bnRpbWUsXG4gICAgICAgIGFyY2hpdGVjdHVyZSxcbiAgICAgICAgYmluYXJ5TmFtZTogcHJvcHMuYmluYXJ5TmFtZSxcbiAgICAgIH0pLFxuICAgICAgaGFuZGxlcjogaGFuZGxlcixcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIFNlYXJjaGVzIGZvciBhbiBlbnRyeSBmaWxlLiBQcmVmZXJlbmNlIG9yZGVyIGlzIHRoZSBmb2xsb3dpbmc6XG4gKiAxLiBHaXZlbiBlbnRyeSBmaWxlXG4gKiAyLiBBIENhcmdvLnRvbWwgZmlsZSBuYW1lZCBhcyB0aGUgZGVmaW5pbmcgZmlsZSB3aXRoIGlkIGFzIHUgKGRlZmluaW5nLWZpbGUuaWQvQ2FyZ28udG9tbClcbiAqL1xuZnVuY3Rpb24gZmluZEVudHJ5KGlkOiBzdHJpbmcsIGVudHJ5Pzogc3RyaW5nKTogc3RyaW5nIHtcbiAgaWYgKGVudHJ5KSB7XG4gICAgaWYgKCEvXFxDYXJnby50b21sJC8udGVzdChlbnRyeSkpIHtcbiAgICAgIC8vIE9ubHkgQ2FyZ28udG9tbCBmaWxlcyBhcmUgc3VwcG9ydGVkLlxuICAgICAgLy8gSWYgdGhlIHVzZXIgZ2F2ZSBhIHBhdGggdG8gYSBkaXJlY3RvcnksIHdlJ2xsIGxvb2sgZm9yIGEgQ2FyZ28udG9tbCBmaWxlIGluIGl0LlxuICAgICAgLy8gSWYgdGhlIHVzZXIgZ2F2ZSBhIHBhdGggdG8gYSBmaWxlLCB3ZSdsbCB1c2UgaXQgYXMtaXMuXG4gICAgICBlbnRyeSA9IHBhdGguam9pbihlbnRyeSwgJ0NhcmdvLnRvbWwnKTtcbiAgICB9XG5cbiAgICBpZiAoIWZzLmV4aXN0c1N5bmMoZW50cnkpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBmaW5kIG1hbmlmZXN0IGZpbGUgYXQgJHtlbnRyeX1gKTtcbiAgICB9XG5cbiAgICByZXR1cm4gZW50cnk7XG4gIH1cblxuICBjb25zdCBkZWZpbmluZ0ZpbGUgPSBmaW5kRGVmaW5pbmdGaWxlKCk7XG4gIGNvbnN0IGV4dG5hbWUgPSBwYXRoLmV4dG5hbWUoZGVmaW5pbmdGaWxlKTtcblxuICBjb25zdCBtYW5pZmVzdEZpbGUgPSBkZWZpbmluZ0ZpbGUucmVwbGFjZShcbiAgICBuZXcgUmVnRXhwKGAke2V4dG5hbWV9JGApLFxuICAgIGAuJHtpZH0vQ2FyZ28udG9tbGAsXG4gICk7XG4gIGlmIChmcy5leGlzdHNTeW5jKG1hbmlmZXN0RmlsZSkpIHtcbiAgICByZXR1cm4gbWFuaWZlc3RGaWxlO1xuICB9XG5cbiAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZmluZCBtYW5pZmVzdCBmaWxlIC0gQ2FyZ28udG9tbCcpO1xufVxuXG4vKipcbiAqIEZpbmRzIHRoZSBuYW1lIG9mIHRoZSBmaWxlIHdoZXJlIHRoZSBgUnVzdEZ1bmN0aW9uYCBpcyBkZWZpbmVkXG4gKi9cbmZ1bmN0aW9uIGZpbmREZWZpbmluZ0ZpbGUoKTogc3RyaW5nIHtcbiAgbGV0IGRlZmluaW5nSW5kZXg7XG4gIGNvbnN0IHNpdGVzID0gY2FsbHNpdGVzKCk7XG4gIGZvciAoY29uc3QgW2luZGV4LCBzaXRlXSBvZiBzaXRlcy5lbnRyaWVzKCkpIHtcbiAgICBpZiAoc2l0ZS5nZXRGdW5jdGlvbk5hbWUoKSA9PT0gJ1J1c3RGdW5jdGlvbicpIHtcbiAgICAgIC8vIFRoZSBuZXh0IHNpdGUgaXMgdGhlIHNpdGUgd2hlcmUgdGhlIFJ1c3RGdW5jdGlvbiB3YXMgY3JlYXRlZFxuICAgICAgZGVmaW5pbmdJbmRleCA9IGluZGV4ICsgMTtcbiAgICAgIGJyZWFrO1xuICAgIH1cbiAgfVxuXG4gIGlmICghZGVmaW5pbmdJbmRleCB8fCAhc2l0ZXNbZGVmaW5pbmdJbmRleF0pIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBmaW5kIGRlZmluaW5nIGZpbGUuJyk7XG4gIH1cblxuICByZXR1cm4gc2l0ZXNbZGVmaW5pbmdJbmRleF0uZ2V0RmlsZU5hbWUoKTtcbn1cbiJdfQ==