"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsDeploymentProvider = void 0;
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const cr = require("aws-cdk-lib/custom-resources");
const is_complete_function_1 = require("./is-complete-function");
const on_event_function_1 = require("./on-event-function");
/**
 * A custom resource provider to handle creation of new {@link EcsDeployment}.
 */
class EcsDeploymentProvider extends cr.Provider {
    constructor(scope, id, props) {
        const eventLambda = new on_event_function_1.OnEventFunction(scope, `${id}OnEventLambda`, {
            timeout: cdk.Duration.seconds(60),
        });
        eventLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'codedeploy:GetApplicationRevision',
                'codedeploy:RegisterApplicationRevision',
            ],
            resources: [
                props.deploymentGroup.application.applicationArn,
            ],
        }));
        eventLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'codedeploy:CreateDeployment',
                'codedeploy:StopDeployment',
                'codedeploy:GetDeployment',
            ],
            resources: [
                props.deploymentGroup.deploymentGroupArn,
            ],
        }));
        eventLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'codedeploy:GetDeploymentConfig',
            ],
            resources: [
                props.deploymentGroup.deploymentConfig.deploymentConfigArn,
            ],
        }));
        const completeLambda = new is_complete_function_1.IsCompleteFunction(scope, `${id}IsCompleteLambda`, {
            timeout: cdk.Duration.seconds(60),
        });
        completeLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'codedeploy:GetDeployment',
            ],
            resources: [
                props.deploymentGroup.deploymentGroupArn,
            ],
        }));
        super(scope, id, {
            onEventHandler: eventLambda,
            isCompleteHandler: completeLambda,
            queryInterval: props.queryInterval || cdk.Duration.seconds(15),
            totalTimeout: props.timeout,
            disableWaiterStateMachineLogging: true,
        });
    }
}
exports.EcsDeploymentProvider = EcsDeploymentProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvZWNzLWRlcGxveW1lbnQtcHJvdmlkZXIvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsbUNBQW1DO0FBRW5DLDJDQUEyQztBQUMzQyxtREFBbUQ7QUFFbkQsaUVBQTREO0FBQzVELDJEQUFzRDtBQXdCdEQ7O0dBRUc7QUFDSCxNQUFhLHFCQUFzQixTQUFRLEVBQUUsQ0FBQyxRQUFRO0lBQ3BELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUM7UUFFekUsTUFBTSxXQUFXLEdBQUcsSUFBSSxtQ0FBZSxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsZUFBZSxFQUFFO1lBQ25FLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7U0FDbEMsQ0FBQyxDQUFDO1FBQ0gsV0FBVyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDbEQsT0FBTyxFQUFFO2dCQUNQLG1DQUFtQztnQkFDbkMsd0NBQXdDO2FBQ3pDO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEtBQUssQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDLGNBQWM7YUFDakQ7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLFdBQVcsQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ2xELE9BQU8sRUFBRTtnQkFDUCw2QkFBNkI7Z0JBQzdCLDJCQUEyQjtnQkFDM0IsMEJBQTBCO2FBQzNCO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEtBQUssQ0FBQyxlQUFlLENBQUMsa0JBQWtCO2FBQ3pDO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixXQUFXLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNsRCxPQUFPLEVBQUU7Z0JBQ1AsZ0NBQWdDO2FBQ2pDO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEtBQUssQ0FBQyxlQUFlLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CO2FBQzNEO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLGNBQWMsR0FBRyxJQUFJLHlDQUFrQixDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsa0JBQWtCLEVBQUU7WUFDNUUsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztTQUNsQyxDQUFDLENBQUM7UUFDSCxjQUFjLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNyRCxPQUFPLEVBQUU7Z0JBQ1AsMEJBQTBCO2FBQzNCO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEtBQUssQ0FBQyxlQUFlLENBQUMsa0JBQWtCO2FBQ3pDO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLGNBQWMsRUFBRSxXQUFXO1lBQzNCLGlCQUFpQixFQUFFLGNBQWM7WUFDakMsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhLElBQUksR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzlELFlBQVksRUFBRSxLQUFLLENBQUMsT0FBTztZQUMzQixnQ0FBZ0MsRUFBRSxJQUFJO1NBQ3ZDLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQXJERCxzREFxREMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgY29kZWRlcGxveSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY29kZWRlcGxveSc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjciBmcm9tICdhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgSXNDb21wbGV0ZUZ1bmN0aW9uIH0gZnJvbSAnLi9pcy1jb21wbGV0ZS1mdW5jdGlvbic7XG5pbXBvcnQgeyBPbkV2ZW50RnVuY3Rpb24gfSBmcm9tICcuL29uLWV2ZW50LWZ1bmN0aW9uJztcblxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBvZiB7QGxpbmsgRWNzRGVwbG95bWVudFByb3ZpZGVyfS5cbiAqL1xuaW50ZXJmYWNlIEVjc0RlcGxveW1lbnRQcm92aWRlclByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBkZXBsb3ltZW50IGdyb3VwIHRvIHRhcmdldCBmb3IgdGhpcyBkZXBsb3ltZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95bWVudEdyb3VwOiBjb2RlZGVwbG95LklFY3NEZXBsb3ltZW50R3JvdXA7XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lb3V0IGZvciB0aGUgZGVwbG95bWVudC4gSWYgdGhlIHRpbWVvdXQgaXMgcmVhY2hlZCwgaXQgd2lsbCB0cmlnZ2VyIGEgcm9sbGJhY2sgb2YgdGhlIHN0YWNrLlxuICAgKi9cbiAgcmVhZG9ubHkgdGltZW91dDogY2RrLkR1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgaW50ZXJ2YWwgdG8gcXVlcnkgdGhlIGRlcGxveW1lbnQgdG8gZGV0ZXJtaW5lIHdoZW4gdGhlIGRlcGxveW1lbnQgaXMgY29tcGxldGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAxNSBzZWNvbmRzXG4gICAqL1xuICByZWFkb25seSBxdWVyeUludGVydmFsPzogY2RrLkR1cmF0aW9uO1xufVxuXG4vKipcbiAqIEEgY3VzdG9tIHJlc291cmNlIHByb3ZpZGVyIHRvIGhhbmRsZSBjcmVhdGlvbiBvZiBuZXcge0BsaW5rIEVjc0RlcGxveW1lbnR9LlxuICovXG5leHBvcnQgY2xhc3MgRWNzRGVwbG95bWVudFByb3ZpZGVyIGV4dGVuZHMgY3IuUHJvdmlkZXIge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRWNzRGVwbG95bWVudFByb3ZpZGVyUHJvcHMpIHtcblxuICAgIGNvbnN0IGV2ZW50TGFtYmRhID0gbmV3IE9uRXZlbnRGdW5jdGlvbihzY29wZSwgYCR7aWR9T25FdmVudExhbWJkYWAsIHtcbiAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDYwKSxcbiAgICB9KTtcbiAgICBldmVudExhbWJkYS5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnY29kZWRlcGxveTpHZXRBcHBsaWNhdGlvblJldmlzaW9uJyxcbiAgICAgICAgJ2NvZGVkZXBsb3k6UmVnaXN0ZXJBcHBsaWNhdGlvblJldmlzaW9uJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgcHJvcHMuZGVwbG95bWVudEdyb3VwLmFwcGxpY2F0aW9uLmFwcGxpY2F0aW9uQXJuLFxuICAgICAgXSxcbiAgICB9KSk7XG4gICAgZXZlbnRMYW1iZGEuYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2NvZGVkZXBsb3k6Q3JlYXRlRGVwbG95bWVudCcsXG4gICAgICAgICdjb2RlZGVwbG95OlN0b3BEZXBsb3ltZW50JyxcbiAgICAgICAgJ2NvZGVkZXBsb3k6R2V0RGVwbG95bWVudCcsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgIHByb3BzLmRlcGxveW1lbnRHcm91cC5kZXBsb3ltZW50R3JvdXBBcm4sXG4gICAgICBdLFxuICAgIH0pKTtcbiAgICBldmVudExhbWJkYS5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnY29kZWRlcGxveTpHZXREZXBsb3ltZW50Q29uZmlnJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgcHJvcHMuZGVwbG95bWVudEdyb3VwLmRlcGxveW1lbnRDb25maWcuZGVwbG95bWVudENvbmZpZ0FybixcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgY29uc3QgY29tcGxldGVMYW1iZGEgPSBuZXcgSXNDb21wbGV0ZUZ1bmN0aW9uKHNjb3BlLCBgJHtpZH1Jc0NvbXBsZXRlTGFtYmRhYCwge1xuICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgIH0pO1xuICAgIGNvbXBsZXRlTGFtYmRhLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdjb2RlZGVwbG95OkdldERlcGxveW1lbnQnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICBwcm9wcy5kZXBsb3ltZW50R3JvdXAuZGVwbG95bWVudEdyb3VwQXJuLFxuICAgICAgXSxcbiAgICB9KSk7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBvbkV2ZW50SGFuZGxlcjogZXZlbnRMYW1iZGEsXG4gICAgICBpc0NvbXBsZXRlSGFuZGxlcjogY29tcGxldGVMYW1iZGEsXG4gICAgICBxdWVyeUludGVydmFsOiBwcm9wcy5xdWVyeUludGVydmFsIHx8IGNkay5EdXJhdGlvbi5zZWNvbmRzKDE1KSxcbiAgICAgIHRvdGFsVGltZW91dDogcHJvcHMudGltZW91dCxcbiAgICAgIGRpc2FibGVXYWl0ZXJTdGF0ZU1hY2hpbmVMb2dnaW5nOiB0cnVlLFxuICAgIH0pO1xuICB9XG59XG4iXX0=