"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = exports.DeploymentStatus = void 0;
const client_codedeploy_1 = require("@aws-sdk/client-codedeploy");
const logger_1 = require("./logger");
var DeploymentStatus;
(function (DeploymentStatus) {
    DeploymentStatus["CREATED"] = "Created";
    DeploymentStatus["QUEUED"] = "Queued";
    DeploymentStatus["IN_PROGRESS"] = "InProgress";
    DeploymentStatus["BAKING"] = "Baking";
    DeploymentStatus["SUCCEEDED"] = "Succeeded";
    DeploymentStatus["FAILED"] = "Failed";
    DeploymentStatus["STOPPED"] = "Stopped";
    DeploymentStatus["READY"] = "Ready";
})(DeploymentStatus || (exports.DeploymentStatus = DeploymentStatus = {}));
/**
 * The lambda function called from CloudFormation for this custom resource.
 *
 * @param event
 * @returns whether the deployment is complete
 */
async function handler(event) {
    const logger = new logger_1.Logger();
    const codedeployClient = new client_codedeploy_1.CodeDeployClient({});
    try {
        const resp = await codedeployClient.send(new client_codedeploy_1.GetDeploymentCommand({ deploymentId: event.PhysicalResourceId }));
        let rollbackResp = {};
        if (resp.deploymentInfo?.rollbackInfo?.rollbackDeploymentId) {
            rollbackResp = await codedeployClient.send(new client_codedeploy_1.GetDeploymentCommand({ deploymentId: resp.deploymentInfo?.rollbackInfo?.rollbackDeploymentId }));
        }
        logger.appendKeys({
            stackEvent: event.RequestType,
            deploymentId: event.PhysicalResourceId,
            deploymentStatus: resp.deploymentInfo?.status,
            rollbackStatus: rollbackResp?.deploymentInfo?.status,
        });
        logger.info('Checking deployment');
        // check if deployment id is complete
        switch (event.RequestType) {
            case 'Create':
            case 'Update':
                switch (resp.deploymentInfo?.status) {
                    case DeploymentStatus.SUCCEEDED:
                        logger.info('Deployment finished successfully', { complete: true });
                        return { IsComplete: true };
                    case DeploymentStatus.FAILED:
                    case DeploymentStatus.STOPPED:
                        if (rollbackResp.deploymentInfo?.status) {
                            if (rollbackResp.deploymentInfo?.status == DeploymentStatus.SUCCEEDED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.FAILED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.STOPPED) {
                                const errInfo = resp.deploymentInfo.errorInformation;
                                const error = new Error(`Deployment ${resp.deploymentInfo.status}: [${errInfo?.code}] ${errInfo?.message}`);
                                logger.error('Deployment failed', { complete: true, error });
                                throw error;
                            }
                            logger.info('Waiting for final status from a rollback', { complete: false });
                            return { IsComplete: false }; // waiting for final status from rollback
                        }
                        else {
                            const errInfo = resp.deploymentInfo.errorInformation;
                            const error = new Error(`Deployment ${resp.deploymentInfo.status}: [${errInfo?.code}] ${errInfo?.message}`);
                            logger.error('No rollback to wait for', { complete: true, error });
                            throw error;
                        }
                    default:
                        logger.info('Waiting for final status from deployment', { complete: false });
                        return { IsComplete: false };
                }
            case 'Delete':
                switch (resp.deploymentInfo?.status) {
                    case DeploymentStatus.SUCCEEDED:
                        logger.info('Deployment finished successfully - nothing to delete', { complete: true });
                        return { IsComplete: true };
                    case DeploymentStatus.FAILED:
                    case DeploymentStatus.STOPPED:
                        if (rollbackResp.deploymentInfo?.status) {
                            if (rollbackResp.deploymentInfo?.status == DeploymentStatus.SUCCEEDED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.FAILED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.STOPPED) {
                                logger.info('Rollback in final status', { complete: true });
                                return { IsComplete: true }; // rollback finished, we're deleted
                            }
                            logger.info('Waiting for final status from a rollback', { complete: false });
                            return { IsComplete: false }; // waiting for rollback
                        }
                        logger.info('No rollback to wait for', { complete: true });
                        return { IsComplete: true };
                    default:
                        logger.info('Waiting for final status from deployment', { complete: false });
                        return { IsComplete: false };
                }
            default:
                logger.error('Unknown request type');
                throw new Error(`Unknown request type: ${event.RequestType}`);
        }
    }
    catch (e) {
        logger.error('Unable to determine deployment status', { error: e });
        if (event.RequestType === 'Delete') {
            logger.warn('Ignoring error - nothing to do', { complete: true });
            return { IsComplete: true };
        }
        throw e;
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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