import datetime
import faulthandler
import unittest

import numpy as np

faulthandler.enable()  # to debug seg faults and timeouts

import netCDF4

import cf

VN = cf.CF()


def _make_broken_bounds_cdl(filename):
    with open(filename, mode="w") as f:
        f.write(
            """netcdf broken_bounds {
dimensions:
      lat = 180 ;
      bnds = 2 ;
      lon = 288 ;
      time = UNLIMITED ; // (1825 currently)
variables:
      double lat(lat) ;
           lat:long_name = "latitude" ;
           lat:units = "degrees_north" ;
           lat:axis = "Y" ;
           lat:bounds = "lat_bnds" ;
           lat:standard_name = "latitude" ;
           lat:cell_methods = "time: point" ;
      double lat_bnds(lat, bnds) ;
           lat_bnds:long_name = "latitude bounds" ;
           lat_bnds:units = "degrees_north" ;
           lat_bnds:axis = "Y" ;
      double lon(lon) ;
           lon:long_name = "longitude" ;
           lon:units = "degrees_east" ;
           lon:axis = "X" ;
           lon:bounds = "lon_bnds" ;
           lon:standard_name = "longitude" ;
           lon:cell_methods = "time: point" ;
      double lon_bnds(lon, bnds) ;
           lon_bnds:long_name = "longitude bounds" ;
           lon_bnds:units = "m" ;
           lon_bnds:axis = "X" ;
      float pr(time, lat, lon) ;
           pr:long_name = "Precipitation" ;
           pr:units = "kg m-2 s-1" ;
           pr:missing_value = 1.e+20f ;
           pr:_FillValue = 1.e+20f ;
           pr:cell_methods = "area: time: mean" ;
           pr:cell_measures = "area: areacella" ;
           pr:standard_name = "precipitation_flux" ;
           pr:interp_method = "conserve_order1" ;
           pr:original_name = "pr" ;
      double time(time) ;
           time:long_name = "time" ;
           time:units = "days since 1850-01-01 00:00:00" ;
           time:axis = "T" ;
           time:calendar_type = "noleap" ;
           time:calendar = "noleap" ;
           time:bounds = "time_bnds" ;
           time:standard_name = "time" ;
           time:description = "Temporal mean" ;
      double time_bnds(time, bnds) ;
           time_bnds:long_name = "time axis boundaries" ;
           time_bnds:units = "days since 1850-01-01 00:00:00" ;

// global attributes:
           :external_variables = "areacella" ;
           :Conventions = "CF-"""
            + VN
            + """" ;
           :source = "model" ;
           :comment = "Bounds variable has incompatible units to its parent coordinate variable" ;
}
"""
        )


def _make_regrid_file(filename):
    n = netCDF4.Dataset(filename, "w", format="NETCDF3_CLASSIC")

    n.Conventions = "CF-" + VN

    n.createDimension("time", 2)
    n.createDimension("bounds2", 2)
    n.createDimension("latitude", 30)
    n.createDimension("longitude", 48)
    n.createDimension("time_1", 1)
    n.createDimension("lat", 73)
    n.createDimension("lon", 96)

    latitude = n.createVariable("latitude", "f8", ("latitude",))
    latitude.standard_name = "latitude"
    latitude.units = "degrees_north"
    latitude.bounds = "latitude_bounds"
    latitude[...] = np.arange(-87, 90.0, 6)

    longitude = n.createVariable("longitude", "f8", ("longitude",))
    longitude.standard_name = "longitude"
    longitude.units = "degrees_east"
    longitude.bounds = "longitude_bounds"
    longitude[...] = np.arange(3.75, 360, 7.5)

    lat = n.createVariable("lat", "f8", ("lat",))
    lat.standard_name = "latitude"
    lat.units = "degrees_north"
    lat.bounds = "lat_bounds"
    lat[...] = np.arange(-90, 91.0, 2.5)

    lon = n.createVariable("lon", "f8", ("lon",))
    lon.standard_name = "longitude"
    lon.units = "degrees_east"
    lon.bounds = "lon_bounds"
    lon[...] = np.arange(3.75, 361, 3.75)

    longitude_bounds = n.createVariable(
        "longitude_bounds", "f8", ("longitude", "bounds2")
    )
    longitude_bounds[..., 0] = longitude[...] - 3.75
    longitude_bounds[..., 1] = longitude[...] + 3.75

    latitude_bounds = n.createVariable(
        "latitude_bounds", "f8", ("latitude", "bounds2")
    )
    latitude_bounds[..., 0] = latitude[...] - 3
    latitude_bounds[..., 1] = latitude[...] + 3

    lon_bounds = n.createVariable("lon_bounds", "f8", ("lon", "bounds2"))
    lon_bounds[..., 0] = lon[...] - 1.875
    lon_bounds[..., 1] = lon[...] + 1.875

    lat_bounds = n.createVariable("lat_bounds", "f8", ("lat", "bounds2"))
    lat_bounds[..., 0] = lat[...] - 1.25
    lat_bounds[..., 1] = lat[...] + 1.25

    time = n.createVariable("time", "f4", ("time",))
    time.standard_name = "time"
    time.units = "days since 1860-1-1"
    time.calendar = "360_day"
    time.axis = "T"
    time.bounds = "time_bounds"
    time[...] = [15, 45]

    time_bounds = n.createVariable("time_bounds", "f4", ("time", "bounds2"))
    time_bounds[...] = [
        [
            0,
            30,
        ],
        [30, 60],
    ]

    time_1 = n.createVariable("time_1", "f4", ("time_1",))
    time_1.standard_name = "time"
    time_1.units = "days since 1860-1-1"
    time_1.calendar = "360_day"
    time_1.axis = "T"
    time_1.bounds = "time_1_bounds"
    time_1[...] = 15

    time_1_bounds = n.createVariable(
        "time_1_bounds", "f4", ("time_1", "bounds2")
    )
    time_1_bounds[...] = [0, 30]

    height = n.createVariable("height", "f8", ())
    height.units = "m"
    height.standard_name = "height"
    height.positive = "up"
    height.axis = "Z"
    height[...] = 2

    src = n.createVariable("src", "f8", ("time", "latitude", "longitude"))
    src.standard_name = "air_temperature"
    src.units = "K"
    src.coordinates = "height"
    src.cell_methods = "time: mean"

    # Don't generate this data randomly - it's useful to see the real
    # patterns of global temperature.
    src[...] = [
        [
            [
                243.6,
                242.4,
                241.8,
                241.5,
                241.2,
                240.8,
                240.5,
                240.4,
                240.2,
                240.5,
                241.2,
                241.9,
                242.5,
                243.0,
                243.4,
                244.1,
                245.2,
                246.4,
                247.0,
                247.4,
                248.3,
                250.2,
                251.6,
                252.3,
                253.9,
                255.8,
                257.6,
                258.7,
                258.5,
                257.7,
                256.8,
                255.6,
                254.1,
                252.0,
                251.6,
                252.4,
                254.0,
                255.1,
                254.8,
                254.3,
                253.4,
                252.6,
                251.5,
                249.9,
                248.3,
                246.4,
                244.9,
                244.2,
            ],
            [
                243.3,
                241.0,
                239.8,
                238.3,
                237.4,
                237.4,
                238.9,
                240.3,
                241.3,
                241.2,
                241.2,
                240.8,
                241.0,
                242.6,
                244.4,
                245.5,
                246.4,
                248.2,
                249.8,
                251.0,
                254.3,
                260.7,
                265.4,
                266.1,
                265.9,
                265.9,
                266.4,
                266.1,
                265.3,
                264.2,
                262.8,
                261.8,
                261.0,
                258.7,
                256.9,
                257.3,
                259.5,
                262.7,
                264.4,
                264.9,
                265.4,
                264.2,
                264.7,
                263.4,
                258.3,
                251.8,
                247.2,
                245.4,
            ],
            [
                248.6,
                245.7,
                245.6,
                244.7,
                243.3,
                243.3,
                244.2,
                245.2,
                249.4,
                251.7,
                248.8,
                245.0,
                243.0,
                244.0,
                245.7,
                245.2,
                244.7,
                246.0,
                246.9,
                248.7,
                252.4,
                257.6,
                266.8,
                269.5,
                269.7,
                270.1,
                270.5,
                270.6,
                270.2,
                269.2,
                266.7,
                266.1,
                266.7,
                267.5,
                267.3,
                266.2,
                266.5,
                268.1,
                267.3,
                267.5,
                271.5,
                271.5,
                271.5,
                271.0,
                267.6,
                261.5,
                255.5,
                252.0,
            ],
            [
                270.3,
                269.9,
                269.6,
                268.4,
                266.0,
                264.0,
                258.8,
                256.8,
                262.0,
                265.2,
                263.6,
                257.3,
                254.3,
                253.9,
                255.6,
                256.9,
                255.9,
                253.9,
                254.5,
                261.0,
                263.8,
                267.9,
                270.5,
                271.8,
                272.3,
                272.6,
                273.4,
                273.9,
                273.7,
                273.6,
                273.3,
                273.4,
                273.9,
                273.8,
                273.8,
                273.6,
                274.4,
                274.6,
                271.1,
                268.0,
                271.2,
                271.5,
                271.4,
                271.3,
                271.6,
                271.5,
                270.8,
                270.5,
            ],
            [
                274.5,
                274.3,
                274.4,
                274.3,
                274.5,
                274.5,
                273.4,
                273.1,
                273.2,
                273.3,
                273.0,
                271.5,
                271.6,
                272.5,
                272.8,
                272.8,
                272.6,
                272.5,
                272.1,
                272.5,
                273.5,
                273.6,
                274.0,
                274.8,
                274.9,
                275.0,
                275.0,
                275.0,
                274.9,
                274.7,
                274.8,
                274.9,
                275.1,
                275.2,
                275.5,
                275.8,
                276.0,
                276.2,
                276.0,
                273.4,
                272.8,
                273.9,
                274.5,
                274.6,
                274.7,
                274.7,
                274.7,
                274.6,
            ],
            [
                275.3,
                275.2,
                275.4,
                275.3,
                275.4,
                275.7,
                275.7,
                275.8,
                275.9,
                275.5,
                274.9,
                274.5,
                274.3,
                274.6,
                274.7,
                275.0,
                275.2,
                275.3,
                275.5,
                275.6,
                276.0,
                276.7,
                277.0,
                276.8,
                276.8,
                277.0,
                277.1,
                276.8,
                276.7,
                277.1,
                277.0,
                277.4,
                277.3,
                277.2,
                277.2,
                277.3,
                277.5,
                278.0,
                278.8,
                279.1,
                278.3,
                278.2,
                277.8,
                277.3,
                277.4,
                276.9,
                276.4,
                275.8,
            ],
            [
                278.6,
                278.7,
                278.4,
                278.0,
                277.8,
                278.0,
                278.3,
                278.2,
                278.1,
                278.0,
                278.7,
                279.4,
                279.1,
                279.2,
                279.1,
                278.9,
                279.0,
                279.6,
                279.8,
                280.1,
                280.2,
                280.6,
                281.1,
                280.0,
                280.0,
                280.5,
                281.2,
                281.0,
                280.8,
                281.3,
                281.6,
                281.4,
                281.1,
                280.7,
                280.3,
                280.1,
                280.3,
                280.7,
                280.8,
                283.0,
                281.4,
                280.3,
                279.6,
                279.4,
                279.7,
                279.9,
                279.1,
                278.6,
            ],
            [
                283.9,
                283.2,
                282.5,
                281.8,
                281.6,
                282.0,
                282.4,
                282.3,
                282.3,
                283.0,
                284.3,
                284.2,
                284.0,
                283.7,
                283.6,
                283.7,
                284.0,
                284.2,
                284.3,
                285.0,
                285.2,
                285.1,
                285.2,
                286.9,
                287.3,
                286.5,
                286.2,
                285.7,
                285.6,
                285.7,
                285.5,
                285.2,
                285.0,
                284.5,
                284.0,
                283.4,
                283.0,
                283.1,
                283.0,
                288.0,
                285.5,
                285.2,
                284.0,
                283.2,
                283.4,
                284.7,
                284.9,
                284.6,
            ],
            [
                289.3,
                288.6,
                288.9,
                289.4,
                288.8,
                289.1,
                289.5,
                289.3,
                289.2,
                289.6,
                289.4,
                288.9,
                288.4,
                288.1,
                288.0,
                288.2,
                288.3,
                288.4,
                289.4,
                290.3,
                291.6,
                290.6,
                290.8,
                291.8,
                291.8,
                290.8,
                290.1,
                290.1,
                290.7,
                290.5,
                290.1,
                289.9,
                289.6,
                289.1,
                288.5,
                287.6,
                286.8,
                286.9,
                287.5,
                294.9,
                293.2,
                291.4,
                289.9,
                289.6,
                290.1,
                290.5,
                290.3,
                289.8,
            ],
            [
                292.5,
                292.3,
                293.5,
                292.9,
                296.4,
                295.3,
                294.8,
                294.7,
                294.4,
                293.9,
                293.1,
                292.5,
                291.8,
                291.3,
                291.9,
                293.9,
                292.1,
                293.6,
                297.5,
                295.3,
                295.3,
                295.4,
                295.1,
                294.6,
                294.5,
                294.0,
                294.0,
                294.6,
                295.1,
                295.1,
                295.1,
                295.0,
                294.7,
                294.1,
                293.3,
                292.0,
                290.6,
                289.5,
                293.3,
                301.4,
                299.6,
                296.1,
                294.7,
                294.5,
                294.6,
                294.8,
                294.0,
                293.0,
            ],
            [
                293.4,
                293.4,
                295.6,
                293.0,
                297.2,
                299.2,
                298.2,
                297.2,
                296.2,
                295.3,
                294.6,
                294.2,
                294.3,
                294.2,
                295.8,
                298.7,
                297.5,
                299.4,
                300.3,
                298.5,
                297.7,
                297.8,
                297.4,
                297.0,
                296.9,
                296.7,
                297.0,
                297.1,
                297.3,
                297.4,
                297.2,
                296.8,
                296.2,
                295.6,
                294.8,
                293.6,
                292.1,
                290.7,
                290.9,
                297.7,
                300.3,
                297.1,
                298.1,
                296.9,
                295.9,
                295.2,
                294.4,
                293.7,
            ],
            [
                295.2,
                295.6,
                295.6,
                295.3,
                298.0,
                300.5,
                298.8,
                298.5,
                297.4,
                296.6,
                296.5,
                296.6,
                296.7,
                296.8,
                298.6,
                301.8,
                301.4,
                299.2,
                299.3,
                299.3,
                300.0,
                299.5,
                299.1,
                299.1,
                298.9,
                298.8,
                299.0,
                299.0,
                298.9,
                298.5,
                297.5,
                296.3,
                295.0,
                294.3,
                293.7,
                292.8,
                292.0,
                292.6,
                290.3,
                292.7,
                299.9,
                296.4,
                297.0,
                297.4,
                296.1,
                295.3,
                294.6,
                294.5,
            ],
            [
                297.4,
                296.9,
                294.4,
                294.9,
                296.7,
                300.0,
                299.4,
                299.2,
                298.7,
                298.7,
                299.3,
                299.7,
                299.7,
                299.8,
                300.7,
                301.9,
                301.3,
                300.1,
                301.1,
                301.1,
                301.4,
                301.1,
                300.7,
                300.3,
                300.3,
                300.6,
                300.5,
                300.4,
                300.0,
                299.4,
                298.2,
                296.5,
                294.9,
                293.7,
                292.9,
                292.9,
                293.9,
                293.8,
                289.6,
                297.1,
                298.9,
                296.4,
                296.1,
                298.7,
                297.6,
                296.6,
                296.1,
                296.4,
            ],
            [
                300.5,
                299.9,
                295.9,
                295.1,
                295.0,
                299.8,
                300.3,
                300.1,
                300.2,
                300.4,
                300.7,
                300.8,
                301.0,
                301.1,
                301.5,
                301.9,
                302.1,
                302.2,
                302.0,
                300.7,
                301.9,
                301.7,
                301.5,
                301.3,
                301.3,
                301.3,
                301.1,
                300.9,
                300.6,
                300.0,
                299.0,
                297.6,
                296.4,
                295.7,
                295.5,
                296.0,
                297.4,
                295.2,
                298.4,
                300.2,
                298.8,
                298.2,
                297.6,
                299.5,
                299.2,
                298.7,
                298.8,
                299.4,
            ],
            [
                301.4,
                299.9,
                298.7,
                296.7,
                294.5,
                299.3,
                300.2,
                300.1,
                300.2,
                300.5,
                300.6,
                300.7,
                301.1,
                300.2,
                301.2,
                300.7,
                301.7,
                302.3,
                300.1,
                300.7,
                300.8,
                300.6,
                300.4,
                300.1,
                299.8,
                299.5,
                299.0,
                298.5,
                298.2,
                297.9,
                297.5,
                297.2,
                297.1,
                297.3,
                297.8,
                298.5,
                299.2,
                297.0,
                300.9,
                300.5,
                300.2,
                299.5,
                299.7,
                300.3,
                300.4,
                300.5,
                300.7,
                301.0,
            ],
            [
                301.7,
                298.3,
                298.4,
                297.1,
                296.4,
                299.3,
                299.3,
                299.7,
                300.1,
                300.2,
                300.1,
                300.5,
                301.3,
                299.8,
                301.0,
                299.6,
                301.9,
                301.9,
                301.7,
                301.3,
                300.8,
                300.5,
                300.2,
                300.0,
                299.6,
                299.3,
                299.2,
                298.9,
                298.6,
                298.5,
                298.3,
                298.3,
                298.4,
                298.8,
                299.2,
                299.6,
                299.7,
                297.9,
                298.3,
                299.1,
                300.2,
                299.9,
                299.9,
                300.2,
                300.6,
                301.0,
                301.3,
                301.3,
            ],
            [
                296.5,
                296.5,
                298.6,
                298.1,
                297.5,
                291.4,
                294.6,
                298.1,
                298.8,
                299.6,
                297.7,
                299.7,
                300.3,
                299.2,
                300.4,
                301.3,
                299.5,
                300.9,
                300.8,
                300.5,
                300.2,
                299.9,
                299.5,
                299.3,
                299.1,
                299.1,
                299.1,
                299.0,
                299.0,
                299.1,
                299.3,
                299.4,
                299.8,
                299.8,
                299.3,
                299.6,
                299.1,
                298.2,
                298.7,
                300.1,
                299.2,
                299.0,
                298.6,
                298.6,
                299.0,
                299.9,
                298.2,
                297.5,
            ],
            [
                296.9,
                296.4,
                296.6,
                295.1,
                297.1,
                293.1,
                290.8,
                295.7,
                296.5,
                297.2,
                293.9,
                297.9,
                297.6,
                293.8,
                295.8,
                298.8,
                298.2,
                298.9,
                298.9,
                299.1,
                298.9,
                298.2,
                297.5,
                296.9,
                296.6,
                296.3,
                296.2,
                296.0,
                296.0,
                296.4,
                296.8,
                297.1,
                297.6,
                298.7,
                296.1,
                294.9,
                296.5,
                298.6,
                298.3,
                298.6,
                298.2,
                297.6,
                297.2,
                296.9,
                296.7,
                297.3,
                297.9,
                297.5,
            ],
            [
                290.4,
                289.2,
                288.8,
                289.8,
                293.2,
                290.9,
                288.7,
                291.5,
                295.4,
                292.9,
                289.2,
                292.3,
                289.4,
                284.6,
                287.3,
                293.6,
                294.7,
                293.8,
                294.2,
                295.1,
                295.9,
                296.3,
                296.0,
                295.3,
                294.6,
                294.0,
                293.4,
                293.0,
                292.7,
                292.7,
                292.7,
                292.7,
                293.2,
                293.8,
                284.9,
                292.3,
                296.2,
                295.6,
                296.5,
                297.3,
                296.9,
                296.4,
                295.8,
                295.4,
                295.3,
                294.3,
                292.3,
                292.0,
            ],
            [
                287.2,
                286.3,
                287.4,
                286.9,
                287.2,
                284.8,
                288.8,
                285.3,
                285.1,
                284.1,
                282.3,
                276.0,
                275.1,
                274.6,
                277.2,
                277.9,
                286.6,
                288.8,
                289.0,
                289.7,
                290.8,
                291.9,
                292.5,
                292.4,
                292.0,
                291.4,
                290.7,
                290.2,
                289.9,
                290.0,
                290.0,
                289.9,
                289.7,
                284.1,
                279.8,
                286.4,
                287.8,
                291.0,
                293.1,
                293.7,
                294.0,
                294.2,
                294.1,
                293.8,
                293.4,
                293.4,
                289.8,
                287.6,
            ],
            [
                280.6,
                285.5,
                288.7,
                288.8,
                286.8,
                281.0,
                278.3,
                276.1,
                275.1,
                274.5,
                259.3,
                252.7,
                252.3,
                264.4,
                271.9,
                273.4,
                278.4,
                279.7,
                279.1,
                284.5,
                286.3,
                287.4,
                287.9,
                288.1,
                287.9,
                287.4,
                287.1,
                287.1,
                287.3,
                287.4,
                287.5,
                286.6,
                280.6,
                274.1,
                274.0,
                274.2,
                274.0,
                280.3,
                288.9,
                290.1,
                290.6,
                290.9,
                291.3,
                291.6,
                291.5,
                291.3,
                289.6,
                281.6,
            ],
            [
                283.4,
                283.8,
                281.9,
                278.6,
                274.7,
                270.8,
                275.5,
                275.8,
                273.4,
                263.9,
                261.0,
                262.5,
                259.6,
                261.2,
                263.1,
                263.7,
                257.6,
                263.7,
                270.0,
                273.9,
                278.9,
                279.9,
                281.1,
                282.0,
                282.3,
                282.1,
                282.3,
                283.0,
                283.8,
                284.4,
                284.7,
                279.8,
                267.4,
                263.8,
                267.3,
                266.6,
                266.9,
                269.3,
                279.5,
                284.7,
                286.3,
                286.8,
                287.5,
                288.5,
                289.0,
                288.4,
                285.2,
                278.3,
            ],
            [
                274.1,
                268.7,
                266.7,
                267.4,
                275.8,
                272.9,
                272.3,
                266.8,
                265.1,
                263.9,
                260.3,
                255.9,
                254.0,
                254.2,
                254.2,
                251.0,
                246.6,
                247.9,
                260.6,
                268.4,
                272.8,
                274.1,
                275.4,
                275.8,
                276.4,
                277.3,
                278.2,
                279.1,
                280.3,
                281.3,
                282.2,
                276.3,
                266.5,
                261.3,
                260.0,
                260.9,
                266.2,
                262.4,
                264.5,
                274.0,
                276.3,
                278.6,
                281.2,
                283.9,
                285.7,
                285.4,
                284.7,
                279.6,
            ],
            [
                272.7,
                265.5,
                263.2,
                260.9,
                259.3,
                258.8,
                258.3,
                257.4,
                256.2,
                256.5,
                257.0,
                255.2,
                252.8,
                252.1,
                249.8,
                244.3,
                241.6,
                242.7,
                251.1,
                267.8,
                266.7,
                266.4,
                271.4,
                272.1,
                273.2,
                274.5,
                275.8,
                276.8,
                278.2,
                279.6,
                278.7,
                268.7,
                262.3,
                257.8,
                255.5,
                256.1,
                257.5,
                260.0,
                258.8,
                260.5,
                268.1,
                275.4,
                278.5,
                280.7,
                282.4,
                283.3,
                281.9,
                275.9,
            ],
            [
                276.9,
                267.1,
                268.1,
                255.3,
                249.1,
                247.3,
                247.4,
                249.0,
                248.9,
                249.3,
                249.4,
                250.5,
                250.6,
                247.2,
                243.4,
                239.2,
                236.3,
                236.0,
                243.7,
                256.7,
                256.2,
                249.1,
                261.6,
                264.2,
                268.2,
                271.2,
                270.2,
                272.0,
                273.9,
                273.1,
                266.5,
                262.1,
                260.5,
                256.4,
                251.8,
                250.0,
                252.0,
                257.4,
                252.0,
                254.8,
                270.0,
                275.2,
                278.1,
                280.0,
                281.3,
                282.1,
                282.0,
                275.4,
            ],
            [
                274.8,
                263.2,
                258.6,
                246.4,
                242.3,
                240.9,
                240.9,
                240.7,
                238.5,
                238.7,
                239.0,
                239.7,
                238.6,
                235.6,
                233.2,
                230.9,
                229.4,
                231.4,
                234.8,
                238.1,
                238.7,
                241.6,
                244.2,
                247.5,
                257.9,
                264.2,
                254.9,
                256.2,
                256.3,
                254.9,
                255.7,
                253.4,
                251.6,
                249.1,
                245.6,
                245.1,
                247.2,
                248.4,
                248.9,
                257.5,
                270.1,
                259.8,
                265.1,
                275.8,
                277.7,
                274.3,
                278.0,
                278.9,
            ],
            [
                273.2,
                268.4,
                256.0,
                247.6,
                250.2,
                244.8,
                239.2,
                234.9,
                234.6,
                230.2,
                229.6,
                230.8,
                231.4,
                229.9,
                228.3,
                227.7,
                228.3,
                230.0,
                230.5,
                231.3,
                231.8,
                235.8,
                238.5,
                239.8,
                242.0,
                246.1,
                248.1,
                244.2,
                246.0,
                247.0,
                246.2,
                243.8,
                242.5,
                241.8,
                242.7,
                241.8,
                241.7,
                242.4,
                245.5,
                254.0,
                263.4,
                250.4,
                244.5,
                248.4,
                255.2,
                261.2,
                267.8,
                270.9,
            ],
            [
                268.5,
                269.1,
                268.5,
                265.7,
                262.3,
                243.8,
                236.0,
                236.2,
                235.6,
                235.6,
                234.7,
                229.1,
                228.7,
                227.4,
                228.8,
                233.2,
                233.0,
                235.3,
                235.9,
                236.5,
                236.9,
                237.3,
                237.9,
                238.7,
                239.2,
                239.6,
                241.5,
                239.2,
                239.7,
                240.2,
                241.2,
                240.7,
                240.0,
                240.8,
                242.7,
                240.8,
                239.1,
                245.3,
                251.1,
                258.6,
                247.1,
                239.5,
                237.2,
                235.4,
                241.2,
                246.8,
                247.9,
                258.9,
            ],
            [
                259.5,
                258.8,
                258.0,
                253.9,
                243.8,
                239.9,
                238.3,
                237.7,
                235.8,
                233.8,
                233.3,
                233.2,
                234.4,
                232.8,
                234.4,
                236.1,
                236.5,
                237.4,
                237.5,
                237.6,
                237.8,
                236.8,
                236.3,
                236.2,
                236.1,
                235.9,
                235.7,
                235.3,
                234.9,
                234.6,
                235.6,
                236.4,
                236.5,
                236.5,
                237.7,
                239.6,
                235.8,
                232.6,
                234.3,
                238.0,
                236.9,
                235.3,
                233.4,
                235.4,
                238.5,
                241.4,
                243.3,
                252.2,
            ],
            [
                235.4,
                235.8,
                236.3,
                235.0,
                234.9,
                236.2,
                235.8,
                235.5,
                236.3,
                236.6,
                235.6,
                235.9,
                236.6,
                236.9,
                237.2,
                237.9,
                238.5,
                238.7,
                238.6,
                238.4,
                238.2,
                238.1,
                237.9,
                237.2,
                236.7,
                236.2,
                235.8,
                235.2,
                234.3,
                233.1,
                232.7,
                233.3,
                233.8,
                233.0,
                232.9,
                233.1,
                234.2,
                233.5,
                233.0,
                234.8,
                234.5,
                234.2,
                234.7,
                236.3,
                237.7,
                238.0,
                236.8,
                236.1,
            ],
        ],
        [
            [
                242.9,
                242.6,
                241.9,
                242.3,
                240.2,
                240.4,
                240.3,
                241.3,
                240.0,
                239.5,
                242.1,
                241.0,
                243.0,
                243.7,
                244.3,
                243.8,
                244.9,
                246.1,
                247.5,
                247.8,
                248.7,
                249.7,
                250.9,
                253.1,
                254.7,
                256.3,
                257.2,
                259.0,
                258.2,
                257.7,
                256.6,
                255.9,
                254.4,
                251.2,
                250.6,
                252.5,
                254.6,
                254.5,
                255.0,
                254.9,
                253.3,
                253.3,
                251.8,
                249.6,
                248.8,
                247.2,
                244.7,
                244.3,
            ],
            [
                243.6,
                240.8,
                239.4,
                237.9,
                237.0,
                237.5,
                239.4,
                239.8,
                241.0,
                240.8,
                240.8,
                241.1,
                240.8,
                241.7,
                245.4,
                246.1,
                246.6,
                249.1,
                250.7,
                250.8,
                254.7,
                260.1,
                265.0,
                266.1,
                265.4,
                265.8,
                265.6,
                266.5,
                266.2,
                263.3,
                262.6,
                261.4,
                261.5,
                258.0,
                256.7,
                256.6,
                260.0,
                262.8,
                264.3,
                265.7,
                265.2,
                264.2,
                265.5,
                263.1,
                257.6,
                252.6,
                246.3,
                245.4,
            ],
            [
                248.9,
                245.5,
                246.5,
                244.3,
                242.8,
                243.0,
                243.2,
                244.5,
                249.7,
                252.2,
                249.1,
                244.7,
                242.8,
                244.1,
                245.0,
                245.6,
                245.6,
                245.5,
                247.5,
                249.4,
                252.2,
                257.6,
                266.6,
                269.0,
                270.0,
                271.0,
                270.0,
                270.2,
                271.0,
                269.2,
                266.4,
                266.2,
                266.8,
                267.2,
                268.0,
                265.9,
                266.2,
                268.4,
                268.2,
                267.6,
                271.8,
                271.2,
                272.1,
                271.8,
                267.5,
                261.8,
                255.2,
                252.6,
            ],
            [
                270.4,
                269.6,
                269.1,
                267.5,
                266.9,
                264.2,
                259.1,
                256.5,
                262.1,
                266.1,
                263.7,
                256.9,
                255.1,
                253.5,
                255.6,
                256.0,
                256.6,
                254.5,
                253.5,
                261.3,
                264.7,
                268.4,
                271.0,
                271.3,
                271.5,
                273.1,
                273.2,
                273.8,
                272.9,
                274.1,
                272.8,
                273.2,
                273.5,
                274.4,
                274.7,
                274.6,
                275.0,
                275.3,
                271.3,
                267.3,
                271.9,
                271.2,
                271.7,
                272.3,
                271.2,
                270.5,
                271.0,
                270.8,
            ],
            [
                274.9,
                273.8,
                273.7,
                273.6,
                274.8,
                275.0,
                272.5,
                273.9,
                274.2,
                273.1,
                272.3,
                270.9,
                272.0,
                273.3,
                272.1,
                272.1,
                271.9,
                273.3,
                271.6,
                272.1,
                272.6,
                273.1,
                273.3,
                275.7,
                275.6,
                274.0,
                275.1,
                274.8,
                274.0,
                275.1,
                275.1,
                275.7,
                274.1,
                276.1,
                275.1,
                275.3,
                276.5,
                275.4,
                275.2,
                274.4,
                273.3,
                274.3,
                274.7,
                274.3,
                274.9,
                273.7,
                274.4,
                274.1,
            ],
            [
                275.7,
                275.4,
                275.7,
                275.7,
                275.1,
                276.0,
                275.5,
                276.2,
                276.8,
                276.4,
                274.5,
                274.0,
                275.1,
                274.5,
                273.7,
                274.0,
                274.7,
                274.8,
                275.1,
                275.3,
                275.5,
                277.2,
                277.0,
                276.6,
                277.7,
                277.5,
                276.2,
                277.7,
                275.8,
                277.9,
                277.5,
                276.7,
                277.8,
                276.8,
                277.9,
                277.7,
                278.0,
                278.0,
                279.4,
                278.9,
                278.5,
                278.0,
                278.3,
                277.3,
                276.7,
                276.7,
                277.3,
                276.1,
            ],
            [
                278.9,
                277.7,
                279.3,
                277.1,
                278.2,
                277.6,
                278.7,
                277.7,
                277.1,
                278.5,
                279.2,
                279.3,
                280.0,
                279.7,
                278.1,
                278.9,
                278.1,
                279.8,
                279.1,
                279.4,
                279.6,
                280.6,
                282.1,
                280.3,
                279.7,
                280.1,
                281.1,
                281.5,
                281.2,
                280.9,
                281.0,
                281.9,
                281.7,
                281.3,
                279.7,
                281.1,
                280.6,
                279.9,
                281.6,
                283.4,
                281.9,
                281.2,
                279.1,
                278.9,
                279.7,
                279.8,
                280.1,
                277.7,
            ],
            [
                283.2,
                283.7,
                283.3,
                281.8,
                280.9,
                282.2,
                281.4,
                282.6,
                282.6,
                283.4,
                284.8,
                284.4,
                283.7,
                283.1,
                283.2,
                283.2,
                283.5,
                283.5,
                284.9,
                284.3,
                284.3,
                285.9,
                285.9,
                286.5,
                287.0,
                286.6,
                285.7,
                285.0,
                286.4,
                285.3,
                286.5,
                284.7,
                285.6,
                284.4,
                284.4,
                284.4,
                283.2,
                282.3,
                282.4,
                287.8,
                285.9,
                285.2,
                284.4,
                282.4,
                283.1,
                284.1,
                284.7,
                283.6,
            ],
            [
                289.1,
                288.0,
                288.6,
                289.2,
                288.4,
                290.0,
                289.4,
                290.2,
                289.0,
                290.2,
                288.5,
                288.4,
                288.8,
                288.6,
                288.0,
                288.2,
                287.6,
                288.3,
                289.4,
                290.0,
                292.5,
                290.1,
                290.6,
                291.5,
                290.9,
                291.0,
                290.4,
                289.9,
                290.0,
                290.9,
                289.8,
                289.7,
                289.9,
                289.9,
                289.0,
                288.4,
                286.7,
                286.6,
                287.3,
                294.3,
                294.1,
                291.8,
                289.5,
                288.8,
                289.5,
                290.4,
                290.2,
                289.5,
            ],
            [
                292.9,
                292.7,
                294.2,
                292.1,
                295.9,
                295.1,
                294.9,
                295.3,
                294.4,
                293.0,
                294.0,
                292.5,
                291.2,
                291.9,
                291.3,
                293.6,
                291.4,
                292.9,
                298.4,
                294.4,
                294.8,
                296.3,
                294.1,
                295.1,
                294.3,
                293.7,
                293.0,
                295.5,
                294.5,
                295.8,
                294.5,
                294.1,
                295.2,
                294.6,
                292.5,
                291.9,
                290.4,
                290.1,
                293.1,
                301.1,
                300.5,
                296.7,
                294.3,
                295.1,
                294.9,
                295.8,
                293.6,
                292.4,
            ],
            [
                292.9,
                293.7,
                294.7,
                292.5,
                296.6,
                299.8,
                297.2,
                297.8,
                297.0,
                294.5,
                293.6,
                294.1,
                295.3,
                294.8,
                296.5,
                299.1,
                297.0,
                298.8,
                301.1,
                298.9,
                297.6,
                298.1,
                298.1,
                296.9,
                296.0,
                297.2,
                296.4,
                297.4,
                298.0,
                297.6,
                296.7,
                296.0,
                297.0,
                296.5,
                293.8,
                294.4,
                293.0,
                290.0,
                291.8,
                297.6,
                300.3,
                296.8,
                297.4,
                296.3,
                295.4,
                295.0,
                294.4,
                294.4,
            ],
            [
                294.2,
                296.5,
                295.0,
                294.8,
                298.9,
                301.2,
                298.0,
                297.8,
                297.7,
                297.2,
                296.2,
                296.7,
                295.8,
                297.6,
                298.9,
                301.0,
                301.3,
                299.2,
                299.8,
                298.9,
                299.2,
                298.7,
                299.5,
                299.7,
                299.2,
                299.6,
                298.3,
                298.5,
                299.4,
                298.3,
                297.4,
                297.2,
                294.1,
                294.2,
                293.2,
                293.0,
                291.2,
                293.2,
                289.3,
                293.7,
                300.1,
                296.6,
                297.9,
                298.4,
                296.4,
                295.5,
                293.9,
                293.7,
            ],
            [
                297.7,
                297.4,
                294.5,
                294.2,
                295.9,
                300.2,
                300.0,
                299.2,
                298.5,
                298.0,
                298.8,
                300.5,
                300.2,
                300.3,
                300.8,
                302.5,
                301.6,
                300.5,
                300.4,
                301.9,
                302.1,
                300.4,
                300.7,
                301.3,
                300.3,
                301.6,
                300.2,
                301.3,
                300.6,
                298.6,
                298.8,
                297.1,
                295.0,
                293.9,
                292.3,
                293.1,
                294.7,
                293.0,
                289.4,
                296.3,
                298.8,
                296.1,
                295.2,
                297.8,
                297.1,
                296.7,
                296.4,
                296.2,
            ],
            [
                300.9,
                300.6,
                295.8,
                294.8,
                294.3,
                298.8,
                300.8,
                299.5,
                299.8,
                300.7,
                299.8,
                301.8,
                300.1,
                302.0,
                301.2,
                301.7,
                301.6,
                303.0,
                301.8,
                301.0,
                302.5,
                301.1,
                301.4,
                300.5,
                302.0,
                300.5,
                300.9,
                300.5,
                300.0,
                300.5,
                298.4,
                297.2,
                295.7,
                296.2,
                294.8,
                295.3,
                298.1,
                295.8,
                298.0,
                299.8,
                298.9,
                298.9,
                296.7,
                298.8,
                298.5,
                299.0,
                298.4,
                299.5,
            ],
            [
                300.4,
                298.9,
                298.5,
                296.9,
                294.5,
                299.1,
                300.3,
                300.9,
                299.8,
                299.7,
                301.0,
                301.5,
                301.9,
                300.9,
                302.0,
                301.2,
                302.0,
                302.0,
                300.1,
                300.1,
                300.2,
                299.8,
                301.2,
                300.2,
                299.4,
                299.9,
                299.0,
                298.7,
                297.6,
                298.2,
                298.4,
                298.0,
                297.5,
                296.6,
                297.5,
                297.6,
                300.0,
                297.6,
                301.7,
                299.9,
                300.8,
                300.3,
                300.6,
                300.6,
                299.8,
                300.4,
                300.1,
                301.7,
            ],
            [
                300.8,
                297.9,
                298.4,
                296.2,
                296.4,
                298.5,
                299.0,
                299.9,
                299.2,
                300.2,
                300.1,
                301.3,
                301.3,
                299.8,
                300.0,
                299.7,
                301.7,
                301.9,
                301.5,
                302.0,
                299.9,
                300.8,
                300.8,
                299.0,
                300.0,
                300.0,
                299.1,
                299.4,
                298.5,
                298.9,
                297.4,
                298.9,
                298.3,
                298.2,
                298.7,
                298.7,
                299.5,
                297.3,
                298.7,
                299.8,
                300.7,
                299.8,
                300.5,
                300.8,
                300.7,
                301.3,
                301.1,
                300.3,
            ],
            [
                296.6,
                295.8,
                297.6,
                298.2,
                298.2,
                290.9,
                294.3,
                297.5,
                297.9,
                298.7,
                297.3,
                300.1,
                300.3,
                298.3,
                299.8,
                301.3,
                299.0,
                300.6,
                301.3,
                300.6,
                301.1,
                299.9,
                299.2,
                300.3,
                299.1,
                298.7,
                299.0,
                299.7,
                299.8,
                298.2,
                299.5,
                300.3,
                299.6,
                299.6,
                299.9,
                298.9,
                300.0,
                297.6,
                297.8,
                299.6,
                299.2,
                298.7,
                298.9,
                298.2,
                299.4,
                300.4,
                298.8,
                297.0,
            ],
            [
                296.9,
                296.6,
                296.4,
                295.2,
                297.7,
                292.7,
                289.9,
                296.6,
                295.7,
                297.8,
                294.4,
                298.8,
                297.4,
                294.0,
                296.1,
                298.2,
                297.8,
                299.4,
                298.7,
                298.4,
                299.4,
                298.7,
                296.9,
                296.5,
                295.9,
                295.3,
                296.4,
                296.5,
                296.9,
                295.6,
                297.0,
                296.2,
                296.8,
                297.9,
                295.1,
                294.6,
                295.8,
                298.3,
                298.7,
                297.9,
                297.6,
                297.2,
                297.4,
                297.9,
                296.4,
                296.4,
                298.9,
                296.7,
            ],
            [
                291.3,
                290.2,
                288.0,
                289.4,
                292.3,
                290.7,
                288.5,
                290.8,
                294.6,
                292.4,
                289.6,
                291.6,
                289.7,
                284.3,
                288.0,
                294.4,
                293.9,
                294.1,
                293.4,
                295.5,
                295.5,
                296.2,
                295.2,
                294.9,
                293.9,
                294.2,
                292.7,
                292.6,
                293.6,
                291.7,
                292.2,
                293.6,
                292.9,
                294.3,
                285.4,
                293.1,
                295.9,
                295.9,
                296.1,
                297.0,
                297.1,
                297.1,
                295.0,
                296.2,
                296.1,
                294.4,
                292.3,
                292.1,
            ],
            [
                287.1,
                286.7,
                286.8,
                287.3,
                288.0,
                285.4,
                288.1,
                284.9,
                285.0,
                283.9,
                283.1,
                276.6,
                274.9,
                275.4,
                276.8,
                278.7,
                285.8,
                289.5,
                288.4,
                289.9,
                290.2,
                292.1,
                291.7,
                293.1,
                291.6,
                290.8,
                291.0,
                290.7,
                289.5,
                290.8,
                290.5,
                289.8,
                289.7,
                284.3,
                280.6,
                285.4,
                287.7,
                290.5,
                292.7,
                293.9,
                294.3,
                294.3,
                293.7,
                293.2,
                294.4,
                292.7,
                289.9,
                287.8,
            ],
            [
                280.2,
                285.6,
                288.6,
                289.1,
                287.3,
                280.4,
                277.4,
                276.9,
                274.4,
                273.7,
                259.0,
                252.1,
                252.8,
                264.0,
                272.6,
                274.0,
                278.0,
                279.8,
                278.2,
                283.8,
                286.2,
                286.4,
                287.5,
                288.6,
                287.6,
                286.8,
                287.2,
                287.6,
                288.1,
                287.8,
                288.5,
                286.2,
                280.6,
                273.6,
                274.5,
                274.8,
                273.3,
                280.8,
                288.7,
                290.5,
                289.6,
                291.6,
                291.4,
                291.5,
                292.3,
                291.7,
                290.0,
                281.8,
            ],
            [
                283.8,
                283.9,
                282.7,
                279.1,
                274.5,
                270.1,
                276.5,
                276.1,
                273.0,
                264.4,
                261.2,
                262.8,
                258.6,
                260.2,
                262.5,
                264.3,
                257.8,
                262.8,
                270.9,
                273.5,
                279.0,
                280.7,
                280.7,
                282.3,
                282.4,
                282.9,
                281.4,
                284.0,
                282.9,
                284.8,
                284.4,
                280.3,
                267.5,
                264.5,
                267.8,
                265.6,
                266.5,
                269.4,
                279.7,
                284.7,
                285.4,
                286.4,
                286.7,
                288.1,
                289.2,
                288.4,
                284.5,
                278.6,
            ],
            [
                274.9,
                269.1,
                266.5,
                266.7,
                275.0,
                272.5,
                272.6,
                267.4,
                265.5,
                263.8,
                259.4,
                256.3,
                253.8,
                253.2,
                255.1,
                251.7,
                247.0,
                248.7,
                261.2,
                268.9,
                272.9,
                274.4,
                275.9,
                275.2,
                276.7,
                277.9,
                278.2,
                279.5,
                280.6,
                280.5,
                281.7,
                276.6,
                265.7,
                261.9,
                261.0,
                260.2,
                265.8,
                262.9,
                264.8,
                274.3,
                276.4,
                278.3,
                280.9,
                283.7,
                286.2,
                285.0,
                284.4,
                278.7,
            ],
            [
                272.0,
                266.1,
                263.0,
                260.3,
                258.5,
                258.8,
                258.1,
                257.8,
                256.0,
                256.2,
                257.7,
                255.5,
                252.9,
                251.7,
                250.8,
                243.5,
                241.2,
                243.1,
                250.8,
                268.8,
                266.8,
                267.2,
                271.3,
                272.6,
                273.9,
                273.7,
                275.7,
                275.9,
                277.4,
                280.2,
                279.5,
                268.3,
                262.4,
                258.0,
                255.4,
                255.9,
                257.5,
                259.5,
                259.1,
                259.6,
                269.1,
                275.4,
                278.9,
                279.8,
                282.9,
                283.3,
                282.8,
                275.7,
            ],
            [
                277.0,
                267.4,
                267.9,
                255.7,
                248.9,
                247.9,
                248.2,
                249.4,
                249.1,
                249.9,
                248.7,
                249.5,
                250.6,
                247.9,
                244.2,
                238.8,
                236.1,
                236.8,
                244.3,
                257.5,
                255.3,
                248.7,
                262.3,
                265.0,
                268.4,
                271.1,
                270.5,
                272.9,
                274.2,
                273.9,
                266.9,
                261.4,
                260.6,
                257.2,
                252.1,
                249.6,
                251.7,
                256.5,
                251.5,
                254.6,
                269.6,
                274.4,
                279.0,
                279.3,
                281.8,
                281.5,
                282.5,
                274.8,
            ],
            [
                274.1,
                262.4,
                258.9,
                246.6,
                242.9,
                240.3,
                240.1,
                240.9,
                239.4,
                239.1,
                239.1,
                239.6,
                237.9,
                236.6,
                233.5,
                230.9,
                228.4,
                231.3,
                235.3,
                238.9,
                238.5,
                242.5,
                244.2,
                247.1,
                257.4,
                264.3,
                255.1,
                257.1,
                256.5,
                255.2,
                256.2,
                253.6,
                251.6,
                248.7,
                245.2,
                246.0,
                247.4,
                248.1,
                249.6,
                257.7,
                269.3,
                259.9,
                265.2,
                274.8,
                277.1,
                273.9,
                277.3,
                279.2,
            ],
            [
                272.7,
                268.2,
                256.6,
                248.0,
                249.9,
                245.1,
                240.2,
                234.0,
                234.0,
                231.0,
                229.3,
                231.7,
                231.6,
                229.1,
                228.7,
                228.7,
                228.6,
                230.4,
                231.1,
                231.5,
                232.6,
                236.6,
                238.2,
                240.7,
                241.6,
                245.2,
                247.1,
                244.1,
                246.0,
                248.0,
                245.9,
                242.9,
                242.4,
                241.6,
                242.4,
                241.2,
                242.2,
                241.6,
                245.7,
                254.9,
                263.2,
                250.5,
                243.6,
                248.7,
                254.6,
                261.3,
                267.8,
                271.1,
            ],
            [
                267.5,
                269.3,
                267.7,
                265.7,
                262.4,
                242.9,
                236.2,
                235.2,
                235.7,
                235.3,
                233.8,
                228.7,
                229.6,
                227.4,
                229.6,
                232.4,
                233.1,
                236.1,
                236.5,
                237.0,
                236.0,
                238.1,
                238.4,
                239.0,
                239.5,
                239.2,
                242.1,
                238.3,
                238.9,
                239.6,
                241.2,
                240.3,
                239.0,
                240.0,
                242.4,
                239.9,
                238.9,
                245.3,
                251.6,
                259.2,
                247.7,
                238.6,
                236.8,
                236.2,
                240.5,
                246.3,
                248.2,
                259.4,
            ],
            [
                258.7,
                258.8,
                258.6,
                254.5,
                243.2,
                240.2,
                239.2,
                238.7,
                235.0,
                233.4,
                233.7,
                233.9,
                235.1,
                233.6,
                234.7,
                236.3,
                235.6,
                237.5,
                237.8,
                236.8,
                237.7,
                236.3,
                235.8,
                237.0,
                235.2,
                234.9,
                235.7,
                235.3,
                235.5,
                235.0,
                235.9,
                236.9,
                236.8,
                237.2,
                238.5,
                238.8,
                236.7,
                232.0,
                235.2,
                237.8,
                237.1,
                236.2,
                233.4,
                235.0,
                239.1,
                240.4,
                242.8,
                253.1,
            ],
            [
                235.7,
                235.5,
                235.4,
                235.1,
                234.4,
                236.7,
                235.6,
                234.8,
                236.0,
                236.9,
                235.7,
                235.4,
                237.2,
                237.8,
                237.3,
                237.0,
                237.5,
                239.1,
                239.4,
                238.7,
                237.5,
                238.9,
                238.8,
                237.6,
                235.8,
                236.8,
                236.4,
                235.0,
                234.6,
                233.8,
                233.6,
                233.5,
                233.2,
                233.9,
                233.4,
                233.1,
                234.8,
                234.0,
                233.3,
                234.8,
                235.3,
                234.6,
                233.7,
                235.7,
                238.6,
                239.0,
                237.2,
                236.8,
            ],
        ],
    ]

    dst = n.createVariable("dst", "f4", ("time_1", "lat", "lon"))
    dst.standard_name = "air_temperature"
    dst.units = "K"
    dst.cell_methods = "time_1: mean"

    # Don't generate this data randomly - it's useful to see the real
    # patterns of global temperature.
    dst[...] = [
        [
            [
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
                246.0,
            ],
            [
                243.6,
                243.3,
                243.0,
                242.7,
                242.5,
                242.5,
                242.5,
                242.4,
                242.5,
                242.5,
                242.4,
                242.2,
                241.9,
                241.7,
                241.5,
                241.4,
                241.4,
                241.4,
                241.4,
                241.4,
                241.5,
                241.7,
                241.8,
                241.9,
                242.1,
                242.1,
                241.9,
                241.9,
                241.8,
                242.0,
                242.2,
                242.5,
                243.2,
                243.7,
                244.1,
                244.5,
                244.8,
                244.8,
                244.7,
                244.8,
                244.9,
                245.1,
                245.7,
                246.3,
                247.6,
                248.1,
                248.3,
                247.9,
                247.2,
                246.8,
                246.9,
                247.3,
                247.9,
                248.2,
                248.5,
                249.1,
                249.6,
                249.8,
                250.1,
                250.5,
                250.8,
                250.7,
                250.8,
                250.3,
                250.3,
                250.2,
                249.8,
                249.6,
                249.8,
                250.3,
                250.7,
                251.2,
                252.0,
                252.7,
                252.5,
                251.5,
                250.5,
                249.8,
                248.6,
                248.0,
                247.9,
                248.2,
                248.3,
                248.3,
                248.2,
                247.8,
                247.5,
                247.0,
                246.5,
                246.4,
                246.0,
                245.4,
                244.9,
                244.4,
                244.0,
                243.8,
            ],
            [
                244.1,
                243.6,
                242.8,
                241.9,
                241.3,
                241.0,
                240.8,
                240.8,
                240.5,
                240.2,
                239.8,
                239.5,
                239.4,
                239.5,
                239.5,
                239.3,
                239.1,
                239.0,
                239.1,
                239.5,
                240.1,
                240.7,
                241.2,
                241.7,
                242.0,
                242.5,
                243.1,
                243.5,
                243.6,
                243.9,
                244.3,
                244.8,
                245.3,
                246.0,
                246.8,
                247.5,
                248.0,
                248.4,
                248.7,
                248.9,
                249.3,
                250.2,
                251.5,
                252.8,
                253.6,
                254.2,
                254.3,
                255.1,
                256.9,
                258.4,
                259.8,
                261.2,
                262.7,
                264.0,
                264.8,
                265.3,
                265.1,
                264.4,
                263.6,
                262.6,
                261.6,
                261.1,
                260.4,
                259.4,
                258.2,
                257.2,
                255.3,
                253.5,
                252.7,
                252.5,
                253.0,
                253.5,
                254.1,
                255.2,
                257.1,
                258.0,
                258.3,
                258.3,
                258.7,
                258.5,
                257.9,
                257.0,
                256.0,
                255.5,
                255.0,
                254.2,
                252.9,
                251.7,
                250.8,
                249.6,
                248.3,
                246.9,
                245.7,
                245.0,
                244.5,
                244.3,
            ],
            [
                244.9,
                243.6,
                242.0,
                240.5,
                239.4,
                238.6,
                238.1,
                237.5,
                237.2,
                237.1,
                236.9,
                236.8,
                237.1,
                237.6,
                237.9,
                237.9,
                237.9,
                237.8,
                237.6,
                237.9,
                238.6,
                239.3,
                239.9,
                240.2,
                241.1,
                242.4,
                243.5,
                244.0,
                244.7,
                245.4,
                245.9,
                246.1,
                246.5,
                247.1,
                247.9,
                248.8,
                249.8,
                250.5,
                250.9,
                251.4,
                252.2,
                253.6,
                256.0,
                259.4,
                262.9,
                265.9,
                267.3,
                267.7,
                267.6,
                267.4,
                267.0,
                266.6,
                266.5,
                266.6,
                266.5,
                265.9,
                265.3,
                265.1,
                265.0,
                264.8,
                264.8,
                264.8,
                264.7,
                264.5,
                263.5,
                262.0,
                259.9,
                257.4,
                255.7,
                255.3,
                255.3,
                255.7,
                256.5,
                257.5,
                258.9,
                260.6,
                261.8,
                262.8,
                263.3,
                263.8,
                264.2,
                264.0,
                263.1,
                262.2,
                262.4,
                262.4,
                261.2,
                259.3,
                257.5,
                255.4,
                253.0,
                250.5,
                248.2,
                246.6,
                245.9,
                245.6,
            ],
            [
                244.3,
                242.9,
                241.7,
                241.1,
                240.7,
                240.2,
                239.1,
                238.1,
                237.5,
                237.1,
                236.8,
                237.0,
                237.9,
                239.2,
                240.3,
                241.1,
                241.9,
                242.5,
                242.5,
                241.8,
                241.4,
                241.2,
                240.8,
                240.2,
                239.5,
                239.5,
                240.4,
                241.8,
                243.1,
                244.2,
                245.0,
                245.5,
                245.8,
                246.5,
                247.3,
                248.4,
                249.5,
                250.3,
                250.7,
                251.4,
                253.3,
                255.7,
                259.1,
                263.3,
                265.9,
                266.1,
                266.4,
                266.0,
                265.7,
                265.5,
                265.5,
                265.6,
                265.8,
                266.2,
                266.3,
                265.9,
                265.5,
                265.3,
                264.6,
                264.1,
                263.4,
                262.4,
                261.5,
                261.2,
                261.0,
                260.8,
                260.0,
                258.8,
                257.4,
                256.8,
                256.8,
                257.4,
                258.7,
                260.1,
                261.9,
                263.6,
                265.0,
                265.2,
                264.8,
                264.8,
                265.5,
                265.2,
                264.3,
                262.9,
                263.7,
                265.2,
                266.0,
                265.8,
                263.1,
                260.8,
                256.8,
                252.4,
                249.0,
                247.0,
                245.9,
                245.1,
            ],
            [
                245.2,
                243.7,
                242.2,
                241.3,
                241.5,
                241.7,
                241.3,
                240.1,
                238.9,
                238.5,
                238.6,
                239.0,
                239.9,
                241.3,
                242.4,
                243.3,
                245.4,
                246.5,
                246.4,
                246.4,
                246.4,
                245.9,
                245.2,
                243.8,
                242.2,
                241.2,
                241.2,
                241.7,
                242.5,
                243.3,
                243.8,
                244.2,
                244.4,
                244.5,
                245.4,
                246.5,
                247.4,
                247.6,
                247.6,
                248.0,
                248.9,
                250.8,
                254.1,
                259.1,
                262.6,
                265.6,
                265.6,
                265.2,
                264.8,
                264.8,
                265.4,
                266.1,
                266.6,
                267.2,
                267.2,
                267.2,
                266.8,
                266.7,
                265.6,
                263.3,
                261.6,
                259.5,
                258.2,
                258.3,
                259.4,
                260.8,
                261.7,
                261.7,
                261.5,
                260.9,
                260.2,
                260.3,
                261.3,
                262.4,
                263.9,
                266.1,
                267.0,
                267.7,
                267.5,
                267.2,
                271.5,
                271.5,
                271.6,
                271.7,
                271.6,
                271.5,
                271.6,
                271.4,
                263.7,
                260.6,
                256.3,
                252.6,
                249.8,
                248.1,
                247.1,
                246.2,
            ],
            [
                248.4,
                246.8,
                245.6,
                244.3,
                244.1,
                244.4,
                244.0,
                242.7,
                241.0,
                240.4,
                240.9,
                242.0,
                243.3,
                243.7,
                243.4,
                243.9,
                245.3,
                248.8,
                250.6,
                250.9,
                250.5,
                248.5,
                246.7,
                245.5,
                244.2,
                243.3,
                243.6,
                244.7,
                245.7,
                245.8,
                245.3,
                244.4,
                243.7,
                243.7,
                244.8,
                246.1,
                246.7,
                246.4,
                246.4,
                247.1,
                249.2,
                251.9,
                255.3,
                259.6,
                261.3,
                270.2,
                270.4,
                271.0,
                271.1,
                271.2,
                271.4,
                271.5,
                271.5,
                271.5,
                271.5,
                271.4,
                271.3,
                271.7,
                272.0,
                272.1,
                267.4,
                266.9,
                266.4,
                266.3,
                266.4,
                266.6,
                267.3,
                268.2,
                268.5,
                268.2,
                266.9,
                265.2,
                264.3,
                264.3,
                265.8,
                267.1,
                267.0,
                266.0,
                265.7,
                266.6,
                271.6,
                271.6,
                271.5,
                271.5,
                271.4,
                271.5,
                271.6,
                271.5,
                271.3,
                271.4,
                262.7,
                259.6,
                254.4,
                252.3,
                251.2,
                249.6,
            ],
            [
                257.5,
                254.5,
                252.1,
                249.8,
                249.2,
                250.4,
                251.5,
                250.9,
                250.6,
                250.5,
                249.1,
                248.0,
                247.6,
                247.1,
                246.9,
                247.4,
                249.1,
                253.6,
                256.8,
                257.8,
                255.1,
                251.7,
                247.6,
                245.1,
                244.0,
                243.2,
                243.2,
                244.8,
                246.9,
                247.9,
                247.8,
                247.1,
                246.2,
                245.6,
                245.4,
                245.6,
                246.3,
                246.7,
                248.1,
                250.9,
                253.0,
                253.8,
                254.5,
                254.6,
                256.0,
                270.2,
                270.8,
                271.3,
                271.2,
                271.3,
                271.4,
                271.4,
                271.3,
                271.8,
                272.6,
                272.6,
                271.1,
                271.1,
                271.3,
                271.6,
                271.7,
                271.7,
                271.6,
                271.5,
                271.5,
                271.5,
                271.4,
                271.4,
                271.2,
                271.6,
                271.7,
                271.6,
                271.5,
                272.3,
                272.1,
                271.5,
                271.8,
                268.5,
                265.9,
                264.9,
                271.5,
                271.3,
                271.3,
                271.4,
                271.5,
                271.5,
                271.5,
                271.5,
                271.3,
                271.2,
                271.1,
                271.2,
                271.3,
                263.1,
                261.2,
                259.6,
            ],
            [
                271.3,
                271.4,
                271.5,
                271.6,
                271.6,
                271.5,
                271.3,
                271.3,
                264.0,
                264.2,
                263.1,
                259.6,
                256.1,
                254.7,
                252.4,
                250.5,
                251.9,
                254.3,
                259.4,
                262.6,
                263.1,
                258.8,
                255.1,
                253.2,
                252.1,
                251.8,
                251.8,
                252.1,
                252.9,
                253.4,
                253.9,
                254.0,
                253.7,
                252.8,
                251.7,
                250.5,
                249.8,
                250.2,
                252.7,
                254.8,
                256.8,
                258.2,
                259.7,
                270.3,
                270.2,
                270.5,
                271.2,
                271.1,
                271.3,
                271.5,
                271.6,
                271.6,
                272.2,
                273.2,
                274.1,
                274.1,
                273.8,
                273.9,
                274.0,
                273.9,
                273.4,
                272.8,
                272.5,
                272.7,
                273.6,
                274.0,
                273.8,
                273.6,
                273.7,
                273.8,
                273.2,
                272.8,
                273.5,
                274.4,
                275.4,
                275.3,
                272.6,
                268.4,
                265.9,
                265.2,
                271.4,
                271.3,
                271.0,
                271.2,
                271.5,
                271.5,
                271.3,
                271.2,
                271.2,
                271.2,
                271.2,
                271.3,
                271.4,
                271.4,
                271.3,
                271.3,
            ],
            [
                272.8,
                273.1,
                273.0,
                272.9,
                272.8,
                272.5,
                271.3,
                271.2,
                271.1,
                271.2,
                271.1,
                271.2,
                271.0,
                263.1,
                260.1,
                260.2,
                270.2,
                269.5,
                269.5,
                269.8,
                270.1,
                269.9,
                270.9,
                260.1,
                259.0,
                257.8,
                256.3,
                255.8,
                256.3,
                258.0,
                259.9,
                261.0,
                260.8,
                259.9,
                258.9,
                257.2,
                255.9,
                257.6,
                261.3,
                270.5,
                270.7,
                270.4,
                270.6,
                271.1,
                271.4,
                271.2,
                271.3,
                272.4,
                272.8,
                273.2,
                273.4,
                273.4,
                273.6,
                274.1,
                274.2,
                274.2,
                274.2,
                274.2,
                274.0,
                273.9,
                273.9,
                273.9,
                274.0,
                274.1,
                274.4,
                274.6,
                274.5,
                274.4,
                274.3,
                274.4,
                274.6,
                274.5,
                274.5,
                274.8,
                275.1,
                275.2,
                275.1,
                274.6,
                268.9,
                267.9,
                271.2,
                271.1,
                271.2,
                272.2,
                271.4,
                271.1,
                271.0,
                271.0,
                271.6,
                272.2,
                271.4,
                271.7,
                271.3,
                271.7,
                271.6,
                272.3,
            ],
            [
                274.2,
                274.2,
                274.1,
                274.3,
                274.3,
                274.3,
                274.3,
                274.5,
                274.1,
                274.5,
                274.7,
                274.5,
                271.5,
                271.2,
                271.1,
                270.9,
                270.8,
                270.7,
                270.8,
                270.9,
                271.0,
                271.1,
                271.0,
                270.8,
                270.8,
                270.8,
                270.9,
                270.9,
                271.0,
                271.1,
                271.0,
                271.0,
                271.1,
                271.1,
                271.1,
                271.1,
                271.0,
                271.0,
                270.9,
                270.9,
                271.1,
                271.1,
                271.2,
                271.3,
                271.4,
                272.1,
                273.3,
                273.9,
                274.1,
                274.3,
                274.6,
                274.5,
                274.5,
                274.5,
                274.4,
                274.3,
                274.2,
                274.2,
                274.1,
                274.0,
                273.9,
                274.1,
                274.3,
                274.4,
                274.4,
                274.6,
                274.6,
                274.7,
                274.8,
                274.9,
                275.1,
                275.4,
                275.4,
                275.4,
                275.6,
                275.6,
                275.4,
                275.2,
                275.1,
                269.6,
                271.3,
                271.3,
                272.3,
                272.8,
                272.9,
                273.0,
                273.2,
                273.4,
                273.5,
                273.8,
                273.8,
                274.0,
                274.0,
                274.1,
                274.0,
                274.0,
            ],
            [
                274.7,
                274.6,
                274.4,
                274.4,
                274.5,
                274.4,
                274.3,
                274.4,
                274.4,
                274.5,
                274.8,
                275.0,
                274.8,
                274.4,
                274.0,
                273.8,
                274.0,
                274.5,
                274.5,
                274.4,
                274.3,
                274.3,
                272.9,
                271.3,
                271.2,
                271.3,
                273.0,
                272.9,
                273.0,
                273.6,
                273.6,
                273.4,
                272.9,
                272.7,
                272.9,
                272.8,
                271.8,
                271.6,
                271.6,
                272.3,
                274.1,
                274.6,
                274.6,
                274.5,
                274.8,
                274.7,
                274.9,
                275.2,
                275.1,
                275.0,
                275.0,
                275.0,
                275.1,
                275.2,
                275.2,
                275.2,
                275.2,
                275.0,
                274.9,
                274.8,
                274.9,
                274.9,
                274.9,
                275.0,
                275.0,
                275.1,
                275.1,
                275.2,
                275.4,
                275.5,
                275.8,
                276.0,
                275.9,
                276.1,
                276.4,
                276.5,
                276.4,
                276.1,
                275.5,
                273.0,
                271.6,
                272.0,
                273.1,
                273.8,
                274.5,
                274.8,
                274.9,
                274.9,
                274.3,
                274.8,
                274.9,
                274.8,
                274.7,
                274.8,
                274.8,
                274.8,
            ],
            [
                275.0,
                274.9,
                274.6,
                274.3,
                274.3,
                274.4,
                274.2,
                274.2,
                274.3,
                274.5,
                274.6,
                274.7,
                274.8,
                274.9,
                275.0,
                275.1,
                275.1,
                275.1,
                275.1,
                275.0,
                274.6,
                274.3,
                274.0,
                271.7,
                272.1,
                272.6,
                273.2,
                274.1,
                274.3,
                274.4,
                274.6,
                274.7,
                274.7,
                274.8,
                274.9,
                274.6,
                274.7,
                274.9,
                274.9,
                275.0,
                275.2,
                275.4,
                275.5,
                275.4,
                275.5,
                275.4,
                275.7,
                275.7,
                275.6,
                275.6,
                275.6,
                275.6,
                275.7,
                275.7,
                275.6,
                275.6,
                275.6,
                275.6,
                275.9,
                275.7,
                275.5,
                275.6,
                275.7,
                275.8,
                275.8,
                275.8,
                275.9,
                276.0,
                276.0,
                276.1,
                276.3,
                276.4,
                276.4,
                276.5,
                276.6,
                276.8,
                276.9,
                277.2,
                277.2,
                277.0,
                276.4,
                275.6,
                275.8,
                276.0,
                276.3,
                276.2,
                276.1,
                276.3,
                276.2,
                276.2,
                276.1,
                275.7,
                275.5,
                275.5,
                275.3,
                275.0,
            ],
            [
                275.2,
                275.1,
                274.9,
                274.9,
                275.2,
                275.3,
                275.2,
                275.0,
                275.0,
                275.2,
                275.5,
                275.6,
                275.5,
                275.5,
                275.5,
                275.6,
                275.8,
                275.9,
                275.8,
                275.3,
                274.9,
                274.5,
                274.6,
                274.2,
                274.3,
                273.7,
                273.7,
                273.7,
                273.9,
                274.0,
                274.4,
                274.5,
                274.6,
                274.9,
                275.0,
                275.0,
                275.0,
                275.1,
                275.1,
                275.4,
                275.6,
                275.9,
                276.3,
                276.6,
                277.0,
                276.5,
                276.5,
                276.5,
                276.5,
                276.6,
                276.6,
                276.7,
                277.0,
                276.7,
                276.4,
                276.3,
                276.1,
                276.2,
                277.1,
                276.7,
                276.5,
                276.4,
                276.8,
                277.0,
                276.9,
                276.9,
                277.0,
                277.0,
                276.9,
                276.9,
                277.0,
                276.9,
                276.9,
                277.1,
                277.2,
                277.7,
                277.9,
                278.7,
                279.2,
                279.0,
                278.1,
                278.2,
                278.1,
                278.3,
                278.6,
                278.0,
                277.3,
                277.0,
                277.1,
                277.1,
                277.0,
                276.7,
                276.4,
                276.3,
                276.1,
                275.6,
            ],
            [
                276.0,
                275.8,
                275.7,
                275.8,
                276.1,
                276.2,
                276.2,
                276.1,
                275.9,
                276.0,
                276.3,
                276.4,
                276.3,
                276.4,
                276.3,
                276.3,
                276.5,
                276.5,
                276.3,
                276.0,
                275.7,
                275.1,
                275.7,
                275.6,
                275.7,
                275.7,
                276.0,
                275.6,
                276.0,
                275.4,
                275.8,
                275.7,
                275.6,
                275.6,
                275.9,
                275.9,
                276.2,
                276.5,
                276.2,
                276.1,
                276.3,
                276.1,
                277.0,
                277.3,
                278.2,
                278.2,
                277.7,
                277.5,
                277.5,
                277.6,
                277.5,
                277.8,
                278.2,
                278.2,
                278.1,
                278.0,
                277.6,
                277.2,
                278.1,
                278.1,
                278.0,
                278.0,
                278.8,
                278.7,
                278.4,
                278.3,
                278.2,
                278.2,
                278.1,
                278.0,
                278.1,
                278.1,
                278.3,
                278.4,
                278.5,
                278.9,
                279.5,
                280.2,
                279.7,
                281.3,
                280.2,
                279.7,
                279.5,
                279.3,
                279.1,
                278.3,
                277.8,
                277.9,
                278.3,
                278.3,
                278.2,
                277.7,
                277.4,
                277.2,
                276.7,
                276.3,
            ],
            [
                277.2,
                277.2,
                277.4,
                277.5,
                277.5,
                277.5,
                277.4,
                277.2,
                276.9,
                276.9,
                277.1,
                277.2,
                277.4,
                277.5,
                277.5,
                277.4,
                277.4,
                277.2,
                277.2,
                277.0,
                276.8,
                276.2,
                277.4,
                277.5,
                277.4,
                277.2,
                277.5,
                277.7,
                277.9,
                277.6,
                278.0,
                277.4,
                277.5,
                277.4,
                277.6,
                277.9,
                278.4,
                278.3,
                278.2,
                278.6,
                279.2,
                278.8,
                279.5,
                279.4,
                280.0,
                280.7,
                280.1,
                278.8,
                278.6,
                278.9,
                278.8,
                279.0,
                279.6,
                280.1,
                280.0,
                279.8,
                279.7,
                279.4,
                279.7,
                280.1,
                280.3,
                280.5,
                280.7,
                280.3,
                280.1,
                280.1,
                279.9,
                279.8,
                279.5,
                279.3,
                279.3,
                279.3,
                279.4,
                279.5,
                279.7,
                280.1,
                280.4,
                279.8,
                282.0,
                281.8,
                281.2,
                281.1,
                280.4,
                279.5,
                279.1,
                278.7,
                278.6,
                278.7,
                279.0,
                279.1,
                279.5,
                279.0,
                278.5,
                278.0,
                277.2,
                277.3,
            ],
            [
                279.0,
                279.3,
                279.5,
                279.4,
                279.1,
                278.8,
                278.6,
                278.3,
                278.3,
                278.2,
                278.2,
                278.3,
                278.6,
                278.8,
                278.6,
                278.5,
                278.4,
                278.3,
                278.7,
                278.3,
                278.5,
                280.1,
                281.1,
                280.7,
                280.3,
                280.1,
                280.0,
                280.2,
                280.1,
                279.9,
                279.7,
                279.5,
                279.9,
                279.8,
                280.0,
                280.9,
                280.9,
                280.8,
                280.9,
                281.1,
                281.1,
                280.8,
                281.1,
                281.1,
                281.6,
                281.8,
                281.0,
                280.1,
                280.2,
                280.5,
                281.0,
                281.1,
                281.4,
                282.2,
                281.8,
                281.6,
                281.5,
                281.4,
                281.6,
                282.2,
                282.3,
                282.4,
                282.1,
                281.9,
                281.9,
                281.8,
                281.4,
                281.2,
                280.9,
                280.7,
                280.6,
                280.5,
                280.6,
                280.7,
                280.9,
                281.0,
                281.1,
                279.2,
                284.1,
                283.7,
                282.6,
                281.6,
                280.6,
                280.3,
                280.1,
                279.7,
                279.5,
                279.5,
                279.9,
                279.9,
                280.4,
                280.3,
                280.0,
                279.7,
                279.0,
                279.3,
            ],
            [
                281.7,
                281.7,
                281.4,
                281.2,
                280.9,
                280.5,
                280.2,
                279.8,
                280.0,
                280.0,
                279.8,
                280.0,
                280.2,
                280.3,
                280.0,
                279.9,
                279.9,
                279.8,
                280.4,
                280.0,
                281.9,
                282.4,
                282.6,
                282.2,
                282.3,
                282.2,
                281.9,
                281.9,
                281.7,
                281.5,
                281.4,
                281.6,
                282.0,
                282.0,
                282.3,
                282.7,
                282.4,
                282.5,
                282.5,
                282.5,
                282.7,
                282.6,
                282.7,
                282.8,
                283.2,
                283.1,
                282.7,
                282.0,
                282.5,
                283.3,
                283.9,
                283.5,
                283.7,
                284.1,
                283.9,
                283.6,
                283.4,
                283.4,
                283.6,
                283.9,
                283.9,
                283.8,
                283.6,
                283.6,
                283.6,
                283.4,
                283.1,
                282.8,
                282.6,
                282.3,
                282.2,
                281.9,
                281.8,
                281.9,
                281.9,
                281.9,
                281.9,
                277.8,
                287.9,
                285.9,
                283.8,
                282.2,
                282.2,
                282.4,
                282.3,
                281.9,
                281.1,
                281.0,
                281.1,
                281.0,
                281.5,
                282.3,
                282.3,
                282.3,
                282.0,
                282.0,
            ],
            [
                284.2,
                283.9,
                283.5,
                283.1,
                282.7,
                282.4,
                282.0,
                281.6,
                281.8,
                281.5,
                281.5,
                281.9,
                281.9,
                282.1,
                281.9,
                281.9,
                282.0,
                281.8,
                282.2,
                282.2,
                284.0,
                284.2,
                284.2,
                284.1,
                284.2,
                283.9,
                283.7,
                283.6,
                283.6,
                283.6,
                283.6,
                283.8,
                283.9,
                283.9,
                284.2,
                284.1,
                284.1,
                284.2,
                284.4,
                285.2,
                285.7,
                285.1,
                284.9,
                284.9,
                284.9,
                283.6,
                287.7,
                288.0,
                286.9,
                288.5,
                287.4,
                286.6,
                287.0,
                286.3,
                286.2,
                285.8,
                285.7,
                285.7,
                285.7,
                285.8,
                285.7,
                285.6,
                285.3,
                285.2,
                285.0,
                284.9,
                284.7,
                284.4,
                284.3,
                283.9,
                283.6,
                283.4,
                283.0,
                283.0,
                282.8,
                283.1,
                283.2,
                280.2,
                289.8,
                289.2,
                285.5,
                284.6,
                284.9,
                285.1,
                284.9,
                283.8,
                283.0,
                283.1,
                283.0,
                282.7,
                283.5,
                284.8,
                284.8,
                284.9,
                285.1,
                284.7,
            ],
            [
                286.4,
                286.1,
                285.7,
                285.2,
                284.9,
                284.5,
                284.0,
                283.6,
                283.8,
                283.3,
                283.2,
                284.3,
                284.8,
                285.0,
                285.3,
                285.1,
                284.9,
                284.8,
                285.3,
                285.4,
                286.6,
                286.3,
                286.2,
                286.2,
                286.2,
                286.0,
                285.9,
                285.7,
                285.7,
                285.7,
                285.5,
                285.5,
                285.6,
                285.8,
                285.8,
                285.9,
                286.2,
                286.3,
                286.4,
                287.2,
                287.9,
                287.7,
                287.4,
                287.2,
                287.5,
                288.2,
                286.9,
                291.5,
                291.3,
                290.5,
                289.1,
                288.8,
                288.6,
                287.9,
                287.7,
                287.4,
                287.5,
                287.6,
                287.6,
                287.6,
                287.3,
                287.1,
                286.9,
                286.9,
                286.8,
                286.6,
                286.3,
                286.1,
                286.0,
                285.7,
                285.4,
                285.1,
                284.6,
                284.2,
                284.0,
                284.6,
                284.8,
                281.5,
                289.1,
                290.4,
                287.6,
                289.3,
                289.9,
                287.9,
                286.6,
                286.2,
                286.0,
                285.5,
                285.5,
                286.1,
                287.1,
                287.8,
                287.2,
                287.5,
                287.3,
                287.0,
            ],
            [
                288.7,
                288.5,
                288.1,
                287.6,
                287.1,
                286.8,
                287.2,
                286.7,
                286.5,
                285.8,
                286.0,
                287.4,
                287.8,
                288.3,
                288.7,
                288.2,
                287.8,
                288.0,
                288.5,
                288.7,
                289.1,
                288.6,
                288.5,
                288.2,
                288.0,
                287.9,
                287.6,
                287.4,
                287.4,
                287.4,
                287.3,
                287.3,
                287.5,
                287.6,
                287.7,
                287.8,
                287.9,
                288.0,
                288.1,
                289.0,
                290.0,
                291.0,
                290.2,
                289.5,
                289.6,
                290.0,
                291.1,
                290.8,
                292.5,
                291.4,
                290.5,
                290.1,
                289.7,
                289.4,
                289.2,
                289.3,
                289.6,
                289.9,
                290.0,
                289.6,
                289.3,
                289.2,
                289.0,
                289.0,
                289.0,
                288.7,
                288.4,
                288.2,
                288.0,
                287.6,
                287.2,
                286.8,
                286.3,
                286.0,
                285.8,
                286.3,
                286.6,
                283.4,
                291.9,
                296.0,
                290.3,
                292.7,
                292.4,
                290.3,
                289.5,
                289.2,
                288.9,
                288.8,
                289.2,
                289.7,
                289.9,
                289.6,
                289.5,
                289.6,
                289.4,
                289.1,
            ],
            [
                290.7,
                290.8,
                290.4,
                290.0,
                290.5,
                291.8,
                292.7,
                293.0,
                293.4,
                292.1,
                291.7,
                292.0,
                291.9,
                291.8,
                291.6,
                291.2,
                291.2,
                291.3,
                291.3,
                291.3,
                291.1,
                290.7,
                290.7,
                290.2,
                289.7,
                289.5,
                289.3,
                289.1,
                289.1,
                289.1,
                289.2,
                289.5,
                289.8,
                289.6,
                289.4,
                289.5,
                289.6,
                290.2,
                293.9,
                290.9,
                292.4,
                294.3,
                292.9,
                291.8,
                291.8,
                291.9,
                292.2,
                292.8,
                292.7,
                292.4,
                292.3,
                291.9,
                291.4,
                291.0,
                291.1,
                291.3,
                291.6,
                292.4,
                292.3,
                292.0,
                291.8,
                291.6,
                291.5,
                291.4,
                291.3,
                291.0,
                290.7,
                290.5,
                290.2,
                289.8,
                289.4,
                288.8,
                288.3,
                287.9,
                287.5,
                288.1,
                288.3,
                286.7,
                294.7,
                298.5,
                297.5,
                294.4,
                293.8,
                292.8,
                292.0,
                291.3,
                291.0,
                291.1,
                291.2,
                291.5,
                291.8,
                291.9,
                291.9,
                291.5,
                291.3,
                291.1,
            ],
            [
                292.2,
                292.3,
                292.0,
                292.2,
                292.3,
                291.9,
                293.8,
                295.6,
                296.0,
                296.2,
                295.2,
                293.9,
                293.5,
                293.3,
                293.3,
                293.1,
                293.2,
                293.2,
                293.1,
                292.8,
                292.5,
                292.1,
                292.1,
                291.7,
                291.3,
                291.0,
                290.6,
                290.4,
                290.3,
                290.4,
                290.5,
                290.1,
                291.1,
                291.0,
                290.9,
                291.1,
                291.6,
                297.3,
                298.4,
                294.6,
                291.0,
                296.0,
                294.8,
                294.0,
                294.1,
                294.0,
                293.7,
                294.0,
                293.6,
                293.7,
                293.5,
                293.1,
                292.7,
                292.5,
                292.9,
                293.4,
                293.7,
                294.1,
                294.1,
                294.0,
                293.9,
                293.8,
                293.7,
                293.6,
                293.5,
                293.4,
                293.0,
                292.8,
                292.5,
                292.1,
                291.4,
                290.8,
                290.1,
                289.5,
                289.0,
                289.7,
                288.3,
                292.3,
                297.3,
                301.5,
                301.0,
                299.1,
                295.6,
                294.8,
                293.4,
                293.0,
                292.9,
                293.0,
                293.2,
                293.5,
                293.9,
                294.0,
                293.8,
                293.3,
                292.8,
                292.4,
            ],
            [
                292.9,
                292.5,
                291.9,
                292.0,
                291.5,
                295.9,
                289.6,
                290.0,
                297.4,
                297.2,
                295.9,
                295.2,
                295.1,
                295.2,
                295.3,
                295.2,
                295.1,
                294.9,
                294.6,
                294.2,
                293.8,
                293.4,
                293.2,
                292.8,
                292.3,
                291.8,
                291.6,
                291.1,
                291.5,
                292.2,
                293.2,
                297.0,
                294.3,
                292.2,
                292.0,
                292.5,
                298.2,
                298.2,
                297.6,
                295.0,
                292.4,
                296.8,
                295.8,
                295.7,
                295.8,
                295.4,
                295.0,
                294.9,
                294.6,
                294.8,
                294.5,
                294.1,
                294.0,
                294.2,
                294.6,
                294.9,
                295.3,
                295.4,
                295.4,
                295.4,
                295.5,
                295.4,
                295.4,
                295.4,
                295.4,
                295.3,
                294.9,
                294.5,
                294.2,
                293.8,
                293.2,
                292.4,
                291.6,
                290.8,
                290.5,
                290.7,
                287.0,
                294.5,
                299.9,
                301.8,
                303.2,
                299.2,
                297.4,
                296.8,
                295.7,
                294.8,
                294.9,
                294.9,
                294.8,
                295.0,
                295.3,
                295.2,
                294.9,
                294.3,
                293.7,
                293.3,
            ],
            [
                293.3,
                293.0,
                292.8,
                293.8,
                293.2,
                298.3,
                293.4,
                289.7,
                290.7,
                298.8,
                297.3,
                297.2,
                297.1,
                296.7,
                296.7,
                296.6,
                296.2,
                295.8,
                295.4,
                295.0,
                294.6,
                294.1,
                293.7,
                293.3,
                293.1,
                293.1,
                293.4,
                292.4,
                292.9,
                294.0,
                295.2,
                299.8,
                293.3,
                293.0,
                293.2,
                297.7,
                300.7,
                298.8,
                297.4,
                298.5,
                294.3,
                297.7,
                296.6,
                296.9,
                296.7,
                296.2,
                295.8,
                295.7,
                295.6,
                295.8,
                295.4,
                295.3,
                295.4,
                295.8,
                295.9,
                296.0,
                296.3,
                296.4,
                296.5,
                296.6,
                296.7,
                296.6,
                296.6,
                296.6,
                296.5,
                296.3,
                295.9,
                295.6,
                295.2,
                294.6,
                294.0,
                293.3,
                292.5,
                291.7,
                291.1,
                290.3,
                286.5,
                294.7,
                297.7,
                304.8,
                304.4,
                300.7,
                297.3,
                298.0,
                297.6,
                296.6,
                296.3,
                296.2,
                295.9,
                295.8,
                295.8,
                295.6,
                295.1,
                294.6,
                293.9,
                293.6,
            ],
            [
                293.5,
                293.2,
                293.4,
                293.4,
                291.1,
                297.3,
                294.5,
                292.8,
                290.8,
                299.7,
                299.5,
                299.4,
                299.1,
                298.2,
                297.6,
                297.2,
                296.7,
                296.1,
                295.6,
                295.2,
                294.8,
                294.4,
                294.1,
                293.8,
                293.9,
                294.2,
                294.2,
                293.8,
                294.3,
                295.6,
                297.2,
                300.9,
                295.2,
                296.9,
                298.7,
                299.6,
                301.0,
                300.8,
                300.3,
                300.0,
                295.9,
                298.4,
                297.6,
                297.7,
                297.5,
                297.2,
                296.9,
                296.8,
                296.9,
                296.8,
                296.5,
                296.5,
                296.7,
                296.9,
                296.9,
                296.9,
                297.1,
                297.2,
                297.3,
                297.3,
                297.3,
                297.2,
                297.1,
                297.0,
                296.7,
                296.3,
                296.0,
                295.7,
                295.4,
                294.9,
                294.3,
                293.7,
                293.0,
                292.2,
                291.3,
                290.6,
                290.3,
                293.8,
                283.8,
                303.0,
                303.3,
                301.1,
                295.7,
                296.0,
                299.0,
                298.0,
                297.4,
                297.0,
                296.4,
                296.0,
                295.7,
                295.3,
                294.8,
                294.4,
                293.9,
                293.7,
            ],
            [
                293.7,
                293.6,
                294.4,
                293.9,
                294.2,
                296.8,
                296.3,
                294.3,
                293.1,
                298.0,
                300.5,
                300.5,
                298.9,
                299.1,
                298.0,
                297.7,
                297.0,
                296.4,
                295.9,
                295.5,
                295.1,
                294.9,
                294.9,
                294.7,
                295.0,
                295.3,
                295.1,
                295.0,
                295.6,
                296.9,
                298.5,
                301.8,
                297.3,
                301.1,
                302.4,
                299.7,
                299.6,
                301.0,
                300.8,
                298.4,
                297.0,
                299.1,
                298.5,
                298.5,
                298.3,
                298.1,
                298.0,
                298.0,
                297.9,
                297.6,
                297.5,
                297.6,
                297.7,
                297.8,
                297.7,
                297.7,
                297.7,
                297.9,
                298.0,
                297.8,
                297.6,
                297.4,
                297.1,
                296.7,
                296.3,
                295.9,
                295.6,
                295.4,
                295.2,
                294.7,
                294.2,
                293.6,
                292.9,
                292.1,
                291.3,
                290.9,
                291.9,
                294.1,
                280.6,
                297.3,
                301.2,
                301.0,
                296.7,
                296.5,
                299.8,
                298.7,
                297.8,
                297.1,
                296.3,
                295.8,
                295.4,
                295.0,
                294.6,
                294.3,
                293.9,
                293.8,
            ],
            [
                294.3,
                294.7,
                294.9,
                294.8,
                300.4,
                293.8,
                295.8,
                294.9,
                295.8,
                298.2,
                301.2,
                301.1,
                295.4,
                299.3,
                298.7,
                298.3,
                297.7,
                297.1,
                296.5,
                296.1,
                295.8,
                295.8,
                295.9,
                295.6,
                295.9,
                296.1,
                295.9,
                296.0,
                296.6,
                297.9,
                299.7,
                302.8,
                300.9,
                303.2,
                301.0,
                298.0,
                298.2,
                299.2,
                299.3,
                300.2,
                299.3,
                300.0,
                299.3,
                299.1,
                298.8,
                298.6,
                298.6,
                298.7,
                298.7,
                298.5,
                298.5,
                298.4,
                298.5,
                298.5,
                298.5,
                298.5,
                298.5,
                298.6,
                298.6,
                298.4,
                298.0,
                297.5,
                297.0,
                296.4,
                295.8,
                295.3,
                294.9,
                294.7,
                294.5,
                294.2,
                293.7,
                293.2,
                292.5,
                291.7,
                291.5,
                291.8,
                293.3,
                293.4,
                281.5,
                294.6,
                300.3,
                299.7,
                299.1,
                296.2,
                294.6,
                299.0,
                298.1,
                297.1,
                296.2,
                295.7,
                295.3,
                295.0,
                294.7,
                294.3,
                294.0,
                294.1,
            ],
            [
                295.1,
                295.6,
                295.7,
                295.2,
                294.5,
                294.9,
                296.9,
                295.3,
                294.1,
                297.9,
                301.5,
                300.9,
                301.8,
                298.2,
                298.7,
                298.8,
                298.2,
                297.6,
                297.1,
                296.8,
                296.7,
                296.9,
                297.0,
                297.0,
                297.1,
                297.0,
                296.9,
                297.1,
                297.6,
                298.7,
                300.4,
                302.7,
                302.9,
                301.2,
                299.5,
                299.3,
                299.9,
                299.3,
                298.6,
                297.4,
                300.6,
                300.3,
                299.9,
                299.7,
                299.4,
                299.4,
                299.4,
                299.4,
                299.4,
                299.2,
                299.1,
                299.0,
                299.0,
                299.3,
                299.4,
                299.5,
                299.3,
                299.2,
                299.2,
                298.7,
                298.1,
                297.5,
                296.9,
                296.1,
                295.4,
                294.8,
                294.3,
                294.0,
                293.7,
                293.4,
                293.1,
                292.5,
                292.0,
                291.8,
                292.4,
                293.3,
                293.7,
                292.8,
                281.2,
                293.6,
                299.9,
                300.7,
                298.9,
                296.3,
                294.1,
                296.1,
                298.5,
                297.5,
                296.7,
                296.1,
                295.7,
                295.4,
                295.0,
                294.7,
                294.5,
                294.7,
            ],
            [
                295.7,
                296.3,
                296.6,
                296.3,
                293.9,
                295.3,
                295.8,
                295.6,
                294.4,
                297.6,
                300.6,
                300.6,
                301.4,
                298.0,
                298.9,
                299.1,
                298.6,
                298.0,
                297.7,
                297.6,
                297.7,
                298.2,
                298.5,
                298.5,
                298.5,
                298.5,
                298.2,
                298.4,
                298.8,
                299.7,
                300.9,
                302.2,
                302.6,
                300.0,
                297.9,
                299.1,
                299.7,
                300.4,
                299.2,
                301.1,
                300.7,
                300.8,
                300.9,
                300.7,
                300.4,
                300.2,
                300.0,
                299.7,
                299.6,
                299.5,
                299.7,
                299.8,
                299.8,
                299.8,
                300.0,
                300.1,
                299.9,
                299.6,
                299.3,
                299.1,
                298.4,
                297.7,
                296.8,
                295.8,
                295.0,
                294.4,
                293.9,
                293.5,
                293.1,
                292.8,
                292.6,
                292.2,
                292.2,
                292.7,
                293.3,
                293.3,
                290.7,
                293.3,
                283.6,
                295.9,
                300.0,
                301.3,
                296.9,
                295.9,
                295.1,
                296.2,
                298.6,
                298.3,
                297.6,
                296.9,
                296.4,
                295.9,
                295.6,
                295.3,
                295.2,
                295.3,
            ],
            [
                296.7,
                297.4,
                298.1,
                298.3,
                293.0,
                294.1,
                295.4,
                294.8,
                296.0,
                296.7,
                297.1,
                300.4,
                301.3,
                298.5,
                299.2,
                299.2,
                298.8,
                298.5,
                298.4,
                298.7,
                299.1,
                299.4,
                299.6,
                299.8,
                299.9,
                299.8,
                299.6,
                299.8,
                300.2,
                300.8,
                301.3,
                301.9,
                302.2,
                302.4,
                299.3,
                299.7,
                300.3,
                302.5,
                300.3,
                301.2,
                301.4,
                301.5,
                301.5,
                301.3,
                300.9,
                300.6,
                300.4,
                300.2,
                300.2,
                300.2,
                300.6,
                301.0,
                300.8,
                300.5,
                300.4,
                300.4,
                300.2,
                300.0,
                299.7,
                299.4,
                298.9,
                298.1,
                297.2,
                296.3,
                295.5,
                294.7,
                294.1,
                293.5,
                293.0,
                292.6,
                292.3,
                292.4,
                292.9,
                293.6,
                294.6,
                295.1,
                293.8,
                280.5,
                293.1,
                299.5,
                298.8,
                298.8,
                297.6,
                295.5,
                295.4,
                294.9,
                298.9,
                299.0,
                298.2,
                297.5,
                296.9,
                296.4,
                296.2,
                296.0,
                295.9,
                296.3,
            ],
            [
                298.1,
                298.6,
                299.5,
                299.7,
                293.4,
                293.8,
                294.6,
                293.7,
                293.8,
                296.3,
                297.1,
                300.2,
                300.4,
                299.7,
                299.6,
                299.5,
                299.3,
                299.5,
                299.6,
                299.9,
                300.1,
                300.3,
                300.6,
                300.8,
                300.8,
                300.8,
                300.7,
                300.9,
                301.2,
                301.6,
                301.8,
                302.0,
                302.1,
                302.2,
                302.2,
                302.3,
                300.9,
                302.6,
                301.9,
                301.9,
                302.0,
                301.8,
                301.7,
                301.4,
                301.2,
                301.1,
                301.1,
                301.0,
                300.9,
                300.9,
                301.2,
                301.3,
                301.1,
                300.9,
                300.8,
                300.7,
                300.6,
                300.4,
                300.1,
                299.9,
                299.6,
                299.0,
                298.1,
                297.3,
                296.3,
                295.4,
                294.7,
                294.1,
                293.8,
                293.4,
                293.4,
                293.8,
                294.6,
                295.3,
                296.4,
                296.5,
                286.2,
                293.0,
                299.3,
                299.6,
                298.9,
                298.4,
                298.4,
                297.8,
                295.5,
                294.9,
                298.6,
                299.3,
                298.8,
                298.2,
                297.7,
                297.2,
                296.9,
                296.9,
                297.2,
                297.7,
            ],
            [
                299.3,
                300.1,
                300.9,
                301.0,
                296.5,
                295.0,
                295.4,
                294.5,
                293.6,
                293.2,
                298.3,
                300.1,
                300.6,
                300.2,
                300.2,
                300.2,
                300.1,
                300.3,
                300.4,
                300.5,
                300.6,
                300.7,
                300.8,
                300.9,
                301.0,
                301.1,
                301.2,
                301.2,
                301.4,
                301.6,
                301.8,
                301.9,
                301.9,
                302.0,
                302.2,
                302.2,
                302.4,
                302.3,
                302.5,
                302.3,
                302.1,
                301.8,
                301.7,
                301.6,
                301.5,
                301.5,
                301.4,
                301.3,
                301.3,
                301.4,
                301.4,
                301.3,
                301.2,
                301.1,
                301.1,
                301.0,
                300.9,
                300.7,
                300.5,
                300.2,
                299.8,
                299.2,
                298.4,
                297.5,
                296.7,
                296.1,
                295.6,
                295.3,
                295.1,
                295.0,
                295.1,
                295.5,
                296.1,
                297.1,
                297.6,
                296.6,
                292.3,
                299.6,
                300.4,
                299.8,
                299.4,
                298.4,
                298.4,
                298.9,
                296.7,
                296.7,
                300.1,
                299.5,
                299.2,
                298.8,
                298.4,
                298.2,
                298.2,
                298.4,
                298.7,
                298.9,
            ],
            [
                300.7,
                301.3,
                301.7,
                301.6,
                297.2,
                296.6,
                296.7,
                296.4,
                294.2,
                293.5,
                299.5,
                299.8,
                300.8,
                300.4,
                300.1,
                300.1,
                300.2,
                300.4,
                300.5,
                300.5,
                300.6,
                300.8,
                300.8,
                300.8,
                300.8,
                300.9,
                300.9,
                300.9,
                301.2,
                301.4,
                301.6,
                301.8,
                302.0,
                302.1,
                302.2,
                302.4,
                302.4,
                302.4,
                300.7,
                296.4,
                302.0,
                301.9,
                301.9,
                301.8,
                301.8,
                301.7,
                301.6,
                301.5,
                301.5,
                301.4,
                301.4,
                301.4,
                301.3,
                301.2,
                301.0,
                300.9,
                300.8,
                300.7,
                300.4,
                299.9,
                299.4,
                298.8,
                298.2,
                297.7,
                297.3,
                296.9,
                296.6,
                296.4,
                296.3,
                296.3,
                296.5,
                296.9,
                297.4,
                298.4,
                298.5,
                291.0,
                298.6,
                300.5,
                300.8,
                300.9,
                300.1,
                298.7,
                298.5,
                298.4,
                297.9,
                297.8,
                299.0,
                299.5,
                299.9,
                299.7,
                299.6,
                299.5,
                299.5,
                299.6,
                299.9,
                300.1,
            ],
            [
                301.2,
                301.4,
                301.5,
                301.6,
                298.1,
                298.3,
                298.3,
                297.6,
                294.4,
                293.5,
                297.6,
                299.1,
                300.7,
                300.6,
                300.2,
                299.9,
                300.0,
                300.2,
                300.4,
                300.4,
                300.5,
                300.6,
                300.5,
                300.6,
                300.7,
                300.8,
                300.9,
                301.5,
                299.1,
                302.0,
                302.1,
                302.2,
                302.6,
                302.4,
                302.2,
                302.3,
                302.2,
                298.0,
                295.6,
                301.5,
                301.8,
                301.7,
                301.6,
                301.5,
                301.3,
                301.2,
                301.1,
                301.0,
                301.0,
                300.9,
                300.7,
                300.6,
                300.3,
                300.0,
                299.9,
                299.8,
                299.7,
                299.6,
                299.4,
                299.2,
                298.8,
                298.5,
                298.1,
                297.8,
                297.8,
                297.7,
                297.6,
                297.5,
                297.6,
                297.7,
                297.9,
                298.2,
                298.8,
                299.5,
                299.3,
                294.1,
                300.7,
                301.2,
                301.1,
                300.9,
                300.2,
                299.5,
                298.9,
                298.7,
                298.8,
                297.7,
                300.7,
                300.1,
                300.4,
                300.3,
                300.3,
                300.3,
                300.4,
                300.5,
                300.7,
                300.9,
            ],
            [
                301.2,
                301.3,
                301.4,
                299.1,
                297.8,
                299.4,
                298.8,
                297.3,
                291.6,
                294.2,
                297.2,
                299.3,
                300.3,
                300.2,
                300.3,
                300.2,
                300.1,
                300.2,
                300.3,
                300.5,
                300.5,
                300.6,
                300.6,
                300.6,
                300.8,
                301.1,
                301.2,
                298.7,
                300.5,
                302.1,
                300.4,
                300.7,
                296.8,
                302.3,
                302.2,
                302.3,
                302.6,
                299.3,
                301.6,
                301.2,
                300.7,
                300.4,
                300.3,
                300.2,
                300.1,
                300.0,
                300.0,
                299.7,
                299.7,
                299.4,
                299.3,
                299.2,
                298.8,
                298.6,
                298.3,
                297.9,
                297.7,
                297.5,
                297.4,
                297.5,
                297.3,
                297.2,
                297.1,
                297.0,
                297.2,
                297.0,
                297.3,
                297.4,
                297.7,
                298.0,
                298.0,
                298.6,
                298.9,
                299.3,
                299.3,
                294.4,
                300.5,
                301.2,
                300.9,
                300.7,
                299.8,
                300.0,
                301.0,
                299.5,
                299.3,
                300.7,
                300.7,
                300.2,
                300.4,
                300.5,
                300.4,
                300.5,
                300.6,
                300.7,
                300.9,
                301.0,
            ],
            [
                301.4,
                301.5,
                301.4,
                299.0,
                298.3,
                299.5,
                299.6,
                297.9,
                293.7,
                293.8,
                296.3,
                302.4,
                300.0,
                299.5,
                299.7,
                300.1,
                300.3,
                300.3,
                300.5,
                300.6,
                300.6,
                300.6,
                300.8,
                300.9,
                301.3,
                301.6,
                301.5,
                298.7,
                301.6,
                301.6,
                298.6,
                299.5,
                302.0,
                301.9,
                302.1,
                302.2,
                302.1,
                301.8,
                301.2,
                300.6,
                300.2,
                300.0,
                299.8,
                299.7,
                299.6,
                299.4,
                299.4,
                299.1,
                299.0,
                298.6,
                298.5,
                298.2,
                298.0,
                297.8,
                297.6,
                297.3,
                297.0,
                297.0,
                296.6,
                296.8,
                296.5,
                296.5,
                296.3,
                296.3,
                296.0,
                296.1,
                296.5,
                296.8,
                297.0,
                297.8,
                298.0,
                298.8,
                299.0,
                299.3,
                299.4,
                293.3,
                299.8,
                300.5,
                300.4,
                299.8,
                300.2,
                302.2,
                300.8,
                300.1,
                300.2,
                300.1,
                300.0,
                300.2,
                300.2,
                300.3,
                300.4,
                300.6,
                300.8,
                300.9,
                301.1,
                301.2,
            ],
            [
                301.9,
                301.9,
                301.6,
                298.0,
                297.9,
                299.2,
                298.9,
                297.8,
                295.1,
                295.1,
                300.5,
                301.9,
                300.2,
                299.3,
                299.3,
                299.8,
                299.9,
                300.1,
                300.2,
                300.3,
                300.2,
                300.1,
                300.3,
                300.5,
                301.1,
                301.6,
                301.5,
                299.3,
                300.9,
                301.2,
                299.4,
                297.5,
                301.9,
                301.9,
                301.9,
                301.9,
                301.9,
                301.7,
                301.6,
                301.3,
                301.0,
                300.7,
                300.6,
                300.4,
                300.3,
                300.1,
                299.9,
                299.8,
                299.7,
                299.5,
                299.3,
                299.0,
                299.1,
                299.2,
                299.0,
                298.8,
                298.6,
                298.5,
                298.4,
                298.4,
                298.1,
                298.1,
                298.3,
                298.3,
                298.2,
                298.4,
                298.7,
                298.8,
                299.1,
                299.3,
                299.4,
                299.7,
                299.7,
                299.7,
                299.9,
                300.1,
                293.9,
                299.3,
                300.0,
                298.1,
                302.8,
                301.0,
                299.5,
                300.0,
                299.9,
                299.9,
                300.0,
                300.2,
                300.4,
                300.6,
                300.8,
                301.0,
                301.3,
                301.4,
                301.7,
                301.8,
            ],
            [
                302.3,
                302.5,
                299.8,
                295.9,
                296.7,
                297.9,
                297.1,
                295.9,
                298.0,
                299.7,
                292.1,
                298.2,
                298.1,
                299.2,
                298.9,
                299.5,
                299.8,
                300.0,
                300.1,
                300.0,
                299.8,
                299.7,
                299.8,
                300.2,
                300.7,
                301.2,
                301.3,
                296.9,
                300.8,
                301.4,
                302.1,
                298.0,
                302.2,
                301.9,
                301.5,
                301.8,
                301.8,
                301.7,
                301.6,
                301.5,
                301.4,
                301.3,
                301.2,
                301.1,
                301.0,
                300.8,
                300.7,
                300.6,
                300.4,
                300.3,
                300.1,
                300.0,
                300.1,
                299.9,
                299.9,
                299.9,
                299.8,
                299.7,
                299.7,
                299.8,
                299.7,
                299.6,
                299.5,
                299.4,
                299.4,
                299.6,
                299.7,
                299.8,
                300.0,
                300.1,
                300.1,
                300.1,
                300.1,
                299.9,
                300.1,
                301.1,
                290.3,
                299.6,
                299.7,
                298.0,
                296.0,
                299.6,
                299.4,
                299.8,
                299.6,
                299.6,
                299.9,
                300.1,
                300.4,
                300.7,
                301.0,
                301.3,
                301.6,
                302.2,
                299.7,
                300.3,
            ],
            [
                298.4,
                297.2,
                296.5,
                293.9,
                297.7,
                298.4,
                294.5,
                296.0,
                301.3,
                299.8,
                288.7,
                292.5,
                294.3,
                296.1,
                298.5,
                298.7,
                299.2,
                299.5,
                299.7,
                299.7,
                299.6,
                299.5,
                299.6,
                300.0,
                300.4,
                300.7,
                300.8,
                300.4,
                300.8,
                301.3,
                301.5,
                301.9,
                302.0,
                297.3,
                301.3,
                301.4,
                301.3,
                301.2,
                301.1,
                301.0,
                300.9,
                300.8,
                300.8,
                300.7,
                300.5,
                300.3,
                300.3,
                300.2,
                300.0,
                300.0,
                300.0,
                299.9,
                299.9,
                299.9,
                299.9,
                299.9,
                299.9,
                299.8,
                299.7,
                299.8,
                299.8,
                299.8,
                299.8,
                299.8,
                299.9,
                300.1,
                300.2,
                300.1,
                299.9,
                299.8,
                299.8,
                300.0,
                299.9,
                299.7,
                299.7,
                298.3,
                295.7,
                296.7,
                303.6,
                300.4,
                299.1,
                299.5,
                299.7,
                299.7,
                299.0,
                298.9,
                299.1,
                299.3,
                299.6,
                299.9,
                300.0,
                300.5,
                301.6,
                298.6,
                297.4,
                298.3,
            ],
            [
                296.5,
                294.9,
                293.1,
                296.6,
                300.2,
                300.2,
                298.6,
                299.1,
                301.0,
                299.0,
                290.4,
                291.1,
                290.9,
                290.4,
                297.5,
                297.8,
                298.0,
                298.4,
                299.1,
                299.6,
                299.8,
                293.2,
                299.1,
                299.5,
                299.9,
                300.2,
                300.3,
                297.6,
                298.5,
                301.0,
                301.0,
                301.2,
                301.1,
                298.1,
                300.7,
                300.7,
                300.6,
                300.5,
                300.4,
                300.2,
                299.9,
                299.7,
                299.6,
                299.4,
                299.2,
                299.0,
                298.9,
                298.8,
                298.6,
                298.6,
                298.6,
                298.6,
                298.7,
                298.6,
                298.6,
                298.7,
                298.7,
                298.7,
                298.7,
                298.8,
                299.0,
                299.1,
                299.2,
                299.3,
                299.5,
                299.8,
                299.9,
                299.6,
                299.2,
                299.0,
                299.1,
                299.6,
                300.1,
                300.1,
                294.4,
                299.6,
                300.2,
                299.3,
                301.1,
                300.2,
                299.0,
                299.5,
                299.1,
                298.7,
                298.5,
                298.2,
                298.1,
                298.1,
                298.3,
                298.6,
                298.7,
                299.2,
                300.9,
                295.8,
                295.9,
                296.5,
            ],
            [
                298.7,
                297.5,
                297.7,
                299.4,
                300.9,
                299.8,
                297.2,
                296.9,
                299.0,
                301.1,
                291.8,
                289.3,
                288.2,
                297.1,
                297.1,
                296.8,
                296.8,
                297.3,
                298.3,
                299.3,
                300.1,
                291.1,
                298.5,
                298.8,
                299.1,
                299.3,
                298.9,
                295.9,
                297.2,
                296.0,
                300.4,
                300.4,
                300.1,
                297.6,
                300.2,
                300.1,
                300.1,
                300.1,
                300.0,
                299.8,
                299.6,
                299.3,
                299.0,
                298.6,
                298.3,
                298.1,
                297.9,
                297.7,
                297.6,
                297.5,
                297.5,
                297.4,
                297.5,
                297.5,
                297.4,
                297.2,
                297.2,
                297.3,
                297.2,
                297.5,
                297.8,
                298.1,
                298.3,
                298.6,
                298.8,
                299.2,
                299.5,
                299.4,
                298.8,
                298.1,
                298.0,
                298.6,
                300.1,
                296.6,
                299.2,
                297.7,
                297.0,
                296.4,
                297.7,
                298.7,
                298.7,
                298.8,
                298.4,
                298.2,
                298.1,
                297.8,
                297.5,
                297.3,
                297.2,
                297.5,
                297.7,
                297.9,
                299.8,
                296.7,
                297.8,
                298.6,
            ],
            [
                298.5,
                297.9,
                297.1,
                296.8,
                297.3,
                297.1,
                294.6,
                293.7,
                296.6,
                299.4,
                293.8,
                298.8,
                284.9,
                289.3,
                297.1,
                295.9,
                295.7,
                296.3,
                297.4,
                298.5,
                291.5,
                294.5,
                298.3,
                298.2,
                298.4,
                298.2,
                297.4,
                293.2,
                293.8,
                293.9,
                299.2,
                299.3,
                295.4,
                299.2,
                299.3,
                299.1,
                299.2,
                299.2,
                299.3,
                299.4,
                299.3,
                299.0,
                298.6,
                298.1,
                297.6,
                297.3,
                297.0,
                296.8,
                296.7,
                296.6,
                296.5,
                296.2,
                296.3,
                296.3,
                296.1,
                296.1,
                296.1,
                296.1,
                296.3,
                296.5,
                296.8,
                297.0,
                297.2,
                297.2,
                297.3,
                297.7,
                298.4,
                299.1,
                298.9,
                297.9,
                297.2,
                296.4,
                291.5,
                293.8,
                299.1,
                299.1,
                298.8,
                298.8,
                298.9,
                298.9,
                298.5,
                298.2,
                297.8,
                297.5,
                297.4,
                297.3,
                297.1,
                296.9,
                296.7,
                296.5,
                296.5,
                296.9,
                299.4,
                299.6,
                298.4,
                298.6,
            ],
            [
                295.1,
                294.7,
                293.0,
                292.3,
                293.5,
                294.7,
                292.3,
                293.1,
                295.3,
                296.1,
                294.7,
                300.3,
                284.4,
                288.6,
                289.2,
                295.6,
                294.9,
                295.7,
                296.8,
                298.0,
                291.2,
                291.8,
                290.6,
                298.1,
                298.1,
                297.6,
                288.6,
                289.9,
                289.8,
                296.0,
                297.7,
                298.2,
                298.0,
                297.6,
                297.2,
                297.0,
                297.2,
                297.5,
                297.7,
                297.9,
                298.2,
                298.4,
                298.3,
                298.0,
                297.6,
                297.1,
                296.6,
                296.3,
                295.9,
                295.7,
                295.4,
                295.1,
                295.0,
                294.9,
                294.7,
                294.7,
                294.6,
                294.6,
                294.8,
                295.2,
                295.4,
                295.3,
                295.4,
                295.3,
                295.2,
                295.4,
                296.6,
                298.2,
                298.9,
                289.8,
                287.0,
                289.8,
                294.7,
                298.4,
                298.6,
                298.7,
                299.0,
                299.0,
                298.6,
                298.5,
                298.2,
                297.8,
                297.5,
                297.1,
                296.7,
                296.6,
                296.6,
                296.4,
                296.2,
                296.1,
                296.1,
                295.9,
                296.5,
                296.6,
                295.4,
                295.0,
            ],
            [
                292.2,
                290.9,
                290.4,
                290.4,
                290.2,
                288.6,
                289.6,
                290.2,
                292.8,
                292.5,
                299.2,
                289.1,
                287.6,
                289.6,
                290.8,
                290.7,
                295.2,
                295.5,
                296.5,
                297.5,
                289.3,
                291.2,
                287.7,
                298.1,
                298.2,
                288.9,
                286.8,
                285.9,
                286.0,
                292.4,
                295.0,
                296.9,
                296.4,
                295.5,
                294.8,
                294.5,
                294.8,
                295.2,
                295.6,
                295.9,
                296.3,
                296.7,
                296.9,
                297.0,
                296.8,
                296.5,
                296.1,
                295.8,
                295.3,
                294.9,
                294.6,
                294.3,
                294.1,
                293.8,
                293.6,
                293.4,
                293.2,
                293.1,
                293.0,
                293.1,
                293.1,
                293.0,
                293.0,
                293.0,
                292.9,
                293.4,
                294.6,
                296.7,
                290.7,
                283.2,
                283.4,
                296.3,
                294.1,
                297.6,
                297.4,
                297.6,
                295.5,
                295.2,
                298.1,
                297.8,
                297.6,
                297.3,
                297.1,
                296.8,
                296.3,
                295.9,
                295.8,
                295.7,
                295.6,
                295.5,
                295.5,
                294.7,
                293.2,
                292.9,
                292.1,
                292.2,
            ],
            [
                291.1,
                288.9,
                286.0,
                287.7,
                287.6,
                286.6,
                287.5,
                287.1,
                290.2,
                290.0,
                297.1,
                287.9,
                286.7,
                288.3,
                290.4,
                288.3,
                295.3,
                295.2,
                295.8,
                288.5,
                288.9,
                288.4,
                287.4,
                287.3,
                291.2,
                286.0,
                284.9,
                280.5,
                282.4,
                280.3,
                283.8,
                293.2,
                293.6,
                293.6,
                292.4,
                292.2,
                292.3,
                292.5,
                293.2,
                293.8,
                294.2,
                294.8,
                295.2,
                295.5,
                295.6,
                295.5,
                295.2,
                294.8,
                294.4,
                294.0,
                293.7,
                293.4,
                293.1,
                292.8,
                292.6,
                292.4,
                292.2,
                292.0,
                291.8,
                291.8,
                291.7,
                291.7,
                291.8,
                291.8,
                291.6,
                292.0,
                293.2,
                295.5,
                284.4,
                280.6,
                286.9,
                294.5,
                295.2,
                296.1,
                295.2,
                291.2,
                296.1,
                296.7,
                296.8,
                296.6,
                296.5,
                296.5,
                296.3,
                296.1,
                295.7,
                295.3,
                295.1,
                294.9,
                294.9,
                294.8,
                294.9,
                294.4,
                290.6,
                290.8,
                290.8,
                291.1,
            ],
            [
                290.4,
                288.1,
                285.2,
                285.7,
                286.9,
                287.4,
                287.2,
                286.0,
                287.6,
                289.6,
                290.1,
                285.8,
                284.8,
                287.0,
                289.7,
                288.5,
                294.5,
                294.6,
                286.5,
                286.4,
                284.1,
                285.4,
                287.0,
                288.2,
                287.0,
                282.8,
                281.1,
                278.0,
                278.5,
                280.5,
                279.8,
                276.8,
                289.8,
                292.0,
                290.9,
                290.5,
                290.5,
                290.4,
                290.8,
                291.3,
                291.9,
                292.6,
                293.2,
                293.7,
                294.0,
                294.1,
                294.0,
                293.9,
                293.6,
                293.2,
                292.8,
                292.4,
                292.1,
                291.6,
                291.3,
                291.1,
                290.9,
                290.9,
                290.9,
                290.8,
                290.7,
                290.8,
                290.9,
                290.9,
                290.6,
                290.9,
                292.0,
                293.9,
                279.2,
                280.3,
                285.6,
                293.0,
                293.5,
                293.6,
                292.5,
                293.9,
                294.4,
                294.8,
                295.0,
                295.2,
                295.2,
                295.3,
                295.3,
                295.3,
                295.1,
                294.9,
                294.6,
                294.4,
                294.2,
                294.0,
                294.0,
                294.2,
                294.0,
                289.0,
                289.9,
                290.0,
            ],
            [
                288.3,
                287.8,
                287.2,
                286.6,
                286.8,
                287.7,
                288.2,
                287.0,
                286.6,
                287.5,
                284.8,
                283.8,
                284.1,
                293.8,
                294.4,
                283.1,
                280.0,
                280.8,
                283.8,
                284.8,
                281.4,
                284.1,
                284.0,
                275.8,
                272.7,
                278.9,
                274.1,
                272.2,
                277.0,
                274.2,
                277.1,
                276.9,
                275.1,
                288.0,
                288.3,
                289.1,
                289.0,
                288.8,
                288.8,
                289.2,
                289.6,
                290.3,
                290.9,
                291.6,
                292.0,
                292.2,
                292.2,
                292.2,
                292.1,
                291.9,
                291.6,
                291.3,
                291.0,
                290.6,
                290.3,
                290.0,
                289.7,
                289.6,
                289.7,
                289.9,
                289.8,
                289.7,
                289.8,
                289.8,
                289.5,
                289.8,
                290.7,
                279.3,
                278.3,
                280.4,
                281.1,
                288.7,
                288.6,
                288.4,
                288.4,
                291.6,
                292.2,
                292.7,
                293.0,
                293.3,
                293.5,
                293.8,
                293.9,
                294.1,
                294.1,
                294.0,
                293.8,
                293.7,
                293.6,
                293.4,
                293.1,
                293.1,
                293.5,
                287.8,
                287.2,
                287.1,
            ],
            [
                283.4,
                285.0,
                285.9,
                285.5,
                286.3,
                287.6,
                287.1,
                286.6,
                286.6,
                284.8,
                282.4,
                282.7,
                284.0,
                293.1,
                274.8,
                275.5,
                279.8,
                279.6,
                278.0,
                283.2,
                282.9,
                276.8,
                261.2,
                245.8,
                259.6,
                256.2,
                259.3,
                267.7,
                277.8,
                274.4,
                275.5,
                275.4,
                274.8,
                284.1,
                284.2,
                285.8,
                286.5,
                286.5,
                287.1,
                287.7,
                288.2,
                288.7,
                289.2,
                289.7,
                290.1,
                290.3,
                290.4,
                290.2,
                290.2,
                290.0,
                289.8,
                289.6,
                289.3,
                289.1,
                289.0,
                288.9,
                288.9,
                288.7,
                288.7,
                288.9,
                288.9,
                288.6,
                288.8,
                288.7,
                288.3,
                288.6,
                281.2,
                277.0,
                276.8,
                277.0,
                275.8,
                277.1,
                276.5,
                276.5,
                276.7,
                288.3,
                290.4,
                291.3,
                291.5,
                291.7,
                291.9,
                292.1,
                292.4,
                292.6,
                292.8,
                292.9,
                292.8,
                292.8,
                292.7,
                292.6,
                292.3,
                292.2,
                292.5,
                292.7,
                282.1,
                283.9,
            ],
            [
                279.1,
                280.6,
                283.5,
                283.6,
                289.8,
                290.2,
                290.1,
                290.0,
                290.4,
                290.8,
                280.5,
                281.3,
                283.8,
                276.7,
                274.6,
                276.6,
                276.4,
                275.7,
                272.5,
                281.0,
                278.8,
                254.8,
                251.9,
                253.4,
                256.7,
                247.5,
                257.2,
                263.1,
                270.5,
                272.0,
                273.2,
                273.7,
                274.2,
                280.9,
                279.7,
                282.0,
                282.5,
                282.2,
                284.2,
                285.2,
                286.2,
                287.0,
                287.5,
                287.9,
                288.2,
                288.4,
                288.6,
                288.5,
                288.3,
                288.2,
                287.9,
                287.7,
                287.4,
                287.3,
                287.3,
                287.4,
                287.6,
                287.5,
                287.5,
                287.6,
                287.8,
                287.6,
                287.7,
                287.6,
                287.1,
                279.8,
                279.5,
                273.8,
                273.7,
                274.4,
                274.8,
                275.3,
                275.0,
                275.1,
                274.7,
                284.1,
                288.4,
                289.8,
                290.2,
                290.2,
                290.6,
                290.7,
                290.9,
                291.1,
                291.3,
                291.6,
                291.8,
                291.8,
                291.7,
                291.6,
                291.5,
                291.4,
                291.5,
                291.4,
                282.4,
                278.9,
            ],
            [
                278.8,
                277.5,
                278.0,
                288.2,
                288.2,
                288.1,
                288.2,
                288.6,
                289.0,
                289.3,
                278.3,
                280.4,
                277.2,
                272.0,
                275.3,
                276.3,
                275.6,
                273.5,
                269.1,
                269.9,
                255.7,
                246.4,
                251.9,
                253.4,
                245.8,
                246.2,
                258.6,
                263.1,
                267.9,
                270.2,
                270.5,
                271.0,
                276.2,
                276.7,
                274.4,
                277.2,
                269.0,
                266.8,
                280.2,
                282.3,
                283.3,
                284.2,
                285.0,
                285.6,
                285.8,
                286.0,
                286.3,
                286.5,
                286.4,
                286.3,
                286.1,
                285.9,
                285.7,
                285.6,
                285.5,
                285.7,
                286.1,
                286.1,
                286.2,
                286.4,
                286.5,
                286.5,
                286.6,
                286.7,
                277.9,
                276.5,
                273.3,
                269.3,
                271.2,
                272.5,
                272.2,
                271.8,
                271.7,
                271.3,
                270.9,
                270.7,
                284.5,
                287.6,
                288.5,
                288.9,
                289.5,
                289.8,
                289.6,
                289.7,
                290.0,
                290.2,
                290.5,
                290.7,
                290.8,
                290.7,
                290.7,
                290.5,
                290.1,
                290.0,
                289.2,
                281.4,
            ],
            [
                285.2,
                286.4,
                286.4,
                286.0,
                286.2,
                286.3,
                286.6,
                286.0,
                276.0,
                274.1,
                274.7,
                274.2,
                270.0,
                272.4,
                287.2,
                274.7,
                275.2,
                275.3,
                275.2,
                260.8,
                251.5,
                268.3,
                270.1,
                263.4,
                252.5,
                263.2,
                252.3,
                262.8,
                264.5,
                265.2,
                266.5,
                268.0,
                272.0,
                270.4,
                247.3,
                271.7,
                275.1,
                266.1,
                277.3,
                278.6,
                280.3,
                281.6,
                282.2,
                282.6,
                283.1,
                283.4,
                283.9,
                284.3,
                284.5,
                284.3,
                283.8,
                283.5,
                283.5,
                283.4,
                283.6,
                284.0,
                284.5,
                284.7,
                284.9,
                285.2,
                285.3,
                285.5,
                285.4,
                285.4,
                273.3,
                267.6,
                265.5,
                264.9,
                263.3,
                271.6,
                270.0,
                268.6,
                268.3,
                268.5,
                268.7,
                270.4,
                281.5,
                285.3,
                286.5,
                287.6,
                288.1,
                288.5,
                288.7,
                288.6,
                288.7,
                288.8,
                289.0,
                289.6,
                289.7,
                289.8,
                289.6,
                289.3,
                288.6,
                287.8,
                279.7,
                279.1,
            ],
            [
                273.8,
                284.8,
                284.3,
                283.4,
                284.1,
                284.3,
                269.9,
                276.7,
                275.1,
                272.3,
                266.6,
                264.7,
                269.8,
                274.6,
                283.5,
                272.9,
                272.1,
                273.2,
                272.5,
                263.6,
                259.2,
                256.8,
                262.2,
                266.6,
                266.4,
                261.5,
                258.6,
                264.1,
                262.1,
                262.1,
                260.5,
                263.4,
                255.3,
                246.1,
                253.9,
                266.2,
                270.5,
                272.7,
                262.4,
                270.6,
                276.3,
                277.7,
                277.8,
                277.9,
                278.5,
                279.6,
                280.3,
                280.7,
                281.2,
                281.3,
                281.1,
                281.1,
                281.3,
                281.5,
                281.7,
                282.3,
                282.9,
                283.3,
                283.5,
                284.0,
                284.2,
                284.4,
                284.4,
                278.8,
                269.8,
                264.9,
                264.1,
                262.6,
                262.6,
                268.9,
                265.4,
                264.7,
                264.5,
                265.8,
                266.6,
                265.8,
                267.8,
                279.0,
                281.4,
                283.4,
                284.6,
                285.3,
                285.8,
                286.0,
                286.8,
                286.9,
                287.1,
                288.1,
                288.7,
                288.8,
                288.4,
                288.0,
                287.3,
                286.7,
                277.4,
                276.5,
            ],
            [
                274.9,
                283.4,
                282.1,
                269.1,
                281.3,
                261.9,
                270.2,
                267.7,
                280.6,
                281.8,
                282.8,
                273.9,
                265.5,
                281.3,
                272.6,
                269.8,
                268.9,
                269.3,
                271.0,
                266.4,
                261.9,
                257.4,
                261.4,
                253.0,
                261.0,
                261.9,
                257.6,
                260.1,
                257.9,
                258.3,
                256.1,
                252.3,
                255.7,
                247.3,
                251.4,
                240.9,
                265.6,
                270.0,
                261.5,
                269.7,
                270.6,
                274.3,
                274.8,
                275.4,
                276.4,
                277.5,
                277.5,
                277.6,
                278.1,
                278.5,
                278.6,
                279.0,
                279.3,
                279.7,
                280.0,
                280.5,
                281.1,
                281.7,
                282.1,
                282.5,
                283.0,
                283.1,
                283.4,
                276.3,
                268.0,
                265.9,
                262.4,
                261.1,
                261.6,
                262.4,
                262.7,
                263.0,
                262.7,
                264.7,
                265.1,
                264.7,
                266.2,
                265.9,
                276.9,
                279.0,
                280.8,
                281.2,
                280.9,
                281.5,
                283.2,
                283.6,
                284.3,
                286.1,
                287.3,
                287.6,
                286.8,
                286.8,
                286.5,
                286.3,
                275.0,
                276.1,
            ],
            [
                274.5,
                271.0,
                266.0,
                268.2,
                266.9,
                266.2,
                266.2,
                263.3,
                277.4,
                279.3,
                280.8,
                273.2,
                273.8,
                278.1,
                269.3,
                267.1,
                265.8,
                264.3,
                266.5,
                265.0,
                266.8,
                262.1,
                257.7,
                260.3,
                253.8,
                252.7,
                247.7,
                255.3,
                255.3,
                253.8,
                254.4,
                250.6,
                250.4,
                243.1,
                244.6,
                248.9,
                249.0,
                266.6,
                262.0,
                270.0,
                272.0,
                273.2,
                273.3,
                274.0,
                274.7,
                275.4,
                275.5,
                275.6,
                275.9,
                276.1,
                276.5,
                277.1,
                277.7,
                278.1,
                278.4,
                278.9,
                279.7,
                280.2,
                280.6,
                281.2,
                281.9,
                282.1,
                282.5,
                276.1,
                272.3,
                267.5,
                260.7,
                261.8,
                262.1,
                259.5,
                259.8,
                261.4,
                261.6,
                272.4,
                273.0,
                259.2,
                261.9,
                262.4,
                264.8,
                274.8,
                276.8,
                276.2,
                276.9,
                277.8,
                279.4,
                280.8,
                282.1,
                284.0,
                285.3,
                286.1,
                285.4,
                285.1,
                285.2,
                285.8,
                285.1,
                283.7,
            ],
            [
                273.0,
                274.0,
                266.9,
                261.6,
                264.6,
                265.3,
                264.3,
                262.6,
                262.2,
                264.1,
                268.2,
                268.7,
                267.6,
                266.1,
                263.6,
                262.5,
                260.5,
                260.1,
                259.8,
                258.0,
                258.9,
                259.2,
                256.6,
                251.1,
                247.7,
                252.2,
                249.4,
                249.4,
                253.3,
                249.6,
                249.9,
                246.2,
                246.0,
                244.9,
                242.8,
                246.4,
                249.2,
                250.5,
                264.2,
                269.5,
                271.8,
                271.9,
                271.8,
                272.8,
                273.5,
                273.9,
                273.9,
                274.0,
                274.2,
                274.5,
                275.1,
                275.7,
                276.4,
                276.9,
                277.2,
                277.7,
                278.5,
                279.1,
                279.6,
                280.3,
                281.1,
                281.4,
                281.0,
                275.6,
                271.9,
                266.2,
                263.0,
                260.8,
                257.4,
                257.4,
                256.9,
                258.9,
                259.5,
                269.7,
                252.8,
                258.9,
                261.3,
                262.3,
                264.2,
                272.0,
                273.8,
                268.0,
                274.9,
                276.0,
                277.5,
                279.2,
                280.6,
                282.1,
                283.3,
                284.4,
                284.4,
                284.1,
                285.0,
                285.1,
                284.0,
                282.6,
            ],
            [
                271.1,
                272.8,
                268.2,
                266.4,
                264.6,
                262.8,
                263.0,
                261.6,
                261.5,
                261.0,
                260.6,
                260.5,
                260.5,
                260.2,
                259.6,
                258.6,
                256.7,
                257.0,
                257.7,
                256.5,
                256.8,
                257.7,
                258.7,
                253.6,
                252.2,
                251.6,
                251.0,
                251.5,
                252.9,
                251.5,
                249.3,
                244.7,
                242.7,
                242.7,
                240.6,
                242.4,
                248.5,
                248.8,
                265.8,
                269.7,
                270.9,
                270.4,
                269.2,
                271.3,
                272.1,
                272.4,
                272.5,
                272.7,
                273.0,
                273.5,
                274.1,
                274.7,
                275.4,
                275.9,
                276.3,
                276.8,
                277.6,
                278.3,
                279.1,
                279.8,
                280.5,
                280.6,
                280.3,
                266.9,
                265.4,
                261.0,
                259.6,
                258.2,
                256.0,
                255.5,
                255.1,
                257.3,
                257.7,
                258.2,
                257.1,
                260.2,
                258.5,
                260.7,
                260.2,
                260.6,
                264.8,
                267.5,
                274.2,
                275.5,
                276.9,
                278.5,
                279.6,
                280.7,
                281.8,
                282.9,
                282.9,
                283.2,
                284.6,
                283.9,
                281.7,
                280.7,
            ],
            [
                272.9,
                278.0,
                266.3,
                264.1,
                264.1,
                261.7,
                260.5,
                260.0,
                258.1,
                256.2,
                255.4,
                254.6,
                254.8,
                255.0,
                254.6,
                256.0,
                254.5,
                254.0,
                255.1,
                256.0,
                256.1,
                256.0,
                257.0,
                257.5,
                256.7,
                254.9,
                252.3,
                253.8,
                251.8,
                248.4,
                245.3,
                242.5,
                240.9,
                240.6,
                239.5,
                239.7,
                245.4,
                244.2,
                260.7,
                268.4,
                269.1,
                268.7,
                240.9,
                267.9,
                269.2,
                270.1,
                270.8,
                271.1,
                272.0,
                272.8,
                273.2,
                274.0,
                274.8,
                275.5,
                276.1,
                276.5,
                277.1,
                277.9,
                278.7,
                279.4,
                280.0,
                279.9,
                267.4,
                266.1,
                263.0,
                261.9,
                258.8,
                256.7,
                255.0,
                255.2,
                254.8,
                254.5,
                256.1,
                255.6,
                256.0,
                268.1,
                253.7,
                258.8,
                255.8,
                255.8,
                261.0,
                266.4,
                273.4,
                275.0,
                276.6,
                278.6,
                279.7,
                280.2,
                281.0,
                281.7,
                282.3,
                282.5,
                283.6,
                282.1,
                269.5,
                272.3,
            ],
            [
                279.5,
                278.6,
                276.0,
                264.8,
                272.6,
                272.6,
                259.8,
                257.3,
                254.5,
                251.7,
                250.6,
                250.5,
                250.5,
                250.1,
                250.8,
                252.4,
                253.3,
                251.8,
                252.0,
                252.7,
                252.6,
                252.2,
                252.9,
                254.6,
                255.4,
                255.2,
                255.5,
                251.6,
                249.8,
                244.5,
                243.6,
                239.9,
                238.9,
                237.6,
                238.1,
                239.3,
                240.5,
                250.0,
                255.8,
                266.6,
                266.7,
                267.2,
                245.0,
                243.8,
                265.6,
                267.4,
                268.2,
                268.7,
                270.2,
                271.3,
                272.1,
                272.9,
                273.6,
                274.7,
                275.5,
                276.4,
                277.0,
                277.7,
                278.3,
                278.6,
                278.1,
                269.4,
                266.8,
                263.5,
                263.3,
                262.7,
                260.3,
                257.5,
                254.8,
                253.6,
                252.4,
                251.2,
                251.1,
                251.3,
                262.3,
                268.0,
                249.2,
                256.3,
                252.8,
                250.1,
                256.0,
                271.0,
                273.2,
                275.2,
                277.1,
                278.8,
                279.9,
                280.3,
                280.9,
                281.6,
                282.6,
                282.9,
                283.4,
                282.9,
                280.9,
                270.4,
            ],
            [
                279.7,
                277.9,
                275.4,
                273.1,
                257.7,
                272.2,
                269.6,
                250.4,
                251.0,
                248.2,
                245.6,
                246.1,
                246.9,
                246.0,
                247.1,
                248.1,
                248.7,
                248.0,
                248.3,
                248.7,
                248.9,
                248.9,
                248.9,
                249.6,
                250.4,
                250.2,
                247.7,
                246.4,
                244.8,
                244.6,
                241.7,
                239.0,
                238.3,
                236.0,
                233.8,
                235.6,
                236.8,
                240.2,
                250.0,
                258.6,
                257.6,
                257.7,
                256.0,
                242.0,
                257.2,
                264.2,
                264.5,
                265.3,
                266.7,
                268.3,
                269.3,
                271.1,
                272.2,
                272.0,
                273.1,
                274.7,
                275.8,
                276.5,
                277.1,
                277.2,
                266.4,
                266.3,
                262.2,
                262.1,
                260.9,
                261.2,
                259.3,
                256.6,
                254.0,
                251.6,
                249.0,
                248.4,
                251.3,
                250.0,
                253.9,
                263.0,
                245.9,
                252.7,
                254.5,
                246.1,
                268.6,
                271.9,
                273.7,
                275.7,
                277.1,
                278.2,
                279.5,
                280.0,
                280.7,
                281.3,
                281.8,
                281.8,
                281.7,
                282.2,
                281.6,
                265.3,
            ],
            [
                279.8,
                277.7,
                255.1,
                258.3,
                255.5,
                270.2,
                266.8,
                246.2,
                247.0,
                245.2,
                242.0,
                243.4,
                243.2,
                243.2,
                244.7,
                245.2,
                244.9,
                243.2,
                243.9,
                244.2,
                244.7,
                244.7,
                244.2,
                244.3,
                245.4,
                243.4,
                240.8,
                239.1,
                238.9,
                238.3,
                236.7,
                236.1,
                234.8,
                233.7,
                232.2,
                231.0,
                229.6,
                237.8,
                243.5,
                244.1,
                242.9,
                240.6,
                250.7,
                250.6,
                239.3,
                252.1,
                250.7,
                250.8,
                256.8,
                263.2,
                267.1,
                268.8,
                269.6,
                250.4,
                257.9,
                258.2,
                259.1,
                262.1,
                261.1,
                259.5,
                259.4,
                261.1,
                259.4,
                257.8,
                257.6,
                257.5,
                255.0,
                254.4,
                251.8,
                248.8,
                245.5,
                249.3,
                250.1,
                249.3,
                250.3,
                252.9,
                246.2,
                249.3,
                255.3,
                261.8,
                268.1,
                271.6,
                273.4,
                274.7,
                275.4,
                276.2,
                278.5,
                279.4,
                280.3,
                280.9,
                281.3,
                281.2,
                281.0,
                281.4,
                281.4,
                280.7,
            ],
            [
                278.7,
                277.8,
                264.9,
                264.0,
                254.2,
                267.9,
                249.4,
                242.7,
                244.0,
                242.0,
                240.6,
                241.8,
                241.4,
                241.1,
                241.8,
                241.7,
                240.0,
                238.2,
                238.6,
                239.1,
                239.4,
                239.5,
                239.9,
                240.3,
                240.7,
                238.7,
                236.5,
                235.7,
                234.5,
                233.6,
                232.0,
                229.2,
                230.4,
                227.2,
                226.2,
                229.9,
                232.8,
                235.1,
                237.5,
                239.0,
                239.8,
                240.1,
                240.2,
                241.7,
                245.4,
                245.5,
                246.0,
                244.4,
                255.6,
                260.1,
                262.5,
                266.9,
                267.4,
                250.2,
                259.4,
                258.6,
                258.3,
                257.9,
                256.5,
                254.3,
                254.3,
                256.5,
                256.6,
                254.8,
                251.6,
                252.2,
                251.5,
                249.3,
                246.8,
                245.4,
                244.0,
                242.9,
                248.6,
                248.7,
                248.8,
                248.2,
                249.5,
                252.3,
                246.9,
                258.9,
                267.3,
                270.3,
                272.2,
                248.1,
                253.5,
                273.7,
                276.3,
                277.8,
                279.1,
                279.8,
                281.3,
                281.1,
                280.7,
                280.8,
                280.4,
                279.8,
            ],
            [
                276.4,
                276.5,
                275.5,
                275.0,
                250.6,
                252.0,
                256.1,
                241.6,
                241.3,
                241.2,
                239.5,
                239.0,
                238.2,
                239.2,
                238.2,
                237.3,
                236.0,
                234.8,
                234.5,
                234.7,
                235.0,
                234.5,
                234.9,
                236.3,
                236.9,
                235.5,
                234.5,
                233.6,
                231.4,
                229.8,
                228.6,
                228.8,
                227.9,
                227.4,
                232.3,
                234.1,
                234.0,
                232.7,
                232.6,
                234.1,
                233.7,
                233.4,
                235.4,
                237.4,
                237.3,
                240.4,
                241.4,
                242.2,
                251.4,
                252.7,
                255.0,
                261.0,
                261.3,
                245.5,
                252.5,
                252.0,
                250.5,
                251.8,
                252.1,
                252.0,
                251.6,
                252.7,
                252.8,
                248.1,
                247.8,
                248.3,
                247.9,
                245.9,
                244.3,
                243.9,
                243.3,
                243.7,
                243.5,
                242.5,
                247.1,
                246.3,
                247.3,
                246.6,
                245.6,
                252.1,
                266.7,
                269.9,
                271.9,
                249.9,
                252.0,
                243.4,
                273.0,
                270.7,
                270.1,
                271.5,
                278.6,
                255.6,
                257.5,
                277.9,
                277.3,
                276.7,
            ],
            [
                274.7,
                274.7,
                274.0,
                273.4,
                240.9,
                248.1,
                247.8,
                246.6,
                243.0,
                240.4,
                237.7,
                244.3,
                241.4,
                241.3,
                232.1,
                232.7,
                233.6,
                232.8,
                231.5,
                231.0,
                230.8,
                230.1,
                230.8,
                232.7,
                234.3,
                232.2,
                231.2,
                230.9,
                229.7,
                228.1,
                227.9,
                227.8,
                227.4,
                228.2,
                229.6,
                229.1,
                229.2,
                230.8,
                231.3,
                231.9,
                231.8,
                231.5,
                231.8,
                234.2,
                236.6,
                238.2,
                240.1,
                241.3,
                240.1,
                238.5,
                239.6,
                243.8,
                255.4,
                242.5,
                250.6,
                249.5,
                245.8,
                245.5,
                244.6,
                244.6,
                244.8,
                244.0,
                244.3,
                245.4,
                245.0,
                244.5,
                243.8,
                243.1,
                242.6,
                242.6,
                241.2,
                242.2,
                244.2,
                243.3,
                240.8,
                245.5,
                246.3,
                245.5,
                243.6,
                247.8,
                266.3,
                269.5,
                247.1,
                253.6,
                249.5,
                247.9,
                247.0,
                250.8,
                266.3,
                261.0,
                263.1,
                270.9,
                274.4,
                275.1,
                274.2,
                273.9,
            ],
            [
                272.2,
                272.7,
                272.0,
                271.5,
                270.3,
                270.5,
                242.2,
                243.8,
                247.3,
                263.6,
                258.9,
                246.2,
                240.0,
                240.6,
                239.1,
                235.5,
                236.5,
                238.7,
                229.9,
                227.0,
                227.7,
                226.9,
                226.5,
                229.8,
                232.2,
                230.6,
                230.0,
                229.6,
                229.5,
                228.5,
                227.8,
                227.6,
                227.5,
                228.6,
                229.5,
                229.6,
                229.0,
                228.5,
                228.8,
                229.7,
                230.1,
                230.4,
                231.9,
                238.6,
                238.6,
                238.3,
                238.2,
                237.5,
                239.9,
                244.9,
                246.5,
                245.2,
                248.3,
                255.0,
                234.9,
                238.8,
                242.9,
                248.6,
                248.2,
                250.8,
                250.9,
                247.1,
                248.3,
                241.2,
                241.0,
                240.8,
                240.6,
                240.3,
                239.9,
                244.3,
                242.7,
                240.5,
                241.9,
                241.4,
                237.7,
                237.7,
                244.5,
                244.5,
                247.1,
                252.4,
                266.4,
                269.8,
                249.9,
                253.8,
                245.5,
                238.3,
                238.4,
                242.4,
                247.6,
                249.2,
                245.4,
                255.7,
                264.5,
                264.9,
                266.0,
                269.0,
            ],
            [
                269.0,
                269.4,
                269.4,
                269.8,
                271.1,
                270.8,
                268.2,
                266.5,
                265.6,
                264.8,
                261.3,
                246.1,
                235.9,
                237.4,
                238.2,
                235.0,
                235.9,
                237.4,
                237.4,
                228.2,
                237.7,
                235.9,
                223.6,
                225.8,
                227.0,
                228.1,
                227.4,
                226.5,
                226.1,
                226.3,
                226.9,
                228.1,
                227.3,
                226.7,
                227.5,
                235.2,
                235.5,
                235.8,
                235.5,
                235.1,
                235.1,
                235.6,
                236.4,
                238.5,
                239.4,
                239.6,
                240.6,
                240.2,
                239.7,
                241.4,
                243.6,
                241.9,
                241.5,
                251.9,
                244.3,
                240.9,
                241.6,
                242.6,
                242.3,
                243.6,
                243.1,
                242.9,
                245.9,
                238.0,
                238.6,
                238.3,
                238.3,
                240.3,
                241.8,
                244.4,
                243.1,
                238.5,
                239.1,
                239.7,
                239.8,
                242.1,
                247.8,
                245.0,
                251.2,
                264.4,
                268.3,
                269.9,
                232.1,
                250.7,
                244.8,
                242.9,
                241.2,
                237.9,
                239.4,
                246.4,
                245.0,
                249.2,
                248.3,
                247.0,
                251.1,
                263.6,
            ],
            [
                267.8,
                268.5,
                268.3,
                269.1,
                269.9,
                268.6,
                266.7,
                265.9,
                264.3,
                262.9,
                255.5,
                237.1,
                230.7,
                236.7,
                238.3,
                235.6,
                234.2,
                235.0,
                234.6,
                237.0,
                240.1,
                234.9,
                234.3,
                231.2,
                225.1,
                227.0,
                228.0,
                227.5,
                227.0,
                226.7,
                228.7,
                237.0,
                234.4,
                235.6,
                236.8,
                236.5,
                235.3,
                235.5,
                236.0,
                236.8,
                237.3,
                237.7,
                237.5,
                237.0,
                237.2,
                237.3,
                237.9,
                238.6,
                238.3,
                238.3,
                239.5,
                239.2,
                238.4,
                240.3,
                238.0,
                237.5,
                238.4,
                239.1,
                239.2,
                239.7,
                240.3,
                241.0,
                242.1,
                241.8,
                241.6,
                240.7,
                240.6,
                241.3,
                242.6,
                243.3,
                242.9,
                242.6,
                236.4,
                238.7,
                241.0,
                250.4,
                251.0,
                253.4,
                262.5,
                268.7,
                268.8,
                231.8,
                243.0,
                241.4,
                239.4,
                238.4,
                234.8,
                235.0,
                237.2,
                241.8,
                242.2,
                247.0,
                246.7,
                247.1,
                248.8,
                257.9,
            ],
            [
                266.3,
                268.4,
                267.8,
                267.9,
                267.3,
                266.4,
                265.5,
                264.8,
                262.8,
                262.1,
                259.4,
                245.0,
                231.9,
                235.1,
                237.9,
                236.7,
                235.8,
                235.2,
                235.2,
                236.9,
                236.9,
                234.1,
                232.7,
                231.0,
                231.8,
                233.7,
                234.0,
                225.9,
                227.3,
                236.4,
                237.8,
                239.8,
                236.5,
                236.2,
                236.3,
                236.5,
                236.3,
                237.0,
                237.4,
                237.8,
                238.1,
                237.6,
                236.9,
                236.0,
                235.8,
                236.1,
                236.8,
                237.4,
                237.5,
                237.2,
                237.3,
                237.1,
                237.1,
                237.9,
                237.9,
                238.0,
                238.1,
                237.9,
                237.4,
                237.1,
                237.5,
                238.2,
                238.8,
                239.8,
                240.2,
                240.6,
                240.6,
                240.6,
                240.6,
                241.0,
                242.2,
                243.4,
                236.8,
                239.2,
                240.8,
                241.8,
                249.6,
                254.1,
                240.6,
                239.5,
                239.6,
                236.0,
                232.0,
                229.8,
                229.2,
                228.7,
                227.9,
                229.8,
                234.4,
                238.3,
                241.0,
                250.1,
                247.6,
                247.8,
                249.5,
                254.9,
            ],
            [
                266.7,
                266.9,
                265.6,
                265.9,
                265.2,
                264.6,
                264.6,
                262.5,
                250.8,
                241.4,
                240.2,
                239.9,
                236.9,
                238.9,
                240.3,
                238.4,
                236.9,
                236.5,
                234.4,
                233.6,
                234.0,
                234.2,
                233.2,
                233.4,
                235.1,
                233.9,
                231.8,
                232.0,
                232.3,
                234.2,
                233.9,
                235.5,
                235.1,
                235.7,
                236.7,
                237.4,
                237.2,
                237.1,
                237.0,
                236.9,
                237.4,
                237.5,
                237.4,
                236.3,
                235.6,
                235.6,
                235.7,
                235.6,
                235.7,
                235.7,
                235.7,
                235.6,
                235.7,
                235.7,
                235.5,
                235.5,
                235.5,
                235.4,
                235.2,
                235.1,
                235.5,
                236.5,
                237.0,
                237.1,
                237.2,
                237.3,
                237.3,
                237.2,
                237.6,
                239.0,
                241.1,
                246.3,
                232.2,
                234.5,
                234.2,
                231.2,
                231.1,
                233.8,
                236.8,
                238.4,
                236.6,
                236.1,
                234.8,
                235.3,
                233.6,
                233.4,
                235.0,
                237.4,
                239.0,
                239.4,
                240.2,
                241.5,
                242.2,
                244.1,
                247.4,
                257.0,
            ],
            [
                247.2,
                247.8,
                250.1,
                246.0,
                246.8,
                247.5,
                245.6,
                243.0,
                238.0,
                235.9,
                237.4,
                238.7,
                238.8,
                239.1,
                240.2,
                236.9,
                235.1,
                236.5,
                234.2,
                231.7,
                231.0,
                231.8,
                232.2,
                232.4,
                235.5,
                235.5,
                236.2,
                235.3,
                234.9,
                235.0,
                235.4,
                236.6,
                236.9,
                237.6,
                238.0,
                238.2,
                238.0,
                238.2,
                238.3,
                238.1,
                238.4,
                238.2,
                238.0,
                237.4,
                236.9,
                237.3,
                236.8,
                236.2,
                236.2,
                236.3,
                236.1,
                235.6,
                235.2,
                234.8,
                234.3,
                233.9,
                233.5,
                233.2,
                232.8,
                232.6,
                233.1,
                234.0,
                234.2,
                234.4,
                234.2,
                234.0,
                233.9,
                233.9,
                233.8,
                234.4,
                235.7,
                238.1,
                235.7,
                235.8,
                240.5,
                224.1,
                225.3,
                227.2,
                232.5,
                240.2,
                239.2,
                238.6,
                239.2,
                239.4,
                235.6,
                235.0,
                233.9,
                235.2,
                235.9,
                236.5,
                239.9,
                238.9,
                238.5,
                239.7,
                242.2,
                245.1,
            ],
            [
                237.3,
                237.3,
                238.2,
                237.7,
                238.5,
                239.3,
                238.2,
                236.9,
                235.5,
                236.1,
                237.4,
                237.9,
                237.4,
                237.0,
                237.1,
                236.0,
                235.7,
                236.5,
                236.7,
                236.4,
                235.3,
                234.4,
                234.8,
                234.9,
                235.4,
                235.8,
                236.0,
                235.9,
                236.1,
                236.4,
                236.9,
                237.4,
                237.9,
                238.3,
                238.7,
                238.8,
                238.8,
                238.6,
                238.6,
                238.6,
                238.5,
                238.4,
                238.4,
                238.2,
                238.0,
                238.1,
                237.6,
                236.9,
                236.6,
                236.4,
                236.1,
                235.7,
                235.5,
                235.2,
                234.8,
                234.4,
                234.1,
                233.5,
                232.7,
                231.8,
                231.6,
                232.0,
                232.2,
                232.9,
                233.0,
                232.7,
                232.3,
                232.0,
                231.9,
                232.1,
                232.4,
                233.0,
                233.1,
                234.2,
                236.6,
                232.8,
                232.1,
                232.7,
                233.6,
                236.6,
                235.3,
                235.1,
                236.3,
                234.4,
                234.0,
                234.8,
                235.8,
                237.1,
                238.1,
                239.5,
                240.2,
                240.6,
                239.4,
                238.3,
                238.6,
                238.1,
            ],
            [
                231.9,
                231.5,
                231.7,
                231.8,
                232.1,
                232.1,
                231.8,
                231.6,
                231.6,
                232.3,
                233.4,
                233.9,
                233.7,
                233.5,
                233.6,
                234.0,
                235.2,
                236.4,
                237.1,
                237.5,
                237.2,
                237.2,
                237.4,
                237.6,
                237.9,
                238.2,
                238.5,
                238.7,
                238.8,
                238.8,
                238.9,
                239.0,
                239.1,
                239.1,
                239.1,
                239.0,
                238.9,
                239.0,
                238.5,
                238.3,
                238.1,
                238.1,
                238.2,
                238.2,
                238.1,
                238.0,
                237.9,
                237.8,
                237.6,
                237.3,
                237.1,
                237.2,
                237.1,
                236.9,
                236.6,
                236.4,
                236.2,
                235.6,
                235.0,
                234.7,
                234.3,
                233.6,
                233.3,
                233.8,
                235.0,
                235.9,
                235.3,
                234.1,
                234.0,
                234.1,
                233.5,
                233.1,
                233.1,
                233.7,
                234.0,
                232.8,
                232.8,
                233.1,
                233.4,
                233.7,
                233.0,
                232.6,
                232.8,
                233.1,
                233.6,
                234.3,
                234.8,
                235.2,
                235.4,
                235.3,
                235.0,
                234.6,
                234.0,
                233.5,
                233.5,
                232.7,
            ],
            [
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
                235.4,
            ],
        ]
    ]


def _make_cfa_file(filename):
    n = netCDF4.Dataset(filename, "w", format="NETCDF4")

    n.Conventions = f"CF-{VN} CFA-0.6.2"
    n.comment = (
        "A CFA-netCDF file with non-standarised aggregation instructions"
    )

    n.createDimension("time", 12)
    level = n.createDimension("level", 1)
    lat = n.createDimension("lat", 73)
    lon = n.createDimension("lon", 144)
    n.createDimension("f_time", 2)
    n.createDimension("f_level", 1)
    n.createDimension("f_lat", 1)
    n.createDimension("f_lon", 1)
    n.createDimension("i", 4)
    n.createDimension("j", 2)

    lon = n.createVariable("lon", "f4", ("lon",))
    lon.standard_name = "longitude"
    lon.units = "degrees_east"

    lat = n.createVariable("lat", "f4", ("lat",))
    lat.standard_name = "latitude"
    lat.units = "degrees_north"

    time = n.createVariable("time", "f4", ("time",))
    time.standard_name = "time"
    time.units = "days since 2000-01-01"

    level = n.createVariable("level", "f4", ("level",))

    tas = n.createVariable("tas", "f4", ())
    tas.standard_name = "air_temperature"
    tas.units = "K"
    tas.aggregated_dimensions = "time level lat lon"
    tas.aggregated_data = "location: aggregation_location file: aggregation_file format: aggregation_format address: aggregation_address tracking_id: aggregation_tracking_id"

    loc = n.createVariable("aggregation_location", "i4", ("i", "j"))
    loc[0, :] = 6
    loc[1, 0] = level.size
    loc[2, 0] = lat.size
    loc[3, 0] = lon.size

    fil = n.createVariable(
        "aggregation_file", str, ("f_time", "f_level", "f_lat", "f_lon")
    )
    fil[0, 0, 0, 0] = "January-June.nc"
    fil[1, 0, 0, 0] = "July-December.nc"

    add = n.createVariable(
        "aggregation_address", str, ("f_time", "f_level", "f_lat", "f_lon")
    )
    add[0, 0, 0, 0] = "tas0"
    add[1, 0, 0, 0] = "tas1"

    fmt = n.createVariable("aggregation_format", str, ())
    fmt[()] = "nc"

    tid = n.createVariable(
        "aggregation_tracking_id", str, ("f_time", "f_level", "f_lat", "f_lon")
    )
    tid[0, 0, 0, 0] = "tracking_id0"
    tid[1, 0, 0, 0] = "tracking_id1"

    n.close()

    return filename


broken_bounds_file = _make_broken_bounds_cdl("broken_bounds.cdl")

regrid_file = _make_regrid_file("regrid.nc")

cfa_file = _make_cfa_file("cfa.nc")

if __name__ == "__main__":
    print("Run date:", datetime.datetime.now())
    cf.environment()
    print()
    unittest.main(verbosity=2)
