import datetime
import faulthandler
import os
import unittest

faulthandler.enable()  # to debug seg faults and timeouts

import numpy as np

import cf

esmpy_imported = True
try:
    import esmpy  # noqa: F401
except ImportError:
    esmpy_imported = False


methods = (
    "linear",
    "nearest_stod",
    "patch",
)


# Set numerical comparison tolerances
atol = 2e-12
rtol = 0


def esmpy_regrid(coord_sys, method, src, dst, **kwargs):
    """Helper function that regrids one dimension of Field data using
    pure esmpy.

    Used to verify `cf.Field.regridc`

    :Returns:

        Regridded numpy masked array.

    """
    meshloc = {
        "face": esmpy.MeshLoc.ELEMENT,
        "node": esmpy.MeshLoc.NODE,
    }

    esmpy_regrid = cf.regrid.regrid(
        coord_sys,
        src,
        dst,
        method,
        return_esmpy_regrid_operator=True,
        **kwargs
    )

    src_meshloc = None
    dst_meshloc = None

    domain_topology = src.domain_topology(default=None)
    if domain_topology is not None:
        src_meshloc = meshloc[domain_topology.get_cell()]

    domain_topology = dst.domain_topology(default=None)
    if domain_topology is not None:
        dst_meshloc = meshloc[domain_topology.get_cell()]

    src_field = esmpy.Field(
        esmpy_regrid.srcfield.grid, meshloc=src_meshloc, name="src"
    )
    dst_field = esmpy.Field(
        esmpy_regrid.dstfield.grid, meshloc=dst_meshloc, name="dst"
    )

    fill_value = 1e20

    array = np.squeeze(src.array)
    if coord_sys == "spherical":
        array = array.transpose()

    src_field.data[...] = np.ma.MaskedArray(array, copy=False).filled(
        fill_value
    )
    dst_field.data[...] = fill_value

    esmpy_regrid(src_field, dst_field, zero_region=esmpy.Region.SELECT)

    out = dst_field.data

    return np.ma.MaskedArray(out.copy(), mask=(out == fill_value))


class RegridFeatureTypeTest(unittest.TestCase):
    # Get the test source and destination fields
    src_grid_file = os.path.join(
        os.path.dirname(os.path.abspath(__file__)), "regrid_xyz.nc"
    )
    src_mesh_file = os.path.join(
        os.path.dirname(os.path.abspath(__file__)), "ugrid_global_1.nc"
    )
    dst_featureType_file = os.path.join(
        os.path.dirname(os.path.abspath(__file__)), "dsg_trajectory.nc"
    )
    src_grid = cf.read(src_grid_file)[0]
    src_mesh = cf.read(src_mesh_file)[0]
    dst_featureType = cf.read(dst_featureType_file)[0]

    def setUp(self):
        """Preparations called immediately before each test method."""
        # Disable log messages to silence expected warnings
        cf.log_level("DISABLE")
        # Note: to enable all messages for given methods, lines or calls (those
        # without a 'verbose' option to do the same) e.g. to debug them, wrap
        # them (for methods, start-to-end internally) as follows:
        # cfdm.log_level('DEBUG')
        # < ... test code ... >
        # cfdm.log_level('DISABLE')

    @unittest.skipUnless(esmpy_imported, "Requires esmpy/ESMF package.")
    def test_Field_regrid_grid_to_featureType_3d(self):
        self.assertFalse(cf.regrid_logging())

        dst = self.dst_featureType.copy()
        src = self.src_grid.copy()

        # Mask some destination grid points
        dst[20:25] = cf.masked

        coord_sys = "spherical"

        for src_masked in (False, True):
            if src_masked:
                src = src.copy()
                src[0, 2, 2, 3] = cf.masked

            # Loop over whether or not to use the destination grid
            # masked points
            for use_dst_mask in (False, True):
                kwargs = {
                    "use_dst_mask": use_dst_mask,
                    "z": "air_pressure",
                    "ln_z": True,
                }
                for method in methods:
                    x = src.regrids(dst, method=method, **kwargs)
                    a = x.array

                    y = esmpy_regrid(coord_sys, method, src, dst, **kwargs)

                    self.assertEqual(y.size, a.size)
                    self.assertTrue(np.allclose(y, a, atol=atol, rtol=rtol))

                    if isinstance(a, np.ma.MaskedArray):
                        self.assertTrue((y.mask == a.mask).all())
                    else:
                        self.assertFalse(y.mask.any())

    @unittest.skipUnless(esmpy_imported, "Requires esmpy/ESMF package.")
    def test_Field_regrid_featureType_to_grid_2d(self):
        self.assertFalse(cf.regrid_logging())

        # Create some nice data
        src = self.dst_featureType.copy()
        src.del_construct("cellmethod0")
        src = src[:12]
        src[...] = 273 + np.arange(12)
        x = src.coord("X")
        x[...] = [4, 6, 9, 11, 14, 16, 4, 6, 9, 11, 14, 16]
        y = src.coord("Y")
        y[...] = [41, 41, 31, 31, 21, 21, 39, 39, 29, 29, 19, 19]

        dst = self.src_grid.copy()
        x = dst.coord("X")
        x[...] = [5, 10, 15, 20]
        y = dst.coord("Y")
        y[...] = [10, 20, 30, 40]

        # Mask some destination grid points
        dst[0, 0, 1, 2] = cf.masked

        # Expected destination regridded values
        y0 = np.ma.array(
            [[0, 0, 0, 0], [0, 0, 1122, 0], [0, 1114, 0, 0], [1106, 0, 0, 0]],
            mask=[
                [True, True, True, True],
                [True, True, False, True],
                [True, False, True, True],
                [False, True, True, True],
            ],
        )

        for src_masked in (False, True):
            y = y0.copy()
            if src_masked:
                src = src.copy()
                src[6:8] = cf.masked
                # This following element should be smaller, because it
                # now only has two source cells contributing to it,
                # rather than four.
                y[3, 0] = 547

            # Loop over whether or not to use the destination grid
            # masked points
            for use_dst_mask in (False, True):
                if use_dst_mask:
                    y = y.copy()
                    y[1, 2] = np.ma.masked

                kwargs = {"use_dst_mask": use_dst_mask}
                method = "nearest_dtos"
                for return_operator in (False, True):
                    if return_operator:
                        r = src.regrids(
                            dst, method=method, return_operator=True, **kwargs
                        )
                        x = src.regrids(r)
                    else:
                        x = src.regrids(dst, method=method, **kwargs)

                    a = x.array

                    self.assertEqual(y.size, a.size)
                    self.assertTrue(np.allclose(y, a, atol=atol, rtol=rtol))

                    if isinstance(a, np.ma.MaskedArray):
                        self.assertTrue((y.mask == a.mask).all())
                    else:
                        self.assertFalse(y.mask.any())

    @unittest.skipUnless(esmpy_imported, "Requires esmpy/ESMF package.")
    def test_Field_regrid_grid_to_featureType_2d(self):
        self.assertFalse(cf.regrid_logging())

        dst = self.dst_featureType.copy()
        src = self.src_grid.copy()
        src = src[0, 0]

        # Mask some destination grid points
        dst[20:25] = cf.masked

        coord_sys = "spherical"

        for src_masked in (False, True):
            if src_masked:
                src = src.copy()
                src[0, 0, 2, 3] = cf.masked

            # Loop over whether or not to use the destination grid
            # masked points
            for use_dst_mask in (False, True):
                kwargs = {"use_dst_mask": use_dst_mask}
                for method in methods:
                    x = src.regrids(dst, method=method, **kwargs)
                    a = x.array

                    y = esmpy_regrid(coord_sys, method, src, dst, **kwargs)
                    self.assertEqual(y.size, a.size)
                    self.assertTrue(np.allclose(y, a, atol=atol, rtol=rtol))

                    if isinstance(a, np.ma.MaskedArray):
                        self.assertTrue((y.mask == a.mask).all())
                    else:
                        self.assertFalse(y.mask.any())

    @unittest.skipUnless(esmpy_imported, "Requires esmpy/ESMF package.")
    def test_Field_regrid_mesh_to_featureType_2d(self):
        self.assertFalse(cf.regrid_logging())

        dst = self.dst_featureType.copy()
        src = self.src_mesh.copy()

        # Mask some destination grid points
        dst[20:25] = cf.masked

        coord_sys = "spherical"

        for src_masked in (False, True):
            if src_masked:
                src = src.copy()
                src[20:30] = cf.masked

            # Loop over whether or not to use the destination grid
            # masked points
            for use_dst_mask in (False, True):
                kwargs = {"use_dst_mask": use_dst_mask}
                for method in methods:
                    x = src.regrids(dst, method=method, **kwargs)
                    a = x.array

                    y = esmpy_regrid(coord_sys, method, src, dst, **kwargs)

                    self.assertEqual(y.size, a.size)
                    self.assertTrue(np.allclose(y, a, atol=atol, rtol=rtol))

                    if isinstance(a, np.ma.MaskedArray):
                        self.assertTrue((y.mask == a.mask).all())
                    else:
                        self.assertFalse(y.mask.any())

    @unittest.skipUnless(esmpy_imported, "Requires esmpy/ESMF package.")
    def test_Field_regrid_featureType_cartesian(self):
        self.assertFalse(cf.regrid_logging())

        # Cartesian regridding involving DSG featureTypes is not
        # currently supported
        src = self.src_grid
        dst = self.dst_featureType
        with self.assertRaises(ValueError):
            src.regridc(dst, method="linear")

        src, dst = dst, src
        with self.assertRaises(ValueError):
            src.regridc(dst, method="linear")

    def test_Field_regrid_featureType_bad_methods(self):
        dst = self.dst_featureType.copy()
        src = self.src_grid.copy()

        for method in ("conservative", "conservative_2nd"):
            with self.assertRaises(ValueError):
                src.regrids(dst, method=method)


if __name__ == "__main__":
    print("Run date:", datetime.datetime.now())
    cf.environment()
    print("")
    unittest.main(verbosity=2)
