"""Pydantic model for mapping a raw file to a device."""

from datetime import datetime
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field


class RawFileDeviceMappingBase(BaseModel):
    """Base model for raw file device mapping.

    Attributes:
        raw_file_id: Foreign key reference to RawFile.
        device_id: Foreign key reference to Device.
        source_stream_identifier: Identifier for the source stream.
        input_srid: Spatial reference ID for the input data.
        input_timezone: Timezone for the input data.
    """

    raw_file_id: UUID
    device_id: UUID
    source_stream_identifier: str
    input_srid: int = Field(ge=2000, le=900913)
    input_timezone: str = Field(pattern=r"^UTM([+-]\d+)?$")


class RawFileDeviceMappingCreate(RawFileDeviceMappingBase):
    """Model for creating raw file device mapping data."""


class RawFileDeviceMappingUpdate(RawFileDeviceMappingBase):
    """Model for updating raw file device mapping data.

    Attributes:
        mapping_id: Unique identifier for the mapping.
        created_date: Date and time when the mapping was created.
        updated_date: Date and time when the mapping was last updated.
        updated_by: User who last updated the mapping.

    """


class RawFileDeviceMapping(RawFileDeviceMappingUpdate):
    """Model for raw file device mapping data with database fields.

    Attributes:
        created_date: Date and time when the mapping was created.
        updated_date: Date and time when the mapping was last updated.
        updated_by: User who last updated the mapping.
    """

    mapping_id: UUID
    created_date: datetime
    updated_date: datetime
    updated_by: str

    model_config = ConfigDict(from_attributes=True)


class RawFileDeviceMappingFilter(BaseModel):
    """Model for filtering raw file device mappings.

    Attributes:
        raw_file_id: Raw file ID filter (required).
        device_id: Device ID filter (required).
    """

    raw_file_id: UUID
    device_id: UUID
