# engine/_processors_cy.py
# Copyright (C) 2010-2024 the SQLAlchemy authors and contributors
# <see AUTHORS file>
#
# This module is part of SQLAlchemy and is released under
# the MIT License: https://www.opensource.org/licenses/mit-license.php
# mypy: disable-error-code="misc"
from __future__ import annotations

from datetime import date as date_cls
from datetime import datetime as datetime_cls
from datetime import time as time_cls
from typing import Any
from typing import Optional

# START GENERATED CYTHON IMPORT
# This section is automatically generated by the script tools/cython_imports.py
try:
    # NOTE: the cython compiler needs this "import cython" in the file, it
    # can't be only "from sqlalchemy.util import cython" with the fallback
    # in that module
    import cython
except ModuleNotFoundError:
    from sqlalchemy.util import cython


def _is_compiled() -> bool:
    """Utility function to indicate if this module is compiled or not."""
    return cython.compiled  # type: ignore[no-any-return]


# END GENERATED CYTHON IMPORT


@cython.annotation_typing(False)
def int_to_boolean(value: Any) -> Optional[bool]:
    if value is None:
        return None
    return True if value else False


@cython.annotation_typing(False)
def to_str(value: Any) -> Optional[str]:
    if value is None:
        return None
    return str(value)


@cython.annotation_typing(False)
def to_float(value: Any) -> Optional[float]:
    if value is None:
        return None
    return float(value)


@cython.annotation_typing(False)
def str_to_datetime(value: Optional[str]) -> Optional[datetime_cls]:
    if value is None:
        return None
    return datetime_cls.fromisoformat(value)


@cython.annotation_typing(False)
def str_to_time(value: Optional[str]) -> Optional[time_cls]:
    if value is None:
        return None
    return time_cls.fromisoformat(value)


@cython.annotation_typing(False)
def str_to_date(value: Optional[str]) -> Optional[date_cls]:
    if value is None:
        return None
    return date_cls.fromisoformat(value)


@cython.cclass
class to_decimal_processor_factory:
    type_: type
    format_: str

    __slots__ = ("type_", "format_")

    def __init__(self, type_: type, scale: int):
        self.type_ = type_
        self.format_ = f"%.{scale}f"

    def __call__(self, value: Optional[Any]) -> object:
        if value is None:
            return None
        else:
            return self.type_(self.format_ % value)
