import gradio as gr
import huggingface_hub
from ._theme import seafoam
from ..inference import InferenceSession
from typing import List, Literal, Optional
from gradio.helpers import log_message

from ._base import CHAT_MODE
from ..llama_cpp import LlamaCPParams, LlamaCPPGenerationConfig
from ..interactors._base import BaseInteract

js = """function () {
  gradioURL = window.location.href
  if (!gradioURL.endsWith('?__theme=dark')) {
    window.location.replace(gradioURL + '?__theme=dark');
  }
}"""


class GradioUserInference:
    """
    The `GradioUserInference` class is a Python class that represents a user's interaction with a language
    model. It has an `__init__` method that initializes the class with an `inference_session` object,
    `max_tokens`, and `max_length` parameters. The `inference_session` object is used to perform
    inference with the language model. The `max_tokens` parameter sets the maximum number of tokens that
    can be used in a single query, and the `max_length` parameter sets the maximum length of a sentence.
    """

    def __init__(
            self,
            interactor: BaseInteract,
            use_prefix_for_interactor: bool = True,
            inference_session: Optional[InferenceSession] = None,
            llama_param: Optional[LlamaCPParams] = None,
            generation_config: Optional[LlamaCPPGenerationConfig] = None,
            examples: Optional[list[str]] = None
    ):

        if examples is None:
            examples = [
                "Hello world",
                "What's up?",
                "Can you code?"
            ]
        if generation_config is None and inference_session is not None:
            generation_config = inference_session.generation_config

        if inference_session is None and llama_param is not None:
            inference_session = InferenceSession.create(
                llama_params=llama_param,
                generation_config=generation_config
            )

        self.interactor = interactor
        self.use_prefix_for_interactor = use_prefix_for_interactor
        self.inference_session = inference_session
        self.llama_param = llama_param
        self.generation_config = generation_config
        self.examples = examples

    def sample(
            self,
            prompt: str,
            history: List[List[str]],
            system_prompt: str,
            mode: CHAT_MODE = CHAT_MODE[-1],
            max_tokens: int = 4096,
            temperature: float = 0.8,
            top_p: float = 0.9,
            top_k: int = 50
    ):
        """
        The sample function is the main entry point for a user to interact with the model.
        It takes in a prompt, which can be any string, and returns an iterator over
        strings that are generated by the model.
        The sample function also takes in some optional arguments:

        :param self: Refer to the current object
        :param prompt: str: Pass in the text that you want to generate a response for
        :param history: List[List[str]]: Keep track of the conversation history
        :param system_prompt: str: the model system prompt.
        :param mode: str: represent the mode that model inference be used in (e.g chat or instruction)
        :param max_tokens: int: Limit the number of tokens in a response
        :param temperature: float: Control the randomness of the generated text
        :param top_p: float: Control the probability of sampling from the top k tokens
        :param top_k: int: Control the number of candidates that are considered for each token
        :return: A generator that yields the next token in the sequence
        """

        assert mode in CHAT_MODE, "Requested Mode is not in Available Modes"
        if mode == "Instruction":
            history = []
        if self.inference_session is not None:
            string = self.interactor.format_message(
                prompt=prompt,
                history=history,
                system_message=None if system_prompt == "" else system_prompt,
                prefix=self.interactor.get_prefix_prompt() if self.use_prefix_for_interactor else None,
            )
            history.append([prompt, ""])
            total_response = ""
            for response in self.inference_session.generate(
                    prompt=string,
                    top_k=top_k,
                    top_p=top_p,
                    max_new_tokens=max_tokens,
                    temperature=temperature,
            ):
                total_response += response.predictions.text
                history[-1][-1] = total_response
                yield '', history
        else:
            return [
                [prompt, "Model is not loaded !"]
            ]

    def _load_new_model(
            self,
            repo_id,
            filename,
            hf_token,
            n_gpu_layers,
            main_gpu,
            vocab_only,
            use_mmap,
            use_mlock,
            n_batch,
            n_threads,
            n_threads_batch,
            rope_freq_base,
            rope_freq_scale,
            yarn_ext_factor,
            yarn_attn_factor,
            yarn_beta_fast,
            yarn_beta_slow,
            yarn_orig_ctx,
            mul_mat_q,
            logits_all,
            embedding,
            offload_kqv,
            last_n_tokens_size,
            lora_base,
            lora_scale,
            lora_path,
            numa,
            verbose,
            max_length
    ):

        self.inference_session = None
        log_message(
            f"Model is being Loaded", level="warning"
        )
        self.llama_params = LlamaCPParams(
            model_path=huggingface_hub.hf_hub_download(
                filename=filename,
                repo_id=repo_id,
                hf_token=hf_token
            ),
            num_context=max_length,
            offload_kqv=offload_kqv,
            num_threads=n_threads,
            verbose=verbose,
            lora_path=lora_path,
            main_gpu=main_gpu,
            use_mmap=use_mmap,
            use_mlock=use_mlock,
            lora_base=lora_base,
            embedding=embedding,
            num_batch=n_batch,
            logits_all=logits_all,
            vocab_only=vocab_only,
            lora_scale=lora_scale,
            mul_mat_q=mul_mat_q,
            num_gpu_layers=n_gpu_layers,
            rope_freq_base=rope_freq_base,
            yarn_beta_fast=yarn_beta_fast,
            yarn_beta_slow=yarn_beta_slow,
            yarn_orig_ctx=yarn_orig_ctx,
            rope_freq_scale=rope_freq_scale,
            yarn_attn_factor=yarn_attn_factor,
            num_threads_batch=n_threads_batch,
            yarn_ext_factor=yarn_ext_factor,
            last_num_tokens_size=last_n_tokens_size,
        )
        self.inference_session = InferenceSession.create(
            llama_params=self.llama_params,
            generation_config=self.generation_config
        )

        log_message(
            f"Model is Loaded", level="info"
        )

    def model_tab_interface_components(self):
        """
        The function `model_tab_interface_components` creates a user interface for selecting and loading
        a model with various parameters.
        """
        with gr.Row():
            with gr.Column(scale=3):
                with gr.Row():
                    n_gpu_layers = gr.Number(value=0, label="N GPU Layers")
                    main_gpu = gr.Number(value=0, label="Main GPU ID")
                    max_length = gr.Number(value=2048, label="Max Length")
                    n_batch = gr.Number(value=512, label="number batch")
                    n_threads = gr.Number(value=None, label="number threads")
                    n_threads_batch = gr.Number(value=None, label="N threads batch")
                    rope_freq_base = gr.Number(value=0.0, label="rope freq base")
                    rope_freq_scale = gr.Number(value=0.0, label="rope freq scale")
                    yarn_ext_factor = gr.Number(value=-1.0, label="yarn ext factor")
                    yarn_attn_factor = gr.Number(value=1.0, label="yarn attn factor")
                    yarn_beta_fast = gr.Number(value=32.0, label="yarn beta fast")
                    yarn_beta_slow = gr.Number(value=1.0, label="yarn beta slow")
                    yarn_orig_ctx = gr.Number(value=0, label="yarn orig ctx")
                    last_n_tokens_size = gr.Number(value=64, label="LN tokens size")
                    lora_scale = gr.Number(value=1.0, label="lora scale")
                with gr.Row():
                    lora_base = gr.Textbox(value=None, label="lora base")
                    lora_path = gr.Textbox(value=None, label="lora path")
                with gr.Row():
                    use_mlock = gr.Checkbox(value=False, label="use mlock")
                    numa = gr.Checkbox(value=False, label="numa")
                    verbose = gr.Checkbox(value=True, label="verbose")
                    vocab_only = gr.Checkbox(value=False, label="Vocab Only")
                    use_mmap = gr.Checkbox(value=True, label="use mmap")
                    mul_mat_q = gr.Checkbox(value=True, label="mul mat q")
                    logits_all = gr.Checkbox(value=False, label="logits all")
                    embedding = gr.Checkbox(value=False, label="embedding")
                    offload_kqv = gr.Checkbox(value=False, label="offload kqv")
            with gr.Column(scale=2):
                repo_id = gr.Text(
                    value="erfanzar/LinguaMatic-2.7B-GGUF",
                    label="Huggingface Repo ID",
                )
                filename = gr.Text(
                    value="LinguaMatic-2.7B-GGUF.Q4_K_M.gguf",
                    label="Checkpoint Path in given repo"
                )
                hf_token = gr.Text(
                    label="HuggingFace Access Token",
                    placeholder="Leave Empty in case of using Public Models"
                )
                load_button = gr.Button(
                    value="Load Model",
                    variant="primary"
                )

        load_button.click(
            self._load_new_model,
            inputs=[
                repo_id,
                filename,
                hf_token,
                n_gpu_layers,
                main_gpu,
                vocab_only,
                use_mmap,
                use_mlock,
                n_batch,
                n_threads,
                n_threads_batch,
                rope_freq_base,
                rope_freq_scale,
                yarn_ext_factor,
                yarn_attn_factor,
                yarn_beta_fast,
                yarn_beta_slow,
                yarn_orig_ctx,
                mul_mat_q,
                logits_all,
                embedding,
                offload_kqv,
                last_n_tokens_size,
                lora_base,
                lora_scale,
                lora_path,
                numa,
                verbose,
                max_length
            ]
        )

    def chat_interface_components(self):
        """
        The function `chat_interface_components` creates the components for a chat interface, including
        a chat history, message box, buttons for submitting, stopping, and clearing the conversation,
        and sliders for advanced options.
        """
        with gr.Column("100%"):
            gr.Markdown(
                "# <h1><center style='color:white;'>Powered by "
                "[EasyDeL](https://github.com/erfanzar/EasyDel)</center></h1>",
            )
            history = gr.Chatbot(
                elem_id="EasyDel",
                label="EasyDel",
                container=True,
                height="65vh",
            )
            prompt = gr.Textbox(
                show_label=False, placeholder='Enter Your Prompt Here.', container=False
            )
            with gr.Row():
                submit = gr.Button(
                    value="Run",
                    variant="primary"
                )
                stop = gr.Button(
                    value='Stop'
                )
                clear = gr.Button(
                    value='Clear Conversation'
                )
            with gr.Accordion(open=False, label="Advanced Options"):
                system_prompt = gr.Textbox(
                    value="",
                    show_label=True,
                    label="System Prompt",
                    placeholder='System Prompt',
                    container=False
                )
                max_tokens = gr.Slider(
                    value=self.generation_config.max_new_tokens,
                    maximum=10000,
                    minimum=0,
                    label='Max Tokens',
                    step=1
                )
                temperature = gr.Slider(
                    value=0.8,
                    maximum=1,
                    minimum=0.1,
                    label='Temperature',
                    step=0.01
                )
                top_p = gr.Slider(
                    value=0.9,
                    maximum=1,
                    minimum=0.1,
                    label='Top P',
                    step=0.01
                )
                top_k = gr.Slider(
                    value=50,
                    maximum=100,
                    minimum=1,
                    label='Top K',
                    step=1
                )
                mode = gr.Dropdown(
                    choices=CHAT_MODE,
                    value=CHAT_MODE[1],
                    label="Mode",
                    multiselect=False
                )

        inputs = [
            prompt,
            history,
            system_prompt,
            mode,
            max_tokens,
            temperature,
            top_p,
            top_k
        ]

        clear.click(fn=lambda: [], outputs=[history])
        sub_event = submit.click(
            fn=self.sample, inputs=inputs, outputs=[prompt, history]
        )
        txt_event = prompt.submit(
            fn=self.sample, inputs=inputs, outputs=[prompt, history]
        )

        stop.click(fn=None, inputs=None, outputs=None,
                   cancels=[txt_event, sub_event])

    def build_model_tab_interface(self) -> gr.Blocks:
        """
        The function "build_model_tab_interface" returns a gr.Blocks object that contains the components
        of a model tab interface.
        :return: a gr.Blocks object.
        """

        with gr.Blocks(
                theme=seafoam,
                title="Model Tab",

        ) as block:
            self.model_tab_interface_components()
        return block

    def build_chat_interface(self) -> gr.Blocks:
        """
        The build function is the main entry point for your app.
        It should return a single gr.Block instance that will be displayed in the browser.

        :param self: Make the class methods work with an instance of the class
        :return: A block, which is then queued
        """
        with gr.Blocks(
                theme=seafoam,
                title="Chat",
                css="footer {visibility: hidden}"
        ) as block:
            self.chat_interface_components()
        block.queue()
        return block

    def build_inference(self) -> gr.Blocks:
        """
        The function "build_inference" returns a gr.Blocks object that contains two tabs, one for model
        interface components and one for chat interface components.
        :return: a gr.Blocks object.
        """
        with gr.Blocks(
                theme=seafoam,
                css="footer {visibility: hidden}"
        ) as block:
            with gr.Tab("Model", ):
                self.model_tab_interface_components()
            with gr.Tab("Chat"):
                self.chat_interface_components()
        return block
