"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EdgeFunction = void 0;
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const edge_function_provider_1 = require("./edge-function-provider");
class EdgeFunction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        const { region, node } = this.stack;
        const { role, code, name = node.uniqueId, parameterName = `/cloudcomponents/edge-lambda/${name}`, } = props;
        const lambdaAtEdgeStack = region !== 'us-east-1'
            ? this.getOrCreateCrossRegionSupportStack()
            : this.stack;
        this.role = role !== null && role !== void 0 ? role : new aws_iam_1.Role(lambdaAtEdgeStack, 'LambdaAtEdgeExecutionRole', {
            assumedBy: new aws_iam_1.CompositePrincipal(new aws_iam_1.ServicePrincipal('edgelambda.amazonaws.com'), new aws_iam_1.ServicePrincipal('lambda.amazonaws.com')),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const lambdaFunction = new aws_lambda_1.Function(lambdaAtEdgeStack, name, {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code,
            role: this.role,
        });
        this.parameter = new aws_ssm_1.StringParameter(lambdaAtEdgeStack, 'StringParameter', {
            parameterName,
            description: 'Parameter stored for cross region Lambda@Edge',
            stringValue: lambdaFunction.functionArn,
        });
    }
    retrieveEdgeFunction(scope) {
        const { lambdaFunction } = new edge_function_provider_1.EdgeFunctionProvider(scope, 'Retrieve', {
            parameter: this.parameter,
        });
        return lambdaFunction;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK app');
        }
        return app;
    }
    getOrCreateCrossRegionSupportStack() {
        const { account, stackName } = this.stack;
        const stackId = `lambda-at-edge-support-stack`;
        const app = this.requireApp();
        let supportStack = app.node.tryFindChild(stackId);
        if (!supportStack) {
            supportStack = new core_1.Stack(app, stackId, {
                stackName: `${stackName}-support-lambda-at-edge`,
                env: {
                    account,
                    region: 'us-east-1',
                },
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
            // the stack containing the edge lambdas must be deployed before
            this.stack.addDependency(supportStack);
        }
        return supportStack;
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer
                    .cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
}
exports.EdgeFunction = EdgeFunction;
//# sourceMappingURL=data:application/json;base64,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