"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpHeaders = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const edge_function_1 = require("./edge-function");
const with_configuration_1 = require("./with-configuration");
// Blacklisted headers aren't exposed and can't be added by Lambda@Edge functions:
// https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-requirements-limits.html#lambda-blacklisted-headers
const BLACKLISTED_HEADERS = [
    /^connection$/i,
    /^expect$/i,
    /^keep-alive$/i,
    /^proxy-authenticate$/i,
    /^proxy-authorization$/i,
    /^proxy-connection$/i,
    /^trailer$/i,
    /^upgrade$/i,
    /^x-accel-buffering$/i,
    /^x-accel-charset$/i,
    /^x-accel-limit-rate$/i,
    /^x-accel-redirect$/i,
    /^X-Amz-Cf-.*/i,
    /^X-Amzn-.*/i,
    /^X-Cache.*/i,
    /^X-Edge-.*/i,
    /^X-Forwarded-Proto.*/i,
    /^X-Real-IP$/i,
];
class HttpHeaders extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        Object.keys(props.httpHeaders).forEach((header) => {
            if (BLACKLISTED_HEADERS.some((blheader) => blheader.test(header))) {
                throw new Error(`HttpHeader ${header} is blacklisted and can't be added by Lambda@Edge functions`);
            }
        });
        const edgeFunction = new edge_function_1.EdgeFunction(this, 'EdgeFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'lambdas', 'http-headers')),
        });
        const lambdaWithConfig = new with_configuration_1.WithConfiguration(this, 'WithConfiguration', {
            function: edgeFunction.retrieveEdgeFunction(this),
            configuration: {
                logLevel: (_a = props.logLevel) !== null && _a !== void 0 ? _a : with_configuration_1.LogLevel.WARN,
                httpHeaders: props.httpHeaders,
            },
        });
        this.eventType = aws_cloudfront_1.LambdaEdgeEventType.ORIGIN_RESPONSE;
        this.lambdaFunction = lambdaWithConfig.lambdaFunction;
    }
}
exports.HttpHeaders = HttpHeaders;
//# sourceMappingURL=data:application/json;base64,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