"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EdgeFunction = void 0;
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const edge_function_provider_1 = require("./edge-function-provider");
class EdgeFunction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        const { region, node } = this.stack;
        const { role, code, name, parameterName = `/cloudcomponents/edge-lambda/${node.uniqueId}/${name}`, } = props;
        const lambdaAtEdgeStack = region !== 'us-east-1'
            ? this.getOrCreateCrossRegionSupportStack()
            : this.stack;
        this.role = role !== null && role !== void 0 ? role : new aws_iam_1.Role(lambdaAtEdgeStack, 'LambdaAtEdgeExecutionRole', {
            assumedBy: new aws_iam_1.CompositePrincipal(new aws_iam_1.ServicePrincipal('edgelambda.amazonaws.com'), new aws_iam_1.ServicePrincipal('lambda.amazonaws.com')),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const lambdaFunction = new aws_lambda_1.Function(lambdaAtEdgeStack, name, {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code,
            role: this.role,
        });
        this.parameter = new aws_ssm_1.StringParameter(lambdaAtEdgeStack, 'StringParameter', {
            parameterName,
            description: 'Parameter stored for cross region Lambda@Edge',
            stringValue: lambdaFunction.functionArn,
        });
    }
    retrieveEdgeFunction(scope) {
        const { lambdaFunction } = new edge_function_provider_1.EdgeFunctionProvider(scope, 'Retrieve', {
            parameter: this.parameter,
        });
        return lambdaFunction;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Stacks which uses edge functions must be part of a CDK app');
        }
        return app;
    }
    getOrCreateCrossRegionSupportStack() {
        const { account, stackName } = this.stack;
        const stackId = `lambda-at-edge-support-stack`;
        const app = this.requireApp();
        let supportStack = app.node.tryFindChild(stackId);
        if (!supportStack) {
            supportStack = new core_1.Stack(app, stackId, {
                stackName: `${stackName}-support-lambda-at-edge`,
                env: {
                    account,
                    region: 'us-east-1',
                },
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
            // the stack containing the edge lambdas must be deployed before
            this.stack.addDependency(supportStack);
        }
        return supportStack;
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer
                    .cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
}
exports.EdgeFunction = EdgeFunction;
//# sourceMappingURL=data:application/json;base64,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