"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpHeaders = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const edge_function_1 = require("./edge-function");
const with_configuration_1 = require("./with-configuration");
// Blacklisted headers aren't exposed and can't be added by Lambda@Edge functions:
// https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-requirements-limits.html#lambda-blacklisted-headers
const BLACKLISTED_HEADERS = [
    /^connection$/i,
    /^expect$/i,
    /^keep-alive$/i,
    /^proxy-authenticate$/i,
    /^proxy-authorization$/i,
    /^proxy-connection$/i,
    /^trailer$/i,
    /^upgrade$/i,
    /^x-accel-buffering$/i,
    /^x-accel-charset$/i,
    /^x-accel-limit-rate$/i,
    /^x-accel-redirect$/i,
    /^X-Amz-Cf-.*/i,
    /^X-Amzn-.*/i,
    /^X-Cache.*/i,
    /^X-Edge-.*/i,
    /^X-Forwarded-Proto.*/i,
    /^X-Real-IP$/i,
];
class HttpHeaders extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        Object.keys(props.httpHeaders).forEach((header) => {
            if (BLACKLISTED_HEADERS.some((blheader) => blheader.test(header))) {
                throw new Error(`HttpHeader ${header} is blacklisted and can't be added by Lambda@Edge functions`);
            }
        });
        const edgeFunction = new edge_function_1.EdgeFunction(this, 'EdgeFunction', {
            name: 'http-headers',
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'lambdas', 'http-headers')),
        });
        const lambdaWithConfig = new with_configuration_1.WithConfiguration(this, 'WithConfiguration', {
            function: edgeFunction.retrieveEdgeFunction(this),
            configuration: {
                logLevel: (_a = props.logLevel) !== null && _a !== void 0 ? _a : with_configuration_1.LogLevel.WARN,
                httpHeaders: props.httpHeaders,
            },
        });
        this.eventType = aws_cloudfront_1.LambdaEdgeEventType.ORIGIN_RESPONSE;
        this.lambdaFunction = lambdaWithConfig.lambdaFunction;
    }
}
exports.HttpHeaders = HttpHeaders;
//# sourceMappingURL=data:application/json;base64,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