"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseEdgeConstruct = void 0;
const core_1 = require("@aws-cdk/core");
class BaseEdgeConstruct extends core_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        this.edgeStack = this.stack.region !== 'us-east-1' ? this.getOrCreateCrossRegionSupportStack() : this.stack;
    }
    getOrCreateCrossRegionSupportStack() {
        const { account, stackName } = this.stack;
        const stackId = `lambda-at-edge-support-stack`;
        const app = this.requireApp();
        let supportStack = app.node.tryFindChild(stackId);
        if (!supportStack) {
            supportStack = new core_1.Stack(app, stackId, {
                stackName: `${stackName}-support-lambda-at-edge`,
                env: {
                    account,
                    region: 'us-east-1',
                },
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
            // the stack containing the edge lambdas must be deployed before
            this.stack.addDependency(supportStack);
        }
        return supportStack;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Stacks which uses edge constructs must be part of a CDK app');
        }
        return app;
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
}
exports.BaseEdgeConstruct = BaseEdgeConstruct;
//# sourceMappingURL=data:application/json;base64,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