"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PullRequestCheck = void 0;
const path = require("path");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
/**
 * Represents a reference to a PullRequestCheck.
 */
class PullRequestCheck extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { repository, buildSpec, buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_4_0, computeType = buildImage.defaultComputeType, privileged = false, updateApprovalState = true, postComment = true, projectName = `${repository.repositoryName}-pull-request`, role, } = props;
        this.pullRequestProject = new aws_codebuild_1.Project(this, 'PullRequestProject', {
            projectName,
            source: aws_codebuild_1.Source.codeCommit({
                repository,
            }),
            environment: {
                buildImage,
                computeType,
                privileged,
            },
            buildSpec,
            role,
        });
        if (updateApprovalState || postComment) {
            const codeBuildResultFunction = new aws_lambda_1.Function(this, 'CodeBuildResultFunction', {
                runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                code: aws_lambda_1.Code.asset(path.join(__dirname, 'lambdas', 'code-build-result')),
                handler: 'index.handler',
                environment: {
                    UPDATE_APPROVAL_STATE: updateApprovalState ? 'TRUE' : 'FALSE',
                    POST_COMMENT: postComment ? 'TRUE' : 'FALSE',
                },
            });
            codeBuildResultFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: [repository.repositoryArn],
                actions: ['codecommit:PostCommentForPullRequest', 'codecommit:UpdatePullRequestApprovalState'],
            }));
            this.pullRequestProject.onStateChange('PullRequestValidationRule', {
                target: new aws_events_targets_1.LambdaFunction(codeBuildResultFunction),
            });
        }
        const rule = repository.onPullRequestStateChange(`${this.node.uniqueId}Rule`, {
            eventPattern: {
                detail: {
                    event: ['pullRequestSourceBranchUpdated', 'pullRequestCreated'],
                },
            },
        });
        rule.addTarget(new aws_events_targets_1.CodeBuildProject(this.pullRequestProject, {
            event: aws_events_1.RuleTargetInput.fromObject({
                sourceVersion: aws_events_1.EventField.fromPath('$.detail.sourceCommit'),
                artifactsOverride: { type: 'NO_ARTIFACTS' },
                environmentVariablesOverride: [
                    {
                        name: 'pullRequestId',
                        value: aws_events_1.EventField.fromPath('$.detail.pullRequestId'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'repositoryName',
                        value: aws_events_1.EventField.fromPath('$.detail.repositoryNames[0]'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'sourceCommit',
                        value: aws_events_1.EventField.fromPath('$.detail.sourceCommit'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'destinationCommit',
                        value: aws_events_1.EventField.fromPath('$.detail.destinationCommit'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'revisionId',
                        value: aws_events_1.EventField.fromPath('$.detail.revisionId'),
                        type: 'PLAINTEXT',
                    },
                ],
            }),
        }));
    }
    /**
     * Defines an event rule which triggers when a check fails.
     */
    onCheckFailed(id, options) {
        return this.pullRequestProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a check starts.
     */
    onCheckStarted(id, options) {
        return this.pullRequestProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a check complets successfully.
     */
    onCheckSucceeded(id, options) {
        return this.pullRequestProject.onBuildSucceeded(id, options);
    }
    /**
     * Add a permission only if there's a policy attached.
     *
     * @param statement The permissions statement to add.
     */
    addToRolePolicy(statement) {
        this.pullRequestProject.addToRolePolicy(statement);
    }
}
exports.PullRequestCheck = PullRequestCheck;
//# sourceMappingURL=data:application/json;base64,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