"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const dns_1 = require("./dns");
const CUSTOM_HTTP_HEADER = 'X_Request_From_CloudFront';
class Application extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        this.domainName = props.domainName;
        this.domainZone = props.domainZone;
        this.cloudFrontHashHeader =
            props.cloudFrontHashHeader ?? Buffer.from(`${stack.stackName}.${this.domainName}`).toString('base64');
        this.removalPolicy = props.removalPolicy;
        const cluster = new aws_cdk_lib_1.aws_ecs.Cluster(this, 'Cluster', {
            containerInsights: true,
            vpc: props.vpc,
        });
        this.targetGroup = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationTargetGroup(this, 'TargetGroup', {
            vpc: props.vpc,
            port: 80,
            targetType: aws_cdk_lib_1.aws_elasticloadbalancingv2.TargetType.IP,
            stickinessCookieDuration: aws_cdk_lib_1.Duration.days(7),
        });
        this.targetGroup.configureHealthCheck({
            path: '/wp-includes/images/blank.gif',
            interval: aws_cdk_lib_1.Duration.minutes(1),
        });
        const loadBalancer = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer(this, 'Loadbalancer', {
            vpc: props.vpc,
            internetFacing: true,
            http2Enabled: true,
        });
        this.listener = loadBalancer.addListener('Listener', {
            port: 443,
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTPS,
            certificates: [
                aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerCertificate.fromCertificateManager(new aws_cdk_lib_1.aws_certificatemanager.Certificate(this, 'LBCertificate', {
                    domainName: this.domainName,
                    validation: aws_cdk_lib_1.aws_certificatemanager.CertificateValidation.fromDns(this.domainZone),
                })),
            ],
            sslPolicy: aws_cdk_lib_1.aws_elasticloadbalancingv2.SslPolicy.FORWARD_SECRECY_TLS12,
            defaultAction: aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerAction.fixedResponse(403, {
                contentType: 'text/plain',
                messageBody: 'Access denied',
            }),
        });
        this.listener.addAction('Cloudfornt', {
            action: aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerAction.forward([this.targetGroup]),
            conditions: [
                aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerCondition.httpHeader(CUSTOM_HTTP_HEADER, [this.cloudFrontHashHeader]),
            ],
            priority: 100,
        });
        const taskDefinition = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, 'TaskDefinition', {
            memoryLimitMiB: props.memoryLimitMiB ?? 512,
            cpu: 256,
            volumes: [props.volume],
        });
        const container = taskDefinition.addContainer('Wordpress', {
            image: props.image ?? aws_cdk_lib_1.aws_ecs.ContainerImage.fromRegistry('wordpress:5.8-apache'),
            environment: {
                ...props.environment,
                ...props.database.environment,
            },
            secrets: {
                ...props.secrets,
                ...props.database.secrets,
            },
            logging: props.logDriver ??
                aws_cdk_lib_1.aws_ecs.LogDriver.awsLogs({
                    streamPrefix: `${stack.stackName}WordpressContainerLog`,
                    logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
                }),
        });
        container.addPortMappings({
            containerPort: 80,
        });
        container.addMountPoints({
            containerPath: '/var/www/html',
            readOnly: false,
            sourceVolume: props.volume.name,
        });
        this.service = new aws_cdk_lib_1.aws_ecs.FargateService(this, 'Service', {
            cluster,
            serviceName: props.serviceName,
            taskDefinition,
            desiredCount: 2,
        });
        this.service.connections.allowFrom(loadBalancer, aws_cdk_lib_1.aws_ec2.Port.tcp(80));
        this.targetGroup.addTarget(this.service);
        const origin = new aws_cdk_lib_1.aws_cloudfront_origins.LoadBalancerV2Origin(loadBalancer, {
            originSslProtocols: [aws_cdk_lib_1.aws_cloudfront.OriginSslPolicy.TLS_V1_2],
            customHeaders: {
                [CUSTOM_HTTP_HEADER]: this.cloudFrontHashHeader,
            },
            readTimeout: aws_cdk_lib_1.Duration.seconds(60),
            protocolPolicy: aws_cdk_lib_1.aws_cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
        });
        this.distribution = new aws_cdk_lib_1.aws_cloudfront.Distribution(this, 'WorpdressDistribution', {
            comment: 'wordpress cdn',
            defaultBehavior: {
                origin,
                originRequestPolicy: new aws_cdk_lib_1.aws_cloudfront.OriginRequestPolicy(this, 'OriginRequestPolicy', {
                    originRequestPolicyName: 'WordpressDefaultBehavior',
                    cookieBehavior: aws_cdk_lib_1.aws_cloudfront.OriginRequestCookieBehavior.allowList('comment_*', 'wordpress_*', 'wp-settings-*'),
                    headerBehavior: aws_cdk_lib_1.aws_cloudfront.OriginRequestHeaderBehavior.allowList('Host', 'CloudFront-Forwarded-Proto', 'CloudFront-Is-Mobile-Viewer', 'CloudFront-Is-Tablet-Viewer', 'CloudFront-Is-Desktop-Viewer'),
                    queryStringBehavior: aws_cdk_lib_1.aws_cloudfront.OriginRequestQueryStringBehavior.all(),
                }),
                allowedMethods: aws_cdk_lib_1.aws_cloudfront.AllowedMethods.ALLOW_ALL,
                cachedMethods: aws_cdk_lib_1.aws_cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                viewerProtocolPolicy: aws_cdk_lib_1.aws_cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            additionalBehaviors: {
                'wp-admin/*': {
                    origin,
                    originRequestPolicy: aws_cdk_lib_1.aws_cloudfront.OriginRequestPolicy.ALL_VIEWER,
                    allowedMethods: aws_cdk_lib_1.aws_cloudfront.AllowedMethods.ALLOW_ALL,
                    cachedMethods: aws_cdk_lib_1.aws_cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                    viewerProtocolPolicy: aws_cdk_lib_1.aws_cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                },
                'wp-login.php': {
                    origin,
                    originRequestPolicy: aws_cdk_lib_1.aws_cloudfront.OriginRequestPolicy.ALL_VIEWER,
                    allowedMethods: aws_cdk_lib_1.aws_cloudfront.AllowedMethods.ALLOW_ALL,
                    cachedMethods: aws_cdk_lib_1.aws_cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                    viewerProtocolPolicy: aws_cdk_lib_1.aws_cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                },
            },
            enableIpv6: true,
            httpVersion: aws_cdk_lib_1.aws_cloudfront.HttpVersion.HTTP2,
            certificate: props.certificate,
            domainNames: [this.domainName],
            priceClass: aws_cdk_lib_1.aws_cloudfront.PriceClass.PRICE_CLASS_100,
        });
        new dns_1.Dns(this, 'WordpressDns', {
            domainName: this.domainName,
            domainZone: this.domainZone,
            distribution: this.distribution,
        });
    }
    enableStaticContentOffload(domainName, certificate) {
        const bucket = new aws_cdk_lib_1.aws_s3.Bucket(this, 'Bucket', {
            encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
            versioned: true,
            removalPolicy: this.removalPolicy,
            autoDeleteObjects: this.removalPolicy === aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        bucket.grantReadWrite(this.service.taskDefinition.taskRole);
        this.service.taskDefinition.taskRole.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: ['s3:GetBucketLocation'],
            resources: [bucket.bucketArn],
        }));
        const distribution = new aws_cdk_lib_1.aws_cloudfront.Distribution(this, 'StaticContentDistribution', {
            comment: 'static content cdn',
            defaultBehavior: {
                origin: new aws_cdk_lib_1.aws_cloudfront_origins.S3Origin(bucket),
                allowedMethods: aws_cdk_lib_1.aws_cloudfront.AllowedMethods.ALLOW_GET_HEAD,
                cachedMethods: aws_cdk_lib_1.aws_cloudfront.CachedMethods.CACHE_GET_HEAD,
                viewerProtocolPolicy: aws_cdk_lib_1.aws_cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            enableIpv6: true,
            httpVersion: aws_cdk_lib_1.aws_cloudfront.HttpVersion.HTTP2,
            certificate,
            domainNames: [domainName],
            priceClass: aws_cdk_lib_1.aws_cloudfront.PriceClass.PRICE_CLASS_100,
        });
        new dns_1.Dns(this, 'StaticContentDns', {
            domainName: domainName,
            domainZone: this.domainZone,
            distribution,
        });
        return {
            domainName,
            distribution,
        };
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@cloudcomponents/cdk-wordpress.Application", version: "2.2.0" };
//# sourceMappingURL=data:application/json;base64,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