from __future__ import annotations

import re
import time
from enum import Enum
from functools import cached_property
from logging import Logger
from typing import ClassVar

import attr

from cloudshell.cp.core.utils.name_generator import NameGenerator

from cloudshell.cp.cloudstack.api_client.cloudstack_api import CloudStackAPIClient
from cloudshell.cp.cloudstack.exceptions import (
    CloudstackNetworkException,
    InstanceErrorState,
    InstanceNotFound,
    PortIsNotAttached,
)
from cloudshell.cp.cloudstack.models.deploy_app import VMFromTemplateDeployApp
from cloudshell.cp.cloudstack.models.resource_config import CloudstackResourceConfig

generator = NameGenerator()


@attr.s(auto_attribs=True)
class Interface:
    nic_id: str
    ip_address: str
    public_ip_address: str
    network_id: str
    network: str
    mac_address: str
    interface_index: int


class VMState(Enum):
    Running = "Running"
    Error = "Error"

    @classmethod
    def _missing_(cls, value):
        if isinstance(value, str):
            return cls(value.lower())


@attr.s(auto_attribs=True, str=False)
class CloudstackVirtualMachine:
    RESOURCE_TYPE = "VirtualMachine"
    VM_DESTROY_TYPE = "VM.DESTROY"
    VM_CREATE_TYPE = "VM.CREATE"
    NIC_CREATE_TYPE = "NIC.CREATE"
    NIC_DELETE_TYPE = "NIC.DELETE"

    _api: ClassVar[CloudStackAPIClient]
    _logger: ClassVar[Logger]

    vm_uuid: str
    _does_vm_exist: bool = True

    @property
    def num_cpu(self):
        return self.vm_data.get("cpunumber")

    @property
    def memory_size(self):
        return self.vm_data.get("memory")

    @property
    def guest_os(self):
        return self.vm_data.get("osdisplayname")

    @property
    def _creation_date(self):
        return self.vm_data.get("created")

    @cached_property
    def vm_name(self):
        return self.get_vm_data()["displayname"]

    @cached_property
    def zone_id(self):
        return self.get_vm_data()["zoneid"]

    @cached_property
    def vm_data(self):
        return self.get_vm_data()

    @classmethod
    def get(cls, id_: str) -> CloudstackVirtualMachine:
        cls._logger.debug(f"Getting a network with ID '{id_}'")
        return cls(vm_uuid=id_)

    @classmethod
    def create(
        cls,
        deploy_action: VMFromTemplateDeployApp,
        resource_conf: CloudstackResourceConfig,
    ) -> CloudstackVirtualMachine:
        so_id = deploy_action.service_offering_id
        template_id = deploy_action.template_id
        zone_id = deploy_action.zone_id

        inputs = {
            "command": "deployVirtualMachine",
            "serviceofferingid": so_id,
            "templateid": template_id,
            "zoneid": zone_id,
        }

        pod_id = deploy_action.pod_id
        group = deploy_action.group
        network_id = deploy_action.mgmt_network_id or resource_conf.mgmt_network_id
        clusterid = deploy_action.cluster_id
        diskofferingid = deploy_action.disk_offering_id

        if pod_id:
            inputs["podid"] = pod_id
        if group:
            inputs["group"] = group
        if network_id:
            inputs["networkids"] = network_id
        if clusterid:
            inputs["clusterid"] = clusterid
        if diskofferingid:
            inputs["diskofferingid"] = diskofferingid

        vm_name = deploy_action.app_name
        if deploy_action.autogenerated_name:
            vm_name = generator(deploy_action.app_name)
        inputs["displayname"] = vm_name

        cls._logger.info(inputs)
        response = cls._api.send_request(inputs)

        if response.status_code != 200 and response.status_code != 201:
            raise InstanceNotFound(name=deploy_action.app_name)
        id_ = response.json()["deployvirtualmachineresponse"]["id"]
        state = cls._api.wait_for_event_completed(
            id_, cls.RESOURCE_TYPE, cls.VM_CREATE_TYPE
        )
        if state.get("state") != "Completed":
            raise InstanceNotFound(name=vm_name)
        return cls(vm_uuid=response.json()["deployvirtualmachineresponse"]["id"])

    def wait_for_vm_to_start(self, timeout: int = 400) -> None:
        interval = 5
        time_elapsed = 0
        state = self.get_vm_state()
        while time_elapsed < timeout:
            state = self.get_vm_state()
            if state == VMState.Running.value:
                return
            if state == VMState.Error.value:
                raise InstanceErrorState(self, "Instance deployment failed.")
            time_elapsed += interval
            time.sleep(interval)
        raise InstanceErrorState(self, f"Timeout waiting for VM state to be {state}.")

    def get_vm_data(self):
        inputs = {"command": "listVirtualMachines", "id": self.vm_uuid}

        response = self._api.send_request(inputs)
        json_response = response.json()
        if (
            response.status_code != 200
            and response.status_code != 201
            or not json_response.get("listvirtualmachinesresponse")
        ):
            self._logger.warning(
                f"Instance deployment failed with: http status:"
                f" {response.status_code}, response: {response.text}"
            )
            raise InstanceNotFound(id_=self.vm_uuid)
        return json_response["listvirtualmachinesresponse"]["virtualmachine"][0]

    def get_nics(self, network_id=None):
        inputs = {"command": "listNics", "virtualmachineid": self.vm_uuid}
        if network_id:
            inputs["networkid"] = network_id
        response = self._api.send_request(inputs)
        json_response = response.json()
        if (
            response.status_code != 200
            and response.status_code != 201
            or not json_response.get("listnicsresponse")
        ):
            self._logger.warning(
                f"Failed to download list of attached vNICs. "
                f"Http status:"
                f" {response.status_code}, error: {response.text}"
            )
        return json_response.get("listnicsresponse", {}).get("nic", [])

    def get_vm_state(self):
        if not self._does_vm_exist:
            raise Exception(
                f"CloudstackVirtualMachine does not exist. uid: {self.vm_uuid}"
            )
        return self.get_vm_data()["state"]

    def power_on_vm(self):
        if not self._does_vm_exist:
            raise Exception(
                f"CloudstackVirtualMachine does not exist. uid: {self.vm_uuid}"
            )
        inputs = {"command": "startVirtualMachine", "id": self.vm_uuid}

        self._logger.info(inputs)
        response = self._api.send_request(inputs)
        self._logger.info(response)

        if response.status_code != 200 and response.status_code != 201:
            raise Exception(
                "Unable to power on CloudstackVirtualMachine. uid: {}".format(
                    self.vm_uuid
                )
            )

    def power_off_vm(self):
        if not self._does_vm_exist:
            raise Exception(
                f"CloudstackVirtualMachine does not exist. uid: {self.vm_uuid}"
            )
        inputs = {"command": "stopVirtualMachine", "id": self.vm_uuid}

        self._logger.info(inputs)
        response = self._api.send_request(inputs)
        self._logger.info(response)

        if response.status_code != 200 and response.status_code != 201:
            raise Exception(
                "Unable to power off CloudstackVirtualMachine. uid: {}".format(
                    self.vm_uuid
                )
            )

    def get_vm_ip(self, pattern: str) -> Interface:
        if not self._does_vm_exist:
            raise Exception(
                f"CloudstackVirtualMachine does not exist. uid: {self.vm_uuid}"
            )
        ipv4_pattern = pattern or r"^(?:[0-9]{1,3}\.){3}[0-9]{1,3}$"
        ipv4_match = re.compile(ipv4_pattern)
        interval = 5
        time_elapsed = 0
        timeout = 600
        ip = None
        while time_elapsed < timeout and not ip:

            for nic in self.get_interfaces():
                temp_ip = nic.ip_address

                if temp_ip:
                    if ipv4_match.match(temp_ip):
                        return nic

            time_elapsed += interval
            time.sleep(interval)

    def detach_network(self, network_id, max_retries=10):
        if not self._does_vm_exist:
            raise CloudstackNetworkException(
                f"CloudstackVirtualMachine does not exist. uid: {self.vm_uuid}"
            )

        nic = next(
            (x for x in self.get_interfaces() if x.network_id == network_id), None
        )
        if not nic:
            return

        inputs = {
            "command": "removeNicFromVirtualMachine",
            "virtualmachineid": self.vm_uuid,
            "nicid": nic.nic_id,
        }
        state = {}
        while nic in self.get_interfaces(network_id) and max_retries > 0:
            self._logger.info(inputs)
            response = self._api.send_request(inputs)
            self._logger.info(response)

            if response.status_code != 200 and response.status_code != 201:
                raise Exception(
                    f"Unable to detach network from {self.vm_name}. "
                    f"Error: {response.text}"
                )

            state = self._api.wait_for_untyped_event_completed(
                resource_id=nic.nic_id,
                other_id=self.vm_uuid,
                event_type=self.NIC_DELETE_TYPE,
            )
            if state.get("state") != "Completed":
                self._logger.warning(
                    f"Unable to detach {self.vm_name} from {nic.network}."
                    f"Job state: {state.get('description')}"
                    f"Trying again. Retries left: {max_retries}"
                )

            time.sleep(6)
            max_retries -= 1
        else:
            if nic in self.get_interfaces(network_id):
                self._logger.warning(
                    f"Failed to detach {self.vm_name} from {nic.network}."
                    f"Job state: {state.get('description')}"
                )

    def _check_interface_still_exists(self, nic):
        return nic in self.get_interfaces()

    def attach_network(self, network_id):
        if not self._does_vm_exist:
            raise Exception(
                f"CloudstackVirtualMachine does not exist. uid: {self.vm_uuid}"
            )
        inputs = {
            "command": "addNicToVirtualMachine",
            "virtualmachineid": self.vm_uuid,
            "networkid": network_id,
        }

        self._logger.info(inputs)
        response = self._api.send_request(inputs)
        self._logger.info(response)

        if response.status_code != 200 and response.status_code != 201:
            self._logger.error(
                f"Unable to attach {self.vm_name} to {network_id}."
                f"Job state: {response.text}"
            )
            PortIsNotAttached(network_id, self)

        state = self._api.wait_for_untyped_event_completed(
            resource_id=network_id,
            other_id=self.vm_uuid,
            event_type=self.NIC_CREATE_TYPE,
        )
        interface = None
        if "Completed" in state.get("state", ""):
            interface = next(
                (x for x in self.get_interfaces() if x.network_id == network_id), None
            )
        if not interface:
            self._logger.warning(
                f"Unable to attach {self.vm_name} to {network_id}."
                f"Job state: {state.get('description')}"
            )
            raise PortIsNotAttached(network_id, self)

        return interface

    def get_interfaces(self, network_id=None):
        return [
            Interface(
                nic_id=nic.get("id"),
                ip_address=nic.get("ipaddress"),
                public_ip_address=nic.get("publicip"),
                network_id=nic.get("networkid"),
                network=nic.get("networkname"),
                mac_address=nic.get("macaddress"),
                interface_index=int(nic.get("deviceid")),
            )
            for nic in self.get_nics(network_id=network_id)
        ]

    def get_interface_by_mac(self, mac_address):
        return next(
            (x for x in self.get_interfaces() if x.mac_address == mac_address), None
        )

    def delete_vm(self):
        try:
            data = self.get_vm_data()
            self._does_vm_exist = data["state"] == "Destroyed"
        except Exception:
            self._logger.warning(
                f"CloudstackVirtualMachine does not exist. uid: {self.vm_uuid}"
            )
            self._does_vm_exist = False
            return
        inputs = {
            "command": "destroyVirtualMachine",
            "id": self.vm_uuid,
            "expunge": "true",
        }

        self._logger.info(inputs)
        response = self._api.send_request(inputs)
        self._logger.info(response)

        if response.status_code != 200 and response.status_code != 201:
            InstanceErrorState(
                self, f"Unable to delete instance with uid: {self.vm_uuid}"
            )

        state = self._api.wait_for_event_completed(
            self.vm_uuid, self.RESOURCE_TYPE, self.VM_DESTROY_TYPE
        )
        if state.get("state") != "Completed":
            InstanceErrorState(
                self,
                "Unable to delete instance with uid: {} "
                "with error: {}".format(self.vm_uuid, state.get("description")),
            )
