from dataclasses import dataclass
from typing import Any, TypeVar, Union, Literal
from typing_extensions import TypeAlias

from openai.types.responses import (
    Response,
    ResponseOutputMessage,
    ResponseInputItemParam,
    ResponseOutputItem,
    ResponseStreamEvent,
    ResponseFunctionToolCall,
    ResponseFunctionToolCallOutputItem,
)
from openai.types.responses.response_input_message_content_list_param import (
    ResponseInputTextParam, # noqa: F401
    ResponseInputImageParam, # noqa: F401
    ResponseInputFileParam, # noqa: F401
    ResponseInputAudioParam # noqa: F401
)
from openai.types.responses.response_reasoning_item import ResponseReasoningItem
from pydantic import BaseModel


TResponse = Response
"""A type alias for the Response type from the OpenAI SDK."""


class FunctionToolCallProgressItem(BaseModel):
    call_id: str
    """The unique ID of the function tool call generated by the model."""

    progress: float = 0
    """The current progress value of the function call. (e.g. 24)"""

    total: float = 0
    """The total value of the function call. (e.g. 100)"""

    message: str = ""
    """The human readable progress information. (e.g. Starting render...)"""

    type: Literal["function_call_progress"]
    """The type of the function tool call progress. Always `function_call_progress`."""


TResponseInputItem: TypeAlias = Union[
    ResponseInputItemParam,
    ResponseFunctionToolCallOutputItem,
    FunctionToolCallProgressItem,
]
"""A type alias for the ResponseInputItemParam type from the OpenAI SDK."""

TResponseOutputItem = ResponseOutputItem
"""A type alias for the ResponseOutputItem type from the OpenAI SDK."""

TResponseStreamEvent = ResponseStreamEvent
"""A type alias for the ResponseStreamEvent type from the OpenAI SDK."""

T = TypeVar("T", bound=Union[TResponseOutputItem, TResponseInputItem])


@dataclass
class FunctionTool:
    """A tool that wraps a function. In most cases, you should use  the `tool` helpers to
    create a FunctionTool, as they let you easily wrap a Python function.
    """

    name: str
    """The name of the tool, as shown to the LLM. Generally the name of the function."""

    description: str
    """A description of the tool, as shown to the LLM."""

    params_json_schema: dict[str, Any]
    """The JSON schema for the tool's parameters."""


Tool = FunctionTool
"""A type alias for the FunctionTool class."""


@dataclass
class MessageOutputItem:
    """Represents a message from the LLM."""

    raw_item: ResponseOutputMessage
    """The raw response output message."""

    type: Literal["message_output_item"] = "message_output_item"


@dataclass
class ToolCallItem:
    """Represents a tool call e.g. a function call or computer action call."""

    raw_item: ResponseFunctionToolCall
    """The raw tool call item."""

    type: Literal["tool_call_item"] = "tool_call_item"


@dataclass
class ToolCallOutputItem:
    """Represents the output of a tool call."""

    raw_item: ResponseFunctionToolCallOutputItem
    """The raw item from the model."""

    output: Any
    """The output of the tool call. This is whatever the tool call returned; the `raw_item`
    contains a string representation of the output.
    """

    type: Literal["tool_call_output_item"] = "tool_call_output_item"


@dataclass
class ToolCallProgressItem:
    """Represents the intermediate progress of a tool call."""

    raw_item: FunctionToolCallProgressItem
    """The raw item generated by the tool."""

    type: Literal["tool_call_progress_item"] = "tool_call_progress_item"


@dataclass
class ReasoningItem:
    """Represents a reasoning item."""

    raw_item: ResponseReasoningItem
    """The raw reasoning item."""

    type: Literal["reasoning_item"] = "reasoning_item"


StreamEvent: TypeAlias = Union[
    MessageOutputItem,
    ToolCallItem,
    ToolCallOutputItem,
    ToolCallProgressItem,
    ReasoningItem,
]
"""A type that represents an streaming output event."""
