"""Base parser class for import and function extraction."""

from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from pathlib import Path
from typing import Optional


@dataclass
class ImportInfo:
    """Represents an import statement."""

    module: str  # The imported module path
    import_type: str  # static, dynamic, require
    line: int  # Line number in source file

    def is_relative(self) -> bool:
        """Check if this is a relative import."""
        return self.module.startswith(".") or self.module.startswith("@/")

    def is_external(self) -> bool:
        """Check if this is likely an external package."""
        if self.is_relative():
            return False
        # External packages don't start with . or have path separators at start
        first_part = self.module.split("/")[0].split("\\")[0]
        # Common external indicators
        return not first_part.startswith(".")


@dataclass
class FunctionInfo:
    """Represents a function or method definition."""

    name: str
    signature: str  # Full signature (without body)
    start_line: int
    end_line: int
    calls: list[str] = field(default_factory=list)  # Functions called within
    is_method: bool = False  # True if it's a class method
    is_async: bool = False
    docstring: Optional[str] = None


@dataclass
class ClassInfo:
    """Represents a class definition."""

    name: str
    signature: str  # Class declaration line
    start_line: int
    end_line: int
    methods: list[FunctionInfo] = field(default_factory=list)
    bases: list[str] = field(default_factory=list)  # Base classes
    docstring: Optional[str] = None


@dataclass
class ParseResult:
    """Complete parsing result for a file."""

    imports: list[ImportInfo] = field(default_factory=list)
    functions: list[FunctionInfo] = field(default_factory=list)
    classes: list[ClassInfo] = field(default_factory=list)


class BaseParser(ABC):
    """Abstract base class for language-specific parsers."""

    supported_extensions: list[str] = []

    @abstractmethod
    def parse(self, content: str, file_path: Path) -> list[ImportInfo]:
        """
        Parse source code and extract import statements.

        Args:
            content: Source code content
            file_path: Path to the source file (for context)

        Returns:
            List of ImportInfo objects
        """
        pass

    def parse_full(self, content: str, file_path: Path) -> ParseResult:
        """
        Parse source code and extract all information.

        Args:
            content: Source code content
            file_path: Path to the source file (for context)

        Returns:
            ParseResult with imports, functions, and classes
        """
        # Default implementation: only parse imports
        return ParseResult(imports=self.parse(content, file_path))

    def extract_functions(
        self,
        content: str,
        file_path: Path
    ) -> list[FunctionInfo]:
        """
        Extract function definitions from source code.

        Args:
            content: Source code content
            file_path: Path to the source file (for context)

        Returns:
            List of FunctionInfo objects
        """
        result = self.parse_full(content, file_path)
        return result.functions

    def extract_classes(
        self,
        content: str,
        file_path: Path
    ) -> list[ClassInfo]:
        """
        Extract class definitions from source code.

        Args:
            content: Source code content
            file_path: Path to the source file (for context)

        Returns:
            List of ClassInfo objects
        """
        result = self.parse_full(content, file_path)
        return result.classes

    @classmethod
    def can_parse(cls, file_path: Path) -> bool:
        """Check if this parser can handle the given file."""
        return file_path.suffix.lower() in cls.supported_extensions
