"""Related code context MCP tools.

Tools for getting related code context (Repo Map).
"""

from pathlib import Path
from fastmcp import FastMCP

from core.services import RelatedContextService
from .base import get_storage


def register_context_tools(mcp: FastMCP) -> None:
    """Register context tools with the MCP server."""

    @mcp.tool
    def get_related_code_context(
        project_path: str, 
        file_path: str, 
        hops: int = 1,
        include_external: bool = False
    ) -> dict:
        """获取文件的关联代码上下文(Repo Map)，包含目标文件及其依赖文件的签名和摘要。
        
        Get related code context (Repo Map) with signatures and summaries of target file and its dependencies.
        
        Args:
            project_path: 项目路径 | Project path
            file_path: 目标文件相对路径 | Target file relative path
            hops: 依赖跳数(1-3) | Dependency hops (1-3)
            include_external: 是否包含第三方依赖 | Whether to include external dependencies
        """
        project_root = Path(project_path)
        if not project_root.exists():
            return {"error": "NOT_FOUND", "details": f"Path does not exist: {project_path}"}
        
        hops = max(1, min(3, hops))  # Clamp to 1-3
        
        storage = get_storage()
        service = RelatedContextService(storage, project_root)
        
        result = service.get_related_context(
            str(project_root.resolve()), 
            file_path, 
            hops=hops,
            include_external=include_external
        )
        return service.to_dict(result)
