"""Statistics and analysis MCP tools.

Tools for file statistics, reference ranking, and depth analysis.
"""

from pathlib import Path
from fastmcp import FastMCP

from .base import get_stats_service, get_function_analysis, filter_external_deps


def register_stats_tools(mcp: FastMCP) -> None:
    """Register statistics tools with the MCP server."""

    @mcp.tool
    def get_file_stats(path: str, subdirectory: str | None = None) -> dict:
        """获取项目文件类型统计 | Get project file type statistics.
        
        Args:
            path: 项目路径 | Project path
            subdirectory: 子目录过滤（可选）| Subdirectory filter (optional)
        """
        service = get_stats_service()
        response = service.get_file_type_stats(str(Path(path).resolve()), subdirectory=subdirectory)
        
        if response is None:
            return {"error": "NOT_FOUND", "details": f"Project not found: {path}"}
        
        return {
            "project_path": response.project_path,
            "subdirectory": subdirectory,
            "total_files": response.total_files,
            "stats": [
                {
                    "type": s.file_type,
                    "count": s.count,
                    "percentage": s.percentage,
                    "total_size": s.total_size
                }
                for s in response.stats
            ]
        }

    @mcp.tool
    def get_reference_ranking(
        path: str, 
        limit: int = 20, 
        file_type: str | None = None,
        include_external: bool = False
    ) -> dict:
        """获取被引用最多的文件排名 | Get top referenced files ranking.
        
        Args:
            path: 项目路径 | Project path
            limit: 返回结果数量限制 | Limit of results
            file_type: 文件类型过滤（可选）| File type filter (optional)
            include_external: 是否包含第三方依赖 | Whether to include external dependencies
        """
        service = get_stats_service()
        response = service.get_reference_ranking(
            str(Path(path).resolve()), 
            limit=limit, 
            file_type=file_type,
            include_external=include_external
        )
        
        if response is None:
            return {"error": "NOT_FOUND", "details": f"Project not found: {path}"}
        
        return {
            "project_path": response.project_path,
            "limit": limit,
            "file_type_filter": file_type,
            "include_external": include_external,
            "total_results": response.total_results,
            "results": [
                {
                    "file": r.file_path,
                    "count": r.reference_count,
                    "references": r.referencing_files
                }
                for r in response.results
            ]
        }

    @mcp.tool
    def get_depth_analysis(path: str, subdirectory: str | None = None) -> dict:
        """获取目录和文件层级分析 | Get directory and file depth analysis.
        
        Args:
            path: 项目路径 | Project path
            subdirectory: 子目录过滤（可选）| Subdirectory filter (optional)
        """
        service = get_stats_service()
        response = service.get_depth_analysis(str(Path(path).resolve()), subdirectory=subdirectory)
        
        if response is None:
            return {"error": "NOT_FOUND", "details": f"Project not found: {path}"}
        
        return {
            "project_path": response.project_path,
            "subdirectory": subdirectory,
            "directory_depth": response.directory_depth,
            "file_depth": response.file_depth
        }

    @mcp.tool
    def get_function_relations(files: list[str], include_external: bool = False) -> dict:
        """获取指定文件间的函数调用关系（最多10个文件）| Get function call relations between files (max 10).
        
        Args:
            files: 要分析的文件路径列表 | List of file paths to analyze
            include_external: 是否包含第三方依赖调用 | Whether to include external dependency calls
        """
        if len(files) > 10:
            return {"error": "LIMIT_EXCEEDED", "details": "Maximum 10 files allowed"}
        
        try:
            service = get_function_analysis()
            result = service.analyze_files(files)
            data = service.to_dict(result)
            return filter_external_deps(data, include_external)
        except FileNotFoundError as e:
            return {"error": "NOT_FOUND", "details": str(e)}
