"""Project management service.

This module provides high-level operations for scanning,
storing, and managing code projects.
"""

import logging
from dataclasses import dataclass
from pathlib import Path
from typing import Optional

from ..scanner import CodeScanner
from ..graph import GraphBuilder
from ..storage import StorageBackend

logger = logging.getLogger(__name__)


@dataclass
class ScanResult:
    """Result of a project scan operation."""

    project_id: int
    project_path: str
    project_name: str
    total_files: int
    file_types: dict[str, int]
    external_deps: list[str]
    scan_mode: str  # 'full' or 'incremental'
    stats: Optional[dict] = None  # For incremental updates


class ProjectService:
    """Project management service.

    Provides high-level methods for scanning projects,
    managing storage, and querying project data.
    """

    def __init__(self, storage: StorageBackend):
        """Initialize project service.

        Args:
            storage: Storage backend instance
        """
        self.storage = storage

    def scan_project(
        self,
        project_path: str | Path,
        incremental: bool = True
    ) -> ScanResult:
        """Scan a project and store the results.

        Args:
            project_path: Path to the project directory
            incremental: If True, only update modified files

        Returns:
            ScanResult with scan statistics
        """
        path = Path(project_path).resolve()

        if not path.exists():
            raise FileNotFoundError(f"Project path does not exist: {path}")

        if not path.is_dir():
            raise ValueError(f"Project path is not a directory: {path}")

        logger.info(f"Scanning project: {path}")

        # Scan files
        scanner = CodeScanner(path)
        files = scanner.scan()

        # Build dependency graph
        builder = GraphBuilder(path)
        graph = builder.build(files)
        graph_dict = graph.to_dict()

        # Count file types
        file_types: dict[str, int] = {}
        for file_info in files:
            ft = file_info.file_type
            file_types[ft] = file_types.get(ft, 0) + 1

        # Save to storage
        if incremental:
            project_id, stats = self.storage.save_project_incremental(
                path, files, graph_dict
            )
            scan_mode = "incremental"
        else:
            project_id = self.storage.save_project(path, files, graph_dict)
            stats = {"added": len(files), "updated": 0, "unchanged": 0, "removed": 0}
            scan_mode = "full"

        return ScanResult(
            project_id=project_id,
            project_path=str(path),
            project_name=path.name,
            total_files=len(files),
            file_types=file_types,
            external_deps=list(graph.external_deps),
            scan_mode=scan_mode,
            stats=stats
        )

    def rescan_project(
        self,
        project_path: str | Path
    ) -> ScanResult:
        """Force a full rescan of a project.

        Args:
            project_path: Path to the project directory

        Returns:
            ScanResult with scan statistics
        """
        return self.scan_project(project_path, incremental=False)

    def get_project_info(
        self,
        project_path: str
    ) -> Optional[dict]:
        """Get basic information about a stored project.

        Args:
            project_path: Path to the project

        Returns:
            Dictionary with project info or None if not found
        """
        project = self.storage.get_project(project_path)
        if not project:
            return None

        return {
            "id": project.id,
            "path": project.path,
            "name": project.name,
            "last_scanned": project.last_scanned.isoformat(),
            "file_count": project.file_count
        }

    def list_projects(self) -> list[dict]:
        """List all stored projects.

        Returns:
            List of project info dictionaries
        """
        # This would require adding a list_projects method to storage
        # For now, return empty list
        return []

    def delete_project(
        self,
        project_path: str
    ) -> bool:
        """Delete a project from storage.

        Args:
            project_path: Path to the project

        Returns:
            True if deleted, False if not found
        """
        project = self.storage.get_project(project_path)
        if not project:
            return False

        return self.storage.delete_project(project.id)

    def get_file_info(
        self,
        project_path: str,
        file_path: str
    ) -> Optional[dict]:
        """Get information about a specific file.

        Args:
            project_path: Path to the project
            file_path: Relative path to the file

        Returns:
            Dictionary with file info or None if not found
        """
        project = self.storage.get_project(project_path)
        if not project:
            return None

        file_record = self.storage.get_file_by_path(project.id, file_path)
        if not file_record:
            return None

        imports = self.storage.get_imports_by_file(file_record.id)
        functions = self.storage.get_functions_by_file(file_record.id)
        summaries = self.storage.get_summaries_by_file(file_record.id)

        return {
            "id": file_record.id,
            "relative_path": file_record.relative_path,
            "file_type": file_record.file_type,
            "size": file_record.size,
            "depth": file_record.depth,
            "modified_time": file_record.modified_time.isoformat()
                if hasattr(file_record.modified_time, 'isoformat')
                else str(file_record.modified_time),
            "imports": [
                {
                    "module": imp.module,
                    "type": imp.import_type,
                    "line": imp.line,
                    "is_external": imp.is_external()
                }
                for imp in imports
            ],
            "functions": [
                {
                    "name": f.name,
                    "signature": f.signature,
                    "start_line": f.start_line,
                    "end_line": f.end_line
                }
                for f in functions
            ],
            "summaries": [
                {
                    "entity_name": s.entity_name,
                    "entity_type": s.entity_type,
                    "summary": s.summary
                }
                for s in summaries
            ]
        }

    def get_dependency_graph(
        self,
        project_path: str
    ) -> Optional[dict]:
        """Get the dependency graph for a project.

        Args:
            project_path: Path to the project

        Returns:
            Dictionary with nodes and edges, or None if not found
        """
        project = self.storage.get_project(project_path)
        if not project:
            return None

        files = self.storage.get_files_by_project(project.id)

        nodes = []
        edges = []

        for file_record in files:
            nodes.append({
                "id": file_record.relative_path,
                "type": file_record.file_type,
                "size": file_record.size
            })

            imports = self.storage.get_file_imports(
                project.id,
                file_record.relative_path
            )
            for imp_path in imports:
                edges.append({
                    "source": file_record.relative_path,
                    "target": imp_path
                })

        return {
            "nodes": nodes,
            "edges": edges,
            "node_count": len(nodes),
            "edge_count": len(edges)
        }
