"""Tests for storage module.

Feature: code-knowledge-graph-enhancement
"""

import os
import tempfile
from dataclasses import dataclass
from pathlib import Path
from typing import Optional

import pytest
from hypothesis import given, settings, strategies as st

from core.parsers.base import ImportInfo
from core.scanner import FileInfo
from core.storage import (
    SQLiteStorage,
    ProjectRecord,
    FileRecord,
    FileTypeStats,
    DepthStats,
    ReferenceRankingItem,
)


# ============================================================
# Test Fixtures
# ============================================================

@pytest.fixture
def temp_db():
    """Create a temporary database for testing."""
    fd, path = tempfile.mkstemp(suffix=".db")
    os.close(fd)
    storage = SQLiteStorage(path)
    yield storage
    storage.close()
    os.unlink(path)


@pytest.fixture
def sample_project_path(tmp_path):
    """Create a sample project structure for testing."""
    # Create some files
    (tmp_path / "main.py").write_text("from utils import helper\nimport os")
    (tmp_path / "utils.py").write_text("def helper(): pass")
    (tmp_path / "src").mkdir()
    (tmp_path / "src" / "app.py").write_text("from ..main import something")
    (tmp_path / "src" / "components").mkdir()
    (tmp_path / "src" / "components" / "button.tsx").write_text(
        "import React from 'react'"
    )
    return tmp_path


def create_file_info(
    path: Path,
    relative_path: str,
    file_type: str,
    size: int = 100,
    imports: Optional[list] = None
) -> FileInfo:
    """Helper to create FileInfo objects."""
    return FileInfo(
        path=path,
        relative_path=relative_path,
        file_type=file_type,
        size=size,
        imports=imports or []
    )


# ============================================================
# Unit Tests - Basic CRUD Operations
# ============================================================

class TestSQLiteStorageBasic:
    """Unit tests for basic SQLite storage operations."""

    def test_create_storage(self, temp_db):
        """Test storage initialization creates tables."""
        cursor = temp_db._get_cursor()
        cursor.execute(
            "SELECT name FROM sqlite_master WHERE type='table'"
        )
        tables = {row[0] for row in cursor.fetchall()}
        assert "projects" in tables
        assert "files" in tables
        assert "imports" in tables
        assert "functions" in tables
        assert "function_calls" in tables
        assert "code_summaries" in tables

    def test_save_and_get_project(self, temp_db, sample_project_path):
        """Test saving and retrieving a project."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python",
                100
            )
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        assert project_id > 0

        project = temp_db.get_project(str(sample_project_path.resolve()))
        assert project is not None
        assert project.id == project_id
        assert project.name == sample_project_path.name
        assert project.file_count == 1

    def test_get_project_not_found(self, temp_db):
        """Test getting a non-existent project."""
        project = temp_db.get_project("/nonexistent/path")
        assert project is None

    def test_get_files_by_project(self, temp_db, sample_project_path):
        """Test retrieving files for a project."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python",
                100
            ),
            create_file_info(
                sample_project_path / "utils.py",
                "utils.py",
                "python",
                200
            ),
            create_file_info(
                sample_project_path / "app.tsx",
                "app.tsx",
                "typescript",
                300
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        retrieved_files = temp_db.get_files_by_project(project_id)

        assert len(retrieved_files) == 3
        paths = {f.relative_path for f in retrieved_files}
        assert paths == {"main.py", "utils.py", "app.tsx"}

    def test_get_files_by_type(self, temp_db, sample_project_path):
        """Test filtering files by type."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
            create_file_info(
                sample_project_path / "app.tsx",
                "app.tsx",
                "typescript"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        python_files = temp_db.get_files_by_project(
            project_id,
            file_type="python"
        )

        assert len(python_files) == 1
        assert python_files[0].relative_path == "main.py"

    def test_get_file_by_path(self, temp_db, sample_project_path):
        """Test getting a specific file by path."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        file_record = temp_db.get_file_by_path(project_id, "main.py")

        assert file_record is not None
        assert file_record.relative_path == "main.py"
        assert file_record.file_type == "python"

    def test_save_project_with_imports(self, temp_db, sample_project_path):
        """Test saving project with import information."""
        imports = [
            ImportInfo(module="os", import_type="static", line=1),
            ImportInfo(module="./utils", import_type="static", line=2),
        ]
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python",
                100,
                imports
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        file_record = temp_db.get_file_by_path(project_id, "main.py")
        import_records = temp_db.get_imports_by_file(file_record.id)

        assert len(import_records) == 2
        modules = {imp.module for imp in import_records}
        assert "os" in modules
        assert "./utils" in modules

    def test_update_project(self, temp_db, sample_project_path):
        """Test updating an existing project."""
        # First save
        files1 = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}
        project_id1 = temp_db.save_project(sample_project_path, files1, graph)

        # Second save (update)
        files2 = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
            create_file_info(
                sample_project_path / "new.py",
                "new.py",
                "python"
            ),
        ]
        project_id2 = temp_db.save_project(sample_project_path, files2, graph)

        assert project_id1 == project_id2
        project = temp_db.get_project_by_id(project_id1)
        assert project.file_count == 2

    def test_delete_project(self, temp_db, sample_project_path):
        """Test deleting a project."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        result = temp_db.delete_project(project_id)

        assert result is True
        project = temp_db.get_project_by_id(project_id)
        assert project is None


class TestIncrementalUpdate:
    """Tests for incremental update functionality."""

    def test_incremental_first_scan(self, temp_db, sample_project_path):
        """Test incremental update on first scan (should behave like regular save)."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
            create_file_info(
                sample_project_path / "utils.py",
                "utils.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id, stats = temp_db.save_project_incremental(
            sample_project_path, files, graph
        )

        assert project_id > 0
        assert stats["added"] == 2
        assert stats["updated"] == 0
        assert stats["unchanged"] == 0
        assert stats["removed"] == 0

    def test_incremental_unchanged_files(self, temp_db, sample_project_path):
        """Test incremental update with no changes."""
        # Create actual files so mtime comparison works
        (sample_project_path / "main.py").write_text("print('hello')")
        (sample_project_path / "utils.py").write_text("def helper(): pass")

        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
            create_file_info(
                sample_project_path / "utils.py",
                "utils.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        # First scan
        project_id1, _ = temp_db.save_project_incremental(
            sample_project_path, files, graph
        )

        # Second scan without changes
        import time
        time.sleep(0.1)  # Small delay to ensure different scan time

        project_id2, stats = temp_db.save_project_incremental(
            sample_project_path, files, graph
        )

        assert project_id1 == project_id2
        assert stats["added"] == 0
        assert stats["updated"] == 0
        assert stats["unchanged"] == 2
        assert stats["removed"] == 0

    def test_incremental_new_file(self, temp_db, sample_project_path):
        """Test incremental update with a new file added."""
        # Create actual files
        (sample_project_path / "main.py").write_text("print('hello')")

        files1 = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        # First scan
        temp_db.save_project_incremental(sample_project_path, files1, graph)

        # Add new file
        (sample_project_path / "new.py").write_text("# new file")
        files2 = files1 + [
            create_file_info(
                sample_project_path / "new.py",
                "new.py",
                "python"
            ),
        ]

        # Second scan
        project_id, stats = temp_db.save_project_incremental(
            sample_project_path, files2, graph
        )

        assert stats["added"] == 1
        assert stats["unchanged"] == 1
        assert stats["removed"] == 0

        # Verify new file exists
        retrieved = temp_db.get_files_by_project(project_id)
        paths = {f.relative_path for f in retrieved}
        assert "new.py" in paths

    def test_incremental_removed_file(self, temp_db, sample_project_path):
        """Test incremental update with a file removed."""
        # Create actual files
        (sample_project_path / "main.py").write_text("print('hello')")
        (sample_project_path / "old.py").write_text("# will be removed")

        files1 = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
            create_file_info(
                sample_project_path / "old.py",
                "old.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        # First scan
        temp_db.save_project_incremental(sample_project_path, files1, graph)

        # Remove old.py from file list (simulating deletion)
        files2 = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]

        # Second scan
        project_id, stats = temp_db.save_project_incremental(
            sample_project_path, files2, graph
        )

        assert stats["removed"] == 1
        assert stats["unchanged"] == 1

        # Verify old file is gone
        retrieved = temp_db.get_files_by_project(project_id)
        paths = {f.relative_path for f in retrieved}
        assert "old.py" not in paths

    def test_incremental_modified_file(self, temp_db, sample_project_path):
        """Test incremental update with a modified file."""
        # Create actual file
        (sample_project_path / "main.py").write_text("print('hello')")

        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        # First scan
        temp_db.save_project_incremental(sample_project_path, files, graph)

        # Modify file (change mtime by rewriting)
        import time
        time.sleep(1.5)  # Ensure mtime difference > 1 second tolerance
        (sample_project_path / "main.py").write_text("print('modified')")

        # Second scan
        project_id, stats = temp_db.save_project_incremental(
            sample_project_path, files, graph
        )

        assert stats["updated"] == 1
        assert stats["unchanged"] == 0

    def test_get_files_needing_update(self, temp_db, sample_project_path):
        """Test getting list of files that need updates."""
        # Create actual files
        (sample_project_path / "main.py").write_text("print('hello')")
        (sample_project_path / "utils.py").write_text("def helper(): pass")

        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
            create_file_info(
                sample_project_path / "utils.py",
                "utils.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        # First scan
        temp_db.save_project_incremental(sample_project_path, files, graph)

        # Check - no files need update
        needs_update = temp_db.get_files_needing_update(
            sample_project_path,
            ["main.py", "utils.py"]
        )
        assert len(needs_update) == 0

        # Modify one file
        import time
        time.sleep(1.5)
        (sample_project_path / "main.py").write_text("print('modified')")

        # Check again - main.py should need update
        needs_update = temp_db.get_files_needing_update(
            sample_project_path,
            ["main.py", "utils.py"]
        )
        assert "main.py" in needs_update
        assert "utils.py" not in needs_update

        # New file should also be in needs_update
        needs_update = temp_db.get_files_needing_update(
            sample_project_path,
            ["main.py", "new.py"]
        )
        assert "new.py" in needs_update


class TestFileTypeStats:
    """Tests for file type statistics."""

    def test_get_file_stats(self, temp_db, sample_project_path):
        """Test getting file type statistics."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python",
                100
            ),
            create_file_info(
                sample_project_path / "utils.py",
                "utils.py",
                "python",
                200
            ),
            create_file_info(
                sample_project_path / "app.tsx",
                "app.tsx",
                "typescript",
                300
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        stats = temp_db.get_file_stats(project_id)

        assert len(stats) == 2

        # Check Python stats
        python_stats = next(s for s in stats if s.file_type == "python")
        assert python_stats.count == 2
        assert abs(python_stats.percentage - 66.67) < 1

        # Check TypeScript stats
        ts_stats = next(s for s in stats if s.file_type == "typescript")
        assert ts_stats.count == 1
        assert abs(ts_stats.percentage - 33.33) < 1


class TestDepthStats:
    """Tests for depth statistics."""

    def test_get_depth_stats(self, temp_db, sample_project_path):
        """Test getting depth statistics."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),  # depth 0
            create_file_info(
                sample_project_path / "src" / "app.py",
                "src/app.py",
                "python"
            ),  # depth 1
            create_file_info(
                sample_project_path / "src" / "components" / "button.tsx",
                "src/components/button.tsx",
                "typescript"
            ),  # depth 2
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        stats = temp_db.get_depth_stats(project_id)

        assert stats.min_file_depth == 0
        assert stats.max_file_depth == 2
        assert 0 < stats.avg_file_depth < 2
        assert sum(stats.depth_distribution.values()) == 3


class TestReferenceRanking:
    """Tests for reference ranking."""

    def test_get_reference_ranking_with_imports(self, temp_db, sample_project_path):
        """Test getting reference ranking with resolved imports."""
        # Create files with imports
        utils_imports = []
        main_imports = [
            ImportInfo(module="./utils", import_type="static", line=1),
        ]
        app_imports = [
            ImportInfo(module="./utils", import_type="static", line=1),
        ]

        files = [
            create_file_info(
                sample_project_path / "utils.py",
                "utils.py",
                "python",
                100,
                utils_imports
            ),
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python",
                100,
                main_imports
            ),
            create_file_info(
                sample_project_path / "app.py",
                "app.py",
                "python",
                100,
                app_imports
            ),
        ]

        # Create graph with edges
        graph = {
            "nodes": [],
            "edges": [
                {"source": "main.py", "target": "utils.py"},
                {"source": "app.py", "target": "utils.py"},
            ]
        }

        project_id = temp_db.save_project(sample_project_path, files, graph)
        ranking = temp_db.get_reference_ranking(project_id, limit=10)

        # utils.py should be the most referenced
        if ranking:
            assert ranking[0].file_path == "utils.py"
            assert ranking[0].reference_count == 2


class TestFunctionStorage:
    """Tests for function and function call storage."""

    def test_save_and_get_functions(self, temp_db, sample_project_path):
        """Test saving and retrieving functions."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        file_record = temp_db.get_file_by_path(project_id, "main.py")

        functions = [
            {
                "name": "main",
                "signature": "def main() -> None",
                "start_line": 1,
                "end_line": 10
            },
            {
                "name": "helper",
                "signature": "def helper(x: int) -> int",
                "start_line": 12,
                "end_line": 15
            },
        ]

        function_ids = temp_db.save_functions(file_record.id, functions)
        assert len(function_ids) == 2

        retrieved = temp_db.get_functions_by_file(file_record.id)
        assert len(retrieved) == 2
        names = {f.name for f in retrieved}
        assert names == {"main", "helper"}

    def test_save_and_get_function_calls(self, temp_db, sample_project_path):
        """Test saving and retrieving function calls."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        file_record = temp_db.get_file_by_path(project_id, "main.py")

        function_ids = temp_db.save_functions(file_record.id, [
            {"name": "main", "signature": "def main()", "start_line": 1, "end_line": 10}
        ])

        calls = [
            {"callee_name": "print", "line": 5},
            {"callee_name": "helper", "line": 7},
        ]

        call_ids = temp_db.save_function_calls(function_ids[0], calls)
        assert len(call_ids) == 2

        retrieved = temp_db.get_function_calls(function_ids[0])
        assert len(retrieved) == 2
        callee_names = {c.callee_name for c in retrieved}
        assert callee_names == {"print", "helper"}


class TestCodeSummaries:
    """Tests for code summary storage."""

    def test_save_and_get_summaries(self, temp_db, sample_project_path):
        """Test saving and retrieving code summaries."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        file_record = temp_db.get_file_by_path(project_id, "main.py")

        summaries = [
            {
                "file_id": file_record.id,
                "entity_type": "function",
                "entity_name": "main",
                "signature": "def main() -> None",
                "summary": "Entry point of the application.",
                "line_number": 1
            },
        ]

        summary_ids = temp_db.save_summaries(summaries)
        assert len(summary_ids) == 1

        summary = temp_db.get_summary(file_record.id, "main")
        assert summary is not None
        assert summary.entity_name == "main"
        assert summary.summary == "Entry point of the application."

    def test_get_summaries_by_file(self, temp_db, sample_project_path):
        """Test getting all summaries for a file."""
        files = [
            create_file_info(
                sample_project_path / "main.py",
                "main.py",
                "python"
            ),
        ]
        graph = {"nodes": [], "edges": []}

        project_id = temp_db.save_project(sample_project_path, files, graph)
        file_record = temp_db.get_file_by_path(project_id, "main.py")

        summaries = [
            {
                "file_id": file_record.id,
                "entity_type": "function",
                "entity_name": "main",
                "signature": "def main()",
                "summary": "Main function.",
                "line_number": 1
            },
            {
                "file_id": file_record.id,
                "entity_type": "class",
                "entity_name": "App",
                "signature": "class App",
                "summary": "Application class.",
                "line_number": 10
            },
        ]

        temp_db.save_summaries(summaries)
        retrieved = temp_db.get_summaries_by_file(file_record.id)

        assert len(retrieved) == 2
        names = {s.entity_name for s in retrieved}
        assert names == {"main", "App"}


# ============================================================
# Property-Based Tests
# ============================================================

# Strategies for generating test data
file_type_strategy = st.sampled_from(["python", "javascript", "typescript", "vue"])
path_component_strategy = st.text(
    alphabet=st.sampled_from("abcdefghijklmnopqrstuvwxyz0123456789_"),
    min_size=1,
    max_size=20
)


@st.composite
def file_info_strategy(draw, base_path: Path = Path("/tmp/test")):
    """Generate random FileInfo objects."""
    # Generate path components
    depth = draw(st.integers(min_value=0, max_value=4))
    components = [draw(path_component_strategy) for _ in range(depth)]

    file_name = draw(path_component_strategy)
    file_type = draw(file_type_strategy)

    ext_map = {
        "python": ".py",
        "javascript": ".js",
        "typescript": ".ts",
        "vue": ".vue"
    }
    file_name = file_name + ext_map.get(file_type, ".py")

    if components:
        relative_path = "/".join(components) + "/" + file_name
    else:
        relative_path = file_name

    size = draw(st.integers(min_value=0, max_value=1000000))

    return FileInfo(
        path=base_path / relative_path,
        relative_path=relative_path,
        file_type=file_type,
        size=size,
        imports=[]
    )


@st.composite
def file_list_strategy(draw, base_path: Path = Path("/tmp/test")):
    """Generate a list of unique FileInfo objects."""
    count = draw(st.integers(min_value=1, max_value=20))
    files = []
    seen_paths = set()

    for _ in range(count):
        file_info = draw(file_info_strategy(base_path))
        if file_info.relative_path not in seen_paths:
            files.append(file_info)
            seen_paths.add(file_info.relative_path)

    return files if files else [draw(file_info_strategy(base_path))]


class TestStorageRoundTrip:
    """Property-based tests for storage round-trip consistency.

    Feature: code-knowledge-graph-enhancement, Property 1: Storage Round-Trip Consistency
    Validates: Requirements 1.1, 1.3, 1.4
    """

    @given(files=file_list_strategy())
    @settings(max_examples=50, deadline=None)
    def test_file_count_preserved(self, files):
        """All files saved should be retrievable."""
        fd, path = tempfile.mkstemp(suffix=".db")
        os.close(fd)

        try:
            storage = SQLiteStorage(path)
            project_path = Path("/tmp/test_project")

            graph = {"nodes": [], "edges": []}
            project_id = storage.save_project(project_path, files, graph)

            # Verify file count
            project = storage.get_project_by_id(project_id)
            assert project.file_count == len(files)

            # Verify all files are retrievable
            retrieved = storage.get_files_by_project(project_id)
            assert len(retrieved) == len(files)

            storage.close()
        finally:
            os.unlink(path)

    @given(files=file_list_strategy())
    @settings(max_examples=50, deadline=None)
    def test_file_metadata_preserved(self, files):
        """File metadata should be preserved exactly."""
        fd, path = tempfile.mkstemp(suffix=".db")
        os.close(fd)

        try:
            storage = SQLiteStorage(path)
            project_path = Path("/tmp/test_project")

            graph = {"nodes": [], "edges": []}
            project_id = storage.save_project(project_path, files, graph)

            # Verify each file's metadata
            for original in files:
                retrieved = storage.get_file_by_path(
                    project_id,
                    original.relative_path
                )
                assert retrieved is not None
                assert retrieved.relative_path == original.relative_path
                assert retrieved.file_type == original.file_type
                assert retrieved.size == original.size

            storage.close()
        finally:
            os.unlink(path)


class TestFileStatsInvariant:
    """Property-based tests for file statistics invariant.

    Feature: code-knowledge-graph-enhancement, Property 2: File Statistics Invariant
    Validates: Requirements 2.1, 2.2, 2.3, 2.4
    """

    @given(files=file_list_strategy())
    @settings(max_examples=50, deadline=None)
    def test_stats_count_equals_total(self, files):
        """Sum of file counts across all types should equal total files."""
        fd, path = tempfile.mkstemp(suffix=".db")
        os.close(fd)

        try:
            storage = SQLiteStorage(path)
            project_path = Path("/tmp/test_project")

            graph = {"nodes": [], "edges": []}
            project_id = storage.save_project(project_path, files, graph)

            stats = storage.get_file_stats(project_id)
            total_from_stats = sum(s.count for s in stats)

            assert total_from_stats == len(files)

            storage.close()
        finally:
            os.unlink(path)

    @given(files=file_list_strategy())
    @settings(max_examples=50, deadline=None)
    def test_percentages_sum_to_100(self, files):
        """All percentages should sum to 100%."""
        fd, path = tempfile.mkstemp(suffix=".db")
        os.close(fd)

        try:
            storage = SQLiteStorage(path)
            project_path = Path("/tmp/test_project")

            graph = {"nodes": [], "edges": []}
            project_id = storage.save_project(project_path, files, graph)

            stats = storage.get_file_stats(project_id)
            total_percentage = sum(s.percentage for s in stats)

            # Allow for floating-point tolerance
            assert abs(total_percentage - 100.0) < 0.1

            storage.close()
        finally:
            os.unlink(path)


class TestDepthStatsInvariant:
    """Property-based tests for depth statistics invariant.

    Feature: code-knowledge-graph-enhancement, Property 4: Depth Statistics Invariant
    Validates: Requirements 4.1, 4.2, 4.3, 4.4, 4.5
    """

    @given(files=file_list_strategy())
    @settings(max_examples=50, deadline=None)
    def test_depth_bounds(self, files):
        """min_depth <= avg_depth <= max_depth."""
        fd, path = tempfile.mkstemp(suffix=".db")
        os.close(fd)

        try:
            storage = SQLiteStorage(path)
            project_path = Path("/tmp/test_project")

            graph = {"nodes": [], "edges": []}
            project_id = storage.save_project(project_path, files, graph)

            stats = storage.get_depth_stats(project_id)

            assert stats.min_file_depth <= stats.avg_file_depth
            assert stats.avg_file_depth <= stats.max_file_depth

            storage.close()
        finally:
            os.unlink(path)

    @given(files=file_list_strategy())
    @settings(max_examples=50, deadline=None)
    def test_distribution_count_equals_total(self, files):
        """Sum of depth distribution counts should equal total file count."""
        fd, path = tempfile.mkstemp(suffix=".db")
        os.close(fd)

        try:
            storage = SQLiteStorage(path)
            project_path = Path("/tmp/test_project")

            graph = {"nodes": [], "edges": []}
            project_id = storage.save_project(project_path, files, graph)

            stats = storage.get_depth_stats(project_id)
            total_from_distribution = sum(stats.depth_distribution.values())

            assert total_from_distribution == len(files)

            storage.close()
        finally:
            os.unlink(path)


class TestReferenceRankingCorrectness:
    """Property-based tests for reference ranking correctness.

    Feature: code-knowledge-graph-enhancement, Property 3: Reference Ranking Correctness
    Validates: Requirements 3.1, 3.2, 3.3, 3.4, 3.5
    """

    @given(limit=st.integers(min_value=1, max_value=100))
    @settings(max_examples=20, deadline=None)
    def test_ranking_respects_limit(self, limit):
        """Result count should be <= specified limit."""
        fd, path = tempfile.mkstemp(suffix=".db")
        os.close(fd)

        try:
            storage = SQLiteStorage(path)
            project_path = Path("/tmp/test_project")

            # Create more files than the limit
            files = [
                create_file_info(
                    project_path / f"file{i}.py",
                    f"file{i}.py",
                    "python"
                )
                for i in range(limit + 10)
            ]
            graph = {"nodes": [], "edges": []}
            project_id = storage.save_project(project_path, files, graph)

            ranking = storage.get_reference_ranking(project_id, limit=limit)
            assert len(ranking) <= limit

            storage.close()
        finally:
            os.unlink(path)

    def test_ranking_sorted_descending(self, temp_db, sample_project_path):
        """Results should be sorted in descending order by reference count."""
        # Create files with known reference counts
        imports1 = [ImportInfo(module="./utils", import_type="static", line=1)]
        imports2 = [ImportInfo(module="./utils", import_type="static", line=1)]
        imports3 = [ImportInfo(module="./utils", import_type="static", line=1)]

        files = [
            create_file_info(
                sample_project_path / "utils.py",
                "utils.py",
                "python",
                imports=[]
            ),
            create_file_info(
                sample_project_path / "a.py",
                "a.py",
                "python",
                imports=imports1
            ),
            create_file_info(
                sample_project_path / "b.py",
                "b.py",
                "python",
                imports=imports2
            ),
            create_file_info(
                sample_project_path / "c.py",
                "c.py",
                "python",
                imports=imports3
            ),
        ]

        graph = {
            "nodes": [],
            "edges": [
                {"source": "a.py", "target": "utils.py"},
                {"source": "b.py", "target": "utils.py"},
                {"source": "c.py", "target": "utils.py"},
            ]
        }

        project_id = temp_db.save_project(sample_project_path, files, graph)
        ranking = temp_db.get_reference_ranking(project_id)

        # Verify sorted descending
        for i in range(len(ranking) - 1):
            assert ranking[i].reference_count >= ranking[i + 1].reference_count
