"""MCP Server implementation for Code Knowledge Graph.

This module implements the Model Context Protocol (MCP) server
that exposes code knowledge graph tools for AI integration.
"""

import json
from pathlib import Path
from typing import Any, Optional

from core.storage import StorageBackend, SQLiteStorage
from core.services import (
    StatsService,
    ProjectService,
    FunctionAnalysisService,
    RelatedContextService,
    LLMConfig,
    EmbeddingConfig,
    EmbeddingProviderType,
    CodeSummarizer,
    create_llm_provider,
    create_embedding_provider,
    create_summarizer,
)


class MCPServer:
    """MCP Protocol Server for Code Knowledge Graph.

    Provides tools for:
    - Project scanning and analysis
    - File type statistics
    - Reference ranking
    - Depth analysis
    - Function relations
    - Related code context (Repo Map)
    - LLM-powered code summarization (optional)
    - Semantic code search via embeddings (optional)
    """

    def __init__(
        self,
        storage: Optional[StorageBackend] = None,
        db_path: str = "code_knowledge.db",
        llm_config: Optional[LLMConfig] = None,
        embedding_config: Optional[EmbeddingConfig] = None
    ):
        """Initialize MCP server.

        Args:
            storage: Optional storage backend instance
            db_path: Path to SQLite database (used if storage not provided)
            llm_config: Optional LLM configuration for code summarization
                       If not provided, summarization will be disabled
            embedding_config: Optional embedding configuration for semantic search
                             If not provided, embedding will be disabled
        """
        self.storage = storage or SQLiteStorage(db_path)
        self.stats_service = StatsService(self.storage)
        self.project_service = ProjectService(self.storage)
        self.function_analysis = FunctionAnalysisService()
        self.related_context_service = None  # Initialized per project

        # Store configurations
        self.llm_config = llm_config
        self.embedding_config = embedding_config

        # Create summarizer with optional LLM and embedding providers
        self.summarizer = create_summarizer(
            storage=self.storage,
            llm_config=llm_config,
            embedding_config=embedding_config
        )

    @classmethod
    def from_config(
        cls,
        db_path: str = "code_knowledge.db",
        llm_api_url: Optional[str] = None,
        llm_api_key: Optional[str] = None,
        llm_model: Optional[str] = None,
        embedding_provider_type: str = "openai",
        embedding_api_url: Optional[str] = None,
        embedding_api_key: Optional[str] = None,
        embedding_model: Optional[str] = None
    ) -> "MCPServer":
        """Create MCPServer from configuration parameters.

        This is a convenience method for creating the server with
        configuration passed as individual parameters.

        Args:
            db_path: Path to SQLite database
            llm_api_url: LLM API base URL (OpenAI-compatible)
            llm_api_key: LLM API key
            llm_model: LLM model name
            embedding_provider_type: "openai" or "ollama"
            embedding_api_url: Embedding API base URL
            embedding_api_key: Embedding API key (not required for Ollama)
            embedding_model: Embedding model name

        Returns:
            Configured MCPServer instance
        """
        # Build LLM config if all required params provided
        llm_config = None
        if llm_api_url and llm_api_key and llm_model:
            llm_config = LLMConfig(
                api_url=llm_api_url,
                api_key=llm_api_key,
                model=llm_model
            )

        # Build embedding config if required params provided
        embedding_config = None
        if embedding_api_url and embedding_model:
            provider_type = EmbeddingProviderType(embedding_provider_type)
            embedding_config = EmbeddingConfig(
                provider_type=provider_type,
                api_url=embedding_api_url,
                api_key=embedding_api_key or "",
                model=embedding_model
            )

        return cls(
            db_path=db_path,
            llm_config=llm_config,
            embedding_config=embedding_config
        )

    def get_tools(self) -> list[dict]:
        """Return list of available MCP tools.

        Returns:
            List of tool definitions with name, description, and input schema
        """
        tools = [
            {
                "name": "scan_project",
                "description": "扫描并分析项目代码依赖 | Scan and analyze project code dependencies",
                "inputSchema": {
                    "type": "object",
                    "properties": {
                        "path": {
                            "type": "string",
                            "description": "项目路径 | Project path"
                        },
                        "generate_summaries": {
                            "type": "boolean",
                            "default": False,
                            "description": "是否生成代码摘要（需要配置LLM）| Whether to generate code summaries (requires LLM config)"
                        }
                    },
                    "required": ["path"]
                }
            },
            {
                "name": "get_file_stats",
                "description": "获取项目文件类型统计 | Get project file type statistics",
                "inputSchema": {
                    "type": "object",
                    "properties": {
                        "path": {
                            "type": "string",
                            "description": "项目路径 | Project path"
                        },
                        "subdirectory": {
                            "type": "string",
                            "description": "子目录过滤（可选）| Subdirectory filter (optional)"
                        }
                    },
                    "required": ["path"]
                }
            },
            {
                "name": "get_reference_ranking",
                "description": "获取被引用最多的文件排名 | Get top referenced files ranking",
                "inputSchema": {
                    "type": "object",
                    "properties": {
                        "path": {
                            "type": "string",
                            "description": "项目路径 | Project path"
                        },
                        "limit": {
                            "type": "integer",
                            "default": 20,
                            "description": "返回结果数量限制 | Limit of results"
                        },
                        "file_type": {
                            "type": "string",
                            "description": "文件类型过滤（可选）| File type filter (optional)"
                        }
                    },
                    "required": ["path"]
                }
            },
            {
                "name": "get_depth_analysis",
                "description": "获取目录和文件层级分析 | Get directory and file depth analysis",
                "inputSchema": {
                    "type": "object",
                    "properties": {
                        "path": {
                            "type": "string",
                            "description": "项目路径 | Project path"
                        },
                        "subdirectory": {
                            "type": "string",
                            "description": "子目录过滤（可选）| Subdirectory filter (optional)"
                        }
                    },
                    "required": ["path"]
                }
            },
            {
                "name": "get_function_relations",
                "description": "获取指定文件间的函数调用关系（最多10个文件）| Get function call relations between files (max 10)",
                "inputSchema": {
                    "type": "object",
                    "properties": {
                        "files": {
                            "type": "array",
                            "items": {"type": "string"},
                            "maxItems": 10,
                            "description": "要分析的文件路径列表 | List of file paths to analyze"
                        }
                    },
                    "required": ["files"]
                }
            },
            {
                "name": "get_related_code_context",
                "description": "获取文件的关联代码上下文(Repo Map)，包含目标文件及其依赖文件的签名和摘要 | Get related code context (Repo Map) with signatures and summaries",
                "inputSchema": {
                    "type": "object",
                    "properties": {
                        "project_path": {
                            "type": "string",
                            "description": "项目路径 | Project path"
                        },
                        "file_path": {
                            "type": "string",
                            "description": "目标文件相对路径 | Target file relative path"
                        },
                        "hops": {
                            "type": "integer",
                            "default": 1,
                            "minimum": 1,
                            "maximum": 3,
                            "description": "依赖跳数 | Dependency hops"
                        }
                    },
                    "required": ["project_path", "file_path"]
                }
            },
            {
                "name": "generate_summaries",
                "description": "为项目生成代码摘要（需要配置LLM）| Generate code summaries for project (requires LLM config)",
                "inputSchema": {
                    "type": "object",
                    "properties": {
                        "path": {
                            "type": "string",
                            "description": "项目路径 | Project path"
                        },
                        "file_path": {
                            "type": "string",
                            "description": "指定文件路径（可选，不指定则处理引用最多的文件）| Specific file path (optional)"
                        },
                        "force_rescan": {
                            "type": "boolean",
                            "default": False,
                            "description": "是否强制重新生成 | Whether to force regeneration"
                        }
                    },
                    "required": ["path"]
                }
            },
            {
                "name": "semantic_search",
                "description": "语义搜索代码（需要配置Embedding）| Semantic code search (requires embedding config)",
                "inputSchema": {
                    "type": "object",
                    "properties": {
                        "path": {
                            "type": "string",
                            "description": "项目路径 | Project path"
                        },
                        "query": {
                            "type": "string",
                            "description": "搜索查询（自然语言）| Search query (natural language)"
                        },
                        "limit": {
                            "type": "integer",
                            "default": 10,
                            "description": "返回结果数量限制 | Limit of results"
                        }
                    },
                    "required": ["path", "query"]
                }
            },
            {
                "name": "get_summary_status",
                "description": "获取LLM摘要和嵌入配置状态 | Get LLM summarization and embedding configuration status",
                "inputSchema": {
                    "type": "object",
                    "properties": {},
                    "required": []
                }
            }
        ]

        return tools

    async def call_tool(self, name: str, arguments: dict) -> dict[str, Any]:
        """Execute a tool call.

        Args:
            name: Tool name
            arguments: Tool arguments

        Returns:
            Tool result as dictionary

        Raises:
            ValueError: If tool name is unknown or arguments are invalid
        """
        handlers = {
            "scan_project": self._handle_scan_project,
            "get_file_stats": self._handle_get_file_stats,
            "get_reference_ranking": self._handle_get_reference_ranking,
            "get_depth_analysis": self._handle_get_depth_analysis,
            "get_function_relations": self._handle_get_function_relations,
            "get_related_code_context": self._handle_get_related_code_context,
            "generate_summaries": self._handle_generate_summaries,
            "semantic_search": self._handle_semantic_search,
            "get_summary_status": self._handle_get_summary_status,
        }

        handler = handlers.get(name)
        if not handler:
            return self._error_response(
                "Unknown tool",
                "UNKNOWN_TOOL",
                f"Tool '{name}' is not available"
            )

        try:
            return await handler(arguments)
        except ValueError as e:
            return self._error_response(
                "Invalid parameters",
                "INVALID_PARAMS",
                str(e)
            )
        except Exception as e:
            return self._error_response(
                "Internal error",
                "INTERNAL",
                str(e)
            )

    async def _handle_scan_project(self, args: dict) -> dict:
        """Handle scan_project tool call."""
        path = args.get("path")
        if not path:
            raise ValueError("Missing required parameter: path")

        project_path = Path(path)
        if not project_path.exists():
            return self._error_response(
                "Project not found",
                "NOT_FOUND",
                f"Path does not exist: {path}"
            )

        generate_summaries = args.get("generate_summaries", False)
        result = self.project_service.scan_project(str(project_path.resolve()))

        response = {
            "success": True,
            "project_path": result.project_path,
            "project_name": result.project_name,
            "file_count": result.total_files,
            "last_scanned": result.scan_mode,
            "file_types": result.file_types,
            "llm_enabled": self.summarizer.is_llm_enabled,
            "embedding_enabled": self.summarizer.is_embedding_enabled
        }

        # Generate summaries if requested and LLM is configured
        if generate_summaries:
            if not self.summarizer.is_llm_enabled:
                response["summary_warning"] = "LLM not configured, summaries not generated | LLM未配置，摘要未生成"
            else:
                try:
                    stats = await self.summarizer.summarize_project(
                        result.project_id,
                        result.project_path
                    )
                    response["summarization"] = {
                        "total_entities": stats.total_entities,
                        "successful": stats.successful,
                        "failed": stats.failed,
                        "files_processed": stats.files_processed
                    }
                except Exception as e:
                    response["summary_error"] = str(e)

        return response

    async def _handle_get_file_stats(self, args: dict) -> dict:
        """Handle get_file_stats tool call."""
        path = args.get("path")
        if not path:
            raise ValueError("Missing required parameter: path")

        subdirectory = args.get("subdirectory")

        response = self.stats_service.get_file_type_stats(
            str(Path(path).resolve()),
            subdirectory=subdirectory
        )

        if response is None:
            return self._error_response(
                "Project not found",
                "NOT_FOUND",
                f"Project not found: {path}"
            )

        return {
            "project_path": response.project_path,
            "subdirectory": subdirectory,
            "total_files": response.total_files,
            "stats": [
                {
                    "type": s.file_type,
                    "count": s.count,
                    "percentage": s.percentage,
                    "total_size": s.total_size
                }
                for s in response.stats
            ]
        }

    async def _handle_get_reference_ranking(self, args: dict) -> dict:
        """Handle get_reference_ranking tool call."""
        path = args.get("path")
        if not path:
            raise ValueError("Missing required parameter: path")

        limit = args.get("limit", 20)
        file_type = args.get("file_type")

        response = self.stats_service.get_reference_ranking(
            str(Path(path).resolve()),
            limit=limit,
            file_type=file_type
        )

        if response is None:
            return self._error_response(
                "Project not found",
                "NOT_FOUND",
                f"Project not found: {path}"
            )

        return {
            "project_path": response.project_path,
            "limit": limit,
            "file_type_filter": file_type,
            "results": [
                {
                    "file": r.file_path,
                    "count": r.reference_count,
                    "references": r.referencing_files
                }
                for r in response.results
            ]
        }

    async def _handle_get_depth_analysis(self, args: dict) -> dict:
        """Handle get_depth_analysis tool call."""
        path = args.get("path")
        if not path:
            raise ValueError("Missing required parameter: path")

        subdirectory = args.get("subdirectory")

        response = self.stats_service.get_depth_analysis(
            str(Path(path).resolve()),
            subdirectory=subdirectory
        )

        if response is None:
            return self._error_response(
                "Project not found",
                "NOT_FOUND",
                f"Project not found: {path}"
            )

        return {
            "project_path": response.project_path,
            "subdirectory": subdirectory,
            "directory_depth": response.directory_depth,
            "file_depth": response.file_depth
        }

    async def _handle_get_function_relations(self, args: dict) -> dict:
        """Handle get_function_relations tool call."""
        files = args.get("files")
        if not files:
            raise ValueError("Missing required parameter: files")

        if len(files) > 10:
            return self._error_response(
                "Too many files",
                "LIMIT_EXCEEDED",
                "Maximum 10 files allowed for function analysis"
            )

        try:
            result = self.function_analysis.analyze_files(files)
            return self.function_analysis.to_dict(result)
        except FileNotFoundError as e:
            return self._error_response(
                "File not found",
                "NOT_FOUND",
                str(e)
            )

    async def _handle_get_related_code_context(self, args: dict) -> dict:
        """Handle get_related_code_context tool call."""
        project_path = args.get("project_path")
        file_path = args.get("file_path")
        hops = args.get("hops", 1)

        if not project_path:
            raise ValueError("Missing required parameter: project_path")
        if not file_path:
            raise ValueError("Missing required parameter: file_path")

        project_root = Path(project_path)
        if not project_root.exists():
            return self._error_response(
                "Project not found",
                "NOT_FOUND",
                f"Path does not exist: {project_path}"
            )

        # Create service for this project
        context_service = RelatedContextService(self.storage, project_root)

        result = context_service.get_related_context(
            str(project_root.resolve()),
            file_path,
            hops=hops
        )

        return context_service.to_dict(result)

    async def _handle_generate_summaries(self, args: dict) -> dict:
        """Handle generate_summaries tool call."""
        path = args.get("path")
        if not path:
            raise ValueError("Missing required parameter: path")

        project_path = Path(path)
        if not project_path.exists():
            return self._error_response(
                "Project not found",
                "NOT_FOUND",
                f"Path does not exist: {path}"
            )

        if not self.summarizer.is_llm_enabled:
            return self._error_response(
                "LLM not configured | LLM未配置",
                "LLM_NOT_CONFIGURED",
                "Please configure LLM API URL, key, and model to use summarization | 请配置LLM的API URL、密钥和模型以使用摘要功能"
            )

        # Get project from storage
        project = self.storage.get_project(str(project_path.resolve()))
        if not project:
            return self._error_response(
                "Project not scanned",
                "NOT_SCANNED",
                "Please scan the project first using scan_project | 请先使用scan_project扫描项目"
            )

        file_path = args.get("file_path")
        force_rescan = args.get("force_rescan", False)

        try:
            if file_path:
                # Summarize specific file
                stats = await self.summarizer.summarize_file(
                    project.id,
                    str(project_path.resolve()),
                    file_path,
                    force_rescan=force_rescan
                )
            else:
                # Summarize top-referenced files
                stats = await self.summarizer.summarize_project(
                    project.id,
                    str(project_path.resolve()),
                    force_rescan=force_rescan
                )

            return {
                "success": True,
                "project_path": str(project_path.resolve()),
                "file_path": file_path,
                "total_entities": stats.total_entities,
                "successful": stats.successful,
                "failed": stats.failed,
                "skipped": stats.skipped,
                "files_processed": stats.files_processed,
                "message_en": f"Successfully generated {stats.successful} summaries",
                "message_zh": f"成功生成{stats.successful}个摘要"
            }
        except FileNotFoundError as e:
            return self._error_response(
                "File not found",
                "NOT_FOUND",
                str(e)
            )

    async def _handle_semantic_search(self, args: dict) -> dict:
        """Handle semantic_search tool call."""
        path = args.get("path")
        query = args.get("query")

        if not path:
            raise ValueError("Missing required parameter: path")
        if not query:
            raise ValueError("Missing required parameter: query")

        project_path = Path(path)
        if not project_path.exists():
            return self._error_response(
                "Project not found",
                "NOT_FOUND",
                f"Path does not exist: {path}"
            )

        if not self.summarizer.is_embedding_enabled:
            return self._error_response(
                "Embedding not configured | Embedding未配置",
                "EMBEDDING_NOT_CONFIGURED",
                "Please configure embedding API URL and model to use semantic search | 请配置Embedding的API URL和模型以使用语义搜索"
            )

        # Get project from storage
        project = self.storage.get_project(str(project_path.resolve()))
        if not project:
            return self._error_response(
                "Project not scanned",
                "NOT_SCANNED",
                "Please scan the project first using scan_project | 请先使用scan_project扫描项目"
            )

        limit = args.get("limit", 10)

        try:
            results = await self.summarizer.get_summary_by_embedding(
                project.id,
                query,
                limit=limit
            )

            return {
                "success": True,
                "project_path": str(project_path.resolve()),
                "query": query,
                "results": [
                    {
                        "entity_type": r.entity_type,
                        "entity_name": r.entity_name,
                        "signature": r.signature,
                        "summary": r.summary,
                        "summary_en": r.summary_en,
                        "summary_zh": r.summary_zh,
                        "line_number": r.line_number
                    }
                    for r in results
                ],
                "total_results": len(results)
            }
        except Exception as e:
            return self._error_response(
                "Search failed",
                "SEARCH_ERROR",
                str(e)
            )

    async def _handle_get_summary_status(self, args: dict) -> dict:
        """Handle get_summary_status tool call."""
        return {
            "llm_enabled": self.summarizer.is_llm_enabled,
            "embedding_enabled": self.summarizer.is_embedding_enabled,
            "llm_config": {
                "configured": self.llm_config is not None and self.llm_config.is_valid(),
                "api_url": self.llm_config.api_url if self.llm_config else None,
                "model": self.llm_config.model if self.llm_config else None
            } if self.llm_config else {"configured": False},
            "embedding_config": {
                "configured": self.embedding_config is not None and self.embedding_config.is_valid(),
                "provider_type": self.embedding_config.provider_type.value if self.embedding_config else None,
                "api_url": self.embedding_config.api_url if self.embedding_config else None,
                "model": self.embedding_config.model if self.embedding_config else None
            } if self.embedding_config else {"configured": False},
            "message_en": "LLM and embedding status",
            "message_zh": "LLM和嵌入模型状态"
        }

    def _error_response(
        self,
        error: str,
        code: str,
        details: Optional[str] = None
    ) -> dict:
        """Create an error response.

        Args:
            error: Error message
            code: Error code
            details: Optional detailed message

        Returns:
            Error response dictionary
        """
        response = {
            "error": error,
            "code": code
        }
        if details:
            response["details"] = details
        return response

    async def close_async(self) -> None:
        """Close the MCP server and release resources asynchronously."""
        if self.summarizer:
            await self.summarizer.close()
        if hasattr(self.storage, 'close'):
            self.storage.close()

    def close(self) -> None:
        """Close the MCP server and release resources."""
        if hasattr(self.storage, 'close'):
            self.storage.close()
