"""Tests for MCP Server.

This module tests the MCPServer for correctness
according to Property 6: MCP Response Correctness.
"""

import asyncio
from pathlib import Path

import pytest
from hypothesis import given, settings, strategies as st

from mcp_server.server import MCPServer
from core.storage.sqlite import SQLiteStorage
from core.scanner import CodeScanner
from core.graph import GraphBuilder


def scan_and_save_project(project_dir: Path, storage: SQLiteStorage):
    """Helper function to scan a project and save to storage."""
    scanner = CodeScanner(project_dir)
    files = scanner.scan()
    graph_builder = GraphBuilder(project_dir)
    graph_data = graph_builder.build(files)
    graph = graph_data.to_dict()
    storage.save_project(project_dir, files, graph)
    return files, graph


class TestMCPServerTools:
    """Tests for MCP Server tool definitions."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    @pytest.fixture
    def mcp_server(self, temp_db):
        """Create MCP server with temp storage."""
        server = MCPServer(storage=temp_db)
        yield server

    @pytest.fixture
    def sample_project(self, tmp_path):
        """Create a sample project."""
        project_dir = tmp_path / "sample_project"
        project_dir.mkdir()

        (project_dir / "main.py").write_text("""
from utils import helper

def main():
    return helper("test")
""")

        (project_dir / "utils.py").write_text("""
def helper(s: str) -> str:
    \"\"\"Helper function.\"\"\"
    return s.upper()

def another():
    return "another"
""")

        (project_dir / "config.py").write_text("""
VALUE = 42
""")

        return project_dir

    def test_get_tools_returns_list(self, mcp_server):
        """Test that get_tools returns a list of tool definitions."""
        tools = mcp_server.get_tools()

        assert isinstance(tools, list)
        assert len(tools) > 0

    def test_tools_have_required_fields(self, mcp_server):
        """Test that each tool has required fields.

        Property 6: Each tool should have name, description, and inputSchema.
        """
        tools = mcp_server.get_tools()

        for tool in tools:
            assert "name" in tool, f"Tool missing 'name': {tool}"
            assert "description" in tool, f"Tool missing 'description': {tool}"
            assert "inputSchema" in tool, f"Tool missing 'inputSchema': {tool}"

    def test_tool_names_are_unique(self, mcp_server):
        """Test that all tool names are unique."""
        tools = mcp_server.get_tools()
        names = [t["name"] for t in tools]

        assert len(names) == len(set(names)), "Duplicate tool names found"

    def test_expected_tools_present(self, mcp_server):
        """Test that all expected tools are present."""
        tools = mcp_server.get_tools()
        tool_names = {t["name"] for t in tools}

        expected_tools = {
            "scan_project",
            "get_file_stats",
            "get_reference_ranking",
            "get_depth_analysis",
            "get_function_relations",
            "get_related_code_context"
        }

        for expected in expected_tools:
            assert expected in tool_names, f"Missing tool: {expected}"


class TestMCPServerScanProject:
    """Tests for scan_project tool."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    @pytest.fixture
    def mcp_server(self, temp_db):
        """Create MCP server with temp storage."""
        server = MCPServer(storage=temp_db)
        yield server

    @pytest.fixture
    def sample_project(self, tmp_path):
        """Create a sample project."""
        project_dir = tmp_path / "sample_project"
        project_dir.mkdir()

        (project_dir / "main.py").write_text("def main(): pass")
        (project_dir / "utils.py").write_text("def util(): pass")

        return project_dir

    @pytest.mark.asyncio
    async def test_scan_project_success(self, mcp_server, sample_project):
        """Test successful project scan.

        Property 6: scan_project response should contain file count and type statistics.
        """
        result = await mcp_server.call_tool(
            "scan_project",
            {"path": str(sample_project)}
        )

        assert "error" not in result
        assert result.get("success") is True
        assert "file_count" in result
        assert result["file_count"] >= 2
        assert "project_name" in result
        assert "last_scanned" in result

    @pytest.mark.asyncio
    async def test_scan_project_not_found(self, mcp_server):
        """Test scan_project with non-existent path."""
        result = await mcp_server.call_tool(
            "scan_project",
            {"path": "/nonexistent/path"}
        )

        assert "error" in result
        assert result["code"] == "NOT_FOUND"

    @pytest.mark.asyncio
    async def test_scan_project_missing_path(self, mcp_server):
        """Test scan_project with missing path parameter."""
        result = await mcp_server.call_tool(
            "scan_project",
            {}
        )

        assert "error" in result
        assert result["code"] == "INVALID_PARAMS"


class TestMCPServerGetFileStats:
    """Tests for get_file_stats tool."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    @pytest.fixture
    def mcp_server(self, temp_db):
        """Create MCP server with temp storage."""
        server = MCPServer(storage=temp_db)
        yield server

    @pytest.fixture
    def sample_project(self, tmp_path, temp_db):
        """Create and scan a sample project."""
        project_dir = tmp_path / "sample_project"
        project_dir.mkdir()

        (project_dir / "main.py").write_text("def main(): pass")
        (project_dir / "utils.py").write_text("def util(): pass")
        (project_dir / "script.js").write_text("function script() {}")

        scan_and_save_project(project_dir, temp_db)
        return project_dir

    @pytest.mark.asyncio
    async def test_get_file_stats_success(self, mcp_server, sample_project):
        """Test get_file_stats returns valid statistics.

        Property 6: get_file_stats response should match stored statistics.
        """
        result = await mcp_server.call_tool(
            "get_file_stats",
            {"path": str(sample_project)}
        )

        assert "error" not in result
        assert "stats" in result
        assert "total_files" in result
        assert result["total_files"] == 3

        # Check stats structure
        for stat in result["stats"]:
            assert "type" in stat
            assert "count" in stat
            assert "percentage" in stat

    @pytest.mark.asyncio
    async def test_get_file_stats_not_found(self, mcp_server):
        """Test get_file_stats with non-scanned project."""
        result = await mcp_server.call_tool(
            "get_file_stats",
            {"path": "/nonexistent/path"}
        )

        assert "error" in result
        assert result["code"] == "NOT_FOUND"


class TestMCPServerGetReferenceRanking:
    """Tests for get_reference_ranking tool."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    @pytest.fixture
    def mcp_server(self, temp_db):
        """Create MCP server with temp storage."""
        server = MCPServer(storage=temp_db)
        yield server

    @pytest.fixture
    def sample_project(self, tmp_path, temp_db):
        """Create and scan a sample project with dependencies."""
        project_dir = tmp_path / "sample_project"
        project_dir.mkdir()

        (project_dir / "utils.py").write_text("def util(): pass")
        (project_dir / "main.py").write_text("from utils import util\ndef main(): pass")
        (project_dir / "app.py").write_text("from utils import util\ndef app(): pass")

        scan_and_save_project(project_dir, temp_db)
        return project_dir

    @pytest.mark.asyncio
    async def test_get_reference_ranking_success(self, mcp_server, sample_project):
        """Test get_reference_ranking returns valid ranking."""
        result = await mcp_server.call_tool(
            "get_reference_ranking",
            {"path": str(sample_project), "limit": 10}
        )

        assert "error" not in result
        assert "results" in result

    @pytest.mark.asyncio
    async def test_get_reference_ranking_respects_limit(self, mcp_server, sample_project):
        """Test get_reference_ranking respects limit parameter.

        Property 6: get_reference_ranking result count should respect limit parameter.
        """
        result = await mcp_server.call_tool(
            "get_reference_ranking",
            {"path": str(sample_project), "limit": 1}
        )

        assert "error" not in result
        assert len(result.get("results", [])) <= 1


class TestMCPServerGetFunctionRelations:
    """Tests for get_function_relations tool."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    @pytest.fixture
    def mcp_server(self, temp_db):
        """Create MCP server with temp storage."""
        server = MCPServer(storage=temp_db)
        yield server

    @pytest.fixture
    def sample_project(self, tmp_path):
        """Create a sample project."""
        project_dir = tmp_path / "sample_project"
        project_dir.mkdir()

        (project_dir / "funcs.py").write_text("""
def caller():
    callee()

def callee():
    pass
""")
        return project_dir

    @pytest.mark.asyncio
    async def test_get_function_relations_success(self, mcp_server, sample_project):
        """Test get_function_relations returns valid call graph.

        Property 6: get_function_relations response should contain call graph structure.
        """
        result = await mcp_server.call_tool(
            "get_function_relations",
            {"files": [str(sample_project / "funcs.py")]}
        )

        assert "error" not in result
        assert "functions" in result or "files" in result
        assert "call_graph" in result

    @pytest.mark.asyncio
    async def test_get_function_relations_too_many_files(self, mcp_server, sample_project):
        """Test get_function_relations rejects more than 10 files."""
        files = [str(sample_project / f"file{i}.py") for i in range(15)]

        result = await mcp_server.call_tool(
            "get_function_relations",
            {"files": files}
        )

        assert "error" in result
        assert result["code"] == "LIMIT_EXCEEDED"


class TestMCPServerGetRelatedCodeContext:
    """Tests for get_related_code_context tool."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    @pytest.fixture
    def mcp_server(self, temp_db):
        """Create MCP server with temp storage."""
        server = MCPServer(storage=temp_db)
        yield server

    @pytest.fixture
    def sample_project(self, tmp_path, temp_db):
        """Create and scan a sample project."""
        project_dir = tmp_path / "sample_project"
        project_dir.mkdir()

        (project_dir / "main.py").write_text("""
from utils import helper

def main():
    return helper()
""")
        (project_dir / "utils.py").write_text("""
def helper():
    return "help"
""")

        scan_and_save_project(project_dir, temp_db)
        return project_dir

    @pytest.mark.asyncio
    async def test_get_related_code_context_success(self, mcp_server, sample_project):
        """Test get_related_code_context returns valid context.

        Property 6: get_related_code_context response should contain target file
        and related files.
        """
        result = await mcp_server.call_tool(
            "get_related_code_context",
            {
                "project_path": str(sample_project),
                "file_path": "main.py",
                "hops": 1
            }
        )

        assert "error" not in result
        assert "target_file" in result
        assert "related_files" in result
        assert result["target_file"]["file_path"] == "main.py"

    @pytest.mark.asyncio
    async def test_get_related_code_context_not_found(self, mcp_server):
        """Test get_related_code_context with non-existent project."""
        result = await mcp_server.call_tool(
            "get_related_code_context",
            {
                "project_path": "/nonexistent/path",
                "file_path": "main.py"
            }
        )

        assert "error" in result
        assert result["code"] == "NOT_FOUND"


class TestMCPServerUnknownTool:
    """Tests for unknown tool handling."""

    @pytest.fixture
    def mcp_server(self, tmp_path):
        """Create MCP server with temp storage."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        server = MCPServer(storage=storage)
        yield server
        storage.close()

    @pytest.mark.asyncio
    async def test_unknown_tool_error(self, mcp_server):
        """Test that unknown tool returns error."""
        result = await mcp_server.call_tool(
            "unknown_tool",
            {}
        )

        assert "error" in result
        assert result["code"] == "UNKNOWN_TOOL"


class TestMCPServerResponseFormat:
    """Property-based tests for MCP response format."""

    @pytest.fixture
    def temp_db(self, tmp_path):
        """Create a temporary database."""
        db_path = str(tmp_path / "test.db")
        storage = SQLiteStorage(db_path)
        yield storage
        storage.close()

    @pytest.mark.asyncio
    async def test_all_tools_return_valid_json(self, temp_db, tmp_path):
        """Test that all tool responses are valid JSON-serializable dicts.

        Property 6: Response should be valid JSON.
        """
        import json

        # Create a sample project
        project_dir = tmp_path / "sample_project"
        project_dir.mkdir()
        (project_dir / "main.py").write_text("def main(): pass")

        scan_and_save_project(project_dir, temp_db)

        server = MCPServer(storage=temp_db)

        tools_and_args = [
            ("scan_project", {"path": str(project_dir)}),
            ("get_file_stats", {"path": str(project_dir)}),
            ("get_reference_ranking", {"path": str(project_dir)}),
            ("get_depth_analysis", {"path": str(project_dir)}),
            ("get_function_relations", {"files": [str(project_dir / "main.py")]}),
            (
                "get_related_code_context",
                {
                    "project_path": str(project_dir),
                    "file_path": "main.py"
                }
            ),
        ]

        for tool_name, args in tools_and_args:
            result = await server.call_tool(tool_name, args)

            # Should be serializable to JSON
            try:
                json_str = json.dumps(result)
                assert isinstance(json_str, str)
            except (TypeError, ValueError) as e:
                pytest.fail(f"Tool {tool_name} returned non-JSON-serializable result: {e}")
